import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { pool } from './postgres';
import Papa from 'papaparse';
import axios from 'axios';
import { getFileStorageAdapter } from './file-storage-adapter';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

interface DetailedCSVRow {
  Name: string;
  nazwGen: string;
  ROZKROJE: string;
  'MAG BEZKOL (SYNC)': string;
  Ilość: string;
  'Długość (from MAG BEZKOL (SYNC))': string;
  'Szerokość (from MAG BEZKOL (SYNC))': string;
}

interface AggregatedCSVRow {
  nazwa: string;
  Attachments: string;
  rozmiarPlyty: string;
  nazwGen: string;
  autonumer: string;
  'ROZKROJE FORMATKI': string;
}

function parseDimensions(sizeStr: string): { length: number | null; width: number | null } {
  const match = sizeStr.match(/(\d+)\s*x\s*(\d+)/i);
  if (match) {
    return {
      width: parseInt(match[1]),
      length: parseInt(match[2])
    };
  }
  return { length: null, width: null };
}

function parseBoardDimensions(boardStr: string): { length: number; width: number } {
  const match = boardStr.match(/(\d+)\s*x\s*(\d+)/i);
  if (match) {
    return {
      length: parseInt(match[1]),
      width: parseInt(match[2])
    };
  }
  return { length: 2800, width: 2070 };
}

function extractImageUrl(attachmentsStr: string): string | null {
  if (!attachmentsStr) return null;
  const match = attachmentsStr.match(/\((https?:\/\/[^)]+)\)/);
  return match ? match[1] : null;
}

function normalizeFormatkaSize(formatkaSize: string): string {
  // Normalize formatka size for consistent matching:
  // - Remove leading/trailing whitespace
  // - Replace non-breaking spaces with regular spaces
  // - Collapse multiple spaces to single space
  // - Lowercase for case-insensitive matching
  return formatkaSize
    .trim()
    .replace(/\u00A0/g, ' ') // Non-breaking space
    .replace(/\s+/g, ' ')    // Collapse whitespace
    .toLowerCase();
}

async function downloadAndUploadImage(imageUrl: string, rozkrojCode: string): Promise<string | null> {
  try {
    console.log(`  📥 Downloading image from: ${imageUrl}`);
    
    // Download image from Airtable
    const response = await axios.get(imageUrl, {
      responseType: 'arraybuffer',
      timeout: 30000
    });
    
    const buffer = Buffer.from(response.data);
    const contentType = response.headers['content-type'] || 'image/png';
    
    // Generate filename
    const ext = contentType.includes('jpeg') ? 'jpg' : 'png';
    const filename = `${rozkrojCode}.${ext}`;
    
    // Upload to our file storage
    const adapter = await getFileStorageAdapter();
    const uploadedUrl = await adapter.upload({
      filename,
      buffer,
      mimetype: contentType,
      subfolder: 'production/cutting-patterns',
      metadata: {
        rozkrojCode,
        source: 'airtable',
        timestamp: new Date().toISOString()
      }
    });
    
    console.log(`  ✅ Uploaded to: ${uploadedUrl}`);
    return uploadedUrl;
  } catch (error: any) {
    console.error(`  ❌ Failed to download/upload image: ${error.message}`);
    return null;
  }
}

async function importUnifiedRozkroje() {
  console.log('🚀 Starting unified rozkroje import...');
  
  // Read detailed CSV with quantities using Papa Parse (robust parsing)
  const detailedCsvPath = path.join(__dirname, '..', 'attached_assets', 'ROZKROJE FORMATKI-Grid view (1)_1762944449596.csv');
  const detailedCsvContent = fs.readFileSync(detailedCsvPath, 'utf-8');
  
  const detailedParseResult = Papa.parse<any>(detailedCsvContent, {
    header: true,
    skipEmptyLines: true,
    transformHeader: (header: string) => header.trim().replace(/^\uFEFF/, ''),
  });
  
  const detailedRows: DetailedCSVRow[] = detailedParseResult.data
    .filter(row => row['Name'] && row['nazwGen (from ROZKROJE)'])
    .map(row => ({
      Name: row['Name'] || '',
      nazwGen: row['nazwGen (from ROZKROJE)'] || '',
      ROZKROJE: row['ROZKROJE'] || '',
      'MAG BEZKOL (SYNC)': row['MAG BEZKOL (SYNC)'] || '',
      Ilość: row['Ilość'] || '0',
      'Długość (from MAG BEZKOL (SYNC))': row['Długość (from MAG BEZKOL (SYNC))'] || '',
      'Szerokość (from MAG BEZKOL (SYNC))': row['Szerokość (from MAG BEZKOL (SYNC))'] || ''
    }));
  
  console.log(`📊 Parsed ${detailedRows.length} detailed rows (with quantities)`);
  
  // Create quantity map using dimensions (width, length) as key for robust matching
  // IMPORTANT: In detailed CSV, format "462x279" means Długość=462, Szerokość=279
  // So we map by itemName directly (simpler and more reliable)
  const quantityMap = new Map<string, number>();
  for (const row of detailedRows) {
    const rozkrojCode = row.nazwGen;
    const itemName = row['MAG BEZKOL (SYNC)'] || row.Name; // e.g., "462x279"
    const quantity = parseInt(row['Ilość']) || 0;
    
    if (itemName && quantity > 0) {
      const normalizedName = normalizeFormatkaSize(itemName);
      const key = `${rozkrojCode}:${normalizedName}`;
      quantityMap.set(key, quantity);
      console.log(`📋 Mapped ${key} -> ${quantity}`);
    }
  }
  
  // Read aggregated CSV with images and formatki list
  const aggregatedCsvPath = path.join(__dirname, '..', 'attached_assets', 'ROZKROJE-Grid view (3)_1762945071998.csv');
  const aggregatedCsvContent = fs.readFileSync(aggregatedCsvPath, 'utf-8');
  
  const parseResult = Papa.parse<AggregatedCSVRow>(aggregatedCsvContent, {
    header: true,
    skipEmptyLines: true,
    transformHeader: (header: string) => header.trim().replace(/^\uFEFF/, ''),
  });
  
  const aggregatedRows = parseResult.data.filter(row => row.nazwGen && row['ROZKROJE FORMATKI']);
  console.log(`📊 Parsed ${aggregatedRows.length} aggregated rows (with images)`);
  
  let patternsUpdated = 0;
  let patternsCreated = 0;
  let itemsCreated = 0;
  let imagesDownloaded = 0;
  
  for (const row of aggregatedRows) {
    const rozkrojCode = row.nazwGen;
    const formatkiStr = row['ROZKROJE FORMATKI'];
    const airtableImageUrl = extractImageUrl(row.Attachments);
    const opis = row.rozmiarPlyty?.trim() || null; // Opis z kolumny rozmiarPlyty
    
    if (opis) {
      console.log(`  📝 Rozkrój ${rozkrojCode} ma opis: "${opis}"`);
    }
    
    // Parse board dimensions
    let boardLength = 2800;
    let boardWidth = 2070;
    
    if (row.rozmiarPlyty) {
      const dimensions = parseBoardDimensions(row.rozmiarPlyty);
      boardLength = dimensions.length;
      boardWidth = dimensions.width;
    }
    
    // Check if pattern already exists to get image URL
    const existingResult = await pool.query(
      `SELECT id, image_url FROM production.cut_patterns WHERE code = $1`,
      [rozkrojCode]
    );
    
    // Download and upload image if available and pattern doesn't have one yet
    let localImageUrl: string | null = null;
    const existingImageUrl = existingResult.rows.length > 0 
      ? existingResult.rows[0].image_url 
      : null;
    
    if (airtableImageUrl && !existingImageUrl) {
      localImageUrl = await downloadAndUploadImage(airtableImageUrl, rozkrojCode);
      if (localImageUrl) {
        imagesDownloaded++;
      }
    } else if (existingImageUrl) {
      localImageUrl = existingImageUrl; // Keep existing image
      console.log(`  ⏭️  Skipping image download (already exists): ${rozkrojCode}`);
    }
    
    // Use UPSERT to handle both create and update cases safely
    const upsertResult = await pool.query(
      `INSERT INTO production.cut_patterns (
        code, board_length, board_width, board_thickness, kerf, 
        image_url, opis, status, notes, is_active
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
      ON CONFLICT (code) DO UPDATE SET
        board_length = EXCLUDED.board_length,
        board_width = EXCLUDED.board_width,
        image_url = COALESCE(EXCLUDED.image_url, production.cut_patterns.image_url),
        opis = EXCLUDED.opis,
        updated_at = NOW()
      RETURNING id, (xmax = 0) AS was_inserted`,
      [
        rozkrojCode,
        boardLength,
        boardWidth,
        18,
        4,
        localImageUrl,
        opis,
        'planowany',
        `Agregowane formatki: ${formatkiStr}`,
        true
      ]
    );
    
    const patternId = upsertResult.rows[0].id;
    const wasInserted = upsertResult.rows[0].was_inserted;
    
    if (wasInserted) {
      patternsCreated++;
      console.log(`✓ Created pattern ${rozkrojCode} (ID: ${patternId})`);
    } else {
      patternsUpdated++;
      console.log(`✓ Updated pattern ${rozkrojCode} (ID: ${patternId})`);
    }
    
    // Parse formatki list and create items in a transaction
    const formatkiList = formatkiStr.split(',').map(f => f.trim()).filter(f => f);
    
    // Begin transaction for this pattern's items
    const client = await pool.connect();
    try {
      await client.query('BEGIN');
      
      for (const formatkaSize of formatkiList) {
        const { width, length } = parseDimensions(formatkaSize);
        
        if (!width || !length) {
          console.log(`  ⚠️  Skipping formatka "${formatkaSize}" - invalid format`);
          continue;
        }
        
        // Get quantity from detailed CSV using normalized formatkaSize
        const normalizedSize = normalizeFormatkaSize(formatkaSize);
        const quantityKey = `${rozkrojCode}:${normalizedSize}`;
        const quantity = quantityMap.get(quantityKey) || 1;
        
        if (!quantityMap.has(quantityKey)) {
          console.log(`  ⚠️  No quantity found for ${formatkaSize} (normalized key: ${quantityKey}), using default: 1`);
        }
        
        // Check if item already exists
        const existingItem = await client.query(
          `SELECT id FROM production.cut_pattern_items 
           WHERE pattern_id = $1 AND length = $2 AND width = $3`,
          [patternId, length, width]
        );
        
        if (existingItem.rows.length === 0) {
          // Create new item
          await client.query(
            `INSERT INTO production.cut_pattern_items (
              pattern_id, item_name, length, width, thickness, 
              quantity_requested, status, is_active
            ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)`,
            [
              patternId,
              formatkaSize,
              length,
              width,
              18,
              quantity,
              'zaplanowana',
              true
            ]
          );
          itemsCreated++;
          console.log(`    + Added formatka ${formatkaSize} (${width}x${length}) qty: ${quantity}`);
        } else {
          // Update quantity if different
          await client.query(
            `UPDATE production.cut_pattern_items 
             SET quantity_requested = $1, updated_at = NOW()
             WHERE id = $2`,
            [quantity, existingItem.rows[0].id]
          );
          console.log(`    ✓ Updated formatka ${formatkaSize} qty: ${quantity}`);
        }
      }
      
      await client.query('COMMIT');
    } catch (error) {
      await client.query('ROLLBACK');
      console.error(`  ❌ Error processing items for ${rozkrojCode}:`, error);
      throw error;
    } finally {
      client.release();
    }
  }
  
  console.log(`\n✨ Import completed successfully!`);
  console.log(`📦 Created ${patternsCreated} new cut patterns`);
  console.log(`🔄 Updated ${patternsUpdated} existing cut patterns`);
  console.log(`📋 Created ${itemsCreated} cut pattern items`);
  console.log(`🖼️  Downloaded ${imagesDownloaded} images to alpsys system`);
}

// Run import
importUnifiedRozkroje()
  .then(() => {
    console.log('✅ Import finished');
    process.exit(0);
  })
  .catch((error) => {
    console.error('❌ Import failed:', error);
    process.exit(1);
  });
