import { Pool } from "pg";
import type { 
  ProductionSetting, 
  InsertProductionSetting 
} from "@shared/schema";

export async function getProductionSettings(pool: Pool): Promise<ProductionSetting[]> {
  const result = await pool.query(`
    SELECT * FROM production.production_settings
    ORDER BY category, key ASC
  `);
  
  return result.rows.map(row => ({
    id: row.id,
    key: row.key,
    value: row.value,
    valueType: row.value_type,
    description: row.description,
    category: row.category,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getProductionSettingById(pool: Pool, id: number): Promise<ProductionSetting | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_settings
    WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  return {
    id: row.id,
    key: row.key,
    value: row.value,
    valueType: row.value_type,
    description: row.description,
    category: row.category,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function createProductionSetting(pool: Pool, setting: InsertProductionSetting): Promise<ProductionSetting> {
  const result = await pool.query(`
    INSERT INTO production.production_settings 
      (key, value, value_type, description, category, is_active)
    VALUES ($1, $2, $3, $4, $5, $6)
    RETURNING *
  `, [
    setting.key,
    setting.value,
    setting.valueType ?? 'string',
    setting.description ?? null,
    setting.category ?? 'general',
    setting.isActive ?? true,
  ]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    key: row.key,
    value: row.value,
    valueType: row.value_type,
    description: row.description,
    category: row.category,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function updateProductionSetting(
  pool: Pool, 
  id: number, 
  updates: Partial<InsertProductionSetting>
): Promise<ProductionSetting> {
  const setParts: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;
  
  if (updates.key !== undefined) {
    setParts.push(`key = $${paramIndex++}`);
    values.push(updates.key);
  }
  if (updates.value !== undefined) {
    setParts.push(`value = $${paramIndex++}`);
    values.push(updates.value);
  }
  if (updates.valueType !== undefined) {
    setParts.push(`value_type = $${paramIndex++}`);
    values.push(updates.valueType);
  }
  if (updates.description !== undefined) {
    setParts.push(`description = $${paramIndex++}`);
    values.push(updates.description);
  }
  if (updates.category !== undefined) {
    setParts.push(`category = $${paramIndex++}`);
    values.push(updates.category);
  }
  if (updates.isActive !== undefined) {
    setParts.push(`is_active = $${paramIndex++}`);
    values.push(updates.isActive);
  }
  
  setParts.push(`updated_at = NOW()`);
  values.push(id);
  
  const result = await pool.query(`
    UPDATE production.production_settings
    SET ${setParts.join(', ')}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);
  
  if (result.rows.length === 0) {
    throw new Error(`Production setting with id ${id} not found`);
  }
  
  const row = result.rows[0];
  return {
    id: row.id,
    key: row.key,
    value: row.value,
    valueType: row.value_type,
    description: row.description,
    category: row.category,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function deleteProductionSetting(pool: Pool, id: number): Promise<void> {
  await pool.query(`
    DELETE FROM production.production_settings
    WHERE id = $1
  `, [id]);
}
