import type { Express } from "express";
import { z } from "zod";
import { pool } from "../../postgres";
import * as ordersService from "../../services/production/orders";
import { requirePermission } from "../../auth";

const createOrderSchema = z.object({
  orderNumber: z.string().optional(),
  productId: z.number(),
  bomId: z.number().optional().nullable(),
  routingId: z.number().optional().nullable(),
  status: z.enum(['draft', 'confirmed', 'planned', 'in_progress', 'paused', 'done', 'cancelled']).optional(),
  priority: z.enum(['low', 'normal', 'high', 'urgent']).optional(),
  quantityPlanned: z.number().positive(),
  unitOfMeasure: z.string().optional(),
  plannedStartDate: z.string().optional().nullable(),
  plannedEndDate: z.string().optional().nullable(),
  responsibleUserId: z.number().optional().nullable(),
  sourceOrderNumber: z.string().optional().nullable(),
  locationId: z.number().optional().nullable(),
  notes: z.string().optional().nullable(),
});

const filtersSchema = z.object({
  status: z.string().optional(),
  workflowStage: z.string().optional(),
  priority: z.string().optional(),
  productId: z.coerce.number().optional(),
  responsibleUserId: z.coerce.number().optional(),
  startDate: z.string().optional(),
  endDate: z.string().optional(),
  search: z.string().optional(),
  limit: z.coerce.number().optional(),
  offset: z.coerce.number().optional(),
});

const updateWorkflowStageSchema = z.object({
  workflowStage: z.string(),
});

export function registerOrderRoutes(app: Express) {
  // GET /api/production/orders - Get all orders with filtering
  app.get("/api/production/orders", requirePermission('view_production'), async (req, res) => {
    try {
      const filters = filtersSchema.parse(req.query);
      
      // Convert string dates to Date objects
      const processedFilters = {
        ...filters,
        startDate: filters.startDate ? new Date(filters.startDate) : undefined,
        endDate: filters.endDate ? new Date(filters.endDate) : undefined,
      };

      const orders = await ordersService.getOrders(pool, processedFilters);
      res.json(orders);
    } catch (error) {
      console.error("Error fetching production orders:", error);
      res.status(500).json({ message: "Failed to fetch production orders" });
    }
  });

  // GET /api/production/orders/:id - Get order by ID
  app.get("/api/production/orders/:id", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.getOrderById(pool, id);
      if (!order) {
        return res.status(404).json({ message: "Production order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error fetching production order:", error);
      res.status(500).json({ message: "Failed to fetch production order" });
    }
  });

  // POST /api/production/orders - Create new order
  app.post("/api/production/orders", requirePermission('manage_production'), async (req, res) => {
    try {
      const data = createOrderSchema.parse(req.body);
      
      // Auto-generate order number if not provided
      const orderNumber = data.orderNumber || await ordersService.generateOrderNumber(pool);
      
      const order = await ordersService.createOrder(pool, {
        ...data,
        orderNumber,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : null,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : null,
      });
      
      res.status(201).json(order);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Order with this number already exists" });
      }
      console.error("Error creating production order:", error);
      res.status(500).json({ message: "Failed to create production order" });
    }
  });

  // PATCH /api/production/orders/:id - Update order
  app.patch("/api/production/orders/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const data = createOrderSchema.partial().parse(req.body);
      const order = await ordersService.updateOrder(pool, id, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : undefined,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : undefined,
      });
      
      if (!order) {
        return res.status(404).json({ message: "Production order not found" });
      }

      res.json(order);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Order with this number already exists" });
      }
      console.error("Error updating production order:", error);
      res.status(500).json({ message: "Failed to update production order" });
    }
  });

  // DELETE /api/production/orders/:id - Delete order
  app.delete("/api/production/orders/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const success = await ordersService.deleteOrder(pool, id);
      if (!success) {
        return res.status(404).json({ message: "Production order not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting production order:", error);
      res.status(500).json({ message: "Failed to delete production order" });
    }
  });

  // POST /api/production/orders/:id/start - Start order
  app.post("/api/production/orders/:id/start", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.startOrder(pool, id);
      if (!order) {
        return res.status(400).json({ message: "Cannot start order - invalid status or order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error starting production order:", error);
      res.status(500).json({ message: "Failed to start production order" });
    }
  });

  // POST /api/production/orders/:id/complete - Complete order
  app.post("/api/production/orders/:id/complete", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.completeOrder(pool, id);
      if (!order) {
        return res.status(400).json({ message: "Cannot complete order - must be in_progress or order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error completing production order:", error);
      res.status(500).json({ message: "Failed to complete production order" });
    }
  });

  // POST /api/production/orders/:id/cancel - Cancel order
  app.post("/api/production/orders/:id/cancel", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.cancelOrder(pool, id);
      if (!order) {
        return res.status(400).json({ message: "Cannot cancel order - already done/cancelled or order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error cancelling production order:", error);
      res.status(500).json({ message: "Failed to cancel production order" });
    }
  });

  // PATCH /api/production/orders/:id/workflow-stage - Update workflow stage
  app.patch("/api/production/orders/:id/workflow-stage", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const { workflowStage } = updateWorkflowStageSchema.parse(req.body);
      
      // Get user ID from session (if authenticated)
      const userId = (req as any).user?.id;

      const order = await ordersService.updateWorkflowStage(pool, id, workflowStage, userId);
      if (!order) {
        return res.status(404).json({ message: "Production order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error updating workflow stage:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      res.status(500).json({ message: "Failed to update workflow stage" });
    }
  });
}
