import type { Express } from "express";
import { z } from "zod";
import { pool } from "../../postgres";
import { requirePermission } from "../../auth";

const markDamageSchema = z.object({
  isDamaged: z.boolean(),
  damageNotes: z.string().optional(),
});

export function registerComponentRoutes(app: Express) {
  // PATCH /api/production/components/:id/damage - Mark/unmark component as damaged
  app.patch("/api/production/components/:id/damage", requirePermission('manage_production'), async (req, res) => {
    try {
      const componentId = parseInt(req.params.id);
      if (isNaN(componentId)) {
        return res.status(400).json({ message: "Invalid component ID" });
      }

      const { isDamaged, damageNotes } = markDamageSchema.parse(req.body);

      // Validation: damage notes required when marking as damaged
      if (isDamaged && !damageNotes?.trim()) {
        return res.status(400).json({ 
          message: "Damage notes are required when marking a component as damaged" 
        });
      }

      // Get user ID from session (if authenticated)
      const userId = (req.user as any)?.id || null;

      // Update component damage status
      if (isDamaged) {
        // Mark as damaged
        await pool.query(`
          UPDATE bom.product_components
          SET 
            is_damaged = true,
            damage_notes = $1,
            damaged_at = NOW(),
            damaged_by = $2
          WHERE id = $3
        `, [damageNotes, userId, componentId]);
      } else {
        // Clear damage status
        await pool.query(`
          UPDATE bom.product_components
          SET 
            is_damaged = false,
            damage_notes = NULL,
            damaged_at = NULL,
            damaged_by = NULL
          WHERE id = $1
        `, [componentId]);
      }

      // Fetch updated component
      const result = await pool.query(`
        SELECT 
          id,
          generated_name as "generatedName",
          is_damaged as "isDamaged",
          damage_notes as "damageNotes",
          damaged_at as "damagedAt",
          damaged_by as "damagedBy"
        FROM bom.product_components
        WHERE id = $1
      `, [componentId]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Component not found" });
      }

      const component = result.rows[0];

      console.log(`${isDamaged ? '❌' : '✅'} Component ${componentId} (${component.generatedName}) damage status updated by user ${userId}`);

      res.json(component);

    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ 
          message: "Validation error", 
          errors: error.errors 
        });
      }
      
      console.error("Error updating component damage status:", error);
      res.status(500).json({ 
        message: "Failed to update component damage status",
        error: error instanceof Error ? error.message : 'Unknown error'
      });
    }
  });
}
