import { useQuery, useMutation } from "@tanstack/react-query";
import { useParams, useLocation } from "wouter";
import { ArrowLeft, Edit, Trash2, Loader2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import type { ProductionCarrier, ProductionCarrierGroup } from "@shared/schema";

export default function ProductionCarrierDetail() {
  const { id } = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();

  const { data: carrier, isLoading } = useQuery<ProductionCarrier>({
    queryKey: ["/api/production/carriers", id],
    queryFn: async () => {
      const res = await fetch(`/api/production/carriers/${id}`);
      if (!res.ok) throw new Error("Failed to fetch carrier");
      return res.json();
    },
  });

  const { data: groups = [] } = useQuery<ProductionCarrierGroup[]>({
    queryKey: ["/api/production/carriers/groups"],
  });

  const deleteMutation = useMutation({
    mutationFn: async () => {
      const res = await fetch(`/api/production/carriers/${id}`, { method: "DELETE" });
      if (!res.ok) throw new Error("Failed to delete");
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/carriers"] });
      toast({ title: "Nośnik został usunięty" });
      navigate("/production/carriers");
    },
  });

  const getGroupName = (groupId: number) => {
    return groups.find(g => g.id === groupId)?.name || "Nieznana grupa";
  };

  const getStatusLabel = (status: string) => {
    const labels: Record<string, string> = {
      available: "Dostępny",
      in_use: "W użyciu",
      maintenance: "Konserwacja",
      damaged: "Uszkodzony",
      archived: "Zarchiwizowany",
    };
    return labels[status] || status;
  };

  const getStatusVariant = (status: string) => {
    const variants: Record<string, "default" | "secondary" | "outline"> = {
      available: "default",
      in_use: "secondary",
      maintenance: "outline",
      damaged: "outline",
      archived: "outline",
    };
    return variants[status] || "outline";
  };

  if (isLoading) {
    return (
      <div className="container mx-auto p-6">
        <div className="flex justify-center p-8">
          <Loader2 className="h-8 w-8 animate-spin" data-testid="loader-carrier-detail" />
        </div>
      </div>
    );
  }

  if (!carrier) {
    return (
      <div className="container mx-auto p-6">
        <Card>
          <CardContent className="p-8 text-center text-muted-foreground">
            Nie znaleziono nośnika
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-4">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => navigate("/production/carriers")}
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div>
            <h1 className="text-3xl font-bold" data-testid="text-carrier-name">
              {carrier.name}
            </h1>
            <p className="text-muted-foreground font-mono" data-testid="text-carrier-code">
              {carrier.code}
            </p>
          </div>
        </div>
        <div className="flex gap-2">
          <Button
            variant="outline"
            onClick={() => navigate(`/production/carriers/${id}/edit`)}
            data-testid="button-edit"
          >
            <Edit className="h-4 w-4 mr-2" />
            Edytuj
          </Button>
          <Button
            variant="outline"
            onClick={() => {
              if (confirm("Czy na pewno chcesz usunąć ten nośnik?")) {
                deleteMutation.mutate();
              }
            }}
            disabled={deleteMutation.isPending}
            data-testid="button-delete"
          >
            <Trash2 className="h-4 w-4 mr-2" />
            Usuń
          </Button>
        </div>
      </div>

      <div className="grid gap-6 md:grid-cols-2">
        <Card>
          <CardHeader>
            <CardTitle>Podstawowe Informacje</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <p className="text-sm text-muted-foreground">Kod</p>
              <p className="font-mono" data-testid="detail-code">{carrier.code}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Nazwa</p>
              <p data-testid="detail-name">{carrier.name}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Grupa</p>
              <p data-testid="detail-group">{getGroupName(carrier.carrierGroupId)}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Kod kreskowy</p>
              <p className="font-mono" data-testid="detail-barcode">
                {carrier.barcode || "-"}
              </p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Status</p>
              <div data-testid="detail-status">
                <Badge variant={getStatusVariant(carrier.status || "available")}>
                  {getStatusLabel(carrier.status || "available")}
                </Badge>
              </div>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Aktywny</p>
              <Badge variant={carrier.isActive ? "default" : "secondary"} data-testid="detail-active">
                {carrier.isActive ? "Tak" : "Nie"}
              </Badge>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Pojemność i Specyfikacja</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <p className="text-sm text-muted-foreground">Pojemność</p>
              <p data-testid="detail-capacity">
                {carrier.capacity ? `${carrier.capacity} ${carrier.capacityUnit || ''}` : '-'}
              </p>
            </div>
          </CardContent>
        </Card>

        {carrier.notes && (
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle>Notatki</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground whitespace-pre-wrap" data-testid="detail-notes">
                {carrier.notes}
              </p>
            </CardContent>
          </Card>
        )}
      </div>
    </div>
  );
}
