import { useState, useEffect, useCallback } from 'react';

export interface PlanTab {
  id: string;
  label: string;
  token: string;
  isActive: boolean;
}

interface UsePlanTabsReturn {
  tabs: PlanTab[];
  activeTabs: PlanTab[];
  activeTokens: string[];
  addTab: (value: string) => void;
  toggleTab: (tabId: string) => void;
  removeTab: (tabId: string) => void;
  setTabsFromTokens: (tokens: string[]) => void;
}

const STORAGE_KEY_PREFIX = 'production-plan-tabs';

function getStorageKey(planId: string): string {
  return `${STORAGE_KEY_PREFIX}:${planId}`;
}

function sanitizeToken(value: string): string {
  return value.trim().toUpperCase();
}

export function usePlanTabs(planId: string): UsePlanTabsReturn {
  const [tabs, setTabs] = useState<PlanTab[]>([]);

  useEffect(() => {
    if (typeof window === 'undefined') return;

    const storageKey = getStorageKey(planId);
    const stored = localStorage.getItem(storageKey);
    
    if (stored) {
      try {
        const parsed = JSON.parse(stored) as PlanTab[];
        setTabs(parsed);
      } catch (error) {
        console.error('Failed to parse stored tabs:', error);
        setTabs([]);
      }
    }
  }, [planId]);

  useEffect(() => {
    if (typeof window === 'undefined') return;
    
    const storageKey = getStorageKey(planId);
    localStorage.setItem(storageKey, JSON.stringify(tabs));
  }, [planId, tabs]);

  const addTab = useCallback((value: string) => {
    const token = sanitizeToken(value);
    if (!token) return;

    const existingTab = tabs.find(t => t.token === token);
    if (existingTab) {
      setTabs(prev => prev.map(t => 
        t.id === existingTab.id ? { ...t, isActive: true } : t
      ));
      return;
    }

    const newTab: PlanTab = {
      id: `tab-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      label: token,
      token: token,
      isActive: true
    };

    setTabs(prev => [...prev, newTab]);
  }, [tabs]);

  const toggleTab = useCallback((tabId: string) => {
    setTabs(prev => prev.map(tab => 
      tab.id === tabId ? { ...tab, isActive: !tab.isActive } : tab
    ));
  }, []);

  const removeTab = useCallback((tabId: string) => {
    setTabs(prev => prev.filter(tab => tab.id !== tabId));
  }, []);

  const setTabsFromTokens = useCallback((tokens: string[]) => {
    const sanitizedTokens = tokens.map(t => sanitizeToken(t)).filter(Boolean);
    
    setTabs(prev => {
      const existingTokens = prev.map(t => t.token);
      const missingTokens = sanitizedTokens.filter(token => !existingTokens.includes(token));
      
      const newTabs = missingTokens.map(token => ({
        id: `tab-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        label: token,
        token: token,
        isActive: true
      }));
      
      const updatedTabs = prev.map(tab => ({
        ...tab,
        isActive: sanitizedTokens.includes(tab.token)
      }));
      
      const hasChanges = 
        newTabs.length > 0 || 
        updatedTabs.some((tab, idx) => tab.isActive !== prev[idx].isActive);
      
      if (!hasChanges) return prev;
      
      return [...updatedTabs, ...newTabs];
    });
  }, []);

  const activeTabs = tabs.filter(t => t.isActive);
  const activeTokens = activeTabs.map(t => t.token);

  return {
    tabs,
    activeTabs,
    activeTokens,
    addTab,
    toggleTab,
    removeTab,
    setTabsFromTokens
  };
}
