// CACHE BUSTER: Fixed filteredItems ordering issue - v2
import { ShoppingCart, Package, FileText, Hash, Image as ImageIcon, ArrowLeft, Trash2, Search, Loader2 } from "lucide-react";
import { Card, CardContent } from "@/components/ui/card";
import { Separator } from "@/components/ui/separator";
import { Skeleton } from "@/components/ui/skeleton";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Checkbox } from "@/components/ui/checkbox";
import { useQuery } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { usePlanTabsBar } from "./PlanTabsBar";
import { ManageFiltersDialog } from "./ManageFiltersDialog";
import { Input } from "@/components/ui/input";
import { PlanHeader } from "./PlanHeader";
import { useMemo, useState } from "react";
import { useDroppable } from "@dnd-kit/core";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { useLocation } from "wouter";
import { useToast } from "@/hooks/use-toast";

interface PlanLineWithDetails {
  id: number;
  plan_id: number;
  product_id: number;
  quantity: number;
  source_type: string;
  source_reference: string | null;
  status: string;
  notes: string | null;
  metadata: any;
  bom_count: number;
  order_item_id: number | null;
  
  // Product data
  product_sku: string;
  product_title: string;
  product_color: string | null;
  product_color_options: string[] | null;
  product_length: number | null;
  product_width: number | null;
  product_height: number | null;
  product_type: string | null;
  product_group: string | null;
  doors: string | null;
  legs: string | null;
  material: string | null;
  base_price: number | null;
  currency: string | null;
  product_image: string | null;
  
  // Order data (if source_type = 'order_demand')
  order_number: string | null;
  order_date: string | null;
  marketplace: string | null;
  buyer_first_name: string | null;
  buyer_last_name: string | null;
  buyer_email: string | null;
  order_payment_status: string | null;
  order_total_amount: number | null;
}

interface ProductionPlan {
  id: number;
  planNumber: string;
  name: string;
  description?: string;
  plannedStartDate?: string;
  plannedEndDate?: string;
  status: string;
  priority: string;
  notes?: string;
  createdAt: string;
}

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
  dictionaryType: string;
  color: string | null;
  isActive: boolean;
}

interface PlanItemsPanelProps {
  plan: ProductionPlan;
  onBack: () => void;
  planId: string;
  planNumber?: string;
  statistics: {
    ordersCount: number;
    productsCount: number;
    planLinesCount: number;
    totalQuantity: number;
  };
  searchFilter: string;
  onSearchChange: (value: string) => void;
  isLoading?: boolean;
  onRemoveItem?: (lineId: number) => void;
  onRemoveItems?: (lineIds: number[]) => void;
  isRemoving?: boolean;
}

// Helper functions
const getTypeBackgroundColor = (type: string | null): string => {
  if (!type) return "#666666";
  const hash = type.split("").reduce((acc, char) => char.charCodeAt(0) + ((acc << 5) - acc), 0);
  const hue = Math.abs(hash) % 360;
  return `hsl(${hue}, 60%, 45%)`;
};

const mapSourceLabel = (item: PlanLineWithDetails): { label: string; variant: "default" | "secondary" | "outline" } => {
  const sourceType = item.source_type;
  const marketplace = (item.marketplace || item.metadata?.marketplace || '').toLowerCase();
  
  switch (sourceType) {
    case 'order_demand':
    case 'marketplace_order':
      if (marketplace.includes('allegro')) {
        return { label: 'Allegro', variant: 'default' };
      } else if (marketplace.includes('shoper')) {
        return { label: 'Shoper', variant: 'default' };
      }
      return { label: 'Zamówienie', variant: 'outline' };
    
    case 'catalog_internal':
      return { label: 'Katalog', variant: 'secondary' };
    
    case 'cutting_pattern':
      return { label: 'Wzór cięcia', variant: 'outline' };
    
    default:
      return { label: 'Inne', variant: 'secondary' };
  }
};

const getColorHex = (colorCode: string, colors: DictionaryItem[]): string | null => {
  if (!colorCode) return null;
  const colorItem = colors.find(c => c.code.toLowerCase() === colorCode.toLowerCase());
  return colorItem?.color || null;
};

const getTextColor = (bgColor: string): string => {
  // For HSL colors, extract lightness value
  const hslMatch = bgColor.match(/hsl\((\d+),\s*(\d+)%,\s*(\d+)%\)/);
  if (hslMatch) {
    const lightness = parseInt(hslMatch[3]);
    return lightness > 50 ? "#000000" : "#FFFFFF";
  }
  
  // Fallback for hex colors
  const rgb = bgColor.match(/\d+/g);
  if (!rgb) return "#FFFFFF";
  const brightness = (parseInt(rgb[0]) * 299 + parseInt(rgb[1]) * 587 + parseInt(rgb[2]) * 114) / 1000;
  return brightness > 128 ? "#000000" : "#FFFFFF";
};

const formatCm = (value: number | null): string => {
  if (value === null || value === 0) return "-";
  return `${value}cm`;
};

// Component to render the filter tabs bar with search and settings
function FilterTabsBarContent({
  planId,
  searchFilter,
  onSearchChange
}: {
  planId: string;
  searchFilter: string;
  onSearchChange: (value: string) => void;
}) {
  const { tabs, tabsData } = usePlanTabsBar({
    planId,
    searchFilter,
    onSearchChange
  });

  return (
    <div className="flex-1">
      <div className="flex items-center justify-between gap-4 w-full">
        {/* Left: Filter tabs */}
        <div className="flex-1">
          {tabs}
        </div>
        
        {/* Right: Search box and settings gear icon */}
        <div className="flex items-center gap-2 flex-shrink-0">
          <div className="relative">
            <Search className="absolute left-2 top-1/2 transform -translate-y-1/2 h-3.5 w-3.5 text-muted-foreground" />
            <Input
              type="text"
              placeholder="Wyszukaj..."
              value={searchFilter || ''}
              onChange={(e) => onSearchChange(e.target.value)}
              className="pl-8 h-7 w-48 text-xs"
              data-testid="input-search-filter"
            />
          </div>
          
          <ManageFiltersDialog
            tabs={tabsData.persistentTabs}
            onAddTab={tabsData.addTab}
            onRemoveTab={tabsData.removeTab}
            onToggleTab={tabsData.toggleTab}
            onReorderTabs={tabsData.reorderTabs}
          />
        </div>
      </div>
    </div>
  );
}

export function PlanItemsPanel({
  plan,
  onBack,
  planId,
  planNumber,
  statistics,
  searchFilter,
  onSearchChange,
  isLoading: statsLoading = false,
  onRemoveItem,
  onRemoveItems,
  isRemoving = false,
}: PlanItemsPanelProps) {
  const [selectedItemIds, setSelectedItemIds] = useState<Set<number>>(new Set());
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  
  const handleCopyOrderNumber = (orderNumber: string) => {
    navigator.clipboard.writeText(orderNumber);
    toast({
      description: `Skopiowano numer zamówienia: ${orderNumber}`,
      duration: 2000,
    });
  };
  
  const handleGoToOrder = (orderNumber: string) => {
    setLocation(`/order/${orderNumber}`);
  };

  // Fetch plan items
  const { data: items = [], isLoading: itemsLoading } = useQuery<PlanLineWithDetails[]>({
    queryKey: [`/api/production/planning/plans/${planId}/items`],
    staleTime: 0, // Force fresh fetch
    gcTime: 0, // Don't cache old data
  });

  // Fetch dictionaries for color rendering
  const { data: dictionaries = [] } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries"],
  });

  const colors = dictionaries.filter((d) => d.dictionaryType === "color");

  // Filter items based on search (semicolon-delimited AND logic)
  const filteredItems = useMemo(() => {
    if (!searchFilter.trim()) return items;

    const searchTerms = searchFilter
      .split(";")
      .map((term) => term.trim().toLowerCase())
      .filter((term) => term.length > 0);

    if (searchTerms.length === 0) return items;

    return items.filter((item) => {
      const searchableText = [
        item.product_sku,
        item.product_title,
        item.product_color,
        item.product_type,
        item.product_group,
        item.doors,
        item.legs,
        item.material,
        item.order_number,
        item.buyer_first_name,
        item.buyer_last_name,
        ...(item.product_color_options || []),
      ]
        .filter(Boolean)
        .join(" ")
        .toLowerCase();

      return searchTerms.every((term) => searchableText.includes(term));
    });
  }, [items, searchFilter]);

  // Handlers for selection
  const toggleItemSelection = (itemId: number) => {
    setSelectedItemIds(prev => {
      const newSet = new Set(prev);
      if (newSet.has(itemId)) {
        newSet.delete(itemId);
      } else {
        newSet.add(itemId);
      }
      return newSet;
    });
  };

  const toggleSelectAll = () => {
    if (selectedItemIds.size === filteredItems.length) {
      setSelectedItemIds(new Set());
    } else {
      setSelectedItemIds(new Set(filteredItems.map(item => item.id)));
    }
  };

  const handleBulkRemove = () => {
    if (onRemoveItems && selectedItemIds.size > 0) {
      onRemoveItems(Array.from(selectedItemIds));
      setSelectedItemIds(new Set());
    }
  };

  const isLoading = statsLoading || itemsLoading;

  // Droppable zone for DnD
  const { setNodeRef, isOver } = useDroppable({
    id: 'plan-items-dropzone',
  });

  return (
    <Card 
      ref={setNodeRef}
      className={`h-full flex flex-col transition-all duration-300 relative overflow-hidden ${
        isOver 
          ? 'bg-purple-500/10 border-2 border-purple-500 shadow-lg shadow-purple-500/20' 
          : 'border border-border'
      }`}
      data-testid="plan-items-droppable"
    >
      {/* Drag-over overlay with gradient effect */}
      {isOver && (
        <div className="absolute inset-0 bg-gradient-to-br from-purple-500/20 via-purple-400/10 to-purple-600/20 pointer-events-none z-10 animate-pulse" />
      )}
      
      {/* Single Scrollable Container for Everything */}
      <div className="flex-1 min-h-0 overflow-auto">
        {/* Plan Header - Scrolls with content */}
        <PlanHeader plan={plan} onBack={onBack} />
        
        {/* Statistics Bar - Scrolls with header */}
        <div className="h-[25px] bg-secondary/20 px-4 flex items-center gap-6 text-xs border-b flex-shrink-0">
        {isLoading ? (
          <>
            <Skeleton className="h-4 w-32" />
            <Skeleton className="h-4 w-24" />
            <Skeleton className="h-4 w-24" />
            <Skeleton className="h-4 w-24" />
            <Skeleton className="h-4 w-24" />
          </>
        ) : (
          <>
            {/* Plan Number Badge - Similar to Sales Module */}
            {planNumber && (
              <>
                <Badge 
                  variant="default" 
                  className="bg-primary text-primary-foreground font-semibold text-sm px-3 py-0.5"
                  data-testid="badge-plan-number"
                >
                  {planNumber}
                </Badge>
                <Separator orientation="vertical" className="h-4" />
              </>
            )}
            <div className="flex items-center gap-1.5">
              <ShoppingCart className="h-3.5 w-3.5 text-muted-foreground" />
              <span className="text-muted-foreground">Zamówienia:</span>
              <span className="font-semibold" data-testid="stat-orders-count">
                {statistics.ordersCount}
              </span>
            </div>
            <Separator orientation="vertical" className="h-4" />
            <div className="flex items-center gap-1.5">
              <Package className="h-3.5 w-3.5 text-muted-foreground" />
              <span className="text-muted-foreground">Produkty:</span>
              <span className="font-semibold" data-testid="stat-products-count">
                {statistics.productsCount}
              </span>
            </div>
            <Separator orientation="vertical" className="h-4" />
            <div className="flex items-center gap-1.5">
              <FileText className="h-3.5 w-3.5 text-muted-foreground" />
              <span className="text-muted-foreground">Linie:</span>
              <span className="font-semibold" data-testid="stat-lines-count">
                {statistics.planLinesCount}
              </span>
            </div>
            <Separator orientation="vertical" className="h-4" />
            <div className="flex items-center gap-1.5">
              <Hash className="h-3.5 w-3.5 text-muted-foreground" />
              <span className="text-muted-foreground">Ilość:</span>
              <span className="font-semibold" data-testid="stat-total-quantity">
                {statistics.totalQuantity}
              </span>
            </div>
          </>
          )}
        </div>
        
        {/* Sticky Tabs Bar + Search + Settings + Bulk Actions - sticky within scroll container */}
        <div className="sticky top-0 z-20 bg-background px-4 py-2 border-b border-t flex items-center gap-4">
        <FilterTabsBarContent
          planId={planId}
          searchFilter={searchFilter}
          onSearchChange={onSearchChange}
        />
        {selectedItemIds.size > 0 && onRemoveItems && (
          <Button
            variant="destructive"
            size="sm"
            onClick={handleBulkRemove}
            disabled={isRemoving}
            className="h-7 px-3 text-xs flex items-center gap-2"
            data-testid="button-bulk-remove"
          >
            {isRemoving ? (
              <>
                <Loader2 className="h-3.5 w-3.5 animate-spin" />
                Usuwanie...
              </>
            ) : (
              <>
                <Trash2 className="h-3.5 w-3.5" />
                Usuń zaznaczone ({selectedItemIds.size})
              </>
            )}
          </Button>
        )}
        </div>

        {/* Product Table - Inside scrollable container */}
        <CardContent className="p-0 relative">
          {/* Loading overlay during bulk operations */}
          {isRemoving && (
            <div className="absolute inset-0 bg-background/90 backdrop-blur-sm z-50 flex items-center justify-center">
              <div className="flex flex-col items-center gap-3">
                <Loader2 className="h-10 w-10 animate-spin text-primary" />
                <p className="text-base font-semibold">Przetwarzanie...</p>
              </div>
            </div>
          )}
          
          {itemsLoading ? (
            <div className="p-8 space-y-4">
              {[...Array(5)].map((_, i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : filteredItems.length === 0 ? (
            <div className="text-center p-8 text-muted-foreground">
              <Package className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p className="font-medium">Brak produktów w planie</p>
              <p className="text-sm">Dodaj produkty z dostępnych zamówień</p>
            </div>
          ) : (
            <div className="w-full">
              <Table className="w-full">
                <TableHeader className="bg-background border-b">
                  <TableRow className="h-8">
                    <TableHead className="text-xs font-semibold py-1 w-[30px]">
                  <Checkbox
                    checked={filteredItems.length > 0 && selectedItemIds.size === filteredItems.length}
                    onCheckedChange={toggleSelectAll}
                    data-testid="checkbox-select-all"
                    />
                  </TableHead>
                  <TableHead className="text-xs font-semibold py-1 w-[40px]">Zdj.</TableHead>
                  <TableHead className="text-xs font-semibold py-1">Kupujący</TableHead>
                  <TableHead className="text-xs font-semibold py-1">Zamówienie</TableHead>
                  <TableHead className="text-xs font-semibold py-1">Data</TableHead>
                  <TableHead className="text-xs font-semibold py-1">SKU</TableHead>
                  <TableHead className="text-xs font-semibold py-1">Typ</TableHead>
                  <TableHead className="text-xs font-semibold py-1 w-[80px]">D×S×W</TableHead>
                  <TableHead className="text-xs font-semibold py-1">Kolor</TableHead>
                  <TableHead className="text-xs font-semibold py-1">Opcje koloru</TableHead>
                  <TableHead className="text-xs font-semibold py-1 w-[70px]">Drzwi</TableHead>
                  <TableHead className="text-xs font-semibold py-1 w-[70px]">Nogi</TableHead>
                  <TableHead className="text-xs font-semibold py-1 w-[50px]">Ilość</TableHead>
                  <TableHead className="text-xs font-semibold py-1">Status</TableHead>
                  <TableHead className="text-xs font-semibold py-1 w-[50px]">Akcje</TableHead>
                </TableRow>
              </TableHeader>
            <TableBody>
              {filteredItems.map((item) => {
                const bgColor = getTypeBackgroundColor(item.product_type);
                const textColorClass = getTextColor(bgColor) === "#000000" ? "text-black" : "text-white";

                return (
                  <TableRow 
                    key={item.id} 
                    className="h-10 bg-muted/20 text-muted-foreground hover:bg-muted/30 transition-colors" 
                    data-testid={`row-item-${item.id}`}
                  >
                    <TableCell className="py-1">
                      <Checkbox
                        checked={selectedItemIds.has(item.id)}
                        onCheckedChange={() => toggleItemSelection(item.id)}
                        data-testid={`checkbox-item-${item.id}`}
                      />
                    </TableCell>
                    <TableCell className="py-1">
                      <div className="h-10 w-10 bg-muted/50 flex items-center justify-center overflow-hidden">
                        {item.product_image ? (
                          <img
                            src={item.product_image}
                            alt={item.product_title}
                            className="h-full w-full object-cover"
                            data-testid={`img-product-${item.id}`}
                          />
                        ) : (
                          <ImageIcon className="h-4 w-4 text-muted-foreground/50" />
                        )}
                      </div>
                    </TableCell>
                    <TableCell className="py-1 text-xs" data-testid={`text-buyer-${item.id}`}>
                      {item.buyer_first_name && item.buyer_last_name 
                        ? `${item.buyer_first_name} ${item.buyer_last_name}` 
                        : "-"}
                    </TableCell>
                    <TableCell className="py-1" data-testid={`text-order-number-${item.id}`}>
                      <div className="flex flex-col gap-1">
                        {/* Source Badge */}
                        <Badge 
                          variant={mapSourceLabel(item).variant}
                          className="h-5 px-2 text-[10px] w-fit"
                          data-testid={`badge-source-${item.id}`}
                        >
                          {mapSourceLabel(item).label}
                        </Badge>
                        
                        {/* Order Number */}
                        {item.order_number ? (
                          <div className="flex items-center gap-1">
                            <Badge 
                              className="h-5 px-2 text-[10px] bg-black text-white border-transparent cursor-pointer hover:bg-gray-800 transition-colors"
                              onClick={() => handleCopyOrderNumber(item.order_number!)}
                              title="Kliknij aby skopiować numer zamówienia"
                            >
                              #{item.order_number}
                            </Badge>
                            <button
                              className="w-3 h-3 flex items-center justify-center text-primary hover:text-primary/80 transition-colors"
                              title="Przejdź do zamówienia"
                              onClick={() => handleGoToOrder(item.order_number!)}
                            >
                              <FileText className="w-3 h-3" />
                            </button>
                          </div>
                        ) : (
                          <span className="text-xs text-muted-foreground">-</span>
                        )}
                      </div>
                    </TableCell>
                    <TableCell className="py-1 text-xs" data-testid={`text-order-date-${item.id}`}>
                      {item.order_date 
                        ? format(new Date(item.order_date), "dd.MM.yyyy HH:mm", { locale: pl })
                        : "-"}
                    </TableCell>
                    <TableCell className="py-1 font-mono text-xs" data-testid={`text-sku-${item.id}`}>
                      {item.product_sku}
                    </TableCell>
                    <TableCell className="py-1">
                      {item.product_type ? (
                        <Badge
                          variant="default"
                          className={`text-xs h-5 ${textColorClass} border-transparent`}
                          style={{ backgroundColor: bgColor }}
                          data-testid={`badge-type-${item.id}`}
                        >
                          {item.product_type}
                        </Badge>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </TableCell>
                    <TableCell className="py-1 text-xs text-center" data-testid={`text-dimensions-${item.id}`}>
                      {item.product_length && item.product_width && item.product_height ? (
                        <span className="font-mono">
                          {item.product_length}×{item.product_width}×{item.product_height}
                        </span>
                      ) : (
                        "-"
                      )}
                    </TableCell>
                    <TableCell className="py-1 text-xs" data-testid={`text-color-${item.id}`}>
                      {item.product_color ? (
                        (() => {
                          const colorHex = getColorHex(item.product_color, colors);
                          const textColor = colorHex ? 
                            (getTextColor(`hsl(0, 0%, ${parseInt(colorHex.slice(1, 3), 16) / 255 * 100}%)`) === "#000000" ? '#000' : '#fff') 
                            : '#000';
                          return (
                            <div
                              className="px-2 py-1 rounded text-center text-[10px] font-medium border"
                              style={{
                                backgroundColor: colorHex || '#cccccc',
                                color: textColor,
                                borderColor: colorHex || '#999'
                              }}
                              title={item.product_color}
                            >
                              {item.product_color}
                            </div>
                          );
                        })()
                      ) : (
                        "-"
                      )}
                    </TableCell>
                    <TableCell className="py-1" data-testid={`text-color-options-${item.id}`}>
                      {item.product_color_options && item.product_color_options.length > 0 ? (
                        <div className="flex flex-wrap gap-1">
                          {item.product_color_options.map((option, idx) => {
                            const optionHex = getColorHex(option, colors);
                            const optionTextColor = optionHex ? 
                              (getTextColor(`hsl(0, 0%, ${parseInt(optionHex.slice(1, 3), 16) / 255 * 100}%)`) === "#000000" ? '#000' : '#fff') 
                              : '#000';
                            return (
                              <div
                                key={idx}
                                className="px-2 py-0.5 rounded text-center text-[10px] font-medium border"
                                style={{
                                  backgroundColor: optionHex || '#e0e0e0',
                                  color: optionTextColor,
                                  borderColor: optionHex || '#999'
                                }}
                                title={option}
                              >
                                {option}
                              </div>
                            );
                          })}
                        </div>
                      ) : (
                        <span className="text-muted-foreground text-xs">-</span>
                      )}
                    </TableCell>
                    <TableCell className="py-1 text-xs" data-testid={`text-doors-${item.id}`}>
                      {item.doors ? (
                        <Badge variant="outline" className="h-5 px-2 text-[10px]">
                          {item.doors}
                        </Badge>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </TableCell>
                    <TableCell className="py-1 text-xs" data-testid={`text-legs-${item.id}`}>
                      {item.legs ? (
                        <Badge variant="outline" className="h-5 px-2 text-[10px]">
                          {item.legs}
                        </Badge>
                      ) : (
                        <span className="text-muted-foreground">-</span>
                      )}
                    </TableCell>
                    <TableCell className="py-1 text-center font-semibold" data-testid={`text-quantity-${item.id}`}>
                      {item.quantity}
                    </TableCell>
                    <TableCell className="py-1">
                      <Badge variant="outline" className="text-xs h-5" data-testid={`badge-status-${item.id}`}>
                        {item.status}
                      </Badge>
                    </TableCell>
                    <TableCell className="py-1">
                      {onRemoveItem && (
                        <Button
                          variant="ghost"
                          size="icon"
                          className="h-[25px] w-[25px] p-0 bg-red-600 hover:bg-red-700 text-white flex-shrink-0"
                          onClick={(e) => {
                            e.stopPropagation();
                            onRemoveItem(item.id);
                          }}
                          disabled={isRemoving}
                          data-testid={`button-remove-${item.id}`}
                          title="Usuń z planu produkcyjnego"
                        >
                          <ArrowLeft className="h-4 w-4" />
                        </Button>
                      )}
                    </TableCell>
                  </TableRow>
                );
                })}
              </TableBody>
            </Table>
            </div>
          )}
        </CardContent>
      </div>
    </Card>
  );
}
