from datetime import datetime

import dateutil
from odoo import _, api, fields, models
from odoo.tools import ormcache


class MarketOffer(models.Model):
    _inherit = 'trilab.market.offer'

    allegro_status = fields.Selection(
        [('ACTIVE', 'ACTIVE'), ('ENDED', 'ENDED'), ('ACTIVATING', 'ACTIVATING'), ('INACTIVE', 'INACTIVE')],
        string='Allegro Offer Status',
        help="""ACTIVE - the offer is active
ENDED - the offer was active and is now ended (for whatever reason)
ACTIVATING - the offer is planned for listing or is during the process of activation
INACTIVE - a draft offer""",
        compute='_compute_allegro_status',
        store=True,
        index=True,
    )

    allegro_event_ids = fields.One2many('trilab_allegro.event', 'offer_id', string='Allegro Events')

    allegro_republish = fields.Boolean(
        string='Allegro Republish',
        help="""Manage Allegro parameter to automatically relist offer after its end,
if checked - the offer will be republish with the full stock.""",
    )

    @api.depends('data')
    def _compute_allegro_status(self):
        for offer_id in self:
            offer_id.allegro_status = offer_id.allegro_get_offer_status()

    @api.onchange('allegro_republish')
    def _onchange_allegro_republish(self):
        self.mark_dirty()

    @ormcache('self.id', 'self.account_id.id')
    def is_allegro(self) -> bool:
        return self.account_id.is_allegro()

    def get_offer_price(self):
        self.ensure_one()

        if self.is_allegro():
            return float((((self.data or {}).get('sellingMode', {}) or {}).get('price', {}) or {}).get('amount', 0.0))

        return super().get_offer_price()

    def get_offer_qty(self, data: dict = None, account_id=None) -> int:
        self.ensure_one()

        if (account_id or self.account_id).is_allegro():
            return int(((data or self.data).get('stock', {}) or {}).get('available', 0) or 0)

        return super().get_offer_qty(data, account_id)

    def get_offer_sku(self):
        if self.is_allegro():
            return (self.data.get('external') or {}).get('id')

        return super().get_offer_sku()

    def get_offer_ean(self):
        if self.is_allegro():
            try:
                ean_gtin = self.env['ir.config_parameter'].sudo().get_param('allegro_parameter_ean_gtin')

                # get first product info
                product_data = self.data['productSet'][0]['product']['parameters']

                return next(
                    (parameter['values'][0] for parameter in product_data if parameter['id'] == ean_gtin),
                    None,
                )
            except (KeyError, IndexError):
                return None

        return super().get_offer_ean()

    def get_offer_update_date(self, data: dict = None, account_id=None) -> datetime:
        if (account_id or self.account_id).is_allegro() and (data := data or self.data):
            return dateutil.parser.parse(data.get('updatedAt'), ignoretz=True)

        return super().get_offer_update_date(data, account_id)

    def allegro_get_offer_status(self, data: dict = None, account_id=None) -> bool | None:
        self.ensure_one()

        if (account_id or self.account_id).is_allegro():
            return (data or self.data).get('publication', {}).get('status')

        return None

    def allegro_get_offer_republish(self, data: dict = None, account_id=None) -> bool | None:
        if (account_id or self.account_id).is_allegro() and (data := data or self.data):
            return data.get('publication', {}).get('republish')

        return None

    # noinspection PyMethodMayBeStatic
    def data_changed(self, data: dict) -> bool:
        self.ensure_one()

        if self.is_allegro():
            return not self.data or not data or self.data.get('updatedAt') != data.get('updatedAt')

        return super().data_changed(data)

    def action_sync_offer(self):
        allegro_offer_ids = self.filtered(lambda o_id: o_id.is_allegro())
        other_ids = self - allegro_offer_ids

        result = super(MarketOffer, other_ids).action_sync_offer()

        cnt = 0

        for offer_id in allegro_offer_ids:
            if offer_id.account_id.do_sync_offer(offer_id=offer_id)[0]:
                cnt += 1

        message = _('[Allegro] Total %s/updated %s.', len(allegro_offer_ids), cnt)

        if other_ids:
            message = f'{result["params"]["message"]}\n{message}'

        result['params']['message'] = message

        return result

    def allegro_action_preview_auction(self):
        self.ensure_one()

        return {
            'type': 'ir.actions.act_url',
            'target': 'new',
            'url': self.account_id.get_client().get_auction_url(self.ref),
        }

    def allegro_action_edit_auction(self):
        self.ensure_one()

        return {
            'type': 'ir.actions.act_url',
            'target': 'new',
            'url': self.account_id.get_client().get_auction_edit_url('offer', self.ref),
        }

    def allegro_action_republish(self, republish=True):
        offer_ids = self.filtered(lambda o_id: o_id.is_allegro() and o_id.allegro_republish != republish)
        offer_ids.allegro_republish = republish
        offer_ids.mark_dirty()
