export type UserRole = 'admin' | 'manager' | 'user';

export type Permission =
  | 'view_orders'
  | 'view_order_details'
  | 'view_recent_updates'
  | 'view_statistics'
  | 'manage_sync'
  | 'view_sync_logs'
  | 'manage_settings'
  | 'manage_users'
  | 'manage_credentials'
  | 'manage_api_tokens'
  | 'manage_webhooks'
  | 'view_production'
  | 'manage_production';

// Definicja uprawnień dla każdej roli
export const ROLE_PERMISSIONS: Record<UserRole, Permission[]> = {
  admin: [
    'view_orders',
    'view_order_details',
    'view_recent_updates',
    'view_statistics',
    'manage_sync',
    'view_sync_logs',
    'manage_settings',
    'manage_users',
    'manage_credentials',
    'manage_api_tokens',
    'manage_webhooks',
    'view_production',
    'manage_production',
  ],
  manager: [
    'view_orders',
    'view_order_details',
    'view_recent_updates',
    'view_statistics',
    'manage_sync',
    'view_sync_logs',
    'view_production',
  ],
  user: [
    'view_orders',
    'view_order_details',
    'view_recent_updates',
    'view_statistics',
  ],
};

// Mapowanie stron do wymaganych uprawnień
export const PAGE_PERMISSIONS: Record<string, Permission> = {
  '/': 'view_orders',
  '/recent-updates': 'view_recent_updates',
  '/date-sync': 'manage_sync',
  '/settings': 'manage_settings',
  '/logs': 'view_sync_logs',
  '/users': 'manage_users',
  '/api-tokens': 'manage_api_tokens',
  '/api-tokens/logs': 'manage_api_tokens',
  '/webhooks': 'manage_webhooks',
  '/webhooks/logs': 'manage_webhooks',
};

// Helper function do sprawdzania uprawnień
export function hasPermission(role: UserRole, permission: Permission): boolean {
  return ROLE_PERMISSIONS[role]?.includes(permission) ?? false;
}

// Helper function do sprawdzania dostępu do strony
export function canAccessPage(role: UserRole, path: string): boolean {
  // Obsługa dynamicznych tras jak /order/:id
  const basePath = path.startsWith('/order/') ? '/order' : path;
  
  const requiredPermission = PAGE_PERMISSIONS[basePath];
  
  // Jeśli strona nie wymaga specjalnych uprawnień, zezwól
  if (!requiredPermission) {
    return true;
  }
  
  return hasPermission(role, requiredPermission);
}

// Helper function do filtrowania menu items
export function getAccessiblePages(role: UserRole): string[] {
  return Object.keys(PAGE_PERMISSIONS).filter(path => canAccessPage(role, path));
}
