import type { Express } from "express";
import { z } from "zod";
import { pool } from "../../postgres";
import * as planningService from "../../services/production/planning";
import * as batchEngine from "../../services/production/batch-engine";
import { requirePermission } from "../../auth";

const createPlanSchema = z.object({
  planNumber: z.string().optional(),
  name: z.string().min(1),
  description: z.string().optional().nullable(),
  plannedStartDate: z.string().optional().nullable(),
  plannedEndDate: z.string().optional().nullable(),
  status: z.enum(['draft', 'approved', 'in_progress', 'completed', 'cancelled']).optional(),
  priority: z.enum(['low', 'normal', 'high', 'urgent']).optional(),
  notes: z.string().optional().nullable(),
  metadata: z.any().optional(),
  createdBy: z.number().optional().nullable(),
});

const createPlanLineSchema = z.object({
  planId: z.number(),
  productId: z.number(),
  quantity: z.number().positive(),
  sourceType: z.enum(['sales_order', 'forecast', 'buffer_stock', 'manual', 'order_demand', 'catalog_internal', 'cutting_pattern']).optional().nullable(),
  sourceId: z.number().optional().nullable(),
  sourceReference: z.string().optional().nullable(),
  productionOrderId: z.number().optional().nullable(),
  routingId: z.number().optional().nullable(),
  routingOverride: z.any().optional(),
  bomId: z.number().optional().nullable(),
  plannedStartDate: z.string().optional().nullable(),
  plannedEndDate: z.string().optional().nullable(),
  status: z.enum(['pending', 'scheduled', 'in_progress', 'completed', 'cancelled']).optional(),
  sequence: z.number().optional().nullable(),
  notes: z.string().optional().nullable(),
  metadata: z.any().optional(),
});

const filtersSchema = z.object({
  status: z.string().optional(),
  priority: z.string().optional(),
  createdBy: z.coerce.number().optional(),
  startDate: z.string().optional(),
  endDate: z.string().optional(),
  search: z.string().optional(),
  limit: z.coerce.number().optional(),
  offset: z.coerce.number().optional(),
});

const demandFiltersSchema = z.object({
  startDate: z.string().optional(),
  endDate: z.string().optional(),
  marketplace: z.string().optional(),
  orderStatus: z.string().optional(),
  paymentStatus: z.string().optional(),
});

const availableOrdersFiltersSchema = z.object({
  search: z.string().optional(),
  color: z.string().optional(),
  sku: z.string().optional(),
  minLength: z.coerce.number().optional(),
  maxLength: z.coerce.number().optional(),
  minWidth: z.coerce.number().optional(),
  maxWidth: z.coerce.number().optional(),
  orderNumber: z.string().optional(),
  customerName: z.string().optional(),
  marketplace: z.string().optional(),
  showSetsOnly: z.string().optional(), // 'true' or 'false'
  showCatalogLinked: z.string().optional(), // 'true' or 'false'
  limit: z.coerce.number().min(1).max(500).optional().default(100),
  offset: z.coerce.number().min(0).optional().default(0),
  sortBy: z.enum(['order_date', 'order_number', 'buyer_name', 'total_amount', 'product_sku']).optional().default('order_date'),
  sortOrder: z.enum(['asc', 'desc']).optional().default('desc'),
});

// Helper: Parse semicolon-delimited search tokens into structured filters
interface ParsedSearchTokens {
  marketplaces: string[];
  paymentStatuses: string[];
  searchTerms: string[];
}

function parseSearchTokens(rawSearch: string | undefined): ParsedSearchTokens {
  if (!rawSearch) {
    return { marketplaces: [], paymentStatuses: [], searchTerms: [] };
  }

  // Split by semicolon, trim, uppercase, deduplicate
  const tokens = Array.from(new Set(
    rawSearch.split(';')
      .map(t => t.trim().toUpperCase())
      .filter(Boolean)
  ));

  // Whitelist recognized tokens
  const MARKETPLACE_TOKENS = ['ALLEGRO', 'SHOPER'];
  const PAYMENT_STATUS_TOKENS = ['PAID', 'UNPAID', 'PENDING'];

  const marketplaces: string[] = [];
  const paymentStatuses: string[] = [];
  const searchTerms: string[] = [];

  tokens.forEach(token => {
    if (MARKETPLACE_TOKENS.includes(token)) {
      marketplaces.push(token.toLowerCase()); // Store as lowercase for DB comparison
    } else if (PAYMENT_STATUS_TOKENS.includes(token)) {
      paymentStatuses.push(token.toLowerCase());
    } else {
      searchTerms.push(token); // Keep original case for search
    }
  });

  return { marketplaces, paymentStatuses, searchTerms };
}

export function registerPlanningRoutes(app: Express) {
  // GET /api/production/planning/plans - Get all plans with filtering
  app.get("/api/production/planning/plans", requirePermission('view_production'), async (req, res) => {
    try {
      const filters = filtersSchema.parse(req.query);
      
      const processedFilters = {
        ...filters,
        startDate: filters.startDate ? new Date(filters.startDate) : undefined,
        endDate: filters.endDate ? new Date(filters.endDate) : undefined,
      };

      const plans = await planningService.getPlans(pool, processedFilters);
      res.json(plans);
    } catch (error) {
      console.error("Error fetching production plans:", error);
      res.status(500).json({ message: "Failed to fetch production plans" });
    }
  });

  // GET /api/production/planning/plans/:id - Get plan by ID
  app.get("/api/production/planning/plans/:id", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const plan = await planningService.getPlanById(pool, id);
      if (!plan) {
        return res.status(404).json({ message: "Production plan not found" });
      }

      res.json(plan);
    } catch (error) {
      console.error("Error fetching production plan:", error);
      res.status(500).json({ message: "Failed to fetch production plan" });
    }
  });

  // GET /api/production/planning/plans/:id/items - Get plan lines with full details
  app.get("/api/production/planning/plans/:id/items", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const items = await planningService.getPlanLinesWithDetails(pool, id);
      
      // DEBUG: Log how many items we're returning
      console.log(`📦 [PLAN ITEMS] Returning ${items.length} items for plan ${id}`);
      if (items.length > 0) {
        console.log(`📦 [SAMPLE] First item:`, {
          id: items[0].id,
          product_sku: items[0].product_sku,
          product_title: items[0].product_title,
          order_number: items[0].order_number
        });
      }
      
      // Disable HTTP caching and ETag to ensure fresh data
      res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, private, max-age=0');
      res.setHeader('Pragma', 'no-cache');
      res.setHeader('Expires', '0');
      res.removeHeader('ETag'); // Remove Express auto-generated ETag
      
      res.json(items);
    } catch (error) {
      console.error("Error fetching plan items:", error);
      res.status(500).json({ message: "Failed to fetch plan items" });
    }
  });

  // GET /api/production/demand/aggregate - Aggregate order demand by catalog product
  app.get("/api/production/demand/aggregate", requirePermission('view_production'), async (req, res) => {
    try {
      const filters = demandFiltersSchema.parse(req.query);
      
      const processedFilters: planningService.DemandFilters = {
        startDate: filters.startDate ? new Date(filters.startDate) : undefined,
        endDate: filters.endDate ? new Date(filters.endDate) : undefined,
        marketplace: filters.marketplace,
        orderStatus: filters.orderStatus ? filters.orderStatus.split(',') : undefined,
        paymentStatus: filters.paymentStatus ? filters.paymentStatus.split(',') : undefined,
      };

      const aggregatedDemand = await planningService.aggregateDemand(pool, processedFilters);
      res.json(aggregatedDemand);
    } catch (error) {
      console.error("Error aggregating demand:", error);
      res.status(500).json({ message: "Failed to aggregate demand" });
    }
  });

  // POST /api/production/planning/plans - Create new plan
  app.post("/api/production/planning/plans", requirePermission('manage_production'), async (req, res) => {
    try {
      const data = createPlanSchema.parse(req.body);
      
      const plan = await planningService.createPlan(pool, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : null,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : null,
      });
      
      res.status(201).json(plan);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Plan with this number already exists" });
      }
      console.error("Error creating production plan:", error);
      res.status(500).json({ message: "Failed to create production plan" });
    }
  });

  // PATCH /api/production/planning/plans/:id - Update plan
  app.patch("/api/production/planning/plans/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const data = createPlanSchema.partial().parse(req.body);
      const plan = await planningService.updatePlan(pool, id, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : undefined,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : undefined,
      });
      
      if (!plan) {
        return res.status(404).json({ message: "Production plan not found" });
      }

      res.json(plan);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error updating production plan:", error);
      res.status(500).json({ message: "Failed to update production plan" });
    }
  });

  // DELETE /api/production/planning/plans/:id - Delete plan
  app.delete("/api/production/planning/plans/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const success = await planningService.deletePlan(pool, id);
      if (!success) {
        return res.status(404).json({ message: "Production plan not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting production plan:", error);
      res.status(500).json({ message: "Failed to delete production plan" });
    }
  });

  // GET /api/production/planning/plans/:id/lines - Get all lines for a plan
  app.get("/api/production/planning/plans/:id/lines", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const lines = await planningService.getPlanLines(pool, planId);
      res.json(lines);
    } catch (error) {
      console.error("Error fetching plan lines:", error);
      res.status(500).json({ message: "Failed to fetch plan lines" });
    }
  });

  // GET /api/production/planning/lines/:id - Get plan line by ID
  app.get("/api/production/planning/lines/:id", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const line = await planningService.getPlanLineById(pool, id);
      if (!line) {
        return res.status(404).json({ message: "Plan line not found" });
      }

      res.json(line);
    } catch (error) {
      console.error("Error fetching plan line:", error);
      res.status(500).json({ message: "Failed to fetch plan line" });
    }
  });

  // POST /api/production/planning/lines - Create new plan line
  app.post("/api/production/planning/lines", requirePermission('manage_production'), async (req, res) => {
    try {
      const data = createPlanLineSchema.parse(req.body);
      
      // Check for duplicate based on metadata type
      if (data.planId) {
        let duplicateCheck;
        
        // For components from sets: check by set_id + component_id + order_number
        // Note: Same component from DIFFERENT orders = DIFFERENT products (user requirement)
        if (data.metadata?.set_id && data.metadata?.component_id) {
          if (data.metadata.order_number) {
            // New behavior: Check only rows with same order_number
            // This allows same component from different orders to be added separately
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND (metadata->>'set_id')::int = $2
               AND (metadata->>'component_id')::int = $3
               AND (metadata->>'order_number') = $4
               LIMIT 1`,
              [data.planId, data.metadata.set_id, data.metadata.component_id, data.metadata.order_number]
            );
          } 
          // Legacy behavior: Check only set_id + component_id (for old plan_lines without order_number)
          else {
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND (metadata->>'set_id')::int = $2
               AND (metadata->>'component_id')::int = $3
               LIMIT 1`,
              [data.planId, data.metadata.set_id, data.metadata.component_id]
            );
          }
        }
        // For regular order items: check by catalog_product_id + order_number
        // Note: Same product from DIFFERENT orders = DIFFERENT products (user requirement)
        // Only check for order_demand source type (allow manual/forecast/buffer to add same product)
        else if (data.productId && data.sourceType === 'order_demand') {
          if (data.metadata?.order_number) {
            // Strict matching: Check only rows with same order_number
            // This allows same product from different orders to be added separately
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND product_id = $2
               AND source_type = 'order_demand'
               AND (metadata->>'order_number') = $3
               LIMIT 1`,
              [data.planId, data.productId, data.metadata.order_number]
            );
          } else {
            // Fallback: Check only product_id (should not happen as all new items have order_number)
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND product_id = $2
               AND source_type = 'order_demand'
               LIMIT 1`,
              [data.planId, data.productId]
            );
          }
        }
        
        if (duplicateCheck && duplicateCheck.rows.length > 0) {
          return res.status(409).json({ 
            message: "Product already in the plan" 
          });
        }
      }
      
      const line = await planningService.createPlanLine(pool, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : null,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : null,
      });
      
      res.status(201).json(line);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23503') {
        return res.status(400).json({ message: "Invalid plan ID or product ID" });
      }
      console.error("Error creating plan line:", error);
      res.status(500).json({ message: "Failed to create plan line" });
    }
  });

  // PATCH /api/production/planning/lines/:id - Update plan line
  app.patch("/api/production/planning/lines/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const data = createPlanLineSchema.partial().parse(req.body);
      const line = await planningService.updatePlanLine(pool, id, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : undefined,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : undefined,
      });
      
      if (!line) {
        return res.status(404).json({ message: "Plan line not found" });
      }

      res.json(line);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error updating plan line:", error);
      res.status(500).json({ message: "Failed to update plan line" });
    }
  });

  // DELETE /api/production/planning/lines/:id - Delete plan line
  app.delete("/api/production/planning/lines/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const success = await planningService.deletePlanLine(pool, id);
      if (!success) {
        return res.status(404).json({ message: "Plan line not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting plan line:", error);
      res.status(500).json({ message: "Failed to delete plan line" });
    }
  });

  // POST /api/production/planning/lines/:id/generate-batches - Generate component batches for plan line
  app.post("/api/production/planning/lines/:id/generate-batches", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const batches = await batchEngine.generateBatchesForPlanLine(pool, id);
      
      res.status(201).json({
        message: `Generated ${batches.length} batches`,
        batches: batches.map(b => b.batch),
      });
    } catch (error) {
      console.error("Error generating batches:", error);
      
      if (error instanceof Error) {
        if (error.message.includes('not found')) {
          return res.status(404).json({ message: error.message });
        }
        if (error.message.includes('No active BOM')) {
          return res.status(400).json({ message: error.message });
        }
        if (error.message.includes('No components found')) {
          return res.status(400).json({ message: error.message });
        }
      }
      
      res.status(500).json({ message: "Failed to generate batches" });
    }
  });

  // GET /api/production/planning/lines/:id/batches - Get all batches for a plan line
  app.get("/api/production/planning/lines/:id/batches", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const batches = await batchEngine.getBatchesForPlanLine(pool, id);
      res.json(batches);
    } catch (error) {
      console.error("Error fetching batches:", error);
      res.status(500).json({ message: "Failed to fetch batches" });
    }
  });

  // GET /api/production/planning/plans/:id/available-orders - Get available marketplace orders for production planning
  app.get("/api/production/planning/plans/:id/available-orders", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Coerce empty strings to undefined before parsing (URL params issue)
      const normalizedQuery = Object.fromEntries(
        Object.entries(req.query).map(([key, value]) => [key, value === '' ? undefined : value])
      );

      // Parse and validate query parameters
      const filters = availableOrdersFiltersSchema.parse(normalizedQuery);

      // Parse search tokens into structured filters
      const parsedTokens = parseSearchTokens(filters.search);

      // Whitelist map for safe sorting (prevents SQL injection)
      const sortColumnMap: Record<string, string> = {
        'order_date': 'o.order_date',
        'order_number': 'o.order_number',
        'buyer_name': 'LOWER(o.buyer_last_name || \' \' || o.buyer_first_name)',
        'total_amount': 'o.total_to_pay_amount',
        'product_sku': 'MIN(cp.sku)',
      };

      const orderByColumn = sortColumnMap[filters.sortBy];
      const orderByDirection = filters.sortOrder.toUpperCase(); // 'ASC' or 'DESC'

      // Build WHERE clauses for filtering - use validated filters
      const conditions: string[] = ['1=1'];
      const params: any[] = [];
      let paramIndex = 1; // Start at 1 (no planId in params for count query)

      // Marketplace filter from parsed tokens (supports multiple)
      if (parsedTokens.marketplaces.length > 0) {
        const placeholders = parsedTokens.marketplaces.map((_, i) => `$${paramIndex + i}`).join(', ');
        conditions.push(`o.source IN (${placeholders})`);
        params.push(...parsedTokens.marketplaces);
        paramIndex += parsedTokens.marketplaces.length;
      }

      // Payment status filter from parsed tokens (supports multiple)
      if (parsedTokens.paymentStatuses.length > 0) {
        const placeholders = parsedTokens.paymentStatuses.map((_, i) => `$${paramIndex + i}`).join(', ');
        conditions.push(`o.payment_status IN (${placeholders})`);
        params.push(...parsedTokens.paymentStatuses);
        paramIndex += parsedTokens.paymentStatuses.length;
      }

      // Search terms filter (each term AND-ed together)
      if (parsedTokens.searchTerms.length > 0) {
        parsedTokens.searchTerms.forEach(term => {
          conditions.push(`(
            o.order_number ILIKE $${paramIndex} OR
            o.buyer_first_name ILIKE $${paramIndex} OR
            o.buyer_last_name ILIKE $${paramIndex} OR
            o.buyer_email ILIKE $${paramIndex} OR
            COALESCE(mp.sku, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.sku, ps.sku, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.product_type, sm.product_group, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.title, ps.title, '')::text ILIKE $${paramIndex} OR
            COALESCE(oi.name, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.color, ps.color, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.doors, sm.doors, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.legs, sm.legs, '')::text ILIKE $${paramIndex} OR
            EXISTS (
              SELECT 1 FROM unnest(COALESCE(cp.color_options, ps.color_options, ARRAY[]::text[])) AS co 
              WHERE co ILIKE $${paramIndex}
            )
          )`);
          params.push(`%${term}%`);
          paramIndex++;
        });
      }

      // Order number filter
      if (filters.orderNumber) {
        conditions.push(`o.order_number ILIKE $${paramIndex}`);
        params.push(`%${filters.orderNumber}%`);
        paramIndex++;
      }

      // Customer name filter
      if (filters.customerName) {
        conditions.push(`(
          o.buyer_first_name ILIKE $${paramIndex} OR
          o.buyer_last_name ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.customerName}%`);
        paramIndex++;
      }

      // SKU filter
      if (filters.sku) {
        conditions.push(`(
          COALESCE(mp.sku, '')::text ILIKE $${paramIndex} OR
          COALESCE(cp.sku, ps.sku, '')::text ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.sku}%`);
        paramIndex++;
      }

      // Color filter
      if (filters.color) {
        conditions.push(`COALESCE(cp.color, ps.color, '') ILIKE $${paramIndex}`);
        params.push(`%${filters.color}%`);
        paramIndex++;
      }

      // Length filters
      if (filters.minLength) {
        conditions.push(`cp.length >= $${paramIndex}`);
        params.push(filters.minLength);
        paramIndex++;
      }
      if (filters.maxLength) {
        conditions.push(`cp.length <= $${paramIndex}`);
        params.push(filters.maxLength);
        paramIndex++;
      }

      // Width filters
      if (filters.minWidth) {
        conditions.push(`cp.width >= $${paramIndex}`);
        params.push(filters.minWidth);
        paramIndex++;
      }
      if (filters.maxWidth) {
        conditions.push(`cp.width <= $${paramIndex}`);
        params.push(filters.maxWidth);
        paramIndex++;
      }

      // Legacy marketplace filter (kept for backward compatibility if passed explicitly)
      if (filters.marketplace && filters.marketplace !== 'all' && parsedTokens.marketplaces.length === 0) {
        conditions.push(`o.source = $${paramIndex}`);
        params.push(filters.marketplace);
        paramIndex++;
      }

      // Filter: Show only catalog-linked products or sets
      if (filters.showCatalogLinked === 'true') {
        conditions.push(`(cp.id IS NOT NULL OR ps.id IS NOT NULL)`);
      }

      // Filter: Show only products that are part of sets
      if (filters.showSetsOnly === 'true') {
        conditions.push(`EXISTS (
          SELECT 1 FROM product_creator.set_product_links spl
          WHERE spl.product_id = cp.id
        )`);
      }

      const whereClause = conditions.join(' AND ');

      // Step 1: Get total count for pagination (with same core joins to ensure cp and ps aliases exist)
      const countResult = await pool.query(`
        SELECT COUNT(DISTINCT o.id)::int as total
        FROM commerce.orders o
        LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
        LEFT JOIN commerce.marketplace_products mp ON oi.offer_external_id = mp.offer_external_id AND mp.source = o.source
        LEFT JOIN catalog.product_platform_data ppd ON oi.offer_external_id = ppd.external_id
        LEFT JOIN catalog.products cp ON ppd.product_id = cp.id AND (ppd.link_type = 'product' OR ppd.link_type IS NULL)
        LEFT JOIN product_creator.product_sets ps ON ppd.set_id = ps.id AND ppd.link_type = 'set'
        LEFT JOIN product_creator.set_matrices sm ON ps.set_matrix_id = sm.id
        WHERE ${whereClause}
          AND oi.id IS NOT NULL
      `, params);

      const total = countResult.rows[0]?.total || 0;

      // Step 2: Build WHERE clause for data query with adjusted param indices
      // Data query needs planId as $1, so we rebuild conditions with paramIndex starting from 2
      const dataConditions: string[] = ['1=1'];
      const dataParams: any[] = [planId]; // Start with planId at $1
      let dataParamIndex = 2; // Start filter params at $2

      // Rebuild all filter conditions with adjusted indices
      // Marketplace filter from parsed tokens (supports multiple)
      if (parsedTokens.marketplaces.length > 0) {
        const placeholders = parsedTokens.marketplaces.map((_, i) => `$${dataParamIndex + i}`).join(', ');
        dataConditions.push(`o.source IN (${placeholders})`);
        dataParams.push(...parsedTokens.marketplaces);
        dataParamIndex += parsedTokens.marketplaces.length;
      }

      // Payment status filter from parsed tokens (supports multiple)
      if (parsedTokens.paymentStatuses.length > 0) {
        const placeholders = parsedTokens.paymentStatuses.map((_, i) => `$${dataParamIndex + i}`).join(', ');
        dataConditions.push(`o.payment_status IN (${placeholders})`);
        dataParams.push(...parsedTokens.paymentStatuses);
        dataParamIndex += parsedTokens.paymentStatuses.length;
      }

      // Search terms filter (each term AND-ed together)
      if (parsedTokens.searchTerms.length > 0) {
        parsedTokens.searchTerms.forEach(term => {
          dataConditions.push(`(
            o.order_number ILIKE $${dataParamIndex} OR
            o.buyer_first_name ILIKE $${dataParamIndex} OR
            o.buyer_last_name ILIKE $${dataParamIndex} OR
            o.buyer_email ILIKE $${dataParamIndex} OR
            COALESCE(mp.sku, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.sku, ps.sku, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.product_type, sm.product_group, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.title, ps.title, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(oi.name, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.color, ps.color, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.doors, sm.doors, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.legs, sm.legs, '')::text ILIKE $${dataParamIndex} OR
            EXISTS (
              SELECT 1 FROM unnest(COALESCE(cp.color_options, ps.color_options, ARRAY[]::text[])) AS co 
              WHERE co ILIKE $${dataParamIndex}
            )
          )`);
          dataParams.push(`%${term}%`);
          dataParamIndex++;
        });
      }
      if (filters.orderNumber) {
        dataConditions.push(`o.order_number ILIKE $${dataParamIndex}`);
        dataParams.push(`%${filters.orderNumber}%`);
        dataParamIndex++;
      }
      if (filters.customerName) {
        dataConditions.push(`(
          o.buyer_first_name ILIKE $${dataParamIndex} OR
          o.buyer_last_name ILIKE $${dataParamIndex}
        )`);
        dataParams.push(`%${filters.customerName}%`);
        dataParamIndex++;
      }
      if (filters.sku) {
        dataConditions.push(`(
          COALESCE(mp.sku, '')::text ILIKE $${dataParamIndex} OR
          COALESCE(cp.sku, ps.sku, '')::text ILIKE $${dataParamIndex}
        )`);
        dataParams.push(`%${filters.sku}%`);
        dataParamIndex++;
      }
      if (filters.color) {
        dataConditions.push(`COALESCE(cp.color, ps.color, '') ILIKE $${dataParamIndex}`);
        dataParams.push(`%${filters.color}%`);
        dataParamIndex++;
      }
      if (filters.minLength) {
        dataConditions.push(`cp.length >= $${dataParamIndex}`);
        dataParams.push(filters.minLength);
        dataParamIndex++;
      }
      if (filters.maxLength) {
        dataConditions.push(`cp.length <= $${dataParamIndex}`);
        dataParams.push(filters.maxLength);
        dataParamIndex++;
      }
      if (filters.minWidth) {
        dataConditions.push(`cp.width >= $${dataParamIndex}`);
        dataParams.push(filters.minWidth);
        dataParamIndex++;
      }
      if (filters.maxWidth) {
        dataConditions.push(`cp.width <= $${dataParamIndex}`);
        dataParams.push(filters.maxWidth);
        dataParamIndex++;
      }
      // Legacy marketplace filter (kept for backward compatibility if passed explicitly)
      if (filters.marketplace && filters.marketplace !== 'all' && parsedTokens.marketplaces.length === 0) {
        dataConditions.push(`o.source = $${dataParamIndex}`);
        dataParams.push(filters.marketplace);
        dataParamIndex++;
      }

      // Filter: Show only catalog-linked products or sets
      if (filters.showCatalogLinked === 'true') {
        dataConditions.push(`(cp.id IS NOT NULL OR ps.id IS NOT NULL)`);
      }

      // Filter: Show only products that are part of sets
      if (filters.showSetsOnly === 'true') {
        dataConditions.push(`EXISTS (
          SELECT 1 FROM product_creator.set_product_links spl
          WHERE spl.product_id = cp.id
        )`);
      }

      const dataWhereClause = dataConditions.join(' AND ');
      
      // Include MIN(cp.sku) in SELECT/GROUP BY when sorting by product_sku
      const includeMinSku = filters.sortBy === 'product_sku';
      
      const dataResult = await pool.query(`
        SELECT 
          o.id as order_id,
          o.order_number,
          o.source as marketplace,
          o.buyer_first_name,
          o.buyer_last_name,
          o.buyer_email,
          o.order_date,
          o.payment_status,
          o.total_to_pay_amount,
          o.total_to_pay_currency as currency,
          ${includeMinSku ? 'MIN(cp.sku) as min_sku,' : ''}
          json_agg(
            json_build_object(
              'item_id', oi.id,
              'offer_external_id', oi.offer_external_id,
              'name', oi.name,
              'quantity', oi.quantity,
              'unit_price', oi.unit_price,
              'price', oi.price,
              'image_url', oi.image_url,
              'product_length', COALESCE(cp.length::text, sm.length),
              'product_width', COALESCE(cp.width::text, sm.width),
              'product_height', COALESCE(cp.height::text, sm.height),
              'product_color', COALESCE(cp.color, ps.color),
              'product_color_options', COALESCE(cp.color_options, ps.color_options),
              'product_sku', COALESCE(cp.sku, ps.sku),
              'product_type', COALESCE(cp.product_type, sm.product_group),
              'product_doors', COALESCE(cp.doors, sm.doors),
              'product_legs', COALESCE(cp.legs, sm.legs),
              'marketplace_product_id', mp.id,
              'link_type', ppd.link_type,
              'catalog_product_id', cp.id,
              'catalog_product_sku', cp.sku,
              'catalog_product_title', cp.title,
              'catalog_set_id', ps.id,
              'catalog_set_sku', ps.sku,
              'catalog_set_title', ps.title,
              'platform_link_id', ppd.id,
              'bom_component_count', COALESCE(pb_comp_count.component_count, 0),
              'product_group_id', ppg.id,
              'product_group_name', ppg.name,
              'product_group_color_hex', ppg.color_hex,
              'is_in_plan', CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END,
              'in_plan_number', pp_any.plan_number,
              'in_plan_id', pp_any.id,
              'in_current_plan', CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END,
              'set_components', set_components.components
            ) ORDER BY oi.id
          ) FILTER (WHERE oi.id IS NOT NULL) as items
        FROM commerce.orders o
        LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
        LEFT JOIN commerce.marketplace_products mp ON oi.offer_external_id = mp.offer_external_id AND mp.source = o.source
        LEFT JOIN catalog.product_platform_data ppd ON oi.offer_external_id = ppd.external_id
        LEFT JOIN catalog.products cp ON ppd.product_id = cp.id AND (ppd.link_type = 'product' OR ppd.link_type IS NULL)
        LEFT JOIN product_creator.product_sets ps ON ppd.set_id = ps.id AND ppd.link_type = 'set'
        LEFT JOIN product_creator.set_matrices sm ON ps.set_matrix_id = sm.id
        LEFT JOIN bom.product_boms pb ON pb.product_id = cp.id AND pb.is_active = true
        LEFT JOIN LATERAL (
          SELECT COUNT(*)::int as component_count
          FROM bom.product_components pc
          WHERE pc.product_bom_id = pb.id
        ) pb_comp_count ON true
        LEFT JOIN production.production_product_group_items ppgi ON ppgi.product_id = cp.id
        LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
        LEFT JOIN production.production_plan_lines ppl_current ON (
          ppl_current.plan_id = $1 
          AND ppl_current.source_type = 'order_demand'
          AND ppl_current.product_id = cp.id
          AND (ppd.link_type != 'set' OR ppd.link_type IS NULL)
          AND (ppl_current.metadata->>'order_number') = o.order_number
        )
        LEFT JOIN production.production_plan_lines ppl_any ON (
          ppl_any.source_type = 'order_demand'
          AND ppl_any.product_id = cp.id
          AND (ppd.link_type != 'set' OR ppd.link_type IS NULL)
          AND (ppl_any.metadata->>'order_number') = o.order_number
        )
        LEFT JOIN production.production_plans pp_any ON pp_any.id = ppl_any.plan_id
        LEFT JOIN LATERAL (
          SELECT json_agg(
            json_build_object(
              'component_id', comp_data.component_id,
              'component_sku', comp_data.component_sku,
              'component_title', comp_data.component_title,
              'component_color', comp_data.component_color,
              'component_length', comp_data.component_length,
              'component_width', comp_data.component_width,
              'component_height', comp_data.component_height,
              'component_product_type', comp_data.component_product_type,
              'component_doors', comp_data.component_doors,
              'component_legs', comp_data.component_legs,
              'quantity', comp_data.quantity,
              'primary_image_url', comp_data.primary_image_url,
              'parent_set_image_url', comp_data.parent_set_image_url,
              'is_in_current_plan', comp_data.is_in_current_plan,
              'is_in_any_plan', comp_data.is_in_any_plan,
              'in_plan_number', comp_data.in_plan_number,
              'in_plan_id', comp_data.in_plan_id
            ) ORDER BY comp_data.spl_id
          ) as components
          FROM (
            SELECT DISTINCT ON (comp.id)
              comp.id as component_id,
              comp.sku as component_sku,
              comp.title as component_title,
              comp.color as component_color,
              comp.length as component_length,
              comp.width as component_width,
              comp.height as component_height,
              comp.product_type as component_product_type,
              comp.doors as component_doors,
              comp.legs as component_legs,
              spl.quantity,
              COALESCE(comp.image_url, comp_img.url, oi.image_url) as primary_image_url,
              oi.image_url as parent_set_image_url,
              CASE WHEN ppl_comp_current.id IS NOT NULL THEN true ELSE false END as is_in_current_plan,
              CASE WHEN ppl_comp_any.id IS NOT NULL THEN true ELSE false END as is_in_any_plan,
              pp_comp_any.plan_number as in_plan_number,
              pp_comp_any.id as in_plan_id,
              spl.id as spl_id
            FROM product_creator.set_product_links spl
            JOIN catalog.products comp ON comp.id = spl.product_id
            LEFT JOIN catalog.product_images comp_img ON comp_img.product_id = comp.id AND comp_img.is_primary = true
            LEFT JOIN production.production_plan_lines ppl_comp_current ON (
              ppl_comp_current.plan_id = $1
              AND ppl_comp_current.source_type = 'order_demand'
              AND (ppl_comp_current.metadata->>'set_id')::int = ps.id
              AND (ppl_comp_current.metadata->>'component_id')::int = comp.id
              AND (ppl_comp_current.metadata->>'order_number') = o.order_number
            )
            LEFT JOIN production.production_plan_lines ppl_comp_any ON (
              ppl_comp_any.source_type = 'order_demand'
              AND (ppl_comp_any.metadata->>'set_id')::int = ps.id
              AND (ppl_comp_any.metadata->>'component_id')::int = comp.id
              AND (ppl_comp_any.metadata->>'order_number') = o.order_number
            )
            LEFT JOIN production.production_plans pp_comp_any ON pp_comp_any.id = ppl_comp_any.plan_id
            WHERE spl.set_id = ps.id
            ORDER BY comp.id, ppl_comp_any.id DESC NULLS LAST
          ) comp_data
        ) set_components ON ps.id IS NOT NULL
        WHERE ${dataWhereClause}
        GROUP BY o.id, o.order_number, o.source, o.buyer_first_name, o.buyer_last_name, 
                 o.buyer_email, o.order_date, o.payment_status, o.total_to_pay_amount, o.total_to_pay_currency
        HAVING COUNT(oi.id) > 0
        ORDER BY ${includeMinSku ? 'min_sku' : orderByColumn} ${orderByDirection}
        LIMIT $${dataParamIndex} OFFSET $${dataParamIndex + 1}
      `, [...dataParams, filters.limit, filters.offset]);

      res.json({
        orders: dataResult.rows,
        total,
        limit: filters.limit,
        offset: filters.offset,
      });
    } catch (error) {
      console.error("Error fetching available orders:", error);
      res.status(500).json({ message: "Failed to fetch available orders" });
    }
  });

  // GET /api/production/planning/plans/:id/available-catalog-products - Get catalog products for internal orders
  app.get("/api/production/planning/plans/:id/available-catalog-products", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const normalizedQuery = Object.fromEntries(
        Object.entries(req.query).map(([key, value]) => [key, value === '' ? undefined : value])
      );

      const filters = z.object({
        search: z.string().optional(),
        color: z.string().optional(),
        sku: z.string().optional(),
        minLength: z.coerce.number().optional(),
        maxLength: z.coerce.number().optional(),
        minWidth: z.coerce.number().optional(),
        maxWidth: z.coerce.number().optional(),
        limit: z.coerce.number().min(1).max(500).optional().default(100),
        offset: z.coerce.number().min(0).optional().default(0),
        sortBy: z.enum(['title', 'sku', 'color', 'created_at']).optional().default('created_at'),
        sortOrder: z.enum(['asc', 'desc']).optional().default('desc'),
      }).parse(normalizedQuery);

      const sortColumnMap: Record<string, string> = {
        'title': 'cp.title',
        'sku': 'cp.sku',
        'color': 'cp.color',
        'created_at': 'cp.created_at',
      };

      const orderByColumn = sortColumnMap[filters.sortBy];
      const orderByDirection = filters.sortOrder.toUpperCase();

      const conditions: string[] = ['cp.is_active = true'];
      const params: any[] = [];
      let paramIndex = 1;

      if (filters.search) {
        conditions.push(`(
          cp.title ILIKE $${paramIndex} OR
          cp.sku ILIKE $${paramIndex} OR
          cp.color ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.search}%`);
        paramIndex++;
      }

      if (filters.color) {
        conditions.push(`cp.color ILIKE $${paramIndex}`);
        params.push(`%${filters.color}%`);
        paramIndex++;
      }

      if (filters.sku) {
        conditions.push(`cp.sku ILIKE $${paramIndex}`);
        params.push(`%${filters.sku}%`);
        paramIndex++;
      }

      if (filters.minLength) {
        conditions.push(`cp.length::numeric >= $${paramIndex}`);
        params.push(filters.minLength);
        paramIndex++;
      }

      if (filters.maxLength) {
        conditions.push(`cp.length::numeric <= $${paramIndex}`);
        params.push(filters.maxLength);
        paramIndex++;
      }

      if (filters.minWidth) {
        conditions.push(`cp.width::numeric >= $${paramIndex}`);
        params.push(filters.minWidth);
        paramIndex++;
      }

      if (filters.maxWidth) {
        conditions.push(`cp.width::numeric <= $${paramIndex}`);
        params.push(filters.maxWidth);
        paramIndex++;
      }

      const whereClause = conditions.join(' AND ');

      const countResult = await pool.query(`
        SELECT COUNT(*)::int as total
        FROM catalog.products cp
        WHERE ${whereClause}
      `, params);

      const total = countResult.rows[0].total;

      const dataParams = [...params, planId];
      let dataParamIndex = dataParams.length + 1;

      const dataResult = await pool.query(`
        SELECT DISTINCT ON (cp.id)
          cp.id,
          cp.sku,
          cp.title,
          cp.color,
          cp.color_options,
          cp.length,
          cp.width,
          cp.height,
          cp.product_type,
          cp.doors,
          cp.legs,
          cp.created_at,
          (
            SELECT pi.url
            FROM catalog.product_images pi
            WHERE pi.product_id = cp.id
            ORDER BY pi.is_primary DESC, pi.sort_order ASC
            LIMIT 1
          ) as image_url,
          COALESCE(pb_comp_count.component_count, 0) as bom_component_count,
          (
            SELECT ppg.id
            FROM production.production_product_group_items ppgi
            LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
            WHERE ppgi.product_id = cp.id
            LIMIT 1
          ) as product_group_id,
          (
            SELECT ppg.name
            FROM production.production_product_group_items ppgi
            LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
            WHERE ppgi.product_id = cp.id
            LIMIT 1
          ) as product_group_name,
          (
            SELECT ppg.color_hex
            FROM production.production_product_group_items ppgi
            LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
            WHERE ppgi.product_id = cp.id
            LIMIT 1
          ) as product_group_color_hex,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as is_in_plan,
          (
            SELECT pp_any.plan_number
            FROM production.production_plan_lines ppl_any
            LEFT JOIN production.production_plans pp_any ON pp_any.id = ppl_any.plan_id
            WHERE ppl_any.source_type = 'catalog_internal'
              AND ppl_any.product_id = cp.id
            LIMIT 1
          ) as in_plan_number,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as in_current_plan
        FROM catalog.products cp
        LEFT JOIN bom.product_boms pb ON pb.product_id = cp.id AND pb.is_active = true
        LEFT JOIN LATERAL (
          SELECT COUNT(*)::int as component_count
          FROM bom.product_components pc
          WHERE pc.product_bom_id = pb.id
        ) pb_comp_count ON true
        LEFT JOIN production.production_plan_lines ppl_current ON (
          ppl_current.plan_id = $${dataParams.length}
          AND ppl_current.source_type = 'catalog_internal'
          AND ppl_current.product_id = cp.id
        )
        WHERE ${whereClause}
        ORDER BY cp.id, ${orderByColumn} ${orderByDirection}
        LIMIT $${dataParamIndex} OFFSET $${dataParamIndex + 1}
      `, [...dataParams, filters.limit, filters.offset]);

      res.json({
        products: dataResult.rows,
        total,
        limit: filters.limit,
        offset: filters.offset,
      });
    } catch (error) {
      console.error("Error fetching catalog products:", error);
      res.status(500).json({ message: "Failed to fetch catalog products" });
    }
  });

  // GET /api/production/planning/plans/:id/available-cutting-patterns - Get cutting patterns for internal orders
  app.get("/api/production/planning/plans/:id/available-cutting-patterns", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const normalizedQuery = Object.fromEntries(
        Object.entries(req.query).map(([key, value]) => [key, value === '' ? undefined : value])
      );

      const filters = z.object({
        search: z.string().optional(),
        status: z.string().optional(),
        limit: z.coerce.number().min(1).max(500).optional().default(100),
        offset: z.coerce.number().min(0).optional().default(0),
        sortBy: z.enum(['name', 'code', 'created_at']).optional().default('created_at'),
        sortOrder: z.enum(['asc', 'desc']).optional().default('desc'),
      }).parse(normalizedQuery);

      const sortColumnMap: Record<string, string> = {
        'name': 'cpt.name',
        'code': 'cpt.code',
        'created_at': 'cpt.created_at',
      };

      const orderByColumn = sortColumnMap[filters.sortBy];
      const orderByDirection = filters.sortOrder.toUpperCase();

      const conditions: string[] = ['cpt.is_active = true'];
      const params: any[] = [];
      let paramIndex = 1;

      if (filters.search) {
        conditions.push(`(
          cpt.name ILIKE $${paramIndex} OR
          cpt.code ILIKE $${paramIndex} OR
          cpt.description ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.search}%`);
        paramIndex++;
      }

      const whereClause = conditions.join(' AND ');

      const countResult = await pool.query(`
        SELECT COUNT(*)::int as total
        FROM production.cut_pattern_templates cpt
        WHERE ${whereClause}
      `, params);

      const total = countResult.rows[0].total;

      const dataParams = [...params, planId];
      let dataParamIndex = dataParams.length + 1;

      const dataResult = await pool.query(`
        SELECT 
          cpt.id,
          cpt.code,
          cpt.name,
          cpt.description,
          cpt.is_active,
          cpt.board_length,
          cpt.board_width,
          cpt.board_thickness,
          cpt.kerf,
          cpt.created_at,
          COALESCE(SUM(cpti.quantity_default), 0)::int as total_quantity,
          COUNT(cpti.id)::int as items_count,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as is_in_plan,
          pp_any.plan_number as in_plan_number,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as in_current_plan
        FROM production.cut_pattern_templates cpt
        LEFT JOIN production.cut_pattern_template_items cpti ON cpti.template_id = cpt.id
        LEFT JOIN production.production_plan_lines ppl_current ON (
          ppl_current.plan_id = $${dataParams.length}
          AND ppl_current.source_type = 'cutting_pattern'
          AND (ppl_current.metadata->>'cutting_pattern_id')::int = cpt.id
        )
        LEFT JOIN production.production_plan_lines ppl_any ON (
          ppl_any.source_type = 'cutting_pattern'
          AND (ppl_any.metadata->>'cutting_pattern_id')::int = cpt.id
        )
        LEFT JOIN production.production_plans pp_any ON pp_any.id = ppl_any.plan_id
        WHERE ${whereClause}
        GROUP BY cpt.id, cpt.code, cpt.name, cpt.description, cpt.is_active, 
                 cpt.board_length, cpt.board_width, cpt.board_thickness, cpt.kerf, 
                 cpt.created_at, ppl_current.id, pp_any.plan_number
        ORDER BY ${orderByColumn} ${orderByDirection}
        LIMIT $${dataParamIndex} OFFSET $${dataParamIndex + 1}
      `, [...dataParams, filters.limit, filters.offset]);

      res.json({
        patterns: dataResult.rows,
        total,
        limit: filters.limit,
        offset: filters.offset,
      });
    } catch (error) {
      console.error("Error fetching cutting patterns:", error);
      res.status(500).json({ message: "Failed to fetch cutting patterns" });
    }
  });
}
