import axios from 'axios';
import * as fs from 'fs/promises';
import * as path from 'path';
import { pool } from './postgres.js';

async function getAccessToken(): Promise<string | null> {
  const result = await pool.query(`
    SELECT access_token FROM allegro_connections WHERE is_active = true LIMIT 1
  `);
  return result.rows[0]?.access_token || null;
}

async function downloadFromAllegroOffer(offerId: string, externalId: string, accessToken: string): Promise<string | null> {
  try {
    console.log(`📡 Pobieranie informacji o ofercie ${offerId}...`);
    
    const response = await axios.get(
      `https://api.allegro.pl/sale/product-offers/${offerId}`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        }
      }
    );

    const images = response.data?.images || [];
    if (images.length === 0) {
      console.log(`⚠️ Brak zdjęć dla oferty ${offerId}`);
      return null;
    }

    const imageUrl = images[0].url;
    console.log(`📥 Pobieranie zdjęcia: ${imageUrl}`);

    const imageResponse = await axios.get(imageUrl, {
      responseType: 'arraybuffer',
      timeout: 10000,
    });

    const productsDir = path.join(process.cwd(), 'attached_assets', 'products');
    await fs.mkdir(productsDir, { recursive: true });

    const filename = `${externalId}_1.jpg`;
    const filepath = path.join(productsDir, filename);

    await fs.writeFile(filepath, imageResponse.data);
    console.log(`✅ Zapisano: ${filename}`);

    return `/api/products/images/${filename}`;
  } catch (error: any) {
    console.error(`❌ Błąd dla oferty ${offerId}:`, error.message);
    return null;
  }
}

async function downloadMissingImages() {
  try {
    console.log('🔄 Pobieranie brakujących obrazków z Allegro...\n');

    const accessToken = await getAccessToken();
    if (!accessToken) {
      console.error('❌ Brak access token dla Allegro');
      return;
    }

    // Pobierz produkty bez zdjęć
    const result = await pool.query(`
      SELECT DISTINCT
        oi.offer_external_id,
        o.raw_payload->'lineItems'->0->'offer'->>'id' as offer_id,
        oi.name
      FROM commerce.order_items oi
      JOIN commerce.orders o ON oi.order_id = o.id
      LEFT JOIN commerce.marketplace_products mp 
        ON oi.offer_external_id = mp.offer_external_id 
        AND mp.source = o.source
      WHERE (mp.image_url IS NULL OR mp.image_url = '')
        AND o.source = 'ALLEGRO'
        AND o.raw_payload->'lineItems'->0->'offer'->>'id' IS NOT NULL
      LIMIT 10
    `);

    console.log(`📊 Znaleziono ${result.rows.length} produktów bez zdjęć\n`);

    let success = 0;
    let failed = 0;

    for (const product of result.rows) {
      const localPath = await downloadFromAllegroOffer(
        product.offer_id,
        product.offer_external_id,
        accessToken
      );

      if (localPath) {
        // Zaktualizuj marketplace_products
        await pool.query(`
          UPDATE commerce.marketplace_products
          SET image_url = $1, updated_at = NOW()
          WHERE offer_external_id = $2
        `, [localPath, product.offer_external_id]);

        // Zaktualizuj order_items
        await pool.query(`
          UPDATE commerce.order_items
          SET image_url = $1
          WHERE offer_external_id = $2
        `, [localPath, product.offer_external_id]);

        console.log(`✅ Zaktualizowano bazę dla ${product.name}\n`);
        success++;
      } else {
        failed++;
      }

      await new Promise(resolve => setTimeout(resolve, 300));
    }

    console.log(`\n✅ Zakończono: ${success} sukces, ${failed} niepowodzeń`);

  } catch (error) {
    console.error('❌ Błąd:', error);
  } finally {
    await pool.end();
  }
}

downloadMissingImages();
