import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useToast } from "@/hooks/use-toast";
import { Search, Plus, Edit, Trash2, Loader2, Package, ExternalLink, ArrowRight, FileCheck, Truck, MapPin } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogFooter } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Link, useLocation } from "wouter";

interface StockPanel {
  id: number;
  generatedName: string;
  cz1?: string | null;
  cz2?: string | null;
  furnitureType?: string | null;
  length: string;
  width: string;
  thickness: string;
  boardCode?: string | null;
  edgingCode?: string | null;
  colorCode?: string | null;
  colorName?: string | null;
  colorHex?: string | null;
  edge1: boolean;
  edge2: boolean;
  edge3: boolean;
  edge4: boolean;
  isDrilled: boolean;
  isEdged: boolean;
  source: string;
  sourceReference?: string | null;
  quantity: number;
  location?: string | null;
  notes?: string | null;
  imageUrl?: string | null;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

interface StockPanelsResponse {
  data: StockPanel[];
  pagination: {
    total: number;
    page: number;
    limit: number;
    totalPages: number;
  };
}

const SOURCE_LABELS: Record<string, string> = {
  stock: "Zapas",
  production_surplus: "Nadmiar produkcyjny",
  returns: "Zwroty"
};

export default function FormatkiPage() {
  const { toast } = useToast();
  const [, setLocation] = useLocation();
  
  const [searchQuery, setSearchQuery] = useState("");
  const [page, setPage] = useState(1);
  const [limit, setLimit] = useState(25);
  const [sourceFilter, setSourceFilter] = useState<string>("all");
  const [isDrilledFilter, setIsDrilledFilter] = useState<string>("all");
  
  // Selection & Bulk Operations
  const [selectedPanels, setSelectedPanels] = useState<number[]>([]);
  const [showBulkCarrierDialog, setShowBulkCarrierDialog] = useState(false);
  const [showBulkLocationDialog, setShowBulkLocationDialog] = useState(false);
  const [showBulkInventoryDialog, setShowBulkInventoryDialog] = useState(false);
  
  // Dialog states
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [editingPanel, setEditingPanel] = useState<StockPanel | null>(null);
  
  // Form states
  const [formData, setFormData] = useState({
    generatedName: "",
    cz1: "",
    cz2: "",
    furnitureType: "",
    length: "",
    width: "",
    thickness: "18",
    boardCode: "",
    edgingCode: "",
    colorCode: "",
    edge1: false,
    edge2: false,
    edge3: false,
    edge4: false,
    isDrilled: false,
    isEdged: false,
    source: "stock",
    sourceReference: "",
    quantity: "0",
    location: "",
    notes: ""
  });

  // Build query parameters as object for cache segmentation
  const queryParams = {
    page: page.toString(),
    limit: limit.toString(),
    sortBy: 'created_at',
    sortOrder: 'DESC',
    ...(searchQuery && { search: searchQuery }),
    ...(sourceFilter && sourceFilter !== 'all' && { source: sourceFilter }),
    ...(isDrilledFilter && isDrilledFilter !== 'all' && { isDrilled: isDrilledFilter })
  };

  const { data, isLoading } = useQuery<StockPanelsResponse>({
    queryKey: ['/api/warehouse/stock-panels', queryParams],
  });

  // Auto-reset selection when filters change
  useEffect(() => {
    setSelectedPanels([]);
  }, [searchQuery, page, sourceFilter, isDrilledFilter]);

  // Queries for bulk operations
  const { data: carriersData } = useQuery({
    queryKey: ['/api/warehouse/carriers'],
    enabled: showBulkCarrierDialog,
  });

  const { data: locationsData } = useQuery({
    queryKey: ['/api/warehouse/locations'],
    enabled: showBulkLocationDialog,
  });

  // Bulk update carrier mutation
  const updateCarrierMutation = useMutation({
    mutationFn: async (carrierId: number | null) => {
      return apiRequest("POST", "/api/warehouse/materials/bulk-update", {
        materialIds: selectedPanels,
        updates: {
          carrier_id: carrierId,
        },
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/stock-panels'] });
      toast({
        title: "Przewoźnik zaktualizowany",
        description: `Pomyślnie zaktualizowano przewoźnika dla ${selectedPanels.length} formatek.`,
      });
      setSelectedPanels([]);
      setShowBulkCarrierDialog(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować przewoźnika.",
        variant: "destructive",
      });
    },
  });

  // Bulk update location mutation
  const updateLocationMutation = useMutation({
    mutationFn: async (locationId: number | null) => {
      return apiRequest("POST", "/api/warehouse/materials/bulk-update", {
        materialIds: selectedPanels,
        updates: {
          location_id: locationId,
        },
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/stock-panels'] });
      toast({
        title: "Lokalizacja zaktualizowana",
        description: `Pomyślnie zaktualizowano lokalizację dla ${selectedPanels.length} formatek.`,
      });
      setSelectedPanels([]);
      setShowBulkLocationDialog(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować lokalizacji.",
        variant: "destructive",
      });
    },
  });

  // Create inventory count mutation
  const createInventoryCountMutation = useMutation({
    mutationFn: async (data: { name: string; notes?: string }) => {
      const res = await apiRequest("POST", "/api/warehouse/inventory-counts", {
        name: data.name,
        materialIds: selectedPanels,
        notes: data.notes,
      });
      return await res.json();
    },
    onSuccess: (response) => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      toast({
        title: "Spis utworzony",
        description: `Utworzono spis inwentaryzacyjny z ${selectedPanels.length} formatkami.`,
      });
      setSelectedPanels([]);
      setShowBulkInventoryDialog(false);
      if (response.inventoryCount?.id) {
        setLocation(`/warehouse/inventory-counts/${response.inventoryCount.id}`);
      }
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia spisu",
        description: error.message || "Nie udało się utworzyć spisu inwentaryzacyjnego.",
        variant: "destructive",
      });
    },
  });

  const createMutation = useMutation({
    mutationFn: async (data: any) => {
      return apiRequest("POST", "/api/warehouse/stock-panels", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/stock-panels'] });
      toast({ description: "Formatka została dodana" });
      setIsAddDialogOpen(false);
      resetForm();
    },
    onError: (error: any) => {
      toast({
        variant: "destructive",
        description: error.message || "Nie udało się dodać formatki",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: any }) => {
      return apiRequest("PATCH", `/api/warehouse/stock-panels/${id}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/stock-panels'] });
      toast({ description: "Formatka została zaktualizowana" });
      setIsEditDialogOpen(false);
      setEditingPanel(null);
      resetForm();
    },
    onError: (error: any) => {
      toast({
        variant: "destructive",
        description: error.message || "Nie udało się zaktualizować formatki",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/warehouse/stock-panels/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/stock-panels'] });
      toast({ description: "Formatka została usunięta" });
    },
    onError: (error: any) => {
      toast({
        variant: "destructive",
        description: error.message || "Nie udało się usunąć formatki",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      generatedName: "",
      cz1: "",
      cz2: "",
      furnitureType: "",
      length: "",
      width: "",
      thickness: "18",
      boardCode: "",
      edgingCode: "",
      colorCode: "",
      edge1: false,
      edge2: false,
      edge3: false,
      edge4: false,
      isDrilled: false,
      isEdged: false,
      source: "stock",
      sourceReference: "",
      quantity: "0",
      location: "",
      notes: ""
    });
  };

  // Auto-generate name based on params
  const generateName = () => {
    const { isDrilled, cz1, cz2, furnitureType, length, width, colorCode } = formData;
    
    if (isDrilled) {
      // Wiercone: "LISTWA-T-VB-462x70-KASZMIR" lub "BOCZKI-SZUFLADA-100x260-CZARNY"
      const parts = [];
      if (cz1) parts.push(cz1);
      if (cz2) parts.push(cz2);
      if (furnitureType) parts.push(furnitureType);
      if (length && width) parts.push(`${length}x${width}`);
      if (colorCode) parts.push(colorCode);
      return parts.join('-');
    } else {
      // Niewierscone: "500x300-BIALY"
      const parts = [];
      if (length && width) parts.push(`${length}x${width}`);
      if (colorCode) parts.push(colorCode);
      return parts.join('-');
    }
  };

  const handleEdit = (panel: StockPanel) => {
    setEditingPanel(panel);
    setFormData({
      generatedName: panel.generatedName,
      cz1: panel.cz1 || "",
      cz2: panel.cz2 || "",
      furnitureType: panel.furnitureType || "",
      length: panel.length,
      width: panel.width,
      thickness: panel.thickness,
      boardCode: panel.boardCode || "",
      edgingCode: panel.edgingCode || "",
      colorCode: panel.colorCode || "",
      edge1: panel.edge1,
      edge2: panel.edge2,
      edge3: panel.edge3,
      edge4: panel.edge4,
      isDrilled: panel.isDrilled,
      isEdged: panel.isEdged,
      source: panel.source,
      sourceReference: panel.sourceReference || "",
      quantity: panel.quantity.toString(),
      location: panel.location || "",
      notes: panel.notes || ""
    });
    setIsEditDialogOpen(true);
  };

  const handleSubmit = () => {
    if (!formData.generatedName || !formData.length || !formData.width) {
      toast({
        variant: "destructive",
        description: "Wypełnij wymagane pola: nazwa, długość, szerokość"
      });
      return;
    }

    const submitData = {
      ...formData,
      length: parseFloat(formData.length),
      width: parseFloat(formData.width),
      thickness: parseFloat(formData.thickness),
      quantity: parseInt(formData.quantity)
    };

    if (editingPanel) {
      updateMutation.mutate({ id: editingPanel.id, data: submitData });
    } else {
      createMutation.mutate(submitData);
    }
  };

  const panels = data?.data || [];
  const pagination = data?.pagination;

  return (
    <WarehouseLayout category="formatki">
      <div className="p-6 space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold">Formatki</h1>
            <p className="text-muted-foreground mt-1">
              Magazyn gotowych komponentów meblowych
            </p>
          </div>
          <Button onClick={() => setIsAddDialogOpen(true)} data-testid="button-add-panel">
            <Plus className="w-4 h-4 mr-2" />
            Dodaj formatkę
          </Button>
        </div>

        {/* Filters Bar - 50px height */}
        <div className="flex items-center gap-3 h-[50px] px-4 bg-muted/30 border-b">
          <div className="flex-1 relative">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
            <Input
              placeholder="Szukaj formatek..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-9 h-9"
              data-testid="input-search"
            />
          </div>
          <Select value={sourceFilter} onValueChange={setSourceFilter}>
            <SelectTrigger className="w-48 h-9" data-testid="select-source">
              <SelectValue placeholder="Wszystkie źródła" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">Wszystkie źródła</SelectItem>
              <SelectItem value="stock">Zapas</SelectItem>
              <SelectItem value="production_surplus">Nadmiar produkcyjny</SelectItem>
              <SelectItem value="returns">Zwroty</SelectItem>
            </SelectContent>
          </Select>
          <Select value={isDrilledFilter} onValueChange={setIsDrilledFilter}>
            <SelectTrigger className="w-48 h-9" data-testid="select-drilled">
              <SelectValue placeholder="Wszystkie" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">Wszystkie</SelectItem>
              <SelectItem value="true">Tylko wiercone</SelectItem>
              <SelectItem value="false">Tylko oklejone</SelectItem>
            </SelectContent>
          </Select>
        </div>

        {/* Bulk Operations Bar */}
        {selectedPanels.length > 0 && (
          <div className="flex items-center justify-between px-4 py-2 bg-primary/10 border-b">
            <span className="text-sm font-medium">
              Zaznaczono {selectedPanels.length} {selectedPanels.length === 1 ? 'formatkę' : 'formatek'}
            </span>
            <div className="flex items-center gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setShowBulkCarrierDialog(true)}
                data-testid="button-bulk-carrier"
              >
                <Truck className="w-4 h-4 mr-2" />
                Zmień przewoźnika
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setShowBulkLocationDialog(true)}
                data-testid="button-bulk-location"
              >
                <MapPin className="w-4 h-4 mr-2" />
                Zmień lokalizację
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setShowBulkInventoryDialog(true)}
                data-testid="button-bulk-inventory"
              >
                <FileCheck className="w-4 h-4 mr-2" />
                Utwórz spis
              </Button>
            </div>
          </div>
        )}

        {/* Compact ERP List */}
        <div className="flex-1 overflow-auto">
          <div className="p-4">
            {isLoading ? (
              <div className="flex items-center justify-center py-8">
                <Loader2 className="w-6 h-6 animate-spin" />
              </div>
            ) : panels.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                <Package className="w-12 h-12 mx-auto mb-4 opacity-50" />
                <p>Brak formatek</p>
              </div>
            ) : (
              <div>
                {/* Table Header */}
                <div className="flex items-center gap-2 h-[32px] px-3 bg-muted/40 border-b text-xs font-medium text-muted-foreground">
                  <div className="w-[40px] shrink-0 flex items-center justify-center">
                    <Checkbox
                      checked={selectedPanels.length === panels.length && panels.length > 0}
                      onCheckedChange={(checked) => {
                        if (checked) {
                          setSelectedPanels(panels.map(p => p.id));
                        } else {
                          setSelectedPanels([]);
                        }
                      }}
                      data-testid="checkbox-select-all"
                    />
                  </div>
                  <div className="w-[200px] shrink-0">Nazwa</div>
                  <div className="w-[80px] shrink-0">Długość</div>
                  <div className="w-[80px] shrink-0">Szerokość</div>
                  <div className="w-[60px] shrink-0">Grubość</div>
                  <div className="w-[100px] shrink-0">CZ1</div>
                  <div className="w-[100px] shrink-0">CZ2</div>
                  <div className="w-[100px] shrink-0">Typ mebla</div>
                  <div className="w-[120px] shrink-0">Kolor</div>
                  <div className="w-[80px] shrink-0">Płyta</div>
                  <div className="w-[80px] shrink-0">Obrzeże</div>
                  <div className="w-[100px] shrink-0">Krawędzie</div>
                  <div className="w-[100px] shrink-0">Status</div>
                  <div className="w-[120px] shrink-0">Źródło</div>
                  <div className="w-[150px] shrink-0">Referencja</div>
                  <div className="w-[60px] shrink-0 text-right">Ilość</div>
                  <div className="w-[100px] shrink-0">Lokalizacja</div>
                  <div className="w-[200px] shrink-0">Uwagi</div>
                  <div className="w-[50px] shrink-0"></div>
                </div>
                
                {/* Table Rows */}
                <div className="space-y-1">
                  {panels.map((panel) => (
                    <div
                      key={panel.id}
                      className="flex items-center gap-2 h-[40px] px-3 bg-muted/20 border"
                      data-testid={`panel-item-${panel.id}`}
                    >
                      {/* Checkbox */}
                      <div className="w-[40px] shrink-0 flex items-center justify-center">
                        <Checkbox
                          checked={selectedPanels.includes(panel.id)}
                          onCheckedChange={(checked) => {
                            if (checked) {
                              setSelectedPanels([...selectedPanels, panel.id]);
                            } else {
                              setSelectedPanels(selectedPanels.filter(id => id !== panel.id));
                            }
                          }}
                          data-testid={`checkbox-panel-${panel.id}`}
                        />
                      </div>

                      {/* Name */}
                      <div className="w-[200px] shrink-0 flex items-center gap-2 min-w-0">
                        <span className="font-medium text-sm truncate">{panel.generatedName}</span>
                      </div>
                        
                        {/* Length */}
                        <div className="w-[80px] shrink-0 text-xs">{panel.length}</div>
                        
                        {/* Width */}
                        <div className="w-[80px] shrink-0 text-xs">{panel.width}</div>
                        
                        {/* Thickness */}
                        <div className="w-[60px] shrink-0 text-xs">{panel.thickness}</div>
                        
                        {/* CZ1 */}
                        <div className="w-[100px] shrink-0 text-xs truncate text-muted-foreground">{panel.cz1 || '-'}</div>
                        
                        {/* CZ2 */}
                        <div className="w-[100px] shrink-0 text-xs truncate text-muted-foreground">{panel.cz2 || '-'}</div>
                        
                        {/* Furniture Type */}
                        <div className="w-[100px] shrink-0 text-xs truncate text-muted-foreground">{panel.furnitureType || '-'}</div>
                        
                        {/* Color */}
                        <div className="w-[120px] shrink-0 flex items-center gap-1 min-w-0">
                          {panel.colorName ? (
                            <>
                              <div
                                className="w-3 h-3 rounded-full border shrink-0"
                                style={{ backgroundColor: panel.colorHex || '#888' }}
                              />
                              <span className="text-xs truncate">{panel.colorName}</span>
                            </>
                          ) : (
                            <span className="text-xs text-muted-foreground">-</span>
                          )}
                        </div>
                        
                        {/* Board Code */}
                        <div className="w-[80px] shrink-0 text-xs truncate text-muted-foreground">{panel.boardCode || '-'}</div>
                        
                        {/* Edging Code */}
                        <div className="w-[80px] shrink-0 text-xs truncate text-muted-foreground">{panel.edgingCode || '-'}</div>
                        
                        {/* Edges */}
                        <div className="w-[100px] shrink-0 flex gap-1">
                          {panel.edge1 && <Badge variant="outline" className="h-4 text-[9px] px-1">K1</Badge>}
                          {panel.edge2 && <Badge variant="outline" className="h-4 text-[9px] px-1">K2</Badge>}
                          {panel.edge3 && <Badge variant="outline" className="h-4 text-[9px] px-1">K3</Badge>}
                          {panel.edge4 && <Badge variant="outline" className="h-4 text-[9px] px-1">K4</Badge>}
                          {!panel.edge1 && !panel.edge2 && !panel.edge3 && !panel.edge4 && (
                            <span className="text-xs text-muted-foreground">-</span>
                          )}
                        </div>
                        
                        {/* Status */}
                        <div className="w-[100px] shrink-0 flex gap-1">
                          {panel.isEdged && <Badge variant="secondary" className="h-5 text-[10px]">Okl.</Badge>}
                          {panel.isDrilled && <Badge variant="secondary" className="h-5 text-[10px]">Wier.</Badge>}
                        </div>
                        
                        {/* Source */}
                        <div className="w-[120px] shrink-0 text-xs truncate">{SOURCE_LABELS[panel.source] || panel.source}</div>
                        
                        {/* Source Reference */}
                        <div className="w-[150px] shrink-0 text-xs truncate text-muted-foreground">{panel.sourceReference || '-'}</div>
                        
                        {/* Quantity */}
                        <div className="w-[60px] shrink-0 text-sm font-medium text-right">{panel.quantity}</div>
                        
                        {/* Location */}
                        <div className="w-[100px] shrink-0 text-xs truncate text-muted-foreground">{panel.location || '-'}</div>
                        
                        {/* Notes */}
                        <div className="w-[200px] shrink-0 text-xs truncate text-muted-foreground" title={panel.notes || ''}>
                          {panel.notes || '-'}
                        </div>

                        {/* Navigation Button */}
                        <div className="w-[50px] shrink-0 flex items-center justify-center">
                          <Link href={`/warehouse/formatki/${panel.id}`}>
                            <Button
                              variant="ghost"
                              size="icon"
                              className="h-7 w-7"
                              data-testid={`button-nav-panel-${panel.id}`}
                            >
                              <ArrowRight className="h-4 w-4" />
                            </Button>
                          </Link>
                        </div>
                      </div>
                  ))}
                </div>
              </div>
            )}

            {/* Pagination */}
            {pagination && pagination.totalPages > 1 && (
              <div className="flex items-center justify-between mt-4 pt-4 border-t">
                <div className="text-sm text-muted-foreground">
                  Strona {pagination.page} z {pagination.totalPages} ({pagination.total} formatek)
                </div>
                <div className="flex gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setPage(Math.max(1, page - 1))}
                    disabled={page === 1}
                    data-testid="button-prev-page"
                  >
                    Poprzednia
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setPage(Math.min(pagination.totalPages, page + 1))}
                    disabled={page === pagination.totalPages}
                    data-testid="button-next-page"
                  >
                    Następna
                  </Button>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Add/Edit Dialog */}
      <Dialog open={isAddDialogOpen || isEditDialogOpen} onOpenChange={(open) => {
        if (!open) {
          setIsAddDialogOpen(false);
          setIsEditDialogOpen(false);
          setEditingPanel(null);
          resetForm();
        }
      }}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>
              {editingPanel ? "Edytuj formatkę" : "Dodaj formatkę"}
            </DialogTitle>
            <DialogDescription>
              {formData.isDrilled
                ? "Wiercone formatki wymagają pełnej nazwy z przeznaczeniem"
                : "Niewierscone formatki mogą mieć prostą nazwę wymiary-kolor"}
            </DialogDescription>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            <div className="grid gap-2">
              <div className="flex items-center justify-between">
                <Label htmlFor="generatedName">Nazwa formatki *</Label>
                <Button
                  type="button"
                  variant="outline"
                  size="sm"
                  onClick={() => {
                    const generated = generateName();
                    if (generated) {
                      setFormData({ ...formData, generatedName: generated });
                    }
                  }}
                  data-testid="button-generate-name"
                >
                  Wygeneruj nazwę
                </Button>
              </div>
              <Input
                id="generatedName"
                value={formData.generatedName}
                onChange={(e) => setFormData({ ...formData, generatedName: e.target.value })}
                placeholder={formData.isDrilled ? "LISTWA-T-VB-462x70-KASZMIR" : "500x300-BIALY"}
                data-testid="input-name"
              />
            </div>

            <div className="grid grid-cols-3 gap-4">
              <div className="grid gap-2">
                <Label htmlFor="length">Długość (mm) *</Label>
                <Input
                  id="length"
                  type="number"
                  step="0.01"
                  value={formData.length}
                  onChange={(e) => setFormData({ ...formData, length: e.target.value })}
                  data-testid="input-length"
                />
              </div>
              <div className="grid gap-2">
                <Label htmlFor="width">Szerokość (mm) *</Label>
                <Input
                  id="width"
                  type="number"
                  step="0.01"
                  value={formData.width}
                  onChange={(e) => setFormData({ ...formData, width: e.target.value })}
                  data-testid="input-width"
                />
              </div>
              <div className="grid gap-2">
                <Label htmlFor="thickness">Grubość (mm)</Label>
                <Input
                  id="thickness"
                  type="number"
                  step="0.01"
                  value={formData.thickness}
                  onChange={(e) => setFormData({ ...formData, thickness: e.target.value })}
                  data-testid="input-thickness"
                />
              </div>
            </div>

            {formData.isDrilled && (
              <div className="grid grid-cols-3 gap-4">
                <div className="grid gap-2">
                  <Label htmlFor="cz1">Część 1</Label>
                  <Input
                    id="cz1"
                    value={formData.cz1}
                    onChange={(e) => setFormData({ ...formData, cz1: e.target.value })}
                    placeholder="LISTWA-T"
                    data-testid="input-cz1"
                  />
                </div>
                <div className="grid gap-2">
                  <Label htmlFor="cz2">Część 2</Label>
                  <Input
                    id="cz2"
                    value={formData.cz2}
                    onChange={(e) => setFormData({ ...formData, cz2: e.target.value })}
                    placeholder="VB"
                    data-testid="input-cz2"
                  />
                </div>
                <div className="grid gap-2">
                  <Label htmlFor="furnitureType">Rodzaj mebla</Label>
                  <Input
                    id="furnitureType"
                    value={formData.furnitureType}
                    onChange={(e) => setFormData({ ...formData, furnitureType: e.target.value })}
                    placeholder="VB"
                    data-testid="input-furniture-type"
                  />
                </div>
              </div>
            )}

            <div className="grid grid-cols-2 gap-4">
              <div className="flex items-center space-x-2">
                <Checkbox
                  id="isEdged"
                  checked={formData.isEdged}
                  onCheckedChange={(checked) => setFormData({ ...formData, isEdged: checked as boolean })}
                  data-testid="checkbox-edged"
                />
                <Label htmlFor="isEdged" className="cursor-pointer">Oklejone</Label>
              </div>
              <div className="flex items-center space-x-2">
                <Checkbox
                  id="isDrilled"
                  checked={formData.isDrilled}
                  onCheckedChange={(checked) => setFormData({ ...formData, isDrilled: checked as boolean })}
                  data-testid="checkbox-drilled"
                />
                <Label htmlFor="isDrilled" className="cursor-pointer">Wiercone</Label>
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div className="grid gap-2">
                <Label htmlFor="source">Źródło</Label>
                <Select value={formData.source} onValueChange={(value) => setFormData({ ...formData, source: value })}>
                  <SelectTrigger data-testid="select-source-form">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="stock">Zapas</SelectItem>
                    <SelectItem value="production_surplus">Nadmiar produkcyjny</SelectItem>
                    <SelectItem value="returns">Zwroty</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="grid gap-2">
                <Label htmlFor="quantity">Ilość (szt)</Label>
                <Input
                  id="quantity"
                  type="number"
                  value={formData.quantity}
                  onChange={(e) => setFormData({ ...formData, quantity: e.target.value })}
                  data-testid="input-quantity"
                />
              </div>
            </div>

            <div className="grid gap-2">
              <Label htmlFor="location">Lokalizacja w magazynie</Label>
              <Input
                id="location"
                value={formData.location}
                onChange={(e) => setFormData({ ...formData, location: e.target.value })}
                placeholder="np. A-12-3"
                data-testid="input-location"
              />
            </div>

            <div className="grid gap-2">
              <Label htmlFor="notes">Uwagi</Label>
              <Textarea
                id="notes"
                value={formData.notes}
                onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                rows={3}
                data-testid="textarea-notes"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setIsAddDialogOpen(false);
                setIsEditDialogOpen(false);
                setEditingPanel(null);
                resetForm();
              }}
              data-testid="button-cancel"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={createMutation.isPending || updateMutation.isPending}
              data-testid="button-submit"
            >
              {(createMutation.isPending || updateMutation.isPending) && (
                <Loader2 className="w-4 h-4 mr-2 animate-spin" />
              )}
              {editingPanel ? "Zapisz" : "Dodaj"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Carrier Dialog */}
      <Dialog open={showBulkCarrierDialog} onOpenChange={setShowBulkCarrierDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Zmień przewoźnika ({selectedPanels.length} formatek)</DialogTitle>
            <DialogDescription>
              Wybierz nowego przewoźnika dla zaznaczonych formatek
            </DialogDescription>
          </DialogHeader>
          <div className="py-4">
            <Label>Przewoźnik</Label>
            <Select
              onValueChange={(value) => {
                if (value !== "null") {
                  const parsed = parseInt(value);
                  if (!isNaN(parsed)) {
                    updateCarrierMutation.mutate(parsed);
                  }
                } else {
                  updateCarrierMutation.mutate(null);
                }
              }}
              disabled={updateCarrierMutation.isPending}
            >
              <SelectTrigger data-testid="select-bulk-carrier">
                <SelectValue placeholder="Wybierz przewoźnika" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="null">Brak przewoźnika</SelectItem>
                {(carriersData as any)?.map((carrier: any) => (
                  <SelectItem key={carrier.id} value={carrier.id.toString()}>
                    {carrier.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setShowBulkCarrierDialog(false)}
              disabled={updateCarrierMutation.isPending}
            >
              Anuluj
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Location Dialog */}
      <Dialog open={showBulkLocationDialog} onOpenChange={setShowBulkLocationDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Zmień lokalizację ({selectedPanels.length} formatek)</DialogTitle>
            <DialogDescription>
              Wybierz nową lokalizację dla zaznaczonych formatek
            </DialogDescription>
          </DialogHeader>
          <div className="py-4">
            <Label>Lokalizacja</Label>
            <Select
              onValueChange={(value) => {
                if (value !== "null") {
                  const parsed = parseInt(value);
                  if (!isNaN(parsed)) {
                    updateLocationMutation.mutate(parsed);
                  }
                } else {
                  updateLocationMutation.mutate(null);
                }
              }}
              disabled={updateLocationMutation.isPending}
            >
              <SelectTrigger data-testid="select-bulk-location">
                <SelectValue placeholder="Wybierz lokalizację" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="null">Brak lokalizacji</SelectItem>
                {(locationsData as any)?.map((location: any) => (
                  <SelectItem key={location.id} value={location.id.toString()}>
                    {location.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setShowBulkLocationDialog(false)}
              disabled={updateLocationMutation.isPending}
            >
              Anuluj
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Inventory Count Dialog */}
      <Dialog open={showBulkInventoryDialog} onOpenChange={setShowBulkInventoryDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Utwórz spis inwentaryzacyjny ({selectedPanels.length} formatek)</DialogTitle>
            <DialogDescription>
              Wprowadź nazwę dla nowego spisu inwentaryzacyjnego
            </DialogDescription>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            <div className="grid gap-2">
              <Label htmlFor="inventory-name">Nazwa spisu</Label>
              <Input
                id="inventory-name"
                defaultValue={(() => {
                  const today = new Date();
                  const dd = String(today.getDate()).padStart(2, '0');
                  const mm = String(today.getMonth() + 1).padStart(2, '0');
                  const yyyy = today.getFullYear();
                  return `Spis inwentaryzacyjny ${dd}-${mm}-${yyyy} - formatki`;
                })()}
                data-testid="input-inventory-name"
              />
            </div>
            <div className="grid gap-2">
              <Label htmlFor="inventory-notes">Uwagi (opcjonalne)</Label>
              <Textarea
                id="inventory-notes"
                placeholder="Dodatkowe informacje..."
                data-testid="textarea-inventory-notes"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setShowBulkInventoryDialog(false)}
              disabled={createInventoryCountMutation.isPending}
            >
              Anuluj
            </Button>
            <Button
              onClick={() => {
                const nameInput = document.getElementById('inventory-name') as HTMLInputElement;
                const notesInput = document.getElementById('inventory-notes') as HTMLTextAreaElement;
                createInventoryCountMutation.mutate({
                  name: nameInput?.value || '',
                  notes: notesInput?.value || undefined,
                });
              }}
              disabled={createInventoryCountMutation.isPending}
              data-testid="button-create-inventory"
            >
              {createInventoryCountMutation.isPending && (
                <Loader2 className="w-4 h-4 mr-2 animate-spin" />
              )}
              Utwórz spis
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </WarehouseLayout>
  );
}
