import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ArrowLeft, Package, FileText, History, ExternalLink, Trash2, Plus, MapPin, Building2, Truck } from "lucide-react";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { useState } from "react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";

interface Material {
  id: number;
  name: string;
  internalCode: string;
  supplierCode?: string;
  description?: string;
  specifications?: any;
  unitOfMeasure?: string;
  price?: number;
  quantity?: string;
  gallery?: string[];
  primaryImageUrl?: string;
  isActive: boolean;
  groupName?: string;
  categoryName?: string;
  locationId?: number;
  carrierId?: number;
  locationName?: string;
  carrierName?: string;
}

interface Location {
  id: number;
  name: string;
  level: number;
  parentId: number | null;
}

interface Carrier {
  id: number;
  name: string;
  type: string;
}

interface MaterialUsage {
  id: number;
  productionOrderId?: number;
  productionPlanId?: number;
  quantityUsed: number;
  usedAt: string;
  notes?: string;
}

interface InventoryHistory {
  id: number;
  materialId: number;
  operationType: string;
  quantityChange: string;
  quantityBefore: string;
  quantityAfter: string;
  productionOrderId?: number;
  productionPlanId?: number;
  notes?: string;
  documentNumber?: string;
  performedBy?: string;
  createdAt: string;
}

export default function WarehouseMaterialDetail() {
  const params = useParams<{ id: string }>();
  const [location, setLocation] = useLocation();
  const { toast } = useToast();
  const materialId = parseInt(params.id || "0");
  
  // Inventory adjustment dialog state
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [operationType, setOperationType] = useState<string>("inventory_count");
  const [quantityChange, setQuantityChange] = useState<string>("");
  const [documentNumber, setDocumentNumber] = useState<string>("");
  const [notes, setNotes] = useState<string>("");
  
  // Location editor state
  const [isLocationDialogOpen, setIsLocationDialogOpen] = useState(false);
  const [selectedLocationId, setSelectedLocationId] = useState<string>("");
  const [selectedCarrierId, setSelectedCarrierId] = useState<string>("");

  // Determine back path based on current URL
  const getBackPath = () => {
    if (location.includes('/warehouse/obrzeza/')) {
      return '/warehouse/obrzeza';
    }
    if (location.includes('/warehouse/tkaniny/')) {
      return '/warehouse/tkaniny';
    }
    if (location.includes('/warehouse/okucia/')) {
      return '/warehouse/okucia';
    }
    if (location.includes('/warehouse/pianki/')) {
      return '/warehouse/pianki';
    }
    if (location.includes('/warehouse/sruby/')) {
      return '/warehouse/sruby';
    }
    return '/warehouse/plyty';
  };

  const { data: material, isLoading } = useQuery<Material>({
    queryKey: [`/api/warehouse/materials/${materialId}`],
    enabled: !!materialId,
  });

  const { data: usageHistory } = useQuery<MaterialUsage[]>({
    queryKey: [`/api/warehouse/materials/${materialId}/usage`],
    enabled: !!materialId,
  });

  const { data: inventoryHistory } = useQuery<InventoryHistory[]>({
    queryKey: [`/api/warehouse/materials/${materialId}/inventory-history`],
    enabled: !!materialId,
  });

  // Fetch locations and carriers for selection
  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ["/api/production/locations"],
  });

  const { data: carriers = [] } = useQuery<Carrier[]>({
    queryKey: ["/api/production/carriers"],
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/warehouse/materials/${id}`);
    },
    onSuccess: () => {
      // Invalidate materials cache so list refreshes
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/materials"] });
      toast({
        title: "Sukces",
        description: "Materiał został usunięty",
      });
      setLocation(getBackPath());
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się usunąć materiału",
        variant: "destructive",
      });
    },
  });

  const inventoryAdjustmentMutation = useMutation({
    mutationFn: async (data: {
      operationType: string;
      quantityChange: number;
      documentNumber?: string;
      notes?: string;
    }) => {
      return await apiRequest("POST", `/api/warehouse/materials/${materialId}/inventory-adjustment`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/${materialId}`] });
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/${materialId}/inventory-history`] });
      toast({
        title: "Sukces",
        description: "Wpis inwentaryzacyjny został dodany",
      });
      setIsDialogOpen(false);
      resetDialogForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się dodać wpisu inwentaryzacyjnego",
        variant: "destructive",
      });
    },
  });

  const updateLocationMutation = useMutation({
    mutationFn: async (data: {
      locationId: number | null;
      carrierId: number | null;
    }) => {
      return await apiRequest("PATCH", `/api/warehouse/materials/${materialId}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/${materialId}`] });
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/materials"] });
      toast({
        title: "Sukces",
        description: "Lokalizacja magazynowa została zaktualizowana",
      });
      setIsLocationDialogOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować lokalizacji",
        variant: "destructive",
      });
    },
  });

  const handleDelete = () => {
    if (confirm("Czy na pewno chcesz usunąć ten materiał?")) {
      deleteMutation.mutate(materialId);
    }
  };

  const resetDialogForm = () => {
    setOperationType("inventory_count");
    setQuantityChange("");
    setDocumentNumber("");
    setNotes("");
  };

  const handleSubmitInventoryAdjustment = () => {
    const change = parseFloat(quantityChange);
    if (isNaN(change) || change === 0) {
      toast({
        title: "Błąd",
        description: "Zmiana ilości musi być liczbą różną od zera",
        variant: "destructive",
      });
      return;
    }

    inventoryAdjustmentMutation.mutate({
      operationType,
      quantityChange: change,
      documentNumber: documentNumber || undefined,
      notes: notes || undefined,
    });
  };

  const calculateNewQuantity = () => {
    if (!material) return null;
    const change = parseFloat(quantityChange);
    if (isNaN(change)) return null;
    const currentQty = parseFloat(material.quantity || "0");
    return currentQty + change;
  };

  const handleOpenLocationEditor = () => {
    setSelectedLocationId(material?.locationId?.toString() || "");
    setSelectedCarrierId(material?.carrierId?.toString() || "");
    setIsLocationDialogOpen(true);
  };

  const handleSubmitLocation = () => {
    updateLocationMutation.mutate({
      locationId: selectedLocationId ? parseInt(selectedLocationId) : null,
      carrierId: selectedCarrierId ? parseInt(selectedCarrierId) : null,
    });
  };

  // Determine category based on URL
  const category = location.includes('/warehouse/obrzeza/') ? 'obrzeza' :
                   location.includes('/warehouse/tkaniny/') ? 'tkaniny' :
                   location.includes('/warehouse/okucia/') ? 'okucia' :
                   location.includes('/warehouse/pianki/') ? 'pianki' :
                   location.includes('/warehouse/sruby/') ? 'sruby' :
                   'plyty';

  if (isLoading) {
    return (
      <WarehouseLayout category={category}>
        <div className="container mx-auto p-6 space-y-6">
          <Skeleton className="h-10 w-64" />
          <Skeleton className="h-96 w-full" />
        </div>
      </WarehouseLayout>
    );
  }

  if (!material) {
    return (
      <WarehouseLayout category={category}>
        <div className="container mx-auto p-6">
          <Card>
            <CardContent className="pt-6">
              <p className="text-center text-muted-foreground">Materiał nie znaleziony</p>
            </CardContent>
          </Card>
        </div>
      </WarehouseLayout>
    );
  }

  return (
    <WarehouseLayout category={category}>
      <div className="container mx-auto p-6 space-y-6">
        {/* Header */}
        <div className="flex items-center gap-4">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => setLocation(getBackPath())}
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div className="flex-1">
            <h1 className="text-2xl font-bold">{material.name}</h1>
            <p className="text-sm text-muted-foreground">
              Kod: {material.internalCode}
              {material.supplierCode && ` | Kod dostawcy: ${material.supplierCode}`}
            </p>
          </div>
          <div className="flex items-center gap-2">
            <Badge variant={material.isActive ? "default" : "secondary"}>
              {material.isActive ? "Aktywny" : "Nieaktywny"}
            </Badge>
            <Button
              variant="outline"
              size="sm"
              onClick={handleDelete}
              data-testid="button-delete-material"
            >
              <Trash2 className="h-4 w-4 mr-2" />
              Usuń
            </Button>
          </div>
        </div>

        {/* Location Card */}
        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle className="flex items-center gap-2">
                  <MapPin className="h-5 w-5" />
                  Lokalizacja magazynowa
                </CardTitle>
                <CardDescription className="mt-1">
                  Miejsce składowania materiału w magazynie
                </CardDescription>
              </div>
              <Button
                size="sm"
                variant="outline"
                onClick={handleOpenLocationEditor}
                data-testid="button-edit-location"
              >
                <MapPin className="h-4 w-4 mr-1" />
                Edytuj
              </Button>
            </div>
          </CardHeader>
          <CardContent>
            <div className="flex items-center gap-4">
              {material.locationName ? (
                <>
                  <div className="flex items-center gap-2">
                    <Building2 className="h-5 w-5 text-muted-foreground" />
                    <div>
                      <p className="text-sm text-muted-foreground">Hala</p>
                      <p className="font-medium">{material.locationName}</p>
                    </div>
                  </div>
                  {material.carrierName && (
                    <>
                      <div className="h-8 w-px bg-border" />
                      <div className="flex items-center gap-2">
                        <Truck className="h-5 w-5 text-muted-foreground" />
                        <div>
                          <p className="text-sm text-muted-foreground">Nośnik</p>
                          <p className="font-medium">{material.carrierName}</p>
                        </div>
                      </div>
                    </>
                  )}
                </>
              ) : (
                <p className="text-sm text-muted-foreground">
                  Lokalizacja nie została przypisana
                </p>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Inventory Stock Card */}
        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle className="flex items-center gap-2">
                  <Package className="h-5 w-5" />
                  Stan magazynowy
                </CardTitle>
                <CardDescription className="mt-1">
                  Bieżący stan zapasów dla tego materiału
                </CardDescription>
              </div>
              <div className="flex items-center gap-3">
                <Badge variant="outline" className="text-lg font-bold px-4 py-2">
                  {material.quantity || "0"} {material.unitOfMeasure}
                </Badge>
                <Button
                  size="sm"
                  onClick={() => setIsDialogOpen(true)}
                  data-testid="button-add-inventory-entry"
                >
                  <Plus className="h-4 w-4 mr-1" />
                  Dodaj wpis
                </Button>
              </div>
            </div>
          </CardHeader>
        </Card>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          {/* Main Info Card */}
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Package className="h-5 w-5" />
                Szczegóły materiału
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              {material.description && (
                <div>
                  <h3 className="font-medium mb-2">Opis</h3>
                  <p className="text-sm text-muted-foreground">{material.description}</p>
                </div>
              )}

              {material.specifications && Object.keys(material.specifications).length > 0 && (
                <div>
                  <h3 className="font-medium mb-2">Specyfikacja</h3>
                  <dl className="grid grid-cols-2 gap-2 text-sm">
                    {Object.entries(material.specifications).map(([key, value]) => (
                      <div key={key}>
                        <dt className="text-muted-foreground">{key}:</dt>
                        <dd className="font-medium">{String(value)}</dd>
                      </div>
                    ))}
                  </dl>
                </div>
              )}

              <div className="grid grid-cols-2 gap-4 pt-4 border-t">
                {material.unitOfMeasure && (
                  <div>
                    <p className="text-sm text-muted-foreground">Jednostka miary</p>
                    <p className="font-medium">{material.unitOfMeasure}</p>
                  </div>
                )}
                {material.price && (
                  <div>
                    <p className="text-sm text-muted-foreground">Cena</p>
                    <p className="font-medium">{material.price} PLN</p>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Image Card */}
          {material.primaryImageUrl && (
            <Card>
              <CardHeader>
                <CardTitle>Zdjęcie</CardTitle>
              </CardHeader>
              <CardContent>
                <img
                  src={material.primaryImageUrl}
                  alt={material.name}
                  className="w-full h-auto rounded-md"
                />
              </CardContent>
            </Card>
          )}
        </div>

        {/* Inventory History */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <History className="h-5 w-5" />
              Historia inwentaryzacji
            </CardTitle>
            <CardDescription>
              Wszystkie zmiany stanu magazynowego z powodami (inwentaryzacja, zużycie materiału, reklamacja)
            </CardDescription>
          </CardHeader>
          <CardContent>
            {!inventoryHistory || inventoryHistory.length === 0 ? (
              <p className="text-sm text-muted-foreground text-center py-8">
                Brak historii zmian
              </p>
            ) : (
              <div className="space-y-2">
                {inventoryHistory.map((entry) => (
                  <div
                    key={entry.id}
                    className="flex items-start justify-between p-3 border rounded-md"
                  >
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-1">
                        <Badge variant={
                          entry.operationType === 'inventory_count' ? 'default' :
                          entry.operationType === 'material_usage' ? 'secondary' :
                          entry.operationType === 'complaint' ? 'destructive' :
                          entry.operationType === 'purchase' ? 'default' :
                          entry.operationType === 'adjustment' ? 'outline' :
                          'outline'
                        }>
                          {entry.operationType === 'inventory_count' ? 'Inwentaryzacja' :
                           entry.operationType === 'material_usage' ? 'Zużycie materiału' :
                           entry.operationType === 'complaint' ? 'Reklamacja' :
                           entry.operationType === 'purchase' ? 'Zakup' :
                           entry.operationType === 'adjustment' ? 'Korekta' :
                           entry.operationType}
                        </Badge>
                        <span className="font-medium">
                          {parseFloat(entry.quantityChange) > 0 ? '+' : ''}
                          {entry.quantityChange} {material.unitOfMeasure}
                        </span>
                      </div>
                      <p className="text-sm text-muted-foreground">
                        Stan: {entry.quantityBefore} → {entry.quantityAfter} {material.unitOfMeasure}
                      </p>
                      <p className="text-sm text-muted-foreground">
                        {new Date(entry.createdAt).toLocaleString("pl-PL")}
                        {entry.performedBy && ` • ${entry.performedBy}`}
                      </p>
                      {entry.notes && (
                        <p className="text-sm text-muted-foreground mt-1">{entry.notes}</p>
                      )}
                      {entry.documentNumber && (
                        <p className="text-sm text-muted-foreground">Dokument: {entry.documentNumber}</p>
                      )}
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>

        {/* Usage History */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <History className="h-5 w-5" />
              Historia zużycia w produkcji
            </CardTitle>
            <CardDescription>
              Szczegółowa historia użycia tego materiału w zleceniach produkcyjnych
            </CardDescription>
          </CardHeader>
          <CardContent>
            {!usageHistory || usageHistory.length === 0 ? (
              <p className="text-sm text-muted-foreground text-center py-8">
                Brak historii zużycia
              </p>
            ) : (
              <div className="space-y-2">
                {usageHistory.map((usage) => (
                  <div
                    key={usage.id}
                    className="flex items-center justify-between p-3 border rounded-md"
                  >
                    <div className="flex-1">
                      <p className="font-medium">
                        Zużyto: {usage.quantityUsed} {material.unitOfMeasure}
                      </p>
                      <p className="text-sm text-muted-foreground">
                        {new Date(usage.usedAt).toLocaleString("pl-PL")}
                      </p>
                      {usage.notes && (
                        <p className="text-sm text-muted-foreground mt-1">{usage.notes}</p>
                      )}
                    </div>
                    {usage.productionOrderId && (
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => setLocation(`/production/orders/${usage.productionOrderId}`)}
                        data-testid={`button-view-order-${usage.productionOrderId}`}
                      >
                        <ExternalLink className="h-4 w-4 mr-1" />
                        Zlecenie
                      </Button>
                    )}
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>

        {/* Link to material */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="h-5 w-5" />
              Link do materiału
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex items-center gap-2">
              <code className="flex-1 p-2 bg-muted rounded text-sm">
                {window.location.origin}{location}
              </code>
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  navigator.clipboard.writeText(`${window.location.origin}${location}`);
                }}
                data-testid="button-copy-link"
              >
                Kopiuj
              </Button>
            </div>
          </CardContent>
        </Card>

        {/* Inventory Adjustment Dialog */}
        <Dialog open={isDialogOpen} onOpenChange={(open) => {
          setIsDialogOpen(open);
          if (!open) {
            resetDialogForm();
          }
        }}>
          <DialogContent className="sm:max-w-[500px]">
            <DialogHeader>
              <DialogTitle>Dodaj wpis inwentaryzacyjny</DialogTitle>
              <DialogDescription>
                Zarejestruj zmianę stanu magazynowego dla: {material.name}
              </DialogDescription>
            </DialogHeader>
            
            <div className="space-y-4 py-4">
              <div className="space-y-2">
                <Label htmlFor="operation-type">Typ operacji</Label>
                <Select value={operationType} onValueChange={setOperationType}>
                  <SelectTrigger id="operation-type" data-testid="select-operation-type">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="inventory_count">Inwentaryzacja</SelectItem>
                    <SelectItem value="material_usage">Zużycie materiału</SelectItem>
                    <SelectItem value="complaint">Reklamacja</SelectItem>
                    <SelectItem value="adjustment">Korekta</SelectItem>
                    <SelectItem value="purchase">Zakup</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="quantity-change">
                  Zmiana ilości ({material.unitOfMeasure})
                </Label>
                <Input
                  id="quantity-change"
                  type="number"
                  step="0.001"
                  placeholder="np. +10 lub -5"
                  value={quantityChange}
                  onChange={(e) => setQuantityChange(e.target.value)}
                  data-testid="input-quantity-change"
                />
                {quantityChange && !isNaN(parseFloat(quantityChange)) && (
                  <p className="text-sm text-muted-foreground">
                    Stan: {material.quantity || "0"} → {calculateNewQuantity()} {material.unitOfMeasure}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="document-number">Numer dokumentu (opcjonalnie)</Label>
                <Input
                  id="document-number"
                  placeholder="np. WZ/2025/001"
                  value={documentNumber}
                  onChange={(e) => setDocumentNumber(e.target.value)}
                  data-testid="input-document-number"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="notes">Notatki (opcjonalnie)</Label>
                <Textarea
                  id="notes"
                  placeholder="Dodatkowe informacje..."
                  value={notes}
                  onChange={(e) => setNotes(e.target.value)}
                  data-testid="textarea-notes"
                  rows={3}
                />
              </div>
            </div>

            <DialogFooter>
              <Button
                variant="outline"
                onClick={() => {
                  setIsDialogOpen(false);
                  resetDialogForm();
                }}
                data-testid="button-cancel-inventory"
              >
                Anuluj
              </Button>
              <Button
                onClick={handleSubmitInventoryAdjustment}
                disabled={inventoryAdjustmentMutation.isPending}
                data-testid="button-submit-inventory"
              >
                {inventoryAdjustmentMutation.isPending ? "Zapisywanie..." : "Zapisz"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        {/* Location Editor Dialog */}
        <Dialog open={isLocationDialogOpen} onOpenChange={setIsLocationDialogOpen}>
          <DialogContent className="sm:max-w-[500px]">
            <DialogHeader>
              <DialogTitle>Lokalizacja magazynowa</DialogTitle>
              <DialogDescription>
                Ustaw lokalizację i nośnik dla: {material.name}
              </DialogDescription>
            </DialogHeader>

            <div className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="location-select" className="flex items-center gap-2">
                  <Building2 className="h-4 w-4" />
                  Hala
                </Label>
                <Select value={selectedLocationId || "none"} onValueChange={(val) => setSelectedLocationId(val === "none" ? "" : val)}>
                  <SelectTrigger id="location-select" data-testid="select-location">
                    <SelectValue placeholder="Wybierz halę" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Brak lokalizacji</SelectItem>
                    {locations
                      .filter(loc => loc.level === 1)
                      .map((loc) => (
                        <SelectItem key={loc.id} value={loc.id.toString()}>
                          {loc.name}
                        </SelectItem>
                      ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="carrier-select" className="flex items-center gap-2">
                  <Truck className="h-4 w-4" />
                  Nośnik
                </Label>
                <Select value={selectedCarrierId || "none"} onValueChange={(val) => setSelectedCarrierId(val === "none" ? "" : val)}>
                  <SelectTrigger id="carrier-select" data-testid="select-carrier">
                    <SelectValue placeholder="Wybierz nośnik" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Brak nośnika</SelectItem>
                    {carriers.map((carrier) => (
                      <SelectItem key={carrier.id} value={carrier.id.toString()}>
                        {carrier.name} ({carrier.type})
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>

            <DialogFooter>
              <Button
                variant="outline"
                onClick={() => setIsLocationDialogOpen(false)}
                data-testid="button-cancel-location"
              >
                Anuluj
              </Button>
              <Button
                onClick={handleSubmitLocation}
                disabled={updateLocationMutation.isPending}
                data-testid="button-submit-location"
              >
                {updateLocationMutation.isPending ? "Zapisywanie..." : "Zapisz"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>
    </WarehouseLayout>
  );
}
