import { useQuery } from "@tanstack/react-query";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { FileCheck, Calendar, User, Plus } from "lucide-react";
import { Link, useLocation } from "wouter";

interface InventoryCount {
  id: number;
  name: string;
  status: string;
  notes: string | null;
  createdBy: string;
  createdAt: string;
  finalizedBy: string | null;
  finalizedAt: string | null;
  itemCount: string;
}

export default function WarehouseInventoryCountsPage() {
  const [, setLocation] = useLocation();
  
  const { data: inventoryCounts = [], isLoading } = useQuery<InventoryCount[]>({
    queryKey: ["/api/warehouse/inventory-counts"],
  });

  const draftCounts = inventoryCounts.filter(c => c.status === 'draft');
  const finalizedCounts = inventoryCounts.filter(c => c.status === 'finalized');

  if (isLoading) {
    return (
      <WarehouseLayout category="all">
        <div className="p-6">Ładowanie...</div>
      </WarehouseLayout>
    );
  }

  return (
    <WarehouseLayout category="all">
      <div className="space-y-6 p-6">
        <div className="flex items-center justify-between">
          <h1 className="text-3xl font-bold">Spisy inwentaryzacyjne</h1>
          <Button onClick={() => setLocation("/warehouse/inventory-counts/new")} data-testid="button-create-new-count">
            <Plus className="h-4 w-4 mr-2" />
            Dodaj nowy spis
          </Button>
        </div>

        {draftCounts.length > 0 && (
          <div>
            <h2 className="text-xl font-semibold mb-4">Robocze ({draftCounts.length})</h2>
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {draftCounts.map((count) => (
                <Link key={count.id} href={`/warehouse/inventory-counts/${count.id}`}>
                  <Card className="hover-elevate cursor-pointer" data-testid={`card-count-${count.id}`}>
                    <CardHeader>
                      <CardTitle className="flex items-center justify-between">
                        <span className="flex items-center gap-2">
                          <FileCheck className="h-5 w-5" />
                          {count.name}
                        </span>
                        <Badge variant="secondary" data-testid={`badge-status-${count.id}`}>
                          {count.status === 'draft' ? 'Roboczy' : 'Zamknięty'}
                        </Badge>
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="space-y-2 text-sm text-muted-foreground">
                        <div className="flex items-center gap-2">
                          <FileCheck className="h-4 w-4" />
                          <span>Pozycji: {count.itemCount}</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <User className="h-4 w-4" />
                          <span>Utworzył: {count.createdBy}</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <Calendar className="h-4 w-4" />
                          <span>{new Date(count.createdAt).toLocaleString("pl-PL")}</span>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                </Link>
              ))}
            </div>
          </div>
        )}

        {finalizedCounts.length > 0 && (
          <div>
            <h2 className="text-xl font-semibold mb-4">Zamknięte ({finalizedCounts.length})</h2>
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {finalizedCounts.map((count) => (
                <Link key={count.id} href={`/warehouse/inventory-counts/${count.id}`}>
                  <Card className="hover-elevate cursor-pointer" data-testid={`card-count-${count.id}`}>
                    <CardHeader>
                      <CardTitle className="flex items-center justify-between">
                        <span className="flex items-center gap-2">
                          <FileCheck className="h-5 w-5" />
                          {count.name}
                        </span>
                        <Badge variant="default" data-testid={`badge-status-${count.id}`}>
                          Zamknięty
                        </Badge>
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="space-y-2 text-sm text-muted-foreground">
                        <div className="flex items-center gap-2">
                          <FileCheck className="h-4 w-4" />
                          <span>Pozycji: {count.itemCount}</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <User className="h-4 w-4" />
                          <span>Zamknął: {count.finalizedBy}</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <Calendar className="h-4 w-4" />
                          <span>{count.finalizedAt && new Date(count.finalizedAt).toLocaleString("pl-PL")}</span>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                </Link>
              ))}
            </div>
          </div>
        )}

        {inventoryCounts.length === 0 && (
          <div className="text-center py-12">
            <FileCheck className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
            <p className="text-lg text-muted-foreground">Brak spisów inwentaryzacyjnych</p>
            <p className="text-sm text-muted-foreground mt-2">
              Kliknij "Dodaj nowy spis" aby rozpocząć
            </p>
          </div>
        )}
      </div>
    </WarehouseLayout>
  );
}
