import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { ArrowLeft, Save, CheckCircle, Copy } from "lucide-react";
import { useState } from "react";

interface InventoryCountItem {
  id: number;
  inventoryCountId: number;
  materialId: number;
  systemQuantity: string;
  countedQuantity: string | null;
  difference: string | null;
  notes: string | null;
  materialName: string;
  materialCode: string;
  unitOfMeasure: string;
  groupName: string | null;
}

interface InventoryCount {
  id: number;
  name: string;
  status: string;
  notes: string | null;
  createdBy: string;
  createdAt: string;
  finalizedBy: string | null;
  finalizedAt: string | null;
}

export default function WarehouseInventoryCountDetailPage() {
  const params = useParams<{ id: string }>();
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const countId = parseInt(params.id || "0");

  const [editedQuantities, setEditedQuantities] = useState<Record<number, string>>({});

  const { data, isLoading } = useQuery<{ inventoryCount: InventoryCount; items: InventoryCountItem[] }>({
    queryKey: [`/api/warehouse/inventory-counts/${countId}`],
    enabled: !!countId,
  });

  const updateItemMutation = useMutation({
    mutationFn: async ({ itemId, countedQuantity }: { itemId: number; countedQuantity: string }) => {
      return apiRequest("PATCH", `/api/warehouse/inventory-counts/${countId}/items/${itemId}`, {
        countedQuantity: parseFloat(countedQuantity),
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/inventory-counts/${countId}`] });
      toast({ title: "Sukces", description: "Ilość spisana zaktualizowana" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować ilości",
        variant: "destructive",
      });
    },
  });

  const finalizeMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", `/api/warehouse/inventory-counts/${countId}/finalize`, {});
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/inventory-counts/${countId}`] });
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      toast({
        title: "Sukces",
        description: "Spis inwentaryzacyjny zamknięty i magazyn przeliczony",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zamknąć spisu",
        variant: "destructive",
      });
    },
  });

  const duplicateMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", `/api/warehouse/inventory-counts/${countId}/duplicate`, {
        name: `${data?.inventoryCount.name} (kopia)`,
      });
    },
    onSuccess: (result: any) => {
      toast({
        title: "Sukces",
        description: "Spis inwentaryzacyjny zduplikowany",
      });
      setLocation(`/warehouse/inventory-counts/${result.inventoryCount.id}`);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zduplikować spisu",
        variant: "destructive",
      });
    },
  });

  const handleSaveItem = (itemId: number) => {
    const quantity = editedQuantities[itemId];
    if (quantity !== undefined && quantity.trim() !== "") {
      updateItemMutation.mutate({ itemId, countedQuantity: quantity });
      const newEdited = { ...editedQuantities };
      delete newEdited[itemId];
      setEditedQuantities(newEdited);
    }
  };

  const handleQuantityChange = (itemId: number, value: string) => {
    setEditedQuantities({ ...editedQuantities, [itemId]: value });
  };

  if (isLoading || !data) {
    return (
      <WarehouseLayout>
        <div className="p-6">Ładowanie...</div>
      </WarehouseLayout>
    );
  }

  const { inventoryCount, items } = data;
  const isDraft = inventoryCount.status === 'draft';
  const countedItems = items.filter(i => i.countedQuantity !== null).length;
  const uncountedItems = items.filter(i => i.countedQuantity === null).length;

  return (
    <WarehouseLayout>
      <div className="space-y-6 p-6">
        <div className="flex items-center gap-4">
          <Button variant="ghost" onClick={() => setLocation("/warehouse/inventory-counts")} data-testid="button-back">
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div className="flex-1">
            <h1 className="text-3xl font-bold">{inventoryCount.name}</h1>
            <p className="text-muted-foreground">
              Utworzony przez {inventoryCount.createdBy} • {new Date(inventoryCount.createdAt).toLocaleString("pl-PL")}
            </p>
          </div>
          <Badge variant={isDraft ? "secondary" : "default"} data-testid="badge-status">
            {isDraft ? "Roboczy" : "Zamknięty"}
          </Badge>
        </div>

        <div className="grid gap-4 md:grid-cols-3">
          <Card>
            <CardHeader>
              <CardTitle>Pozycje</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold">{items.length}</p>
              <p className="text-sm text-muted-foreground">wszystkich</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle>Spisane</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-green-600">{countedItems}</p>
              <p className="text-sm text-muted-foreground">pozycji spisanych</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <CardTitle>Do spisania</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-orange-600">{uncountedItems}</p>
              <p className="text-sm text-muted-foreground">pozycji pozostało</p>
            </CardContent>
          </Card>
        </div>

        {isDraft && uncountedItems === 0 && (
          <Card className="border-green-500">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <CheckCircle className="h-5 w-5 text-green-600" />
                  <div>
                    <p className="font-semibold">Wszystkie pozycje zostały spisane</p>
                    <p className="text-sm text-muted-foreground">Możesz teraz zamknąć spis i przel

iczyć magazyn</p>
                  </div>
                </div>
                <Button
                  onClick={() => finalizeMutation.mutate()}
                  disabled={finalizeMutation.isPending}
                  data-testid="button-finalize"
                >
                  {finalizeMutation.isPending ? "Zamykanie..." : "Przelicz magazyn"}
                </Button>
              </div>
            </CardContent>
          </Card>
        )}

        <div className="flex gap-2">
          {!isDraft && (
            <Button
              onClick={() => duplicateMutation.mutate()}
              disabled={duplicateMutation.isPending}
              data-testid="button-duplicate"
            >
              <Copy className="h-4 w-4 mr-2" />
              Duplikuj spis
            </Button>
          )}
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Pozycje spisu</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              {items.map((item) => {
                const hasEdited = editedQuantities[item.id] !== undefined;
                const displayedCountedQty = hasEdited ? editedQuantities[item.id] : (item.countedQuantity || "");
                
                return (
                  <div
                    key={item.id}
                    className="flex items-center gap-4 p-3 border rounded-md"
                    data-testid={`item-${item.id}`}
                  >
                    <div className="flex-1">
                      <p className="font-semibold">{item.materialName}</p>
                      <p className="text-sm text-muted-foreground">
                        Kod: {item.materialCode} {item.groupName && `• Grupa: ${item.groupName}`}
                      </p>
                    </div>
                    <div className="w-32">
                      <p className="text-sm text-muted-foreground">Stan systemowy</p>
                      <p className="font-semibold">{item.systemQuantity} {item.unitOfMeasure}</p>
                    </div>
                    <div className="w-32">
                      <p className="text-sm text-muted-foreground">Spisano</p>
                      {isDraft ? (
                        <Input
                          type="number"
                          step="0.001"
                          placeholder="0"
                          value={displayedCountedQty}
                          onChange={(e) => handleQuantityChange(item.id, e.target.value)}
                          disabled={!isDraft}
                          data-testid={`input-counted-${item.id}`}
                          className="w-full"
                        />
                      ) : (
                        <p className="font-semibold">{item.countedQuantity || "-"} {item.unitOfMeasure}</p>
                      )}
                    </div>
                    {item.difference !== null && (
                      <div className="w-24">
                        <p className="text-sm text-muted-foreground">Różnica</p>
                        <p className={`font-semibold ${parseFloat(item.difference) > 0 ? "text-green-600" : parseFloat(item.difference) < 0 ? "text-red-600" : ""}`}>
                          {parseFloat(item.difference) > 0 ? "+" : ""}{item.difference}
                        </p>
                      </div>
                    )}
                    {isDraft && hasEdited && (
                      <Button
                        size="sm"
                        onClick={() => handleSaveItem(item.id)}
                        disabled={updateItemMutation.isPending}
                        data-testid={`button-save-${item.id}`}
                      >
                        <Save className="h-4 w-4" />
                      </Button>
                    )}
                  </div>
                );
              })}
            </div>
          </CardContent>
        </Card>
      </div>
    </WarehouseLayout>
  );
}
