import { useState } from "react";
import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ArrowLeft, Package, FileText, History, ExternalLink, Check, X, Trash2, MapPin, Building2, Truck, Edit } from "lucide-react";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";

interface StockPanel {
  id: number;
  generatedName: string;
  cz1?: string;
  cz2?: string;
  furnitureType?: string;
  length: string;
  width: string;
  thickness: string;
  boardCode?: string;
  edgingCode?: string;
  colorCode?: string;
  colorName?: string;
  colorHex?: string;
  edge1: boolean;
  edge2: boolean;
  edge3: boolean;
  edge4: boolean;
  isDrilled: boolean;
  isEdged: boolean;
  source: string;
  sourceReference?: string;
  quantity: number;
  location?: string;
  locationId?: number;
  carrierId?: number;
  locationName?: string;
  carrierName?: string;
  notes?: string;
  imageUrl?: string;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

interface PanelUsage {
  id: number;
  productionOrderId?: number;
  productionPlanId?: number;
  quantityUsed: number;
  usedAt: string;
  notes?: string;
}

interface Location {
  id: number;
  name: string;
  level: number;
}

interface Carrier {
  id: number;
  name: string;
  type: string;
}

const SOURCE_LABELS: Record<string, string> = {
  stock: "Zapas",
  production_surplus: "Nadmiar produkcyjny",
  returns: "Zwroty"
};

export default function WarehouseFormatkDetail() {
  const params = useParams<{ id: string }>();
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const panelId = parseInt(params.id || "0");
  
  // Location/Carrier edit state
  const [isLocationDialogOpen, setIsLocationDialogOpen] = useState(false);
  const [selectedLocationId, setSelectedLocationId] = useState<string>("");
  const [selectedCarrierId, setSelectedCarrierId] = useState<string>("");

  const { data: panel, isLoading } = useQuery<StockPanel>({
    queryKey: [`/api/warehouse/stock-panels/${panelId}`],
    enabled: !!panelId,
  });

  const { data: usageHistory } = useQuery<PanelUsage[]>({
    queryKey: [`/api/warehouse/stock-panels/${panelId}/usage`],
    enabled: !!panelId,
  });

  // Fetch locations and carriers for selection
  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ["/api/production/locations"],
  });

  const { data: carriers = [] } = useQuery<Carrier[]>({
    queryKey: ["/api/production/carriers"],
  });

  // Filter for hall locations (level 1)
  const hallLocations = locations.filter(loc => loc.level === 1);

  // Update location/carrier mutation
  const updateLocationMutation = useMutation({
    mutationFn: async (data: {
      locationId: number | null;
      carrierId: number | null;
    }) => {
      const res = await apiRequest("PATCH", `/api/warehouse/stock-panels/${panelId}`, data);
      return await res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/stock-panels/${panelId}`] });
      toast({
        title: "Sukces",
        description: "Lokalizacja i nośnik zostały zaktualizowane",
      });
      setIsLocationDialogOpen(false);
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się zaktualizować lokalizacji",
        variant: "destructive",
      });
    },
  });

  const handleOpenLocationDialog = () => {
    setSelectedLocationId(panel?.locationId?.toString() || "");
    setSelectedCarrierId(panel?.carrierId?.toString() || "");
    setIsLocationDialogOpen(true);
  };

  const handleSaveLocation = () => {
    updateLocationMutation.mutate({
      locationId: selectedLocationId ? parseInt(selectedLocationId) : null,
      carrierId: selectedCarrierId ? parseInt(selectedCarrierId) : null,
    });
  };

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/warehouse/stock-panels/${id}`);
    },
    onSuccess: () => {
      // Invalidate stock panels cache so list refreshes
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/stock-panels"] });
      toast({
        title: "Sukces",
        description: "Formatka została usunięta",
      });
      setLocation("/warehouse/formatki");
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się usunąć formatki",
        variant: "destructive",
      });
    },
  });

  const handleDelete = () => {
    if (confirm("Czy na pewno chcesz usunąć tę formatkę?")) {
      deleteMutation.mutate(panelId);
    }
  };

  if (isLoading) {
    return (
      <WarehouseLayout category="formatki">
        <div className="container mx-auto p-6 space-y-6">
          <Skeleton className="h-10 w-64" />
          <Skeleton className="h-96 w-full" />
        </div>
      </WarehouseLayout>
    );
  }

  if (!panel) {
    return (
      <WarehouseLayout category="formatki">
        <div className="container mx-auto p-6">
          <Card>
            <CardContent className="pt-6">
              <p className="text-center text-muted-foreground">Formatka nie znaleziona</p>
            </CardContent>
          </Card>
        </div>
      </WarehouseLayout>
    );
  }

  return (
    <WarehouseLayout category="formatki">
      <div className="container mx-auto p-6 space-y-6">
        {/* Header */}
        <div className="flex items-center gap-4 flex-wrap">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => setLocation("/warehouse/formatki")}
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div className="flex-1 min-w-0">
            <h1 className="text-2xl font-bold truncate">{panel.generatedName}</h1>
            <p className="text-sm text-muted-foreground">
              {panel.length} × {panel.width} × {panel.thickness} mm
            </p>
          </div>
          <div className="flex items-center gap-2">
            <Badge variant={panel.isActive ? "default" : "secondary"}>
              {panel.isActive ? "Aktywny" : "Nieaktywny"}
            </Badge>
            <Badge variant="outline">
              Ilość: {panel.quantity}
            </Badge>
            <Button
              variant="outline"
              size="sm"
              onClick={handleDelete}
              data-testid="button-delete-panel"
            >
              <Trash2 className="h-4 w-4 mr-2" />
              Usuń
            </Button>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Main Info Card */}
          <Card className="lg:col-span-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Package className="h-5 w-5" />
                Szczegóły formatki
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              {/* Dimensions */}
              <div>
                <h3 className="font-medium mb-3">Wymiary</h3>
                <dl className="grid grid-cols-3 gap-4 text-sm">
                  <div>
                    <dt className="text-muted-foreground">Długość</dt>
                    <dd className="font-medium text-lg">{panel.length} mm</dd>
                  </div>
                  <div>
                    <dt className="text-muted-foreground">Szerokość</dt>
                    <dd className="font-medium text-lg">{panel.width} mm</dd>
                  </div>
                  <div>
                    <dt className="text-muted-foreground">Grubość</dt>
                    <dd className="font-medium text-lg">{panel.thickness} mm</dd>
                  </div>
                </dl>
              </div>

              {/* Material Info */}
              <div className="pt-4 border-t">
                <h3 className="font-medium mb-3">Materiały</h3>
                <dl className="grid grid-cols-2 gap-3 text-sm">
                  {panel.boardCode && (
                    <div>
                      <dt className="text-muted-foreground">Kod płyty</dt>
                      <dd className="font-medium">{panel.boardCode}</dd>
                    </div>
                  )}
                  {panel.edgingCode && (
                    <div>
                      <dt className="text-muted-foreground">Kod obrzeża</dt>
                      <dd className="font-medium">{panel.edgingCode}</dd>
                    </div>
                  )}
                  {panel.colorCode && (
                    <div>
                      <dt className="text-muted-foreground">Kod koloru</dt>
                      <dd className="font-medium flex items-center gap-2">
                        {panel.colorHex && (
                          <span 
                            className="inline-block w-4 h-4 rounded border"
                            style={{ backgroundColor: panel.colorHex }}
                          />
                        )}
                        {panel.colorCode}
                      </dd>
                    </div>
                  )}
                  {panel.colorName && (
                    <div>
                      <dt className="text-muted-foreground">Nazwa koloru</dt>
                      <dd className="font-medium">{panel.colorName}</dd>
                    </div>
                  )}
                </dl>
              </div>

              {/* Edging Info */}
              <div className="pt-4 border-t">
                <h3 className="font-medium mb-3">Obrzeżowanie</h3>
                <div className="flex items-center gap-4 flex-wrap">
                  <div className="flex items-center gap-2">
                    {panel.isEdged ? (
                      <Check className="h-4 w-4 text-green-600" />
                    ) : (
                      <X className="h-4 w-4 text-muted-foreground" />
                    )}
                    <span className="text-sm">Obrzeżowane</span>
                  </div>
                  <div className="flex items-center gap-2">
                    {panel.isDrilled ? (
                      <Check className="h-4 w-4 text-green-600" />
                    ) : (
                      <X className="h-4 w-4 text-muted-foreground" />
                    )}
                    <span className="text-sm">Wiercone</span>
                  </div>
                </div>
                <div className="grid grid-cols-4 gap-2 mt-3">
                  {[1, 2, 3, 4].map((edge) => (
                    <div key={edge} className="text-center p-2 border rounded">
                      <p className="text-xs text-muted-foreground mb-1">Krawędź {edge}</p>
                      {panel[`edge${edge}` as keyof StockPanel] ? (
                        <Check className="h-4 w-4 text-green-600 mx-auto" />
                      ) : (
                        <X className="h-4 w-4 text-muted-foreground mx-auto" />
                      )}
                    </div>
                  ))}
                </div>
              </div>

              {/* Additional Info */}
              <div className="pt-4 border-t grid grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-muted-foreground">Źródło</p>
                  <p className="font-medium">{SOURCE_LABELS[panel.source] || panel.source}</p>
                  {panel.sourceReference && (
                    <p className="text-xs text-muted-foreground mt-1">{panel.sourceReference}</p>
                  )}
                </div>
                {panel.location && (
                  <div>
                    <p className="text-sm text-muted-foreground">Lokalizacja</p>
                    <p className="font-medium">{panel.location}</p>
                  </div>
                )}
                {panel.furnitureType && (
                  <div>
                    <p className="text-sm text-muted-foreground">Typ mebla</p>
                    <p className="font-medium">{panel.furnitureType}</p>
                  </div>
                )}
              </div>

              {panel.notes && (
                <div className="pt-4 border-t">
                  <h3 className="font-medium mb-2">Notatki</h3>
                  <p className="text-sm text-muted-foreground">{panel.notes}</p>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Location & Carrier Card */}
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="flex items-center gap-2">
                  <MapPin className="h-5 w-5" />
                  Lokalizacja magazynowa
                </CardTitle>
                <Button
                  variant="ghost"
                  size="icon"
                  onClick={handleOpenLocationDialog}
                  data-testid="button-edit-location"
                >
                  <Edit className="h-4 w-4" />
                </Button>
              </div>
            </CardHeader>
            <CardContent className="space-y-3">
              <div>
                <p className="text-sm text-muted-foreground mb-1 flex items-center gap-1">
                  <Building2 className="h-3 w-3" />
                  Hala
                </p>
                {panel.locationName ? (
                  <p className="font-medium">{panel.locationName}</p>
                ) : (
                  <p className="text-sm text-muted-foreground italic">Nie przypisano</p>
                )}
              </div>
              <div>
                <p className="text-sm text-muted-foreground mb-1 flex items-center gap-1">
                  <Truck className="h-3 w-3" />
                  Nośnik
                </p>
                {panel.carrierName ? (
                  <p className="font-medium">{panel.carrierName}</p>
                ) : (
                  <p className="text-sm text-muted-foreground italic">Nie przypisano</p>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Image Card */}
          {panel.imageUrl && (
            <Card>
              <CardHeader>
                <CardTitle>Zdjęcie</CardTitle>
              </CardHeader>
              <CardContent>
                <img
                  src={panel.imageUrl}
                  alt={panel.generatedName}
                  className="w-full h-auto rounded-md"
                />
              </CardContent>
            </Card>
          )}
        </div>

        {/* Usage History */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <History className="h-5 w-5" />
              Historia zużycia w produkcji
            </CardTitle>
            <CardDescription>
              Szczegółowa historia użycia tej formatki w zleceniach produkcyjnych
            </CardDescription>
          </CardHeader>
          <CardContent>
            {!usageHistory || usageHistory.length === 0 ? (
              <p className="text-sm text-muted-foreground text-center py-8">
                Brak historii zużycia
              </p>
            ) : (
              <div className="space-y-2">
                {usageHistory.map((usage) => (
                  <div
                    key={usage.id}
                    className="flex items-center justify-between gap-4 p-3 border rounded-md"
                  >
                    <div className="flex-1 min-w-0">
                      <p className="font-medium">
                        Zużyto: {usage.quantityUsed} szt.
                      </p>
                      <p className="text-sm text-muted-foreground">
                        {new Date(usage.usedAt).toLocaleString("pl-PL")}
                      </p>
                      {usage.notes && (
                        <p className="text-sm text-muted-foreground mt-1 truncate">{usage.notes}</p>
                      )}
                    </div>
                    {usage.productionOrderId && (
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => setLocation(`/production/orders/${usage.productionOrderId}`)}
                        data-testid={`button-view-order-${usage.productionOrderId}`}
                      >
                        <ExternalLink className="h-4 w-4 mr-1" />
                        Zlecenie
                      </Button>
                    )}
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>

        {/* Link to panel */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="h-5 w-5" />
              Link do formatki
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex items-center gap-2">
              <code className="flex-1 p-2 bg-muted rounded text-sm overflow-x-auto">
                {window.location.origin}/warehouse/formatki/{panelId}
              </code>
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  navigator.clipboard.writeText(`${window.location.origin}/warehouse/formatki/${panelId}`);
                }}
                data-testid="button-copy-link"
              >
                Kopiuj
              </Button>
            </div>
          </CardContent>
        </Card>

        {/* Location Edit Dialog */}
        <Dialog open={isLocationDialogOpen} onOpenChange={setIsLocationDialogOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Edytuj lokalizację magazynową</DialogTitle>
              <DialogDescription>
                Wybierz halę i nośnik dla tej formatki
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4 py-4">
              <div>
                <Label htmlFor="location-select" className="flex items-center gap-2">
                  <Building2 className="h-4 w-4" />
                  Hala
                </Label>
                <Select 
                  value={selectedLocationId || "none"} 
                  onValueChange={(val) => setSelectedLocationId(val === "none" ? "" : val)}
                >
                  <SelectTrigger id="location-select" data-testid="select-location">
                    <SelectValue placeholder="Wybierz halę" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Brak</SelectItem>
                    {hallLocations.map((location) => (
                      <SelectItem key={location.id} value={location.id.toString()}>
                        {location.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="carrier-select" className="flex items-center gap-2">
                  <Truck className="h-4 w-4" />
                  Nośnik
                </Label>
                <Select 
                  value={selectedCarrierId || "none"} 
                  onValueChange={(val) => setSelectedCarrierId(val === "none" ? "" : val)}
                >
                  <SelectTrigger id="carrier-select" data-testid="select-carrier">
                    <SelectValue placeholder="Wybierz nośnik" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Brak</SelectItem>
                    {carriers.map((carrier) => (
                      <SelectItem key={carrier.id} value={carrier.id.toString()}>
                        {carrier.name} ({carrier.type})
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
            <DialogFooter>
              <Button
                variant="outline"
                onClick={() => setIsLocationDialogOpen(false)}
                data-testid="button-cancel-location"
              >
                Anuluj
              </Button>
              <Button
                onClick={handleSaveLocation}
                disabled={updateLocationMutation.isPending}
                data-testid="button-save-location"
              >
                {updateLocationMutation.isPending ? "Zapisywanie..." : "Zapisz"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>
    </WarehouseLayout>
  );
}
