import { useState, useEffect, useCallback, useMemo, useRef } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation, useRoute } from "wouter";
import { useDropzone } from "react-dropzone";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { useToast } from "@/hooks/use-toast";
import { useAuth } from "@/hooks/use-auth";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { ArrowLeft, Trash2, Plus, Upload, X, Eye, Palette, Check, ChevronsUpDown, Save, Play, Loader2, ExternalLink, Package2, Copy } from "lucide-react";
import { Link } from "wouter";
import { Checkbox } from "@/components/ui/checkbox";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { DictionaryComboboxWithAdd } from "@/components/dictionary-combobox-with-add";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import { cn } from "@/lib/utils";

interface SetMatrix {
  id: number;
  name: string;
  namePrefix?: string | null;
  nameSuffix?: string | null;
  productGroup: string | null;
  length?: string | null;
  width?: string | null;
  height?: string | null;
  doors?: string | null;
  legs?: string | null;
  imageUrl?: string | null;
  modifierType?: string | null; // 'amount' | 'percentage'
  modifierOperation?: string | null; // 'add' | 'subtract'
  modifierValue?: string | null;
  components: {
    name: string;
    componentType: string;
    length: number | null;
    width: number | null;
    quantity: number;
  }[];
  colors?: string[];
  colorImages?: Record<string, string[]>;
  colorOptions?: Record<string, string[]>;
  componentProductOverrides?: Record<string, Record<string, { productId: number }>>;
  selectedColors?: number[];  // Changed to indices like product-matrix-editor
  useEnhancedDescription?: boolean;
}

interface Dictionary {
  id: number;
  dictionaryType?: string;
  code: string;
  name: string;
  shortName?: string | null;
  readableName: string | null;
  color?: string | null;
  isActive: boolean;
}

interface ComponentOverridesDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  activeColorIndex: number | null;
  onActiveColorIndexChange: (index: number | null) => void;
  colors: string[];
  components: Array<{ componentType: string; [key: string]: any }>;
  currentOverrides: Record<string, { productId: number }>;
  onSaveOverrides: (colorIndex: number, overrides: Record<string, { productId: number }>) => void;
}

// Helper function to determine text color based on background luminance
function getTextColorForBackground(hexColor: string | null | undefined): string {
  if (!hexColor || !hexColor.startsWith('#')) return 'black';
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  // Calculate luminance using standard formula
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance > 128 ? 'black' : 'white';
}

// Component Overrides Dialog - Extracted to module scope for stable React identity
function ComponentOverridesDialog({
  open,
  onOpenChange,
  activeColorIndex,
  onActiveColorIndexChange,
  colors,
  components,
  currentOverrides,
  onSaveOverrides
}: ComponentOverridesDialogProps) {
  const { toast } = useToast();
  const [localOverrides, setLocalOverrides] = useState<Record<string, { productId: number }>>({});
  const [searchProductQuery, setSearchProductQuery] = useState("");
  const [selectedComponentType, setSelectedComponentType] = useState<string>("");
  const [selectedProductId, setSelectedProductId] = useState<number | null>(null);
  const [productSearchOpen, setProductSearchOpen] = useState(false);
  const initializedColorRef = useRef<number | null>(null);
  
  // Debounced and normalized search
  const [debouncedSearch, setDebouncedSearch] = useState("");
  
  useEffect(() => {
    const timer = setTimeout(() => {
      const normalized = searchProductQuery.trim().replace(/\s+/g, ' ');
      if (normalized.includes(';')) {
        console.log('🔍 [PRODUCT-SEARCH] Normalized search with semicolon:', {
          original: searchProductQuery,
          normalized,
          hasSemicolon: normalized.includes(';')
        });
      }
      setDebouncedSearch(normalized);
    }, 250);
    return () => clearTimeout(timer);
  }, [searchProductQuery]);
  
  // Fetch products for search
  const { data: productsResponse, isFetching: isLoadingProducts } = useQuery<any>({
    queryKey: ['/api/catalog-products', { search: debouncedSearch, componentType: selectedComponentType }],
    enabled: debouncedSearch.length >= 2 && selectedComponentType.length > 0,
  });
  
  const products = productsResponse?.products || [];
  
  const isProductValid = useMemo(() => {
    const valid = selectedProductId !== null && typeof selectedProductId === 'number';
    console.log('🔍 [PRODUCT-VALID]', {
      selectedProductId,
      selectedComponentType,
      isProductValid: valid,
      isLoadingProducts,
      buttonDisabled: !valid || !selectedComponentType || isLoadingProducts
    });
    return valid;
  }, [selectedProductId, selectedComponentType, isLoadingProducts]);
  
  // Helper component to display product details
  const ProductOverrideDisplay = ({ productId }: { productId: number }) => {
    const { data: product, isLoading } = useQuery<any>({
      queryKey: [`/api/catalog-products/${productId}`],
      enabled: !!productId,
    });
    
    if (isLoading) return <span className="flex-1 text-sm text-muted-foreground">Ładowanie...</span>;
    if (!product) return <span className="flex-1 text-sm">Produkt ID: {productId}</span>;
    
    return (
      <div className="flex-1">
        <div className="text-sm font-medium">{product.name}</div>
        <div className="text-xs text-muted-foreground">SKU: {product.sku} • ID: {product.id}</div>
      </div>
    );
  };
  
  useEffect(() => {
    if (open && activeColorIndex !== null) {
      if (initializedColorRef.current !== activeColorIndex) {
        setLocalOverrides(JSON.parse(JSON.stringify(currentOverrides)));
        setSelectedComponentType("");
        setSelectedProductId(null);
        setSearchProductQuery("");
        setDebouncedSearch("");
        setProductSearchOpen(false);
        initializedColorRef.current = activeColorIndex;
        
        console.log('🔍 [DIALOG-INIT] Initialized overrides for color index:', activeColorIndex, currentOverrides);
      }
    } else if (!open) {
      initializedColorRef.current = null;
    }
  }, [open, activeColorIndex, currentOverrides]);
  
  const handleSaveOverrides = () => {
    if (activeColorIndex === null) return;
    
    const cleanedOverrides: Record<string, { productId: number }> = {};
    Object.entries(localOverrides).forEach(([componentType, value]) => {
      if (value && value.productId) {
        cleanedOverrides[componentType] = value;
      }
    });
    
    onSaveOverrides(activeColorIndex, cleanedOverrides);
    onActiveColorIndexChange(null);
    onOpenChange(false);
  };
  
  const handleRemoveOverride = (componentType: string) => {
    const newLocalOverrides = { ...localOverrides };
    delete newLocalOverrides[componentType];
    setLocalOverrides(newLocalOverrides);
  };
  
  const handleAddOverride = () => {
    if (!selectedProductId || !selectedComponentType) return;
    
    console.log('🔍 [ADD-OVERRIDE] Before:', {
      selectedComponentType,
      selectedProductId,
      currentLocalOverrides: localOverrides,
    });
    
    const overriddenTypes = new Set(Object.keys(localOverrides));
    const isOverwrite = overriddenTypes.has(selectedComponentType);
    
    const newLocalOverrides = { 
      ...localOverrides,
      [selectedComponentType]: { productId: selectedProductId }
    };
    
    console.log('🔍 [ADD-OVERRIDE] After mutation:', {
      newLocalOverrides,
      overrideCount: Object.keys(newLocalOverrides).length
    });
    
    setLocalOverrides(newLocalOverrides);
    
    console.log('🔍 [ADD-OVERRIDE] State updated, next render should show:', Object.keys(newLocalOverrides));
    
    toast({
      title: isOverwrite ? "Nadpisanie zaktualizowane" : "Nadpisanie dodane",
      description: `${selectedComponentType} → Produkt #${selectedProductId}`,
    });
    
    setSelectedComponentType("");
    setSelectedProductId(null);
    setSearchProductQuery("");
    setDebouncedSearch("");
    setProductSearchOpen(false);
  };
  
  const colorName = activeColorIndex !== null ? colors[activeColorIndex] : "";
  const allComponentTypes = components.map(c => c.componentType).filter(Boolean);
  const overriddenTypes = new Set(Object.keys(localOverrides));
  
  console.log('🔍 [RENDER] ComponentOverridesDialog:', {
    localOverridesKeys: Object.keys(localOverrides),
    localOverridesCount: Object.keys(localOverrides).length,
    localOverrides: localOverrides
  });
  
  return (
    <AlertDialog open={open} onOpenChange={onOpenChange}>
      <AlertDialogContent className="max-w-3xl max-h-[80vh] overflow-y-auto">
        <AlertDialogHeader>
          <AlertDialogTitle>Nadpisania produktów - {colorName}</AlertDialogTitle>
          <AlertDialogDescription>
            Przypisz konkretne produkty z katalogu do komponentów dla tego koloru. Nadpisania mają priorytet nad automatycznym wyszukiwaniem.
          </AlertDialogDescription>
        </AlertDialogHeader>
        
        <div className="space-y-4 py-4">
          {Object.keys(localOverrides).length > 0 ? (
            <div className="space-y-2">
              <Label>Przypisane produkty:</Label>
              {Object.entries(localOverrides).map(([componentType, override]) => (
                <div key={componentType} className="flex items-center gap-2 p-2 border rounded">
                  <Badge variant="outline">{componentType}</Badge>
                  <ProductOverrideDisplay productId={override.productId} />
                  <Button
                    variant="ghost"
                    size="icon"
                    className="h-6 w-6"
                    onClick={() => handleRemoveOverride(componentType)}
                    data-testid={`button-remove-override-${componentType}`}
                  >
                    <X className="h-3 w-3" />
                  </Button>
                </div>
              ))}
            </div>
          ) : (
            <p className="text-sm text-muted-foreground text-center py-4">
              Brak nadpisań dla tego koloru
            </p>
          )}
          
          {allComponentTypes.length > 0 && (
            <div className="space-y-3 pt-4 border-t">
              <Label>Dodaj nadpisanie:</Label>
              
              <div className="space-y-2">
                <Label className="text-xs text-muted-foreground">Wybierz komponent:</Label>
                <Select value={selectedComponentType} onValueChange={setSelectedComponentType}>
                  <SelectTrigger data-testid="select-component">
                    <SelectValue placeholder="Wybierz komponent..." />
                  </SelectTrigger>
                  <SelectContent>
                    {allComponentTypes.map(ct => {
                      const hasOverride = overriddenTypes.has(ct);
                      return (
                        <SelectItem key={ct} value={ct} data-testid={`select-option-${ct}`}>
                          <div className="flex items-center gap-2">
                            <span>{ct}</span>
                            {hasOverride && (
                              <Badge variant="secondary" className="text-xs px-1 py-0 h-4">✓</Badge>
                            )}
                          </div>
                        </SelectItem>
                      );
                    })}
                  </SelectContent>
                </Select>
              </div>
              
              <div className="space-y-2">
                <Label className="text-xs text-muted-foreground">Wybierz produkt (wyszukiwanie z średnikiem):</Label>
                <Popover open={productSearchOpen} onOpenChange={setProductSearchOpen}>
                  <PopoverTrigger asChild>
                    <Button
                      variant="outline"
                      role="combobox"
                      aria-expanded={productSearchOpen}
                      className="w-full justify-between"
                      disabled={!selectedComponentType}
                      data-testid="button-open-product-search"
                    >
                      {selectedProductId ? (
                        <span className="truncate">Produkt ID: {selectedProductId}</span>
                      ) : (
                        <span className="text-muted-foreground">Szukaj produktu...</span>
                      )}
                      <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-full p-0" align="start">
                    <Command>
                      <CommandInput
                        placeholder="Wpisz nazwę, SKU, typ... (użyj ; dla AND np. wiesz;wot)"
                        value={searchProductQuery}
                        onValueChange={setSearchProductQuery}
                        data-testid="command-input-product-search"
                      />
                      <CommandList className="max-h-60 overflow-y-auto">
                        <CommandEmpty>
                          {debouncedSearch.length < 2
                            ? "Wpisz min 2 znaki"
                            : isLoadingProducts
                            ? "Ładowanie..."
                            : "Brak wyników"}
                        </CommandEmpty>
                        {products && products.length > 0 && (
                          <CommandGroup>
                            {products.slice(0, 10).map((product: any) => (
                              <CommandItem
                                key={product.id}
                                value={`${product.title}-${product.sku}-${product.id}`}
                                onSelect={() => {
                                  setSelectedProductId(product.id);
                                  setProductSearchOpen(false);
                                }}
                                data-testid={`command-product-${product.id}`}
                              >
                                <div className="flex flex-col flex-1">
                                  <div className="text-sm font-medium">{product.title}</div>
                                  <div className="text-xs text-muted-foreground">
                                    SKU: {product.sku} • ID: {product.id}
                                  </div>
                                </div>
                                {selectedProductId === product.id && (
                                  <Check className="ml-2 h-4 w-4" />
                                )}
                              </CommandItem>
                            ))}
                          </CommandGroup>
                        )}
                      </CommandList>
                    </Command>
                  </PopoverContent>
                </Popover>
              </div>
              
              <Button
                onClick={handleAddOverride}
                disabled={!isProductValid || !selectedComponentType || isLoadingProducts}
                className="w-full"
                data-testid="button-add-override"
              >
                {isLoadingProducts ? (
                  <>
                    <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                    Ładowanie produktów...
                  </>
                ) : (
                  <>
                    <Plus className="h-4 w-4 mr-2" />
                    Dodaj nadpisanie
                  </>
                )}
              </Button>
            </div>
          )}
        </div>
        
        <AlertDialogFooter>
          <AlertDialogCancel data-testid="button-cancel-overrides">Anuluj</AlertDialogCancel>
          <AlertDialogAction
            onClick={handleSaveOverrides}
            data-testid="button-save-overrides"
          >
            Zapisz nadpisania
          </AlertDialogAction>
        </AlertDialogFooter>
      </AlertDialogContent>
    </AlertDialog>
  );
}

// Hook do bulk fetchowania produktów
function useBulkCatalogProducts(productIds: number[]) {
  const sortedIds = useMemo(() => {
    return Array.from(new Set(productIds)).sort((a, b) => a - b);
  }, [productIds]);
  
  const idsString = sortedIds.join(',');
  
  const { data, isLoading } = useQuery<Record<number, { id: number; sku: string; title: string; color: string }>>({
    queryKey: [`/api/catalog-products?ids=${idsString}`],
    enabled: sortedIds.length > 0,
  });
  
  return { products: data || {}, isLoading };
}

export default function SetMatrixSettingsPage() {
  const { toast } = useToast();
  const { user } = useAuth();
  const [, setLocation] = useLocation();
  const [, params] = useRoute("/set-matrices/settings/:id");
  const [showDeleteDialog, setShowDeleteDialog] = useState(false);
  const [expandedColorImages, setExpandedColorImages] = useState<Set<number>>(new Set());
  const [colors, setColors] = useState<string[]>([]);
  
  // Selection state for bulk delete
  const [selectedColorsForDelete, setSelectedColorsForDelete] = useState<number[]>([]);
  
  // Selection state for generation (stores color INDICES, not names, to support duplicate color names)
  const [colorsForGeneration, setColorsForGeneration] = useState<number[]>([]);
  
  // Track original colors for generation (to detect unsaved changes)
  const originalColorsForGenerationRef = useRef<number[]>([]);
  
  const [colorOpen, setColorOpen] = useState(false);
  const [colorSearchInput, setColorSearchInput] = useState("");
  const [colorOpenRows, setColorOpenRows] = useState<Set<number>>(new Set());
  const [colorSearchInputs, setColorSearchInputs] = useState<Map<number, string>>(new Map());
  const [colorOptionsMatrix, setColorOptionsMatrix] = useState<Record<string, string[]>>({});
  
  // State for color options search (per row)
  const [colorOptionSearchInputs, setColorOptionSearchInputs] = useState<Map<number, string>>(new Map());
  const [colorOptionOpenRows, setColorOptionOpenRows] = useState<Set<number>>(new Set());
  const [colorImagesState, setColorImagesState] = useState<Record<string, string[]>>({});
  const [draggedImage, setDraggedImage] = useState<{ color: string; index: number } | null>(null);
  
  // Component product overrides: { "colorIndex": { "componentType": { "productId": number } } }
  const [componentOverrides, setComponentOverrides] = useState<Record<string, Record<string, { productId: number }>>>({});
  const [overridesDialogOpen, setOverridesDialogOpen] = useState(false);
  const [activeOverrideColorIndex, setActiveOverrideColorIndex] = useState<number | null>(null);
  
  // Helper for index-based key management
  const indexKey = (idx: number) => idx.toString();
  
  // Generation states
  const [isGenerating, setIsGenerating] = useState(false);
  const [showPreviewDialog, setShowPreviewDialog] = useState(false);
  const [previewData, setPreviewData] = useState<any>(null);
  
  // Form state
  const [name, setName] = useState("");
  const [namePrefix, setNamePrefix] = useState("");
  const [nameSuffix, setNameSuffix] = useState("");
  const [productGroup, setProductGroup] = useState("");
  const [length, setLength] = useState("");
  const [width, setWidth] = useState("");
  const [height, setHeight] = useState("");
  const [doors, setDoors] = useState("");
  const [legs, setLegs] = useState("");
  const [imageUrl, setImageUrl] = useState("");
  const [modifierType, setModifierType] = useState("amount");
  const [modifierOperation, setModifierOperation] = useState("add");
  const [modifierValue, setModifierValue] = useState("0");
  const [components, setComponents] = useState<{
    name: string;
    componentType: string;
    length: string;
    width: string;
    quantity: number;
    ignoreColorOptions?: boolean;
  }[]>([]);
  const [initialComponents, setInitialComponents] = useState<{
    name: string;
    componentType: string;
    length: string;
    width: string;
    quantity: number;
    ignoreColorOptions?: boolean;
  }[]>([]);
  const [useEnhancedDescription, setUseEnhancedDescription] = useState(false);
  
  const matrixId = params?.id === "new" ? null : params?.id ? parseInt(params.id) : null;
  const isNew = params?.id === "new";

  const { data: matrix, isLoading } = useQuery<SetMatrix>({
    queryKey: [`/api/set-matrices/${matrixId}`],
    enabled: !isNew && !!matrixId && !!user,
  });
  
  // Snapshot for tracking unsaved changes
  const [lastSavedSnapshot, setLastSavedSnapshot] = useState<string>("");
  const [hasUnsavedChanges, setHasUnsavedChanges] = useState(false);
  
  // Helper to create snapshot of current form state
  const createSnapshot = useCallback(() => {
    const snapshot = {
      name,
      namePrefix,
      nameSuffix,
      productGroup,
      length,
      width,
      height,
      doors,
      legs,
      imageUrl,
      modifierType,
      modifierOperation,
      modifierValue,
      components: [...components].sort((a, b) => a.componentType.localeCompare(b.componentType)),
      colors: [...colors].sort(),
      colorOptions: Object.keys(colorOptionsMatrix).sort().reduce((acc, key) => {
        acc[key] = [...(colorOptionsMatrix[key] || [])].sort();
        return acc;
      }, {} as Record<string, string[]>),
      colorImages: Object.keys(colorImagesState).sort().reduce((acc, key) => {
        acc[key] = [...(colorImagesState[key] || [])].sort();
        return acc;
      }, {} as Record<string, string[]>),
      componentOverrides: Object.keys(componentOverrides).sort().reduce((acc, key) => {
        const componentTypes = Object.keys(componentOverrides[key] || {}).sort();
        acc[key] = componentTypes.reduce((innerAcc, compType) => {
          // Deep clone to prevent reference issues in snapshot comparison
          innerAcc[compType] = { productId: componentOverrides[key][compType].productId };
          return innerAcc;
        }, {} as Record<string, { productId: number }>);
        return acc;
      }, {} as Record<string, Record<string, { productId: number }>>),
      selectedColors: [...colorsForGeneration].sort(),
      useEnhancedDescription,
    };
    return JSON.stringify(snapshot);
  }, [name, namePrefix, nameSuffix, productGroup, length, width, height, doors, legs, imageUrl, modifierType, modifierOperation, modifierValue, components, colors, colorOptionsMatrix, colorImagesState, componentOverrides, colorsForGeneration, useEnhancedDescription]);
  
  // Initialize snapshot when data loads
  useEffect(() => {
    if (!isNew && matrix) {
      const snapshot = createSnapshot();
      setLastSavedSnapshot(snapshot);
      setHasUnsavedChanges(false);
    }
  }, [matrix?.id]); // Only on initial load
  
  // Compare current state with saved snapshot
  useEffect(() => {
    if (lastSavedSnapshot) {
      const currentSnapshot = createSnapshot();
      setHasUnsavedChanges(currentSnapshot !== lastSavedSnapshot);
    }
  }, [createSnapshot, lastSavedSnapshot]);

  // Fetch generated sets for this matrix
  interface ProductSet {
    id: number;
    setMatrixId: number | null;
    sku: string;
    title: string;
    color: string | null;
    panelCount: number | null;
    basePrice: string | null;
    calculatedPrice: string | null;
    isActive: boolean;
    createdAt: string;
  }
  
  const { data: generatedSets = [], isLoading: isLoadingSets } = useQuery<ProductSet[]>({
    queryKey: [`/api/set-matrices/${matrixId}/sets`],
    enabled: !isNew && !!matrixId && !!user,
  });

  const { data: colorDictionary = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "color" }],
    enabled: !!user,
  });

  // Fetch product groups dictionary
  const { data: productGroups = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "product_group" }],
    enabled: !!user,
  });

  // Fetch product types dictionary
  const { data: productTypes = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "product_type" }],
    enabled: !!user,
  });

  // Fetch dimension dictionaries
  const { data: dimensionLengths = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "dimension_length" }],
    enabled: !!user,
  });

  const { data: dimensionWidths = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "dimension_width" }],
    enabled: !!user,
  });

  const { data: dimensionHeights = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "dimension_height" }],
    enabled: !!user,
  });

  const { data: doorOptions = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "door" }],
    enabled: !!user,
  });

  const { data: legOptions = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "leg" }],
    enabled: !!user,
  });
  
  // Bulk fetch all product IDs from overrides for efficient rendering
  const allOverrideProductIds = useMemo(() => {
    const ids: number[] = [];
    Object.values(componentOverrides).forEach(overridesByComponent => {
      Object.values(overridesByComponent).forEach(override => {
        if (override?.productId) {
          ids.push(override.productId);
        }
      });
    });
    return ids;
  }, [componentOverrides]);
  
  const { products: bulkProducts, isLoading: isLoadingBulkProducts } = useBulkCatalogProducts(allOverrideProductIds);

  // Fetch component_cz1 and color options for COMPONENT_CZ1-COLOR combinations
  const { data: componentCz1Options = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "component_cz1" }],
    enabled: !!user,
  });

  const { data: colorOptionsDict = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "color" }],
    enabled: !!user,
  });

  // Generate COMPONENT_CZ1-COLOR combinations (e.g., DRZWI-ARTISAN, POLKA-BIAŁY)
  const availableColorOptions = useMemo(() => {
    if (!componentCz1Options || !colorOptionsDict) return [];
    const combinations: Array<{ value: string; label: string }> = [];
    componentCz1Options.forEach(component => {
      colorOptionsDict.forEach(color => {
        combinations.push({
          value: `${component.code}-${color.code}`,
          label: `${component.code}-${color.code}`
        });
      });
    });
    return combinations;
  }, [componentCz1Options, colorOptionsDict]);

  // Generate preview name from current form values
  const previewName = useMemo(() => {
    const nameParts: string[] = [];
    
    // Helper function to get shortName from dictionary
    const getShortName = (code: string, dictionaries: Dictionary[]): string => {
      const item = dictionaries.find(d => d.code === code);
      return item?.shortName || code;
    };
    
    // 1. Prefix
    if (namePrefix.trim()) nameParts.push(namePrefix.trim());
    
    // 2. Grupa produktów (use shortName)
    if (productGroup) {
      const shortName = getShortName(productGroup, productGroups);
      nameParts.push(shortName);
    }
    
    // 3. Długość × Szerokość (konwersja mm -> cm)
    if (length && width) {
      const lengthCm = Math.round(parseFloat(length) / 10);
      const widthCm = Math.round(parseFloat(width) / 10);
      nameParts.push(`${lengthCm}x${widthCm}cm`);
    }
    
    // 4. Rodzaje produktów z komponentów (use shortName)
    const typeNamesList: string[] = [];
    let totalPanelCount = 0;
    
    components.forEach((comp) => {
      const compType = comp.componentType || '';
      const compLength = comp.length ? Math.round(parseFloat(comp.length) / 10) : null;
      
      if (compType === 'PANEL') {
        totalPanelCount += (comp.quantity || 0);
      } else if (compType && compLength) {
        const shortName = getShortName(compType, productTypes);
        typeNamesList.push(`${shortName}${compLength}`);
      } else if (compType) {
        const shortName = getShortName(compType, productTypes);
        typeNamesList.push(shortName);
      }
    });
    
    typeNamesList.forEach(type => nameParts.push(type));
    
    // 5. Liczba paneli (z komponentów)
    if (totalPanelCount > 0) {
      nameParts.push(`${totalPanelCount} Panel${totalPanelCount > 1 ? 'e' : ''}`);
    }
    
    // 6. Drzwi (use shortName)
    if (doors) {
      const shortName = getShortName(doors, doorOptions);
      nameParts.push(shortName);
    }
    
    // 7. Nogi (use shortName)
    if (legs) {
      const shortName = getShortName(legs, legOptions);
      nameParts.push(shortName);
    }
    
    // 8. Kolor - pokazujemy przykładowy pierwszy kolor
    if (colors.length > 0) {
      nameParts.push(colors[0]);
    }
    
    // 9. Suffix
    if (nameSuffix.trim()) nameParts.push(nameSuffix.trim());
    
    return nameParts.filter(Boolean).join(' ').trim();
  }, [namePrefix, productGroup, productGroups, length, width, components, productTypes, doors, doorOptions, legs, legOptions, colors, nameSuffix]);

  // Update form state when matrix data changes
  useEffect(() => {
    if (matrix) {
      setName(matrix.name);
      setNamePrefix(matrix.namePrefix || "");
      setNameSuffix(matrix.nameSuffix || "");
      setProductGroup(matrix.productGroup || "");
      setLength(matrix.length || "");
      setWidth(matrix.width || "");
      setHeight(matrix.height || "");
      setDoors(matrix.doors || "");
      setLegs(matrix.legs || "");
      setImageUrl(matrix.imageUrl || "");
      setModifierType(matrix.modifierType || "amount");
      setModifierOperation(matrix.modifierOperation || "add");
      setModifierValue(matrix.modifierValue || "0");
      const colorArray = matrix.colors || [];
      setColors(colorArray);
      
      // Initialize colorsForGeneration (supports indices, with backward compatibility)
      if (matrix.selectedColors && Array.isArray(matrix.selectedColors) && matrix.selectedColors.length > 0) {
        const indices = matrix.selectedColors
          .map((item: any) => typeof item === 'number' ? item : colorArray.indexOf(item))
          .filter((n: number) => !isNaN(n) && n >= 0 && n < colorArray.length);
        
        setColorsForGeneration(indices);
        originalColorsForGenerationRef.current = indices;
      } else {
        // Select all color indices by default
        const allIndices = colorArray.map((_, idx) => idx);
        setColorsForGeneration(allIndices);
        originalColorsForGenerationRef.current = allIndices;
      }
      
      // Normalize colorOptions to use indices (with backward compatibility for name-based keys)
      const normalizedOptions: Record<string, string[]> = {};
      if (matrix.colorOptions) {
        Object.entries(matrix.colorOptions).forEach(([key, value]) => {
          // If key is numeric, keep it as-is (new format)
          if (!isNaN(Number(key))) {
            normalizedOptions[key] = value;
          } else {
            // Legacy name-based key - convert to index
            const index = (matrix.colors || []).indexOf(key);
            if (index >= 0) {
              normalizedOptions[index.toString()] = value;
            }
          }
        });
      }
      setColorOptionsMatrix(normalizedOptions);
      
      // Normalize colorImages to use indices (same strategy)
      const normalizedImages: Record<string, string[]> = {};
      if (matrix.colorImages) {
        Object.entries(matrix.colorImages).forEach(([key, value]) => {
          if (!isNaN(Number(key))) {
            normalizedImages[key] = value;
          } else {
            const index = (matrix.colors || []).indexOf(key);
            if (index >= 0) {
              normalizedImages[index.toString()] = value;
            }
          }
        });
      }
      setColorImagesState(normalizedImages);
      
      // Initialize component overrides (prune empty inner objects as recommended by architect)
      const pruneEmptyOverrides = (overrides: Record<string, Record<string, { productId: number }>>) => {
        const pruned: Record<string, Record<string, { productId: number }>> = {};
        Object.entries(overrides || {}).forEach(([colorKey, componentMap]) => {
          if (componentMap && Object.keys(componentMap).length > 0) {
            pruned[colorKey] = componentMap;
          }
        });
        return pruned;
      };
      setComponentOverrides(pruneEmptyOverrides((matrix as any).componentProductOverrides || {}));
      
      setUseEnhancedDescription(matrix.useEnhancedDescription || false);
      
      // Convert components for editing
      const componentsData = Array.isArray(matrix.components) 
        ? matrix.components.map(comp => ({
            ...comp,
            length: comp.length ? String(comp.length) : "",
            width: comp.width ? String(comp.width) : "",
          }))
        : [];
      setComponents(componentsData);
      setInitialComponents(JSON.parse(JSON.stringify(componentsData))); // Deep copy
    }
  }, [matrix]);

  const createMutation = useMutation({
    mutationFn: async (data: { name: string; productGroup: string | null; imageUrl: string | null; components: any[]; colors: string[]; colorOptions?: Record<string, string[]> }) => {
      const response = await fetch("/api/set-matrices", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(data),
      });
      if (!response.ok) {
        throw new Error(await response.text());
      }
      return response.json();
    },
    onSuccess: (newMatrix) => {
      queryClient.invalidateQueries({ queryKey: ["/api/set-matrices"] });
      
      // Reset original colors for generation ref (no more unsaved changes)
      originalColorsForGenerationRef.current = [...colorsForGeneration];
      
      // Update saved snapshot
      const snapshot = createSnapshot();
      setLastSavedSnapshot(snapshot);
      setHasUnsavedChanges(false);
      
      toast({
        title: "Sukces",
        description: "Matryca została utworzona",
      });
      setLocation(`/set-matrices/settings/${newMatrix.id}`);
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async (data: { name: string; productGroup: string | null; imageUrl: string | null; components: any[]; colors?: string[]; colorOptions?: Record<string, string[]> }) => {
      const response = await fetch(`/api/set-matrices/${matrixId}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(data),
      });
      if (!response.ok) {
        throw new Error(await response.text());
      }
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/set-matrices"] });
      queryClient.invalidateQueries({ queryKey: [`/api/set-matrices/${matrixId}`] });
      
      // Reset initial components state after successful save
      setInitialComponents(JSON.parse(JSON.stringify(components)));
      
      // Reset original colors for generation ref (no more unsaved changes)
      originalColorsForGenerationRef.current = [...colorsForGeneration];
      
      // Update saved snapshot
      const snapshot = createSnapshot();
      setLastSavedSnapshot(snapshot);
      setHasUnsavedChanges(false);
      
      toast({
        title: "Sukces",
        description: "Matryca została zaktualizowana",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const updateMatrixMutation = useMutation({
    mutationFn: async (updates: Partial<SetMatrix>) => {
      const response = await apiRequest("PUT", `/api/set-matrices/${matrixId}`, updates);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/set-matrices/${matrixId}`] });
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await fetch(`/api/set-matrices/${id}`, {
        method: "DELETE",
      });
      if (!response.ok) {
        throw new Error(await response.text());
      }
      return response.status === 204 ? null : response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/set-matrices"] });
      toast({
        title: "Sukces",
        description: "Matryca została usunięta",
      });
      setLocation("/set-matrices");
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const previewMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("POST", `/api/set-matrices/${id}/preview-generation`, {});
    },
    onSuccess: (data: any) => {
      setPreviewData(data);
      setShowPreviewDialog(true);
      toast({
        title: "Podgląd wygenerowany",
        description: data?.preview?.setName ? `Podgląd dla: ${data.preview.setName}` : "Podgląd wygenerowany pomyślnie",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd generowania podglądu",
        description: error.message || "Nie udało się wygenerować podglądu.",
        variant: "destructive",
      });
    },
  });

  const generateSetsMutation = useMutation({
    mutationFn: async (id: number) => {
      const sessionId = `generate-sets-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
      return apiRequest("POST", `/api/set-matrices/${id}/generate`, { sessionId });
    },
    onSuccess: (data: any) => {
      setIsGenerating(false);
      setShowPreviewDialog(false);
      queryClient.invalidateQueries({ queryKey: [`/api/set-matrices/${matrixId}/sets`] });
      toast({
        title: "Zestawy wygenerowane",
        description: `Pomyślnie wygenerowano ${data.setsGenerated} zestawów produktów.`,
      });
    },
    onError: (error: any) => {
      setIsGenerating(false);
      toast({
        title: "Błąd generowania",
        description: error.message || "Nie udało się wygenerować zestawów.",
        variant: "destructive",
      });
    },
  });

  const handleStartGeneration = () => {
    if (isNew || !matrixId) {
      toast({
        title: "Zapisz najpierw",
        description: "Musisz najpierw zapisać matrycę przed rozpoczęciem generowania.",
        variant: "destructive",
      });
      return;
    }

    setLocation(`/set-matrices/${matrixId}/generate`);
  };

  const handleGenerate = () => {
    if (isNew || !matrixId) {
      toast({
        title: "Zapisz najpierw",
        description: "Musisz najpierw zapisać matrycę przed generowaniem zestawów.",
        variant: "destructive",
      });
      return;
    }

    setIsGenerating(true);
    generateSetsMutation.mutate(matrixId, {
      onSettled: () => setIsGenerating(false),
    });
  };

  const handleSave = () => {
    if (!name.trim()) {
      toast({
        title: "Błąd",
        description: "Nazwa matrycy jest wymagana",
        variant: "destructive",
      });
      return;
    }

    // Convert string values to appropriate types for database
    const processedComponents = components.map(comp => {
      const lengthNum = comp.length ? parseInt(comp.length as any, 10) : null;
      const widthNum = comp.width ? parseInt(comp.width as any, 10) : null;
      
      return {
        name: comp.name || "",
        componentType: comp.componentType,
        length: lengthNum && !isNaN(lengthNum) ? lengthNum : null,
        width: widthNum && !isNaN(widthNum) ? widthNum : null,
        quantity: comp.quantity,
        ignoreColorOptions: comp.ignoreColorOptions || false,
      };
    });

    const data = {
      name: name.trim(),
      namePrefix: namePrefix || null,
      nameSuffix: nameSuffix || null,
      productGroup: productGroup || null,
      length: length || null,
      width: width || null,
      height: height || null,
      doors: doors || null,
      legs: legs || null,
      imageUrl: imageUrl.trim() || null,
      modifierType: modifierType || "amount",
      modifierOperation: modifierOperation || "add",
      modifierValue: modifierValue || "0",
      components: processedComponents,
      colors: colors,
      selectedColors: colorsForGeneration,  // Save selected indices
      colorOptions: colorOptionsMatrix,
      componentProductOverrides: componentOverrides,  // Save component overrides
      useEnhancedDescription: useEnhancedDescription,
    };

    if (isNew) {
      createMutation.mutate(data);
    } else {
      updateMutation.mutate(data);
    }
  };

  const handleDelete = () => {
    if (matrixId) {
      deleteMutation.mutate(matrixId);
    }
  };

  const handleAddColor = (colorCode: string) => {
    if (!colorCode) return;
    
    // Allow duplicate colors (same name with different options)
    const newColors = [...colors, colorCode];
    setColors(newColors);
    
    if (!isNew) {
      updateMatrixMutation.mutate({ colors: newColors });
    }
    setColorOpen(false);
    setColorSearchInput("");
  };

  // Duplicate color with all its options and images
  const duplicateColor = (index: number) => {
    const colorName = colors[index];
    if (!colorName) return;
    
    // Add same color name again
    const newIndex = colors.length;
    const newColors = [...colors, colorName];
    setColors(newColors);
    
    // Copy color options to new index
    const sourceOptions = colorOptionsMatrix[indexKey(index)];
    const newColorOptionsMatrix = { ...colorOptionsMatrix };
    if (sourceOptions && sourceOptions.length > 0) {
      newColorOptionsMatrix[indexKey(newIndex)] = [...sourceOptions];
    }
    setColorOptionsMatrix(newColorOptionsMatrix);
    
    // Copy color images to new index
    const sourceImages = colorImagesState[indexKey(index)];
    const newColorImages = { ...colorImagesState };
    if (sourceImages && sourceImages.length > 0) {
      newColorImages[indexKey(newIndex)] = [...sourceImages];
    }
    setColorImagesState(newColorImages);
    
    // Copy component overrides to new index
    const sourceOverrides = componentOverrides[indexKey(index)];
    const newComponentOverrides = { ...componentOverrides };
    if (sourceOverrides && Object.keys(sourceOverrides).length > 0) {
      newComponentOverrides[indexKey(newIndex)] = JSON.parse(JSON.stringify(sourceOverrides));
    }
    setComponentOverrides(newComponentOverrides);
    
    // Auto-save to backend
    if (!isNew) {
      updateMatrixMutation.mutate({
        colors: newColors,
        colorOptions: newColorOptionsMatrix,
        colorImages: newColorImages,
        componentProductOverrides: newComponentOverrides
      });
    }
  };

  const handleRemoveColor = (index: number) => {
    const newColors = colors.filter((_, i) => i !== index);
    setColors(newColors);
    
    // Reindex colorsForGeneration: remove this index and decrement all higher indices
    const newColorsForGeneration = colorsForGeneration
      .filter(c => c !== index)  // Remove the deleted index
      .map(c => c > index ? c - 1 : c);  // Decrement indices that were after deleted one
    setColorsForGeneration(newColorsForGeneration);
    
    // Reindex colorOptions, colorImages, and componentOverrides
    const newColorOptions: Record<string, string[]> = {};
    const newColorImages: Record<string, string[]> = {};
    const newComponentOverrides: Record<string, Record<string, { productId: number }>> = {};
    
    Object.entries(colorOptionsMatrix).forEach(([key, value]) => {
      const keyIndex = parseInt(key);
      if (keyIndex < index) {
        newColorOptions[key] = value;
      } else if (keyIndex > index) {
        newColorOptions[indexKey(keyIndex - 1)] = value;
      }
    });
    
    Object.entries(colorImagesState).forEach(([key, value]) => {
      const keyIndex = parseInt(key);
      if (keyIndex < index) {
        newColorImages[key] = value;
      } else if (keyIndex > index) {
        newColorImages[indexKey(keyIndex - 1)] = value;
      }
    });
    
    Object.entries(componentOverrides).forEach(([key, value]) => {
      const keyIndex = parseInt(key);
      if (keyIndex < index) {
        newComponentOverrides[key] = value;
      } else if (keyIndex > index) {
        newComponentOverrides[indexKey(keyIndex - 1)] = value;
      }
    });
    
    setColorOptionsMatrix(newColorOptions);
    setColorImagesState(newColorImages);
    setComponentOverrides(newComponentOverrides);
    
    if (!isNew) {
      updateMatrixMutation.mutate({ 
        colors: newColors,
        selectedColors: newColorsForGeneration,
        colorImages: newColorImages,
        colorOptions: newColorOptions,
        componentProductOverrides: newComponentOverrides
      });
    }
  };
  
  // Toggle color for generation (NO auto-save - just local state)
  // colorIndex is the position in colors array
  const toggleColorForGeneration = (colorIndex: number) => {
    const wasSelected = colorsForGeneration.includes(colorIndex);
    const newSelectedIndices = wasSelected
      ? colorsForGeneration.filter(idx => idx !== colorIndex)
      : [...colorsForGeneration, colorIndex];
    
    setColorsForGeneration(newSelectedIndices);
  };

  // Toggle all colors for generation (NO auto-save - will save when clicking "Zapisz")
  const toggleAllColorsForGeneration = () => {
    const newSelectedIndices = colorsForGeneration.length === colors.length 
      ? [] 
      : colors.map((_, idx) => idx);
    
    setColorsForGeneration(newSelectedIndices);
  };

  // Toggle color selection for delete
  const toggleColorSelection = (index: number) => {
    setSelectedColorsForDelete(prev => 
      prev.includes(index) ? prev.filter(i => i !== index) : [...prev, index]
    );
  };

  // Toggle all colors for delete
  const toggleAllColors = () => {
    if (selectedColorsForDelete.length === colors.length) {
      setSelectedColorsForDelete([]);
    } else {
      setSelectedColorsForDelete(Array.from({ length: colors.length }, (_, i) => i));
    }
  };

  // Add color row (adds empty string, user selects color in the row)
  const addColor = () => {
    setColors(prev => [...prev, ""]);
  };

  // Remove selected colors
  const removeSelectedColors = () => {
    if (selectedColorsForDelete.length === 0) return;
    
    // Create index mapping: old index -> new index for colors that remain
    const indexMap = new Map<number, number>();
    let newIndex = 0;
    colors.forEach((_, oldIndex) => {
      if (!selectedColorsForDelete.includes(oldIndex)) {
        indexMap.set(oldIndex, newIndex);
        newIndex++;
      }
    });
    
    // Update main colors list
    const newColors = colors.filter((_, i) => !selectedColorsForDelete.includes(i));
    setColors(newColors);
    
    // Remap colorsForGeneration indices
    setColorsForGeneration(prev => 
      prev
        .filter(idx => !selectedColorsForDelete.includes(idx))
        .map(idx => indexMap.get(idx) ?? idx)
        .filter(idx => idx !== undefined)
    );
    
    // Reindex colorOptions
    const newColorOptions: Record<string, string[]> = {};
    Object.entries(colorOptionsMatrix).forEach(([key, value]) => {
      const keyIndex = parseInt(key);
      if (!selectedColorsForDelete.includes(keyIndex)) {
        const newIdx = indexMap.get(keyIndex);
        if (newIdx !== undefined) {
          newColorOptions[newIdx.toString()] = value;
        }
      }
    });
    setColorOptionsMatrix(newColorOptions);
    
    // Reindex colorImages
    const newColorImages: Record<string, string[]> = {};
    Object.entries(colorImagesState).forEach(([key, value]) => {
      const keyIndex = parseInt(key);
      if (!selectedColorsForDelete.includes(keyIndex)) {
        const newIdx = indexMap.get(keyIndex);
        if (newIdx !== undefined) {
          newColorImages[newIdx.toString()] = value;
        }
      }
    });
    setColorImagesState(newColorImages);
    
    setSelectedColorsForDelete([]);
    toast({ title: `Usunięto ${selectedColorsForDelete.length} kolorów` });
  };

  const handleColorImageUpload = async (colorIndex: number, files: FileList | null) => {
    if (!files || files.length === 0 || !matrixId) return;

    for (const file of Array.from(files)) {
      const formData = new FormData();
      formData.append('image', file);
      formData.append('colorIndex', colorIndex.toString());

      try {
        const response = await fetch(`/api/set-matrices/${matrixId}/color-images`, {
          method: 'POST',
          body: formData,
          credentials: 'include',
        });

        if (!response.ok) {
          const text = await response.text();
          throw new Error(`${response.status}: ${text}`);
        }

        await response.json();
        
        queryClient.invalidateQueries({ queryKey: [`/api/set-matrices/${matrixId}`] });

        toast({
          title: "Zdjęcie dodane",
          description: `Dodano zdjęcie dla koloru ${colors[colorIndex]}`,
        });
      } catch (error: any) {
        toast({
          title: "Błąd uploadu",
          description: error.message || "Nie udało się dodać zdjęcia",
          variant: "destructive",
        });
      }
    }
  };

  const handleColorImageDelete = async (colorIndex: number, imageUrl: string) => {
    if (!matrixId) return;

    const filename = imageUrl.split('/').pop();
    if (!filename) return;

    try {
      const response = await fetch(`/api/set-matrices/${matrixId}/color-images/${colorIndex}/${encodeURIComponent(filename)}`, {
        method: 'DELETE',
        credentials: 'include',
      });

      if (!response.ok) {
        const text = await response.text();
        throw new Error(`${response.status}: ${text}`);
      }

      queryClient.invalidateQueries({ queryKey: [`/api/set-matrices/${matrixId}`] });

      toast({
        title: "Zdjęcie usunięte",
        description: `Usunięto zdjęcie dla koloru ${colors[colorIndex]}`,
      });
    } catch (error: any) {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć zdjęcia",
        variant: "destructive",
      });
    }
  };

  const handleImageReorder = async (colorIndex: number, fromIndex: number, toIndex: number) => {
    if (!matrixId || fromIndex === toIndex) return;
    
    const currentImages = [...(colorImagesState[indexKey(colorIndex)] || [])];
    if (fromIndex < 0 || fromIndex >= currentImages.length || toIndex < 0 || toIndex >= currentImages.length) return;
    
    // Reorder array
    const [movedImage] = currentImages.splice(fromIndex, 1);
    currentImages.splice(toIndex, 0, movedImage);
    
    // Update state with new order
    const updatedColorImages = {
      ...colorImagesState,
      [indexKey(colorIndex)]: currentImages
    };
    setColorImagesState(updatedColorImages);
    
    try {
      await updateMatrixMutation.mutateAsync({ colorImages: updatedColorImages });
      
      toast({
        title: "Kolejność zmieniona",
        description: `Zmieniono kolejność zdjęć dla koloru ${colors[colorIndex]}`,
      });
    } catch (error: any) {
      toast({
        title: "Błąd zmiany kolejności",
        description: error.message || "Nie udało się zmienić kolejności",
        variant: "destructive",
      });
    }
  };

  const handleAddComponent = () => {
    setComponents([
      ...components,
      {
        name: "",
        componentType: "",
        length: "",
        width: "",
        quantity: 1,
        ignoreColorOptions: false,
      },
    ]);
  };

  const handleRemoveComponent = (index: number) => {
    setComponents(components.filter((_, i) => i !== index));
  };

  const handleUpdateComponent = (index: number, field: string, value: any) => {
    const updated = [...components];
    // Ensure value is never undefined/null for string fields
    if ((field === 'length' || field === 'width' || field === 'name' || field === 'componentType') && (value === undefined || value === null)) {
      value = "";
    }
    updated[index] = { ...updated[index], [field]: value };
    setComponents(updated);
  };

  // Check if component has unsaved changes
  const hasComponentChanged = (index: number): boolean => {
    if (index >= initialComponents.length) return true; // New component
    const current = components[index];
    const initial = initialComponents[index];
    if (!initial) return true;
    
    return (
      current.name !== initial.name ||
      current.componentType !== initial.componentType ||
      current.length !== initial.length ||
      current.width !== initial.width ||
      current.quantity !== initial.quantity ||
      (current.ignoreColorOptions || false) !== (initial.ignoreColorOptions || false)
    );
  };

  const onDrop = useCallback((acceptedFiles: File[]) => {
    const file = acceptedFiles[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = () => {
        setImageUrl(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  }, []);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop,
    accept: {
      'image/*': ['.png', '.jpg', '.jpeg', '.gif', '.webp']
    },
    maxFiles: 1,
    multiple: false
  });

  const handleRemoveImage = () => {
    setImageUrl("");
  };

  // Parent-level handler for saving component overrides
  // MUST be before early return to follow Rules of Hooks
  const handleSaveComponentOverrides = useCallback((colorIndex: number, overrides: Record<string, { productId: number }>) => {
    const newComponentOverrides = { ...componentOverrides };
    
    if (Object.keys(overrides).length > 0) {
      newComponentOverrides[indexKey(colorIndex)] = overrides;
    } else {
      delete newComponentOverrides[indexKey(colorIndex)];
    }
    
    setComponentOverrides(newComponentOverrides);
    
    if (!isNew) {
      updateMatrixMutation.mutate({ componentProductOverrides: newComponentOverrides });
    }
  }, [componentOverrides, isNew, updateMatrixMutation]);
  
  // Memoize currentOverrides to prevent unnecessary dialog reinitialization
  // MUST be before early return to follow Rules of Hooks
  const currentOverridesForDialog = useMemo(() => {
    return activeOverrideColorIndex !== null 
      ? (componentOverrides[indexKey(activeOverrideColorIndex)] || {})
      : {};
  }, [activeOverrideColorIndex, componentOverrides]);

  if (isLoading && !isNew) {
    return (
      <div className="p-3 sm:p-6">
        <p className="text-muted-foreground">Ładowanie...</p>
      </div>
    );
  }

  const colorImages = matrix?.colorImages || {};

  const filteredColors = colorDictionary.filter(item => {
    const searchLower = colorSearchInput.toLowerCase();
    return item.code.toLowerCase().includes(searchLower) ||
           (item.readableName || "").toLowerCase().includes(searchLower);
  });

  return (
    <div className="p-3 sm:p-6 space-y-4 sm:space-y-6">
      <div className="flex flex-col sm:flex-row gap-3 sm:gap-0 sm:items-center sm:justify-between">
        <div className="flex items-center gap-3">
          <Button
            variant="outline"
            size="icon"
            onClick={() => setLocation("/set-matrices")}
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div>
            <h1 className="text-2xl sm:text-3xl font-bold">
              {isNew ? "Nowa Matryca Zestawu" : name || "Edytuj matrycę"}
            </h1>
            <p className="text-muted-foreground mt-1 text-sm sm:text-base">
              {isNew ? "Utwórz nowy szablon zestawu produktów" : "Edytuj szablon zestawu produktów"}
            </p>
          </div>
        </div>
        <div className="flex gap-2">
          {!isNew && (
            <Button
              onClick={handleStartGeneration}
              disabled={isGenerating}
              data-testid="button-generate-sets"
              className="flex-1 sm:flex-none"
              variant="default"
            >
              <Play className="h-4 w-4 mr-2" />
              Generuj zestawy
            </Button>
          )}
          <Button
            onClick={handleSave}
            disabled={createMutation.isPending || updateMutation.isPending}
            data-testid="button-save-matrix"
            className={`flex-1 sm:flex-none ${
              !isNew && hasUnsavedChanges
                ? "bg-amber-500 hover:bg-amber-600 text-white"
                : !isNew
                ? "bg-green-600 hover:bg-green-700 text-white"
                : ""
            }`}
          >
            <Save className="h-4 w-4 mr-2" />
            {createMutation.isPending || updateMutation.isPending
              ? "Zapisywanie..."
              : isNew
              ? "Utwórz matrycę"
              : "Zapisz zmiany"}
          </Button>
          {!isNew && (
            <Button
              variant="destructive"
              onClick={() => setShowDeleteDialog(true)}
              data-testid="button-delete-matrix"
              className="flex-1 sm:flex-none"
            >
              <Trash2 className="h-4 w-4 mr-2" />
              Usuń
            </Button>
          )}
        </div>
      </div>

      {/* Podstawowe informacje i zdjęcie */}
      <Card>
        <CardContent className="p-4 sm:p-6">
          {/* 3 kolumny: 2 z polami formularza + 1 ze zdjęciem */}
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-4">
            {/* Kolumna 1 - Nazwa i Prefix */}
            <div className="space-y-4">
              <div>
                <Label htmlFor="name">Nazwa matrycy</Label>
                <Input
                  id="name"
                  value={name}
                  onChange={(e) => setName(e.target.value)}
                  placeholder="np. Zestaw kuchenny podstawowy"
                  data-testid="input-matrix-name"
                />
              </div>

              <div>
                <Label htmlFor="namePrefix">Prefix nazwy</Label>
                <Input
                  id="namePrefix"
                  value={namePrefix}
                  onChange={(e) => setNamePrefix(e.target.value)}
                  placeholder="np. Garderoba"
                  data-testid="input-name-prefix"
                />
              </div>
            </div>

            {/* Kolumna 2 - Grupa i Suffix */}
            <div className="space-y-4">
              <div>
                <Label htmlFor="productGroup">Grupa produktów</Label>
                <DictionaryComboboxWithAdd
                  items={productGroups}
                  value={productGroup}
                  onChange={setProductGroup}
                  dictionaryType="product_group"
                  placeholder="Wybierz grupę produktów"
                  searchPlaceholder="Szukaj grupy..."
                  emptyText="Nie znaleziono grup"
                  testId="select-product-group"
                  valueField="code"
                  displayField="name"
                />
              </div>

              <div>
                <Label htmlFor="nameSuffix">Suffix nazwy</Label>
                <Input
                  id="nameSuffix"
                  value={nameSuffix}
                  onChange={(e) => setNameSuffix(e.target.value)}
                  placeholder="np. Premium"
                  data-testid="input-name-suffix"
                />
              </div>
            </div>

            {/* Kolumna 3 - Zdjęcie poglądowe */}
            <div>
              <Label>Zdjęcie poglądowe</Label>
              {imageUrl ? (
                <div className="space-y-2">
                  <div className="relative inline-block">
                    <img 
                      src={imageUrl} 
                      alt="Podgląd" 
                      className="h-20 w-auto object-contain border"
                    />
                    <Button
                      type="button"
                      variant="destructive"
                      size="icon"
                      className="absolute -top-2 -right-2 h-6 w-6"
                      onClick={handleRemoveImage}
                      data-testid="button-remove-image"
                    >
                      <X className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
              ) : (
                <div
                  {...getRootProps()}
                  className={`border-2 border-dashed p-4 text-center cursor-pointer transition-colors ${
                    isDragActive ? 'border-primary bg-primary/5' : 'border-muted-foreground/25 hover:border-primary/50'
                  }`}
                  data-testid="dropzone-image"
                >
                  <input {...getInputProps()} />
                  <Upload className="h-6 w-6 mx-auto mb-2 text-muted-foreground" />
                  {isDragActive ? (
                    <p className="text-sm text-muted-foreground">Upuść zdjęcie tutaj...</p>
                  ) : (
                    <>
                      <p className="text-sm text-muted-foreground mb-1">
                        Przeciągnij i upuść zdjęcie lub kliknij aby wybrać
                      </p>
                      <p className="text-xs text-muted-foreground">
                        PNG, JPG, GIF, WEBP (maks. 5MB)
                      </p>
                    </>
                  )}
                </div>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* SEKCJA WYMIARÓW, DRZWI, NOGI I CENY */}
      <Card>
        <CardHeader>
          <CardTitle>Parametry zestawu i generowanie nazwy</CardTitle>
          <CardDescription>Wymiary, komponenty i cena - wpływają na nazwę produktu</CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Wymiary */}
          <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
            <div>
              <Label htmlFor="length">Długość (mm)</Label>
              <DictionaryComboboxWithAdd
                items={dimensionLengths}
                value={length}
                onChange={setLength}
                dictionaryType="dimension_length"
                placeholder="Wybierz długość"
                searchPlaceholder="Szukaj długości..."
                emptyText="Nie znaleziono długości"
                testId="select-length"
                valueField="code"
                displayField="name"
              />
            </div>
            <div>
              <Label htmlFor="width">Szerokość (mm)</Label>
              <DictionaryComboboxWithAdd
                items={dimensionWidths}
                value={width}
                onChange={setWidth}
                dictionaryType="dimension_width"
                placeholder="Wybierz szerokość"
                searchPlaceholder="Szukaj szerokości..."
                emptyText="Nie znaleziono szerokości"
                testId="select-width"
                valueField="code"
                displayField="name"
              />
            </div>
            <div>
              <Label htmlFor="height">Wysokość (mm)</Label>
              <DictionaryComboboxWithAdd
                items={dimensionHeights}
                value={height}
                onChange={setHeight}
                dictionaryType="dimension_height"
                placeholder="Wybierz wysokość"
                searchPlaceholder="Szukaj wysokości..."
                emptyText="Nie znaleziono wysokości"
                testId="select-height"
                valueField="code"
                displayField="name"
              />
            </div>
          </div>

          {/* Drzwi i Nogi */}
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
            <div>
              <Label htmlFor="doors">Drzwi</Label>
              <DictionaryComboboxWithAdd
                items={doorOptions}
                value={doors}
                onChange={setDoors}
                dictionaryType="door"
                placeholder="Wybierz drzwi"
                searchPlaceholder="Szukaj drzwi..."
                emptyText="Nie znaleziono drzwi"
                testId="select-doors"
                valueField="code"
                displayField="name"
              />
            </div>
            <div>
              <Label htmlFor="legs">Nogi</Label>
              <DictionaryComboboxWithAdd
                items={legOptions}
                value={legs}
                onChange={setLegs}
                dictionaryType="leg"
                placeholder="Wybierz nogi"
                searchPlaceholder="Szukaj nóg..."
                emptyText="Nie znaleziono nóg"
                testId="select-legs"
                valueField="code"
                displayField="name"
              />
            </div>
          </div>

          {/* Modyfikatory ceny */}
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
            <div className="grid grid-cols-3 gap-4">
              <div>
                <Label htmlFor="modifierType">Typ modyfikatora</Label>
                <Select
                  value={modifierType}
                  onValueChange={setModifierType}
                >
                  <SelectTrigger id="modifierType" data-testid="select-modifier-type">
                    <SelectValue placeholder="Wybierz typ" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="amount">Kwota (PLN)</SelectItem>
                    <SelectItem value="percentage">Procent (%)</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="modifierOperation">Operacja</Label>
                <Select
                  value={modifierOperation}
                  onValueChange={setModifierOperation}
                >
                  <SelectTrigger id="modifierOperation" data-testid="select-modifier-operation">
                    <SelectValue placeholder="Wybierz operację" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="add">Dodaj (+)</SelectItem>
                    <SelectItem value="subtract">Odejmij (-)</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="modifierValue">Wartość</Label>
                <Input
                  id="modifierValue"
                  type="number"
                  step="0.01"
                  value={modifierValue}
                  onChange={(e) => setModifierValue(e.target.value)}
                  placeholder="np. 15"
                  data-testid="input-modifier-value"
                />
              </div>
            </div>
          </div>

          {/* Opcje generowania */}
          <div className="flex items-start space-x-2 pt-4">
            <Checkbox
              id="useEnhancedDescription"
              checked={useEnhancedDescription}
              onCheckedChange={(checked) => setUseEnhancedDescription(checked === true)}
              data-testid="checkbox-use-enhanced-description"
            />
            <div className="space-y-1">
              <Label
                htmlFor="useEnhancedDescription"
                className="text-sm font-medium leading-none cursor-pointer"
              >
                Użyj rozszerzonych opisów AI
              </Label>
              <p className="text-xs text-muted-foreground">
                Automatycznie generuj szczegółowe opisy zestawów na podstawie opisów AI komponentów
              </p>
            </div>
          </div>

          {/* Live preview nazwy */}
          <div className="border-t pt-6">
            <Label className="text-base font-semibold mb-3 block">Podgląd wygenerowanej nazwy</Label>
            <div className="bg-muted p-4 rounded-md space-y-2">
              <div className="font-mono text-sm break-all" data-testid="text-name-preview">
                {previewName || <span className="text-muted-foreground italic">Wprowadź dane aby zobaczyć podgląd...</span>}
              </div>
              <div className="text-xs text-muted-foreground">
                Liczba znaków: <span className="font-semibold">{previewName.length}</span>
                {previewName.length > 100 && <span className="text-destructive ml-2">(Uwaga: nazwa może być zbyt długa)</span>}
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* SEKCJA KOLORÓW */}
      <Card>
        <CardHeader className="p-3 sm:p-4">
          <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-3">
            <Popover open={colorOpen} onOpenChange={setColorOpen}>
              <PopoverTrigger asChild>
                <Button
                  variant="default"
                  size="default"
                  role="combobox"
                  aria-expanded={colorOpen}
                  className="w-full sm:w-auto"
                  data-testid="button-add-color"
                >
                  <Plus className="h-4 w-4 mr-2" />
                  Dodaj kolor
                  <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                </Button>
              </PopoverTrigger>
              <PopoverContent className="w-[250px] p-0">
                <Command>
                  <CommandInput
                    placeholder="Szukaj koloru..."
                    value={colorSearchInput}
                    onValueChange={setColorSearchInput}
                  />
                  <CommandList>
                    <CommandEmpty>Brak wyników</CommandEmpty>
                    <CommandGroup>
                      {filteredColors.map((color) => (
                        <CommandItem
                          key={color.id}
                          value={color.code}
                          onSelect={() => handleAddColor(color.code)}
                        >
                          <Check
                            className={cn(
                              "mr-2 h-4 w-4",
                              colors.includes(color.code) ? "opacity-100" : "opacity-0"
                            )}
                          />
                          <div className="flex flex-col">
                            <span className="font-medium">{color.code}</span>
                            <span className="text-xs text-muted-foreground">
                              {color.readableName}
                            </span>
                          </div>
                        </CommandItem>
                      ))}
                    </CommandGroup>
                  </CommandList>
                </Command>
              </PopoverContent>
            </Popover>
            <div className="flex items-center gap-2">
              <Palette className="h-4 w-4" />
              <div>
                <CardTitle className="text-base sm:text-lg">Kolory ({colors.length})</CardTitle>
                <CardDescription className="text-xs">
                  Zarządzaj kolorami i ich zdjęciami dla generatora zestawów
                </CardDescription>
              </div>
            </div>
          </div>
        </CardHeader>
        <CardContent className="p-3 sm:p-4">
          {colors.length > 0 ? (
            <>
              {selectedColorsForDelete.length > 0 && (
                <div className="mb-4">
                  <Button
                    variant="destructive"
                    size="sm"
                    onClick={removeSelectedColors}
                    data-testid="button-delete-selected-colors"
                  >
                    <Trash2 className="w-4 h-4 md:mr-2" />
                    <span className="hidden md:inline">Usuń zaznaczone</span> ({selectedColorsForDelete.length})
                  </Button>
                </div>
              )}

            <div className="border rounded-lg overflow-x-auto">
              <Table className="min-w-[700px]">
                <TableHeader>
                  <TableRow>
                    <TableHead className="w-10 py-1 px-2 h-8">
                      <Checkbox
                        checked={selectedColorsForDelete.length === colors.length && colors.length > 0}
                        onCheckedChange={toggleAllColors}
                        data-testid="checkbox-select-all-colors"
                      />
                    </TableHead>
                    <TableHead className="w-10 py-1 px-2 h-8">#</TableHead>
                    <TableHead className="min-w-[120px] py-1 px-1.5 h-8 text-xs">Kolor</TableHead>
                    <TableHead className="min-w-[130px] py-1 px-2 text-xs h-8">Zdjęcia</TableHead>
                    <TableHead className="w-16 py-1 px-1 h-8">
                      <div className="flex items-center gap-1 justify-center whitespace-nowrap">
                        <span className="text-xs">Gen.</span>
                        <Checkbox
                          className="h-3.5 w-3.5"
                          checked={colorsForGeneration.length === colors.length && colors.length > 0}
                          onCheckedChange={toggleAllColorsForGeneration}
                          data-testid="checkbox-select-all-for-generation"
                        />
                      </div>
                    </TableHead>
                    <TableHead className="min-w-[80px] py-1 px-2 text-xs h-8">Opcje kolor</TableHead>
                    <TableHead className="min-w-[200px] py-1 px-2 text-xs h-8">Nadpisane produkty</TableHead>
                    <TableHead className="w-20 text-right py-1 px-2 text-xs h-8">Akcje</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {colors.map((color, index) => {
                    const colorDict = colorDictionary.find(c => c.code === color);
                    const isSelectedForGeneration = colorsForGeneration.includes(index);
                    const wasOriginallySelected = originalColorsForGenerationRef.current.includes(index);
                    const hasUnsavedChange = isSelectedForGeneration !== wasOriginallySelected;
                    
                    return (
                    <TableRow 
                      key={index}
                      className={
                        hasUnsavedChange 
                          ? 'bg-red-50 dark:bg-red-900/20 border-l-4 border-l-red-500' 
                          : isSelectedForGeneration 
                            ? 'bg-green-50 dark:bg-green-900/20' 
                            : ''
                      }
                    >
                      <TableCell className="py-1 px-2">
                        <Checkbox
                          checked={selectedColorsForDelete.includes(index)}
                          onCheckedChange={() => toggleColorSelection(index)}
                          data-testid={`checkbox-color-${index}`}
                        />
                      </TableCell>
                      <TableCell className="font-mono text-muted-foreground py-1 px-2">
                        {index + 1}
                      </TableCell>
                      <TableCell className="py-1 px-1.5">
                        <Popover 
                          open={colorOpenRows.has(index)} 
                          onOpenChange={(open) => {
                            if (open) {
                              setColorOpenRows(prev => {
                                const newSet = new Set(prev);
                                newSet.add(index);
                                return newSet;
                              });
                            } else {
                              setColorSearchInputs(prev => {
                                const newMap = new Map(prev);
                                newMap.delete(index);
                                return newMap;
                              });
                              setColorOpenRows(prev => {
                                const newSet = new Set(prev);
                                newSet.delete(index);
                                return newSet;
                              });
                            }
                          }}
                        >
                          <PopoverTrigger asChild>
                            <Button
                              variant="outline"
                              role="combobox"
                              aria-expanded={colorOpenRows.has(index)}
                              className="w-full justify-between"
                              style={colorDict?.color ? {
                                backgroundColor: colorDict.color,
                                color: getTextColorForBackground(colorDict.color)
                              } : undefined}
                              data-testid={`select-color-${index}`}
                            >
                              {color}
                              <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                            </Button>
                          </PopoverTrigger>
                          <PopoverContent className="w-[300px] p-0">
                            <Command shouldFilter={false}>
                              <CommandInput 
                                placeholder="Szukaj koloru..." 
                                value={colorSearchInputs.get(index) || ""}
                                onValueChange={(value) => {
                                  const newMap = new Map(colorSearchInputs);
                                  newMap.set(index, value);
                                  setColorSearchInputs(newMap);
                                }}
                              />
                              <CommandList>
                                <CommandEmpty>Nie znaleziono koloru</CommandEmpty>
                                <CommandGroup>
                                  {colorDictionary
                                    .filter((opt) => {
                                      const searchTerm = (colorSearchInputs.get(index) || "").toLowerCase();
                                      if (!searchTerm) return true;
                                      const name = (opt.name || "").toLowerCase();
                                      const readableName = (opt.readableName || "").toLowerCase();
                                      const code = (opt.code || "").toLowerCase();
                                      return name.includes(searchTerm) || 
                                             readableName.includes(searchTerm) || 
                                             code.includes(searchTerm);
                                    })
                                    .map((opt) => (
                                      <CommandItem
                                        key={opt.id}
                                        value={opt.code}
                                        onSelect={() => {
                                          const newColors = [...colors];
                                          newColors[index] = opt.code;
                                          setColors(newColors);
                                          const newSet = new Set(colorOpenRows);
                                          newSet.delete(index);
                                          setColorOpenRows(newSet);
                                          const newMap = new Map(colorSearchInputs);
                                          newMap.delete(index);
                                          setColorSearchInputs(newMap);
                                        }}
                                      >
                                        <Check
                                          className={`mr-2 h-4 w-4 ${color === opt.code ? "opacity-100" : "opacity-0"}`}
                                        />
                                        {opt.code}
                                      </CommandItem>
                                    ))}
                                </CommandGroup>
                              </CommandList>
                            </Command>
                          </PopoverContent>
                        </Popover>
                      </TableCell>
                      <TableCell className="p-0">
                        <div className="flex items-center gap-1.5 flex-wrap pt-1 px-2">
                          {(() => {
                            const images = colorImagesState[indexKey(index)] || [];
                            const isExpanded = expandedColorImages.has(index);
                            const visibleImages = isExpanded ? images : images.slice(0, 2);
                            const remainingCount = images.length - 2;
                            
                            return (
                              <>
                                {visibleImages.map((imageUrl, imgIndex) => {
                                  const actualIndex = isExpanded ? imgIndex : imgIndex;
                                  const isDragging = draggedImage?.color === index.toString() && draggedImage?.index === actualIndex;
                                  
                                  return (
                                    <div 
                                      key={imgIndex} 
                                      className={`relative group cursor-move flex-shrink-0 w-12 h-12 rounded border overflow-hidden transition-opacity ${isDragging ? 'opacity-50' : ''}`}
                                      draggable={true}
                                      onDragStart={(e) => {
                                        setDraggedImage({ color: index.toString(), index: actualIndex });
                                        e.dataTransfer.effectAllowed = 'move';
                                      }}
                                      onDragOver={(e) => {
                                        e.preventDefault();
                                        e.dataTransfer.dropEffect = 'move';
                                      }}
                                      onDrop={(e) => {
                                        e.preventDefault();
                                        if (draggedImage && draggedImage.color === index.toString() && draggedImage.index !== actualIndex) {
                                          handleImageReorder(index, draggedImage.index, actualIndex);
                                        }
                                        setDraggedImage(null);
                                      }}
                                      onDragEnd={() => setDraggedImage(null)}
                                    >
                                      <img
                                        src={imageUrl}
                                        alt={`${color} ${actualIndex + 1}`}
                                        className="w-full h-full object-cover pointer-events-none"
                                      />
                                      
                                      <div className="absolute inset-0 bg-black/60 rounded opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center pointer-events-none">
                                        <span className="text-white text-2xl font-bold">
                                          {actualIndex + 1}
                                        </span>
                                      </div>
                                      
                                      {actualIndex === 0 && (
                                        <span className="absolute top-1 left-1 bg-primary text-primary-foreground text-xs px-1.5 py-0.5 rounded-sm font-medium z-10 pointer-events-none">
                                          1
                                        </span>
                                      )}
                                      <Button
                                        variant="destructive"
                                        size="icon"
                                        className="absolute bottom-6 left-1 w-5 h-5 opacity-0 group-hover:opacity-100 transition-opacity z-10"
                                        onClick={() => handleColorImageDelete(index, imageUrl)}
                                        data-testid={`button-delete-image-${index}-${actualIndex}`}
                                      >
                                        <X className="w-3 h-3" />
                                      </Button>
                                    </div>
                                  );
                                })}
                                
                                {!isExpanded && remainingCount > 0 && (
                                  <Button
                                    variant="outline"
                                    size="sm"
                                    className="h-12 px-2 text-xs"
                                    onClick={() => {
                                      setExpandedColorImages(prev => {
                                        const newSet = new Set(prev);
                                        newSet.add(index);
                                        return newSet;
                                      });
                                    }}
                                    data-testid={`button-show-more-images-${index}`}
                                  >
                                    +{remainingCount}
                                  </Button>
                                )}
                                
                                {isExpanded && images.length > 2 && (
                                  <Button
                                    variant="outline"
                                    size="sm"
                                    className="h-12 px-2 text-xs"
                                    onClick={() => {
                                      setExpandedColorImages(prev => {
                                        const newSet = new Set(prev);
                                        newSet.delete(index);
                                        return newSet;
                                      });
                                    }}
                                    data-testid={`button-show-less-images-${index}`}
                                  >
                                    <Eye className="w-3 h-3" />
                                  </Button>
                                )}
                                
                                {!isNew && (
                                  <label
                                    className="w-36 h-12 border-2 border-dashed rounded flex items-center justify-center cursor-pointer hover:border-primary hover:bg-accent/50 transition-colors flex-shrink-0"
                                    data-testid={`label-upload-${index}`}
                                    onDragOver={(e) => {
                                      e.preventDefault();
                                      e.currentTarget.classList.add('border-primary', 'bg-accent');
                                    }}
                                    onDragLeave={(e) => {
                                      e.preventDefault();
                                      e.currentTarget.classList.remove('border-primary', 'bg-accent');
                                    }}
                                    onDrop={(e) => {
                                      e.preventDefault();
                                      e.currentTarget.classList.remove('border-primary', 'bg-accent');
                                      handleColorImageUpload(index, e.dataTransfer.files);
                                    }}
                                  >
                                    <Upload className="w-4 h-4 text-muted-foreground" />
                                    <input
                                      type="file"
                                      accept="image/*"
                                      multiple
                                      className="hidden"
                                      onChange={(e) => handleColorImageUpload(index, e.target.files)}
                                      data-testid={`input-upload-${index}`}
                                    />
                                  </label>
                                )}
                              </>
                            );
                          })()}
                        </div>
                      </TableCell>
                      <TableCell className="py-1 px-1 text-center">
                        <Checkbox
                          className="h-3.5 w-3.5"
                          checked={colorsForGeneration.includes(index)}
                          onCheckedChange={() => toggleColorForGeneration(index)}
                          data-testid={`checkbox-generate-${index}`}
                        />
                      </TableCell>
                      <TableCell className="py-1 px-2">
                        <div className="flex flex-wrap gap-1">
                          {(colorOptionsMatrix[indexKey(index)] || []).map((option, optIdx) => (
                            <Badge 
                              key={optIdx} 
                              variant="secondary" 
                              className="text-xs"
                              data-testid={`badge-color-option-${index}-${optIdx}`}
                            >
                              {option}
                              <button
                                className="ml-1 hover:text-destructive"
                                onClick={() => {
                                  const newOptions = [...(colorOptionsMatrix[indexKey(index)] || [])];
                                  newOptions.splice(optIdx, 1);
                                  setColorOptionsMatrix({
                                    ...colorOptionsMatrix,
                                    [indexKey(index)]: newOptions
                                  });
                                }}
                                data-testid={`button-remove-option-${index}-${optIdx}`}
                              >
                                <X className="w-3 h-3" />
                              </button>
                            </Badge>
                          ))}
                          <Popover
                            open={colorOptionOpenRows.has(index)}
                            onOpenChange={(open) => {
                              if (open) {
                                setColorOptionOpenRows(prev => {
                                  const newSet = new Set(prev);
                                  newSet.add(index);
                                  return newSet;
                                });
                              } else {
                                setColorOptionSearchInputs(prev => {
                                  const newMap = new Map(prev);
                                  newMap.delete(index);
                                  return newMap;
                                });
                                setColorOptionOpenRows(prev => {
                                  const newSet = new Set(prev);
                                  newSet.delete(index);
                                  return newSet;
                                });
                              }
                            }}
                          >
                            <PopoverTrigger asChild>
                              <Button
                                variant="outline"
                                size="icon"
                                className="h-6 w-6"
                                data-testid={`button-add-color-option-${index}`}
                              >
                                <Plus className="w-3 h-3" />
                              </Button>
                            </PopoverTrigger>
                            <PopoverContent className="w-80 p-0" align="start">
                              <Command shouldFilter={false}>
                                <CommandInput 
                                  placeholder="Szukaj opcji (użyj ; dla AND)..." 
                                  value={colorOptionSearchInputs.get(index) || ""}
                                  onValueChange={(value) => {
                                    const newMap = new Map(colorOptionSearchInputs);
                                    newMap.set(index, value);
                                    setColorOptionSearchInputs(newMap);
                                  }}
                                />
                                <CommandEmpty>Brak wyników</CommandEmpty>
                                <CommandList className="max-h-60 overflow-y-auto">
                                  <CommandGroup heading="Dostępne opcje COMPONENT-COLOR">
                                    {availableColorOptions
                                      .filter(opt => {
                                        // Already selected
                                        if ((colorOptionsMatrix[indexKey(index)] || []).includes(opt.value)) return false;
                                        
                                        // Semicolon search: "bia;drzw" means contains "bia" AND "drzw"
                                        const searchTerm = (colorOptionSearchInputs.get(index) || "").toLowerCase();
                                        if (!searchTerm) return true;
                                        
                                        const tokens = searchTerm.split(';').map(t => t.trim()).filter(t => t);
                                        if (tokens.length === 0) return true;
                                        
                                        const optValue = opt.value.toLowerCase();
                                        return tokens.every(token => optValue.includes(token));
                                      })
                                      .map((opt) => (
                                        <CommandItem
                                          key={opt.value}
                                          value={opt.value}
                                          onSelect={(selectedValue) => {
                                            const newOptions = [...(colorOptionsMatrix[indexKey(index)] || []), selectedValue];
                                            setColorOptionsMatrix({
                                              ...colorOptionsMatrix,
                                              [indexKey(index)]: newOptions
                                            });
                                            
                                            // Close popover after selection
                                            const newSet = new Set(colorOptionOpenRows);
                                            newSet.delete(index);
                                            setColorOptionOpenRows(newSet);
                                            
                                            // Clear search input
                                            const newMap = new Map(colorOptionSearchInputs);
                                            newMap.delete(index);
                                            setColorOptionSearchInputs(newMap);
                                          }}
                                        >
                                          {opt.label}
                                        </CommandItem>
                                      ))}
                                    {availableColorOptions.length === 0 && (
                                      <CommandItem disabled>
                                        <span className="text-muted-foreground text-xs italic">
                                          Ładowanie opcji...
                                        </span>
                                      </CommandItem>
                                    )}
                                  </CommandGroup>
                                </CommandList>
                              </Command>
                            </PopoverContent>
                          </Popover>
                          <Button
                            variant="outline"
                            size="icon"
                            className="h-6 w-6 relative"
                            onClick={() => {
                              setActiveOverrideColorIndex(index);
                              setOverridesDialogOpen(true);
                            }}
                            data-testid={`button-override-products-${index}`}
                          >
                            <Package2 className="w-3 h-3" />
                            {(() => {
                              const overridesForColor = componentOverrides[indexKey(index)] || {};
                              const count = Object.keys(overridesForColor).length;
                              if (count > 0) {
                                return (
                                  <Badge 
                                    variant="destructive" 
                                    className="absolute -top-1.5 -right-1.5 h-4 min-w-[16px] px-1 text-[10px] flex items-center justify-center"
                                  >
                                    {count}
                                  </Badge>
                                );
                              }
                              return null;
                            })()}
                          </Button>
                        </div>
                      </TableCell>
                      <TableCell className="py-1 px-2">
                        {(() => {
                          const overridesForColor = componentOverrides[indexKey(index)] || {};
                          const entries = Object.entries(overridesForColor);
                          
                          if (entries.length === 0) {
                            return (
                              <span className="text-xs text-muted-foreground italic">Brak nadpisań</span>
                            );
                          }
                          
                          if (isLoadingBulkProducts) {
                            return (
                              <div className="flex items-center gap-1">
                                <Loader2 className="w-3 h-3 animate-spin" />
                                <span className="text-xs text-muted-foreground">Ładowanie...</span>
                              </div>
                            );
                          }
                          
                          return (
                            <div className="flex flex-col gap-1">
                              {entries.map(([componentType, override]) => {
                                const product = bulkProducts[override.productId];
                                
                                if (!product) {
                                  return (
                                    <div key={componentType} className="text-xs text-muted-foreground">
                                      {componentType} ID: {override.productId}
                                    </div>
                                  );
                                }
                                
                                return (
                                  <div key={componentType} className="flex items-center gap-1 text-xs">
                                    <span className="font-medium text-muted-foreground">{componentType}</span>
                                    <Link 
                                      href={`/catalog-products/${product.id}`}
                                      className="text-primary hover:underline"
                                      data-testid={`link-product-${product.id}`}
                                    >
                                      {product.title}
                                    </Link>
                                    <span className="text-muted-foreground">({product.sku})</span>
                                  </div>
                                );
                              })}
                            </div>
                          );
                        })()}
                      </TableCell>
                      <TableCell className="py-1 px-2">
                        <div className="flex gap-1">
                          {(() => {
                            const overrideCount = Object.keys(componentOverrides[indexKey(index)] || {}).length;
                            return (
                              <TooltipProvider>
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <div className="relative">
                                      <Button
                                        variant="ghost"
                                        size="icon"
                                        className="h-6 w-6"
                                        onClick={() => {
                                          setActiveOverrideColorIndex(index);
                                          setOverridesDialogOpen(true);
                                        }}
                                        data-testid={`button-manage-overrides-${index}`}
                                      >
                                        <Package2 className={`h-3 w-3 ${overrideCount > 0 ? 'text-primary' : ''}`} />
                                      </Button>
                                      {overrideCount > 0 && (
                                        <Badge 
                                          variant="default" 
                                          className="absolute -top-1 -right-1 h-3 min-w-[12px] px-0.5 text-[8px] leading-none flex items-center justify-center"
                                          data-testid={`badge-overrides-count-${index}`}
                                        >
                                          {overrideCount}
                                        </Badge>
                                      )}
                                    </div>
                                  </TooltipTrigger>
                                  <TooltipContent>
                                    <p>
                                      {overrideCount > 0 
                                        ? `Nadpisania produktów (${overrideCount})`
                                        : 'Nadpisania produktów'
                                      }
                                    </p>
                                  </TooltipContent>
                                </Tooltip>
                              </TooltipProvider>
                            );
                          })()}
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-6 w-6"
                            onClick={() => duplicateColor(index)}
                            title="Duplikuj kolor z opcjami"
                            data-testid={`button-duplicate-color-${index}`}
                          >
                            <Copy className="h-3 w-3" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-6 w-6"
                            onClick={() => handleRemoveColor(index)}
                            data-testid={`button-remove-color-${index}`}
                          >
                            <X className="h-3 w-3" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  );
                  })}
                </TableBody>
              </Table>
            </div>
            </>
          ) : (
            <div className="p-6 text-center text-sm text-muted-foreground">
              <p className="mb-3">Brak kolorów. Użyj przycisku "Dodaj kolor" powyżej aby dodać kolory ze słownika.</p>
            </div>
          )}
          {colors.length > 0 && !isNew && (
            <p className="text-xs text-muted-foreground mt-3">
              Przeciągnij i upuść zdjęcia lub kliknij ikonę + aby dodać. Kliknij X aby usunąć kolor.
            </p>
          )}
          
          {/* Drugi przycisk "Dodaj kolor" na dole listy */}
          {colors.length > 0 && (
            <div className="mt-4 flex items-center gap-2 flex-wrap">
              <Button
                variant="default"
                size="sm"
                onClick={addColor}
                data-testid="button-add-color-bottom"
              >
                <Plus className="h-4 w-4 mr-2" />
                Dodaj kolor
              </Button>
            </div>
          )}
        </CardContent>
      </Card>

      {/* SEKCJA KOMPONENTÓW */}
      <Card>
        <CardContent className="p-4 sm:p-6 space-y-6">
          <div className="space-y-3">
            <Label>Komponenty zestawu</Label>

            {!Array.isArray(components) || components.length === 0 ? (
              <div className="text-center py-6 border-2 border-dashed">
                <p className="text-sm text-muted-foreground">
                  Brak komponentów. Kliknij "Dodaj komponent" poniżej aby dodać.
                </p>
              </div>
            ) : (
              <div className="space-y-2">
                {components.map((component, index) => {
                  const isChanged = hasComponentChanged(index);
                  return <div 
                    key={index} 
                    className={`flex items-end gap-2 p-2 transition-colors ${
                      isChanged 
                        ? 'bg-yellow-50 dark:bg-yellow-900/20 border-l-4 border-l-yellow-500' 
                        : 'even:bg-muted/30'
                    }`}
                    data-testid={`component-${index}`}
                  >
                    <div className="flex-1 grid grid-cols-1 xl:grid-cols-[minmax(0,2fr)_minmax(0,1fr)_minmax(0,1fr)_minmax(96px,1fr)_auto] gap-2">
                      <div>
                        <Label className="text-xs">Rodzaj</Label>
                        <DictionaryComboboxWithAdd
                          items={productTypes}
                          value={component.componentType}
                          onChange={(value) =>
                            handleUpdateComponent(index, "componentType", value)
                          }
                          dictionaryType="product_type"
                          placeholder="Rodzaj"
                          searchPlaceholder="Szukaj typu..."
                          emptyText="Nie znaleziono"
                          testId={`select-component-type-${index}`}
                          valueField="code"
                          displayField="name"
                          backgroundColor={
                            component.componentType 
                              ? productTypes.find(t => t.code === component.componentType)?.color || undefined
                              : undefined
                          }
                        />
                      </div>
                      <div>
                        <Label className="text-xs">Długość</Label>
                        <DictionaryComboboxWithAdd
                          items={dimensionLengths}
                          value={component.length}
                          onChange={(value) =>
                            handleUpdateComponent(index, "length", value)
                          }
                          dictionaryType="dimension_length"
                          placeholder="Długość"
                          searchPlaceholder="Szukaj..."
                          emptyText="Nie znaleziono"
                          testId={`select-component-length-${index}`}
                          valueField="code"
                          displayField="name"
                        />
                      </div>
                      <div>
                        <Label className="text-xs">Szerokość</Label>
                        <DictionaryComboboxWithAdd
                          items={dimensionWidths}
                          value={component.width}
                          onChange={(value) =>
                            handleUpdateComponent(index, "width", value)
                          }
                          dictionaryType="dimension_width"
                          placeholder="Szerokość"
                          searchPlaceholder="Szukaj..."
                          emptyText="Nie znaleziono"
                          testId={`select-component-width-${index}`}
                          valueField="code"
                          displayField="name"
                        />
                      </div>
                      <div>
                        <Label className="text-xs">Ilość</Label>
                        <Input
                          type="number"
                          min="1"
                          value={component.quantity}
                          onChange={(e) =>
                            handleUpdateComponent(
                              index,
                              "quantity",
                              parseInt(e.target.value) || 1
                            )
                          }
                          data-testid={`input-component-quantity-${index}`}
                          className="h-8"
                        />
                      </div>
                      <div>
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Label htmlFor={`ignore-options-${index}`} className="text-xs whitespace-nowrap cursor-help">
                              Ignoruj opcje
                            </Label>
                          </TooltipTrigger>
                          <TooltipContent>
                            <p className="text-xs">Ignoruj opcje kolorów podczas generowania</p>
                            <p className="text-xs text-muted-foreground">(np. wieszak, panel tapicerowany)</p>
                          </TooltipContent>
                        </Tooltip>
                        <div className="flex items-center h-8">
                          <Checkbox
                            id={`ignore-options-${index}`}
                            checked={component.ignoreColorOptions || false}
                            onCheckedChange={(checked) =>
                              handleUpdateComponent(index, "ignoreColorOptions", !!checked)
                            }
                            data-testid={`checkbox-ignore-options-${index}`}
                            aria-describedby={`ignore-options-tooltip-${index}`}
                          />
                        </div>
                      </div>
                    </div>
                    <Button
                      type="button"
                      variant="ghost"
                      size="icon"
                      onClick={() => handleRemoveComponent(index)}
                      data-testid={`button-remove-component-${index}`}
                      className="h-8 w-8 shrink-0"
                    >
                      <X className="h-4 w-4" />
                    </Button>
                  </div>;
                })}
              </div>
            )}
            
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={handleAddComponent}
              data-testid="button-add-component"
              className="w-full sm:w-auto"
            >
              <Plus className="h-4 w-4 mr-1" />
              Dodaj komponent
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Wygenerowane zestawy w katalogu */}
      {!isNew && (
        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle className="flex items-center gap-2">
                  <Package2 className="h-5 w-5" />
                  Wygenerowane zestawy w katalogu {!isLoadingSets && `(${generatedSets.length})`}
                </CardTitle>
                <CardDescription>
                  Lista zestawów produktów wygenerowanych z tej matrycy
                </CardDescription>
              </div>
              <Link href="/catalog-sets">
                <Button variant="outline" size="sm">
                  <ExternalLink className="h-4 w-4 mr-2" />
                  Wszystkie zestawy
                </Button>
              </Link>
            </div>
          </CardHeader>
          <CardContent>
            {isLoadingSets ? (
              <div className="flex items-center justify-center py-8">
                <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
              </div>
            ) : generatedSets.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                <p>Brak wygenerowanych zestawów dla tej matrycy</p>
                <p className="text-sm mt-2">Użyj przycisku "Generuj zestawy" aby utworzyć zestawy</p>
              </div>
            ) : (
              <div className="rounded-md border">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>SKU</TableHead>
                      <TableHead>Tytuł</TableHead>
                      <TableHead>Kolor</TableHead>
                      <TableHead className="text-center">Panele</TableHead>
                      <TableHead className="text-right">Cena</TableHead>
                      <TableHead className="text-center">Status</TableHead>
                      <TableHead className="text-right">Akcje</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {generatedSets.map((set) => (
                      <TableRow key={set.id}>
                        <TableCell className="font-mono text-xs">{set.sku}</TableCell>
                        <TableCell>
                          <Link href={`/catalog-sets/${set.id}`}>
                            <span className="hover-elevate cursor-pointer px-1 font-medium">
                              {set.title}
                            </span>
                          </Link>
                        </TableCell>
                        <TableCell>
                          {set.color ? (() => {
                            const colorDict = colorDictionary.find(c => c.code === set.color);
                            const bgColor = colorDict?.color || '#94A3B8';
                            const textColor = getTextColorForBackground(bgColor);
                            return (
                              <Badge 
                                variant="outline" 
                                className="text-xs h-5 px-1.5 border-transparent" 
                                style={{ backgroundColor: bgColor, color: textColor }}
                              >
                                {set.color}
                              </Badge>
                            );
                          })() : (
                            <span className="text-muted-foreground text-xs">-</span>
                          )}
                        </TableCell>
                        <TableCell className="text-center">
                          {set.panelCount || '-'}
                        </TableCell>
                        <TableCell className="text-right">
                          {set.basePrice || set.calculatedPrice ? (
                            <span className="text-sm">
                              {Number(set.basePrice || set.calculatedPrice).toFixed(2)} PLN
                            </span>
                          ) : (
                            <span className="text-muted-foreground text-xs">-</span>
                          )}
                        </TableCell>
                        <TableCell className="text-center">
                          <Badge variant={set.isActive ? "default" : "secondary"} className="text-xs">
                            {set.isActive ? "Aktywny" : "Nieaktywny"}
                          </Badge>
                        </TableCell>
                        <TableCell className="text-right">
                          <Link href={`/catalog-sets/${set.id}`}>
                            <Button variant="ghost" size="sm" data-testid={`button-view-set-${set.id}`}>
                              <Eye className="h-4 w-4 mr-1" />
                              Zobacz
                            </Button>
                          </Link>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            )}
          </CardContent>
        </Card>
      )}

      {/* Przyciski na dole strony */}
      <div className="flex flex-col sm:flex-row gap-3 sm:gap-0 sm:justify-end pt-4 border-t">
        <div className="flex gap-2">
          {!isNew && (
            <Button
              onClick={handleStartGeneration}
              disabled={isGenerating}
              data-testid="button-generate-sets-bottom"
              className="flex-1 sm:flex-none"
              variant="default"
            >
              <Play className="h-4 w-4 mr-2" />
              Generuj zestawy
            </Button>
          )}
          <Button
            onClick={handleSave}
            disabled={createMutation.isPending || updateMutation.isPending}
            data-testid="button-save-matrix-bottom"
            className="flex-1 sm:flex-none"
          >
            <Save className="h-4 w-4 mr-2" />
            {createMutation.isPending || updateMutation.isPending
              ? "Zapisywanie..."
              : isNew
              ? "Utwórz matrycę"
              : "Zapisz zmiany"}
          </Button>
          {!isNew && (
            <Button
              variant="destructive"
              onClick={() => setShowDeleteDialog(true)}
              data-testid="button-delete-matrix-bottom"
              className="flex-1 sm:flex-none"
            >
              <Trash2 className="h-4 w-4 mr-2" />
              Usuń
            </Button>
          )}
        </div>
      </div>

      <AlertDialog open={showDeleteDialog} onOpenChange={setShowDeleteDialog}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Czy na pewno usunąć?</AlertDialogTitle>
            <AlertDialogDescription>
              Ta operacja jest nieodwracalna. Matryca "{name}" zostanie trwale usunięta.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDelete}
              data-testid="button-confirm-delete"
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
            >
              Usuń
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      {/* Preview Dialog */}
      <AlertDialog open={showPreviewDialog} onOpenChange={setShowPreviewDialog}>
        <AlertDialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
          <AlertDialogHeader>
            <AlertDialogTitle>Podgląd generowania zestawu</AlertDialogTitle>
            <AlertDialogDescription>
              Sprawdź szczegóły przed wygenerowaniem zestawów produktów
            </AlertDialogDescription>
          </AlertDialogHeader>
          
          {!previewData || !previewData.preview ? (
            <div className="py-8 text-center text-muted-foreground">
              Ładowanie podglądu...
            </div>
          ) : (
            <div className="space-y-4 py-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label className="text-muted-foreground">Nazwa zestawu:</Label>
                  <div className="font-semibold">{previewData.preview.setName}</div>
                </div>
                <div>
                  <Label className="text-muted-foreground">SKU:</Label>
                  <div className="font-mono text-sm">{previewData.preview.sku}</div>
                </div>
                <div>
                  <Label className="text-muted-foreground">Kolor:</Label>
                  <div>{previewData.preview.color || "Brak"}</div>
                </div>
                <div>
                  <Label className="text-muted-foreground">Szacowana cena:</Label>
                  <div className="font-semibold">{previewData.preview.estimatedPrice} PLN</div>
                </div>
              </div>

              <div className="border-t pt-4">
                <Label className="text-muted-foreground mb-2 block">
                  Komponenty zestawu ({previewData.preview.totalComponents}):
                </Label>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="w-12">#</TableHead>
                      <TableHead>Nazwa</TableHead>
                      <TableHead>Typ</TableHead>
                      <TableHead>Wymiary</TableHead>
                      <TableHead className="w-20">Ilość</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {previewData.preview.components.map((comp: any, idx: number) => (
                      <TableRow key={idx}>
                        <TableCell className="font-medium">{comp.position}</TableCell>
                        <TableCell>{comp.name}</TableCell>
                        <TableCell>
                          <Badge variant="outline" className="text-xs">
                            {comp.componentType}
                          </Badge>
                        </TableCell>
                        <TableCell>
                          {comp.length && comp.width 
                            ? `${comp.length} × ${comp.width} mm`
                            : comp.length 
                            ? `${comp.length} mm`
                            : "–"}
                        </TableCell>
                        <TableCell className="text-center">{comp.quantity}</TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>

              <div className="bg-accent/20 border border-accent/40 rounded-lg p-4 text-sm">
                <div className="font-medium mb-1">ℹ️ Informacja</div>
                <div className="text-muted-foreground">
                  Zostanie wygenerowany zestaw dla każdego zaznaczonego koloru.
                  Jeśli nie zaznaczono żadnego koloru, zostanie wygenerowany jeden podstawowy zestaw.
                </div>
              </div>
            </div>
          )}

          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-preview">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleGenerate}
              data-testid="button-confirm-generate"
              disabled={isGenerating}
            >
              {isGenerating ? (
                <>
                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  Generowanie...
                </>
              ) : (
                <>
                  <Play className="h-4 w-4 mr-2" />
                  Generuj zestawy
                </>
              )}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      {/* Component Overrides Dialog */}
      <ComponentOverridesDialog 
        open={overridesDialogOpen}
        onOpenChange={setOverridesDialogOpen}
        activeColorIndex={activeOverrideColorIndex}
        onActiveColorIndexChange={setActiveOverrideColorIndex}
        colors={colors}
        components={components}
        currentOverrides={currentOverridesForDialog}
        onSaveOverrides={handleSaveComponentOverrides}
      />
    </div>
  );
}
