import { useParams, useLocation, Link } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useState, useEffect, useMemo, useRef } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Progress } from "@/components/ui/progress";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from "@/components/ui/command";
import { useToast } from "@/hooks/use-toast";
import { ArrowLeft, Save, Loader2, Plus, X, Copy, Trash2, Play, Sparkles, Upload, Image as ImageIcon, Check, ChevronsUpDown, Eye } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { RichTextEditor } from "@/components/rich-text-editor";
import { DictionaryComboboxWithAdd } from "@/components/dictionary-combobox-with-add";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";

interface ProductMatrix {
  id: number;
  name: string;
  namePrefix: string | null;
  nameSuffix: string | null;
  productType: string | null;
  doors: string | null;
  legs: string | null;
  lengths: string[];
  widths: string[];
  heights: string[];
  productGroups: string[];
  priceModifiers: string[];
  priceModifierTypes: string[];
  colors: string[];
  colorImages: Record<string, string[]>;
  selectedColors: string[];
  colorLastGenerated: Record<string, string> | null;
  colorOptions: Record<string, string[]> | null;
  hasWarranty: boolean;
  descriptionHtml: string | null;
  descriptionDoc: any | null;
  useAiGeneration: boolean;
  aiPrompt: string | null;
  defaultPrice: string | null;
  createdAt: Date | null;
  updatedAt: Date | null;
}

interface ProductCreatorDictionary {
  id: number;
  dictionaryType: string;
  code: string;
  name: string;
  readableName: string | null;
  color: string | null;
  isActive: boolean;
}

interface AIGeneratedDescriptionResponse {
  descriptionHtml: string;
  tokens: {
    prompt: number;
    completion: number;
    total: number;
  };
  model: string;
  cost: number;
}

const getDimensionValues = (options: ProductCreatorDictionary[] | undefined): string[] => {
  if (!options) return [];
  return options
    .filter(opt => opt.isActive)
    .map(opt => opt.name || opt.code)
    .sort((a, b) => {
      const numA = parseInt(a);
      const numB = parseInt(b);
      if (!isNaN(numA) && !isNaN(numB)) return numA - numB;
      return a.localeCompare(b);
    });
};

const getDefaultAiPrompt = (section: string): string => {
  const defaultPrompts: Record<string, string> = {
    intro: 'Napisz krótkie, profesjonalne wprowadzenie do produktu {productType} w kolorze {color} o wymiarach {length}x{width}x{height} cm. Podkreśl elegancję, funkcjonalność i uniwersalność zastosowania. Użyj ciepłego, zachęcającego tonu. Maksymalnie 3-4 zdania.',
    features: 'Wymień 5-7 najważniejszych cech produktu {productType} w formie zwięzłych punktów. Uwzględnij: jakość wykonania, materiał, wymiary ({length}x{width}x{height} cm), kolor ({color}), funkcjonalność, trwałość i zalety użytkowe. Każdy punkt powinien być konkretny i przekonujący.',
    safety: 'Opisz kwestie bezpieczeństwa związane z produktem {productType} wykonanym z materiału {material} o wymiarach {length}x{width}x{height} cm. Uwzględnij: bezpieczne użytkowanie, ostrzeżenia dotyczące stabilności i montażu, zalecenia dotyczące rozmieszczenia i obciążenia, informacje o certyfikatach i normach bezpieczeństwa. Podaj praktyczne wskazówki w formie 3-5 zwięzłych punktów. Użyj jasnego, profesjonalnego języka.',
    care: 'Opisz sposoby pielęgnacji i konserwacji produktu {productType} wykonanego z materiału {material} w kolorze {color}. Uwzględnij: zalecane metody czyszczenia, środki czystości odpowiednie dla tego materiału, czynności które należy unikać, częstotliwość pielęgnacji, sposoby zachowania świeżego wyglądu. Podaj praktyczne wskazówki w formie 4-6 zwięzłych punktów.',
  };
  return defaultPrompts[section] || '';
};

// Helper function to determine text color based on background color
const getContrastColor = (hexColor: string): string => {
  // Remove # if present
  const hex = hexColor.replace('#', '');
  
  // Convert to RGB
  const r = parseInt(hex.substring(0, 2), 16);
  const g = parseInt(hex.substring(2, 4), 16);
  const b = parseInt(hex.substring(4, 6), 16);
  
  // Calculate relative luminance
  const luminance = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
  
  // Return black for light colors, white for dark colors
  return luminance > 0.5 ? '#000000' : '#FFFFFF';
};


export default function ProductMatrixEditor() {
  const { id } = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const isNew = id === "new";
  const [activeTab, setActiveTab] = useState("podstawowe");
  const [isGenerating, setIsGenerating] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  
  // SSE logs state
  const [generationLogs, setGenerationLogs] = useState<Array<{
    timestamp: string;
    message: string;
    type: 'info' | 'success' | 'error' | 'warning';
  }>>([]);
  const [sessionId, setSessionId] = useState("");
  const eventSourceRef = useRef<EventSource | null>(null);
  const logsScrollRef = useRef<HTMLDivElement>(null);

  // Form state
  const [name, setName] = useState("");
  const [namePrefix, setNamePrefix] = useState("");
  const [nameSuffix, setNameSuffix] = useState("");
  const [productType, setProductType] = useState("");
  const [doors, setDoors] = useState("");
  const [legs, setLegs] = useState("");
  const [material, setMaterial] = useState("");
  const [lengths, setLengths] = useState<string[]>([]);
  const [widths, setWidths] = useState<string[]>([]);
  const [heights, setHeights] = useState<string[]>([]);
  const [productGroupsArray, setProductGroupsArray] = useState<string[]>([]);
  const [priceModifiersArray, setPriceModifiersArray] = useState<string[]>([]);
  const [priceModifierTypesArray, setPriceModifierTypesArray] = useState<string[]>([]);
  const [colors, setColors] = useState<string[]>([]);
  const [colorImages, setColorImages] = useState<Record<string, string[]>>({});
  const [colorOptionsMatrix, setColorOptionsMatrix] = useState<Record<string, string[]>>({});
  const [draggedImage, setDraggedImage] = useState<{ colorIndex: number; index: number } | null>(null);
  const [descriptionHtml, setDescriptionHtml] = useState("");
  const [descriptionDoc, setDescriptionDoc] = useState<any>(null);
  const [useAiGeneration, setUseAiGeneration] = useState(false);
  const [aiPrompt, setAiPrompt] = useState("");
  const [defaultPrice, setDefaultPrice] = useState("");
  
  // Template system
  const [templateId, setTemplateId] = useState<number | null>(null);
  const [aiConfig, setAiConfig] = useState<Record<string, {
    enabled: boolean;
    prompt: string;
    tone?: string;
    language?: string;
    maxLength?: number;
  }>>({});

  // Temporary inputs
  const [newLengthInput, setNewLengthInput] = useState("");
  const [newWidthInput, setNewWidthInput] = useState("");
  const [newHeightInput, setNewHeightInput] = useState("");

  // Preview generation state
  const [isPreviewOpen, setIsPreviewOpen] = useState(false);
  const [previewData, setPreviewData] = useState<any>(null);
  const [isLoadingPreview, setIsLoadingPreview] = useState(false);

  // Selection state for bulk delete
  const [selectedSizes, setSelectedSizes] = useState<number[]>([]);
  const [selectedColorsForDelete, setSelectedColorsForDelete] = useState<number[]>([]);
  
  // Selection state for generation (stores color INDICES, not names, to support duplicate color names)
  const [colorsForGeneration, setColorsForGeneration] = useState<number[]>([]);
  
  // Track original colors for generation (to detect unsaved changes)
  const originalColorsForGenerationRef = useRef<number[]>([]);
  
  // Track previous colors state to detect truly new additions (not initial hydration)
  const prevColorsRef = useRef<string[]>([]);
  
  // Expanded images state for each color
  const [expandedColorImages, setExpandedColorImages] = useState<Set<number>>(new Set());
  
  // Combobox state for searchable product type
  
  // Combobox state for product groups (per row)
  const [productGroupOpenRows, setProductGroupOpenRows] = useState<Set<number>>(new Set());
  const [isNewGroupDialogOpen, setIsNewGroupDialogOpen] = useState(false);
  const [newGroupCode, setNewGroupCode] = useState("");
  const [newGroupName, setNewGroupName] = useState("");
  const [newGroupReadableName, setNewGroupReadableName] = useState("");
  
  // Combobox state for colors (per row)
  const [colorOpenRows, setColorOpenRows] = useState<Set<number>>(new Set());
  const [colorSearchInputs, setColorSearchInputs] = useState<Map<number, string>>(new Map());
  
  // State for color options search (per row)
  const [colorOptionSearchInputs, setColorOptionSearchInputs] = useState<Map<number, string>>(new Map());
  const [colorOptionOpenRows, setColorOptionOpenRows] = useState<Set<number>>(new Set());

  // Fetch matrix data if editing
  const { data: matrix, isLoading } = useQuery<ProductMatrix>({
    queryKey: [`/api/product-matrices/${id}`],
    enabled: !isNew,
  });

  // Fetch generated products from this matrix
  const { data: generatedProducts = [] } = useQuery<Array<{
    id: number;
    sku: string;
    title: string;
    combination_key: string;
    base_price: string | null;
    is_active: boolean;
  }>>({
    queryKey: [`/api/product-matrices/${id}/products`],
    enabled: !isNew,
  });

  // Fetch dictionaries
  const { data: productTypes } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=product_type"],
  });
  const { data: productGroups } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=product_group"],
  });
  const { data: doorsOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=door"],
  });
  const { data: legsOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=leg"],
  });
  const { data: materialOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=material"],
  });
  const { data: dimensionLengths } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_length"],
  });
  const { data: dimensionWidths } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_width"],
  });
  const { data: dimensionHeights } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_height"],
  });
  const { data: colorOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=color"],
  });
  const { data: componentCz1Options } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=component_cz1"],
  });
  
  // Generate COMPONENT_CZ1-COLOR combinations (after componentCz1Options and colorOptions are declared)
  const availableColorOptions = useMemo(() => {
    if (!componentCz1Options || !colorOptions) return [];
    const combinations: Array<{ value: string; label: string }> = [];
    componentCz1Options.forEach(component => {
      colorOptions.forEach(color => {
        combinations.push({
          value: `${component.code}-${color.code}`,
          label: `${component.code}-${color.code}`
        });
      });
    });
    return combinations;
  }, [componentCz1Options, colorOptions]);
  
  // Fetch description templates
  const { data: templates } = useQuery<Array<{
    id: number;
    name: string;
    categoryId: number | null;
    categoryName?: string;
  }>>({
    queryKey: ["/api/description-templates"],
  });

  // Sort color options alphabetically by name
  const sortedColorOptions = useMemo(() => {
    if (!colorOptions) return [];
    return [...colorOptions]
      .filter(opt => opt.isActive)
      .sort((a, b) => {
        const nameA = (a.name || a.code || '').toLowerCase();
        const nameB = (b.name || b.code || '').toLowerCase();
        return nameA.localeCompare(nameB, 'pl');
      });
  }, [colorOptions]);

  // Load matrix data into form when it arrives
  useEffect(() => {
    if (matrix && !isNew) {
      setName(matrix.name || "");
      setNamePrefix(matrix.namePrefix || "");
      setNameSuffix(matrix.nameSuffix || "");
      setProductType(matrix.productType || "none");
      setDoors(matrix.doors || "none");
      setLegs(matrix.legs || "none");
      setMaterial((matrix as any).material || "none");
      setLengths((matrix.lengths || []).map(v => String(v)));
      setWidths((matrix.widths || []).map(v => String(v)));
      setHeights((matrix.heights || []).map(v => String(v)));
      setProductGroupsArray((matrix.productGroups || []).map(pg => pg || "none"));
      setPriceModifiersArray((matrix.priceModifiers || []).map(v => String(v)));
      setPriceModifierTypesArray(((matrix as any).priceModifierTypes || []).map((t: string) => t || "fixed"));
      setColors(matrix.colors || []);
      setColorImages(matrix.colorImages || {});
      setColorOptionsMatrix((matrix as any).colorOptions || {});
      // Convert selectedColors (indices as strings) to number indices
      // If selectedColors exists, parse strings to numbers; otherwise select all color indices
      const colorArray = matrix.colors || [];
      if (matrix.selectedColors && matrix.selectedColors.length > 0) {
        // Parse string indices to numbers (e.g., ["0", "3", "5"] -> [0, 3, 5])
        const indices = matrix.selectedColors
          .map(s => parseInt(s, 10))
          .filter(n => !isNaN(n) && n >= 0 && n < colorArray.length);
        
        setColorsForGeneration(indices);
        originalColorsForGenerationRef.current = indices;
      } else {
        // Select all color indices by default
        const allIndices = colorArray.map((_, idx) => idx);
        setColorsForGeneration(allIndices);
        originalColorsForGenerationRef.current = allIndices;
      }
      setDescriptionHtml(matrix.descriptionHtml || "");
      setDescriptionDoc(matrix.descriptionDoc);
      setUseAiGeneration(matrix.useAiGeneration || false);
      setAiPrompt(matrix.aiPrompt || "");
      setDefaultPrice(matrix.defaultPrice || "");
      
      // Template system
      setTemplateId((matrix as any).templateId || null);
      setAiConfig((matrix as any).aiConfig || {});
    }
  }, [matrix, isNew]);

  // Auto-select newly added colors for generation
  // When user adds a new color, it should be selected by default
  // BUT: Skip on initial hydration to preserve saved selectedColors
  useEffect(() => {
    // Skip if this is the first run (initial hydration from matrix data)
    if (prevColorsRef.current.length === 0 && colors.length > 0) {
      prevColorsRef.current = colors;
      return;
    }
    
    // Find indices of colors that were added since last update
    const newColorIndices: number[] = [];
    colors.forEach((color, idx) => {
      if (!prevColorsRef.current.includes(color)) {
        newColorIndices.push(idx);
      }
    });
    
    // If there are new colors, add their indices to colorsForGeneration
    if (newColorIndices.length > 0) {
      setColorsForGeneration(prev => [...prev, ...newColorIndices]);
    }
    
    // Update ref for next comparison
    prevColorsRef.current = colors;
  }, [colors]); // Only watch 'colors' array, not colorsForGeneration to avoid infinite loop

  // Setup SSE connection for generation logs
  useEffect(() => {
    if (!isGenerating || !sessionId) {
      // Close connection if stopped generating
      if (eventSourceRef.current) {
        eventSourceRef.current.close();
        eventSourceRef.current = null;
      }
      return;
    }

    console.log('📡 Connecting to SSE stream for session:', sessionId);
    
    const eventSource = new EventSource(`/api/generation-logs/stream/${sessionId}`);
    eventSourceRef.current = eventSource;

    eventSource.onmessage = (event) => {
      try {
        const logData = JSON.parse(event.data);
        setGenerationLogs((prev) => [...prev, logData]);
      } catch (error) {
        console.error('Error parsing log data:', error);
      }
    };

    eventSource.onerror = (error) => {
      console.error('❌ SSE connection error:', error);
      eventSource.close();
    };

    return () => {
      eventSource.close();
    };
  }, [isGenerating, sessionId]);

  // Auto-scroll logs to bottom
  useEffect(() => {
    if (logsScrollRef.current) {
      logsScrollRef.current.scrollTop = logsScrollRef.current.scrollHeight;
    }
  }, [generationLogs]);

  // Save mutation
  const saveMutation = useMutation({
    mutationFn: async (data: any) => {
      console.log('🟡 Mutation function called!', { isNew, id });
      if (isNew) {
        console.log('🟡 POST to /api/product-matrices');
        return apiRequest("POST", "/api/product-matrices", data);
      } else {
        console.log('🟡 PUT to /api/product-matrices/' + id);
        const response = await apiRequest("PUT", `/api/product-matrices/${id}`, data);
        console.log('🟢 PUT response:', response);
        return response;
      }
    },
    onSuccess: async (response: any) => {
      console.log('✅ Mutation SUCCESS!', response);
      
      // Parse response if it's a Response object
      const data = response instanceof Response ? await response.json() : response;
      
      // Reset original colors for generation ref (no more unsaved changes)
      originalColorsForGenerationRef.current = [...colorsForGeneration];
      
      toast({
        title: isNew ? "Matryca utworzona" : "Matryca zaktualizowana",
        description: `Pomyślnie ${isNew ? "utworzono" : "zaktualizowano"} matrycę.`,
      });
      
      // Invalidate both the list and the specific matrix
      queryClient.invalidateQueries({ queryKey: ["/api/product-matrices"] });
      if (!isNew && id) {
        queryClient.invalidateQueries({ queryKey: ["/api/product-matrices", parseInt(id)] });
      }
      
      if (isNew && data.id) {
        navigate(`/product-matrices/${data.id}/edit`);
      }
    },
    onError: (error: any) => {
      console.error('❌ Mutation ERROR!', error);
      toast({
        title: "Błąd zapisu",
        description: error.message || "Nie udało się zapisać matryce.",
        variant: "destructive",
      });
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("DELETE", `/api/product-matrices/${id}`, {});
    },
    onSuccess: () => {
      toast({
        title: "Matryca usunięta",
        description: "Pomyślnie usunięto matrycę.",
      });
      navigate("/product-matrices");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć matryce.",
        variant: "destructive",
      });
    },
  });

  // Generate products mutation with progress tracking
  const [generationProgress, setGenerationProgress] = useState({
    isGenerating: false,
    currentStep: '',
    productsCount: 0,
    colorsCount: 0
  });

  const generateMutation = useMutation({
    mutationFn: async ({ id, sessionId }: { id: number; sessionId: string }) => {
      // Calculate actual Cartesian product count matching backend logic
      // Normalize (trim + parse) and deduplicate dimensions
      const normalizeAndDedupe = (arr: string[]): Set<number | null> => {
        const normalized = arr
          .map(v => v?.trim())
          .filter(v => v !== '')
          .map(v => {
            const num = parseFloat(v);
            return isNaN(num) ? null : num;
          })
          .filter(v => v !== null) as number[];
        
        return new Set(normalized);
      };
      
      const uniqueLengths = normalizeAndDedupe(lengths);
      const uniqueWidths = normalizeAndDedupe(widths);
      const uniqueHeights = normalizeAndDedupe(heights);
      
      // Use selected colors if available, otherwise all colors (matching backend logic)
      // Convert indices to color names
      const selectedColorNames = colorsForGeneration
        .map(idx => colors[idx])
        .filter((c: string) => c && c.trim() !== '');
      const allColorsFiltered = colors.filter((c: string) => c && c.trim() !== '');
      const colorsToUse = selectedColorNames.length > 0 ? selectedColorNames : allColorsFiltered;
      
      // Default to 1 if empty (matching backend [null] placeholder logic)
      const lengthCount = uniqueLengths.size || 1;
      const widthCount = uniqueWidths.size || 1;
      const heightCount = uniqueHeights.size || 1;
      const colorCount = colorsToUse.length || 1;
      
      const totalProducts = lengthCount * widthCount * heightCount * colorCount;
      
      setGenerationProgress({
        isGenerating: true,
        currentStep: 'Przygotowanie danych...',
        productsCount: totalProducts,
        colorsCount: colorsToUse.length
      });
      
      // Simulate progress steps for better UX
      setTimeout(() => {
        setGenerationProgress(prev => ({ ...prev, currentStep: 'Generowanie produktów...' }));
      }, 300);
      
      return apiRequest("POST", `/api/product-matrices/${id}/generate`, { sessionId });
    },
    onSuccess: (data: any) => {
      setGenerationProgress(prev => ({ ...prev, currentStep: 'Finalizacja...' }));
      
      setTimeout(() => {
        setGenerationProgress({ isGenerating: false, currentStep: '', productsCount: 0, colorsCount: 0 });
        toast({
          title: "Produkty wygenerowane",
          description: `Pomyślnie wygenerowano ${data.productsGenerated} produktów.`,
        });
        navigate("/catalog-products");
      }, 500);
    },
    onError: (error: any) => {
      setGenerationProgress({ isGenerating: false, currentStep: '', productsCount: 0, colorsCount: 0 });
      toast({
        title: "Błąd generowania",
        description: error.message || "Nie udało się wygenerować produktów.",
        variant: "destructive",
      });
    },
  });

  // Create new product group mutation
  const createGroupMutation = useMutation({
    mutationFn: async (data: { code: string; name: string; readableName: string | null }) => {
      return apiRequest("POST", "/api/dictionaries", {
        dictionaryType: "product_group",
        code: data.code,
        name: data.name,
        readableName: data.readableName,
        isActive: true
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries?type=product_group"] });
      toast({
        title: "Grupa utworzona",
        description: "Nowa grupa produktu została dodana do bazy.",
      });
      setIsNewGroupDialogOpen(false);
      setNewGroupCode("");
      setNewGroupName("");
      setNewGroupReadableName("");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć grupy.",
        variant: "destructive",
      });
    },
  });

  // Handle color image upload
  const handleColorImageUpload = async (colorIndex: number, files: FileList | null) => {
    if (!files || files.length === 0 || isNew || !id) return;
    
    const colorName = colors[colorIndex] || "unknown";

    for (const file of Array.from(files)) {
      const formData = new FormData();
      formData.append('image', file);
      formData.append('colorIndex', colorIndex.toString());

      try {
        const response = await fetch(`/api/product-matrices/${id}/color-images`, {
          method: 'POST',
          body: formData,
          credentials: 'include',
        });

        if (!response.ok) {
          const text = await response.text();
          throw new Error(`${response.status}: ${text}`);
        }

        const result = await response.json();

        // 🔧 FIX: Merguj colorImages zamiast zastępować cały state
        // Backend zwraca pełny obiekt, ale dla bezpieczeństwa zawsze merge
        setColorImages(prev => ({
          ...prev,
          ...result.colorImages
        }));

        toast({
          title: "Zdjęcie dodane",
          description: `Dodano zdjęcie dla koloru ${colorName}`,
        });
      } catch (error: any) {
        toast({
          title: "Błąd uploadu",
          description: error.message || "Nie udało się dodać zdjęcia",
          variant: "destructive",
        });
      }
    }
  };

  // Handle color image delete
  const handleColorImageDelete = async (colorIndex: number, imageUrl: string) => {
    if (isNew || !id) return;

    const filename = imageUrl.split('/').pop();
    if (!filename) return;
    
    const colorName = colors[colorIndex] || "unknown";

    try {
      const response = await fetch(`/api/product-matrices/${id}/color-images/${colorIndex}/${filename}`, {
        method: 'DELETE',
        credentials: 'include',
      });

      if (!response.ok) {
        const text = await response.text();
        throw new Error(`${response.status}: ${text}`);
      }

      const result = await response.json();
      
      // 🔧 FIX: Merguj colorImages zamiast zastępować cały state
      setColorImages(prev => ({
        ...prev,
        ...result.colorImages
      }));

      toast({
        title: "Zdjęcie usunięte",
        description: `Usunięto zdjęcie dla koloru ${colorName}`,
      });
    } catch (error: any) {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć zdjęcia",
        variant: "destructive",
      });
    }
  };

  // Handle image reorder (drag & drop)
  const handleImageReorder = async (colorIndex: number, fromIndex: number, toIndex: number) => {
    // Guard against invalid inputs
    if (isNew || !id || fromIndex === toIndex || fromIndex === undefined || toIndex === undefined) return;

    // 🔧 FIX: Zapisz oryginalny stan przed zmianą dla ewentualnego revert
    const originalImages = [...(colorImages[colorIndex.toString()] || [])];
    const images = [...originalImages];
    
    // Guard against out of bounds
    if (fromIndex < 0 || fromIndex >= images.length || toIndex < 0 || toIndex >= images.length) return;

    const [movedImage] = images.splice(fromIndex, 1);
    images.splice(toIndex, 0, movedImage);

    // 🔧 FIX: Oblicz nowy state ale używając functional update aby uniknąć race conditions
    let updatedColorImages: Record<string, string[]>;
    setColorImages(prev => {
      updatedColorImages = {
        ...prev,
        [colorIndex.toString()]: images,
      };
      return updatedColorImages;
    });

    // Save to backend
    try {
      await saveMutation.mutateAsync({
        name,
        namePrefix: namePrefix || null,
        nameSuffix: nameSuffix || null,
        productType: productType === "none" ? null : productType,
        doors: doors === "none" ? null : doors,
        legs: legs === "none" ? null : legs,
        material: material === "none" ? null : material,
        lengths,
        widths,
        heights,
        productGroups: productGroupsArray,
        priceModifiers: priceModifiersArray,
        priceModifierTypes: priceModifierTypesArray,
        colors,
        colorImages: updatedColorImages!,
        descriptionHtml,
        descriptionDoc,
        useAiGeneration,
        aiPrompt,
        defaultPrice,
      });
    } catch (error: any) {
      // 🔧 FIX: Revert używając functional update i oryginalnego stanu
      setColorImages(prev => ({
        ...prev,
        [colorIndex.toString()]: originalImages,
      }));
      toast({
        title: "Błąd zmiany kolejności",
        description: error.message || "Nie udało się zmienić kolejności zdjęć",
        variant: "destructive",
      });
    }
  };

  const handleSave = () => {
    console.log('🔵 handleSave called!');
    console.log('🔵 ProductType:', productType);
    console.log('🔵 Material:', material);
    console.log('🔵 TemplateId:', templateId);
    console.log('🔵 AiConfig:', aiConfig);
    
    // Validation - Product Type is required
    if (!productType || productType === "none") {
      console.log('❌ Product Type validation failed');
      toast({
        title: "Błąd walidacji",
        description: "Typ produktu jest polem wymaganym. Wybierz typ produktu z listy.",
        variant: "destructive",
      });
      return;
    }
    
    // Validation - Material is required
    if (!material || material === "none") {
      console.log('❌ Material validation failed');
      toast({
        title: "Błąd walidacji",
        description: "Materiał jest polem wymaganym. Wybierz materiał z listy.",
        variant: "destructive",
      });
      return;
    }

    const data = {
      name,
      namePrefix: namePrefix || null,
      nameSuffix: nameSuffix || null,
      productType: productType === "none" ? null : (productType || null),
      doors: doors === "none" ? null : (doors || null),
      legs: legs === "none" ? null : (legs || null),
      material: material === "none" ? null : (material || null),
      lengths,
      widths,
      heights,
      productGroups: productGroupsArray.map(pg => pg === "none" ? "" : pg),
      priceModifiers: priceModifiersArray,
      priceModifierTypes: priceModifierTypesArray,
      colors,
      colorImages,
      colorOptions: colorOptionsMatrix,
      selectedColors: colorsForGeneration.map(idx => String(idx)), // Save indices as strings, not names
      descriptionHtml,
      descriptionDoc,
      useAiGeneration,
      aiPrompt: aiPrompt || null,
      defaultPrice: defaultPrice || null,
      
      // Template system
      templateId: templateId || null,
      aiConfig: Object.keys(aiConfig).length > 0 ? aiConfig : null,
      hasWarranty: true, // Always true - controlled by AI config warranty checkbox
    };
    console.log('✅ Sending data to backend:', data);
    saveMutation.mutate(data);
  };

  const handlePreviewGeneration = async () => {
    if (isNew || !id) {
      toast({
        title: "Zapisz najpierw",
        description: "Musisz najpierw zapisać matrycę przed podglądem generowania.",
        variant: "destructive",
      });
      return;
    }

    if (!templateId) {
      toast({
        title: "Brak szablonu",
        description: "Matryca musi mieć przypisany szablon aby wygenerować podgląd.",
        variant: "destructive",
      });
      return;
    }

    try {
      setIsLoadingPreview(true);
      const response = await apiRequest("POST", `/api/product-matrices/${id}/preview-generation`, {}) as any;
      setPreviewData(response);
      setIsPreviewOpen(true);
      toast({
        title: "Podgląd wygenerowany",
        description: `Podgląd dla: ${response.preview.productName}`,
      });
    } catch (error: any) {
      console.error("Error generating preview:", error);
      toast({
        title: "Błąd generowania podglądu",
        description: error.message || "Nie udało się wygenerować podglądu.",
        variant: "destructive",
      });
    } finally {
      setIsLoadingPreview(false);
    }
  };

  const handleGenerate = () => {
    if (isNew || !id) {
      toast({
        title: "Zapisz najpierw",
        description: "Musisz najpierw zapisać matrycę przed generowaniem produktów.",
        variant: "destructive",
      });
      return;
    }
    
    // Generate unique session ID for log streaming
    const newSessionId = `generate-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
    setSessionId(newSessionId);
    setGenerationLogs([]); // Clear previous logs
    setIsGenerating(true);
    
    generateMutation.mutate({ id: parseInt(id), sessionId: newSessionId }, {
      onSettled: () => setIsGenerating(false),
    });
  };

  const handleDelete = () => {
    deleteMutation.mutate();
  };

  // Add dimension row
  const addDimensionRow = () => {
    if (!newLengthInput && !newWidthInput && !newHeightInput) {
      toast({
        title: "Brak wymiarów",
        description: "Wprowadź przynajmniej jeden wymiar.",
        variant: "destructive",
      });
      return;
    }

    const newLengths = [...lengths, newLengthInput || "0"];
    const newWidths = [...widths, newWidthInput || "0"];
    const newHeights = [...heights, newHeightInput || "0"];
    const newProductGroups = [...productGroupsArray, ""];
    const newPriceModifiers = [...priceModifiersArray, "0"];
    const newPriceModifierTypes = [...priceModifierTypesArray, "fixed"];

    setLengths(newLengths);
    setWidths(newWidths);
    setHeights(newHeights);
    setProductGroupsArray(newProductGroups);
    setPriceModifiersArray(newPriceModifiers);
    setPriceModifierTypesArray(newPriceModifierTypes);

    setNewLengthInput("");
    setNewWidthInput("");
    setNewHeightInput("");
  };

  // Add predefined sizes for specific height
  const addPredefinedSizes = (height: number) => {
    const predefinedLengths = [500, 600, 800, 1000];
    const predefinedWidths = [300, 360]; // Always add both widths for all heights

    // Check for existing combinations to avoid duplicates
    const existingCombinations = new Set<string>();
    for (let i = 0; i < Math.max(lengths.length, widths.length, heights.length); i++) {
      const key = `${lengths[i] || ''}-${widths[i] || ''}-${heights[i] || ''}`;
      existingCombinations.add(key);
    }

    const newLengths: string[] = [];
    const newWidths: string[] = [];
    const newHeights: string[] = [];
    const newProductGroups: string[] = [];
    const newPriceModifiers: string[] = [];
    const newPriceModifierTypes: string[] = [];
    let skippedCount = 0;

    predefinedLengths.forEach(length => {
      predefinedWidths.forEach(width => {
        const key = `${length}-${width}-${height}`;
        
        if (existingCombinations.has(key)) {
          skippedCount++;
          return; // Skip this combination
        }
        
        newLengths.push(String(length));
        newWidths.push(String(width));
        newHeights.push(String(height));
        newProductGroups.push("");
        newPriceModifiers.push("0");
        newPriceModifierTypes.push("fixed");
      });
    });
    
    const addedCount = newLengths.length;
    
    if (addedCount > 0) {
      // Update all state arrays at once
      setLengths(prev => [...prev, ...newLengths]);
      setWidths(prev => [...prev, ...newWidths]);
      setHeights(prev => [...prev, ...newHeights]);
      setProductGroupsArray(prev => [...prev, ...newProductGroups]);
      setPriceModifiersArray(prev => [...prev, ...newPriceModifiers]);
      setPriceModifierTypesArray(prev => [...prev, ...newPriceModifierTypes]);
      
      toast({ 
        title: `Dodano ${addedCount} nowych kombinacji`,
        description: skippedCount > 0 
          ? `${predefinedLengths.length} długości × ${predefinedWidths.length} szerokości dla H=${height}mm (pominięto ${skippedCount} duplikatów)`
          : `${predefinedLengths.length} długości × ${predefinedWidths.length} szerokości dla H=${height}mm`
      });
    } else {
      toast({ 
        title: "Wszystkie kombinacje już istnieją",
        description: `Brak nowych kombinacji do dodania dla H=${height}mm`,
        variant: "default"
      });
    }
  };

  // Remove dimension row
  const removeDimensionRow = (index: number) => {
    setLengths(prev => prev.filter((_, i) => i !== index));
    setWidths(prev => prev.filter((_, i) => i !== index));
    setHeights(prev => prev.filter((_, i) => i !== index));
    setProductGroupsArray(prev => prev.filter((_, i) => i !== index));
    setPriceModifiersArray(prev => prev.filter((_, i) => i !== index));
    setPriceModifierTypesArray(prev => prev.filter((_, i) => i !== index));
  };

  // Copy dimension row
  const copyDimensionRow = (index: number) => {
    setLengths(prev => [...prev, prev[index]]);
    setWidths(prev => [...prev, prev[index]]);
    setHeights(prev => [...prev, prev[index]]);
    setProductGroupsArray(prev => [...prev, prev[index]]);
    setPriceModifiersArray(prev => [...prev, prev[index]]);
    setPriceModifierTypesArray(prev => [...prev, prev[index] || "fixed"]);
  };

  // Update dimension at index
  const updateDimension = (index: number, field: 'length' | 'width' | 'height' | 'productGroup' | 'priceModifier' | 'priceModifierType', value: string) => {
    if (field === 'length') {
      const newLengths = [...lengths];
      newLengths[index] = value;
      setLengths(newLengths);
    } else if (field === 'width') {
      const newWidths = [...widths];
      newWidths[index] = value;
      setWidths(newWidths);
    } else if (field === 'height') {
      const newHeights = [...heights];
      newHeights[index] = value;
      setHeights(newHeights);
    } else if (field === 'productGroup') {
      const newGroups = [...productGroupsArray];
      newGroups[index] = value;
      setProductGroupsArray(newGroups);
    } else if (field === 'priceModifier') {
      const newModifiers = [...priceModifiersArray];
      newModifiers[index] = value;
      setPriceModifiersArray(newModifiers);
    } else if (field === 'priceModifierType') {
      const newTypes = [...priceModifierTypesArray];
      newTypes[index] = value;
      setPriceModifierTypesArray(newTypes);
    }
  };

  // Add color row
  const addColor = () => {
    setColors(prev => [...prev, ""]);
  };

  // Duplicate color with all its options and images
  const duplicateColor = (index: number) => {
    const colorName = colors[index];
    if (!colorName) return;
    
    // Add same color name again
    const newIndex = colors.length;
    setColors(prev => [...prev, colorName]);
    
    // Copy color images to new index
    const sourceImages = colorImages[index.toString()];
    if (sourceImages && sourceImages.length > 0) {
      setColorImages(prev => ({
        ...prev,
        [newIndex.toString()]: [...sourceImages]
      }));
    }
    
    // Copy color options to new index
    const sourceOptions = colorOptionsMatrix[index.toString()];
    if (sourceOptions && sourceOptions.length > 0) {
      setColorOptionsMatrix(prev => ({
        ...prev,
        [newIndex.toString()]: [...sourceOptions]
      }));
    }
    
    toast({ title: `Skopiowano kolor: ${colorName}` });
  };

  // Remove color
  const removeColor = (index: number) => {
    // Remove from colors array
    setColors(prev => prev.filter((_, i) => i !== index));
    
    // Remove from colorsForGeneration and shift down indices > removed index
    setColorsForGeneration(prev => 
      prev
        .filter(idx => idx !== index) // Remove deleted index
        .map(idx => idx > index ? idx - 1 : idx) // Shift down higher indices
    );
    
    // Reindex colorImages: remove deleted index and shift all higher indices down
    setColorImages(prev => {
      const updated: Record<string, string[]> = {};
      Object.keys(prev).forEach(key => {
        const numKey = parseInt(key);
        if (numKey < index) {
          // Keep indices before deleted index unchanged
          updated[key] = prev[key];
        } else if (numKey > index) {
          // Shift indices after deleted index down by 1
          updated[(numKey - 1).toString()] = prev[key];
        }
        // Skip the deleted index
      });
      return updated;
    });
    
    // Reindex colorOptionsMatrix: same logic as colorImages
    setColorOptionsMatrix(prev => {
      const updated: Record<string, string[]> = {};
      Object.keys(prev).forEach(key => {
        const numKey = parseInt(key);
        if (numKey < index) {
          updated[key] = prev[key];
        } else if (numKey > index) {
          updated[(numKey - 1).toString()] = prev[key];
        }
      });
      return updated;
    });
    
    // Clear selection
    setSelectedColorsForDelete(prev => prev.filter(i => i !== index).map(i => i > index ? i - 1 : i));
  };

  // Copy settings from first record to all others
  const copySettingsFromFirstRecord = () => {
    // Check if there are any records
    const recordCount = Math.max(
      productGroupsArray.length,
      priceModifiersArray.length,
      priceModifierTypesArray.length,
      lengths.length,
      widths.length,
      heights.length
    );

    if (recordCount === 0) {
      toast({ 
        title: "Brak rekordów",
        description: "Dodaj najpierw rozmiary do tabeli",
        variant: "destructive"
      });
      return;
    }

    const firstProductGroup = productGroupsArray[0] || "";
    const firstPriceModifier = priceModifiersArray[0] || "0";
    const firstPriceModifierType = priceModifierTypesArray[0] || "fixed";

    // Copy to all other records
    setProductGroupsArray(prev => prev.map(() => firstProductGroup));
    setPriceModifiersArray(prev => prev.map(() => firstPriceModifier));
    setPriceModifierTypesArray(prev => prev.map(() => firstPriceModifierType));

    const affectedCount = productGroupsArray.length > 0 ? productGroupsArray.length - 1 : 0;
    
    toast({ 
      title: "Skopiowano ustawienia",
      description: `Grupa produktu i modyfikator ceny z pierwszego rekordu zostały skopiowane do ${affectedCount} pozostałych rekordów`
    });
  };

  // Bulk delete selected sizes
  const removeSelectedSizes = () => {
    if (selectedSizes.length === 0) return;
    
    const indicesToKeep = Array.from({ length: maxDimensions }, (_, i) => i).filter(i => !selectedSizes.includes(i));
    
    setLengths(prev => indicesToKeep.map(i => prev[i] || ""));
    setWidths(prev => indicesToKeep.map(i => prev[i] || ""));
    setHeights(prev => indicesToKeep.map(i => prev[i] || ""));
    setProductGroupsArray(prev => indicesToKeep.map(i => prev[i] || ""));
    setPriceModifiersArray(prev => indicesToKeep.map(i => prev[i] || ""));
    setPriceModifierTypesArray(prev => indicesToKeep.map(i => prev[i] || ""));
    
    setSelectedSizes([]);
    toast({ title: `Usunięto ${selectedSizes.length} rozmiarów` });
  };

  // Bulk delete selected colors
  const removeSelectedColors = () => {
    if (selectedColorsForDelete.length === 0) return;
    
    // Create index mapping: old index -> new index for colors that remain
    const indexMap = new Map<number, number>();
    let newIndex = 0;
    colors.forEach((_, oldIndex) => {
      if (!selectedColorsForDelete.includes(oldIndex)) {
        indexMap.set(oldIndex, newIndex);
        newIndex++;
      }
    });
    
    // Update main colors list
    setColors(prev => prev.filter((_, i) => !selectedColorsForDelete.includes(i)));
    
    // Remap colorsForGeneration indices using the index mapping
    setColorsForGeneration(prev => 
      prev
        .filter(idx => !selectedColorsForDelete.includes(idx)) // Remove deleted indices
        .map(idx => indexMap.get(idx) ?? idx) // Remap to new indices
        .filter(idx => idx !== undefined)
    );
    
    // Reindex colorImages using the index mapping
    setColorImages(prev => {
      const updated: Record<string, string[]> = {};
      Object.keys(prev).forEach(key => {
        const oldIndex = parseInt(key);
        const newIdx = indexMap.get(oldIndex);
        if (newIdx !== undefined) {
          updated[newIdx.toString()] = prev[key];
        }
      });
      return updated;
    });
    
    // Reindex colorOptionsMatrix using the index mapping
    setColorOptionsMatrix(prev => {
      const updated: Record<string, string[]> = {};
      Object.keys(prev).forEach(key => {
        const oldIndex = parseInt(key);
        const newIdx = indexMap.get(oldIndex);
        if (newIdx !== undefined) {
          updated[newIdx.toString()] = prev[key];
        }
      });
      return updated;
    });
    
    setSelectedColorsForDelete([]);
    toast({ title: `Usunięto ${selectedColorsForDelete.length} kolorów` });
  };

  // Toggle size selection
  const toggleSizeSelection = (index: number) => {
    setSelectedSizes(prev => 
      prev.includes(index) ? prev.filter(i => i !== index) : [...prev, index]
    );
  };

  // Toggle color selection
  const toggleColorSelection = (index: number) => {
    setSelectedColorsForDelete(prev => 
      prev.includes(index) ? prev.filter(i => i !== index) : [...prev, index]
    );
  };

  // Toggle all sizes
  const toggleAllSizes = () => {
    if (selectedSizes.length === maxDimensions) {
      setSelectedSizes([]);
    } else {
      setSelectedSizes(Array.from({ length: maxDimensions }, (_, i) => i));
    }
  };

  // Toggle all colors
  const toggleAllColors = () => {
    if (selectedColorsForDelete.length === colors.length) {
      setSelectedColorsForDelete([]);
    } else {
      setSelectedColorsForDelete(Array.from({ length: colors.length }, (_, i) => i));
    }
  };

  // Load all colors from dictionary
  const loadAllColors = () => {
    if (!colorOptions) return;
    const allColors = colorOptions
      .filter(opt => opt.isActive)
      .map(opt => opt.name || opt.code);
    setColors(allColors);
  };

  // Toggle color for generation (NO auto-save - just local state)
  // colorIndex is the position in colors array
  const toggleColorForGeneration = (colorIndex: number) => {
    const wasSelected = colorsForGeneration.includes(colorIndex);
    const newSelectedIndices = wasSelected
      ? colorsForGeneration.filter(idx => idx !== colorIndex)
      : [...colorsForGeneration, colorIndex];
    
    setColorsForGeneration(newSelectedIndices);
  };

  // Toggle all colors for generation (NO auto-save - will save when clicking "Zapisz")
  const toggleAllColorsForGeneration = () => {
    const newSelectedIndices = colorsForGeneration.length === colors.length 
      ? [] 
      : colors.map((_, idx) => idx);
    
    setColorsForGeneration(newSelectedIndices);
  };

  // Clean up orphaned selectedColors (indices that are out of bounds)
  const cleanupSelectedColors = () => {
    // Filter to only include valid indices (within bounds of colors array)
    const validSelectedIndices = colorsForGeneration.filter(idx => idx >= 0 && idx < colors.length);
    const removedCount = colorsForGeneration.length - validSelectedIndices.length;
    
    if (removedCount === 0) {
      toast({
        title: "Brak starych kolorów",
        description: "Wszystkie zaznaczone kolory istnieją w aktualnej liście"
      });
      return;
    }
    
    setColorsForGeneration(validSelectedIndices);
    
    toast({
      title: "Wyczyszczono stare kolory",
      description: `Usunięto ${removedCount} nieistniejących kolorów z zaznaczenia. Kliknij "Zapisz" aby zachować zmiany.`
    });
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <Loader2 className="w-8 h-8 animate-spin" />
      </div>
    );
  }

  const maxDimensions = Math.max(
    lengths.length,
    widths.length,
    heights.length,
    1
  );

  return (
    <div className="w-full px-6 py-6">
      {/* Progress Dialog with Live Logs */}
      <Dialog open={generationProgress.isGenerating} onOpenChange={() => {}}>
        <DialogContent className="max-w-4xl max-h-[80vh] flex flex-col" onPointerDownOutside={(e) => e.preventDefault()} onEscapeKeyDown={(e) => e.preventDefault()}>
          <DialogHeader className="flex-shrink-0">
            <DialogTitle>Generowanie produktów</DialogTitle>
            <DialogDescription>
              Proszę czekać, trwa generowanie wariantów produktów...
            </DialogDescription>
          </DialogHeader>
          
          <div className="flex-1 min-h-0 flex flex-col gap-4">
            {/* Stats Section */}
            <div className="flex-shrink-0 space-y-2">
              <div className="flex justify-between text-sm">
                <span className="text-muted-foreground">Status:</span>
                <span className="font-medium">{generationProgress.currentStep}</span>
              </div>
              <Progress value={undefined} className="h-2" />
              <div className="grid grid-cols-3 gap-4">
                <div className="space-y-1">
                  <p className="text-xs text-muted-foreground">Rozmiarów</p>
                  <p className="text-xl font-bold">{Math.max(lengths.length, widths.length, heights.length)}</p>
                </div>
                <div className="space-y-1">
                  <p className="text-xs text-muted-foreground">Kolorów</p>
                  <p className="text-xl font-bold">{generationProgress.colorsCount || 0}</p>
                </div>
                <div className="space-y-1">
                  <p className="text-xs text-muted-foreground">Razem</p>
                  <p className="text-xl font-bold text-primary">{generationProgress.productsCount}</p>
                </div>
              </div>
            </div>
            
            {/* Live Logs Section */}
            <div className="flex-1 min-h-0 bg-muted/30 rounded-lg border p-3 flex flex-col">
              <p className="text-xs font-medium text-muted-foreground mb-2 flex-shrink-0">Logi na żywo:</p>
              <div 
                ref={logsScrollRef}
                className="flex-1 min-h-0 overflow-y-auto space-y-1 text-xs font-mono"
              >
                {generationLogs.length === 0 ? (
                  <p className="text-muted-foreground text-center py-4">Oczekiwanie na logi...</p>
                ) : (
                  generationLogs.map((log, index) => (
                    <div key={index} className="flex items-start gap-2">
                      <span className="text-muted-foreground flex-shrink-0">
                        {new Date(log.timestamp).toLocaleTimeString('pl-PL', {
                          hour: '2-digit',
                          minute: '2-digit',
                          second: '2-digit',
                        })}
                      </span>
                      <span className="flex-shrink-0">
                        {log.type === 'success' && '✅'}
                        {log.type === 'error' && '❌'}
                        {log.type === 'warning' && '⚠️'}
                        {log.type === 'info' && '📝'}
                      </span>
                      <span className={
                        log.type === 'success' ? 'text-green-600 dark:text-green-400' :
                        log.type === 'error' ? 'text-red-600 dark:text-red-400' :
                        log.type === 'warning' ? 'text-yellow-600 dark:text-yellow-400' :
                        'text-foreground/80'
                      }>
                        {log.message}
                      </span>
                    </div>
                  ))
                )}
              </div>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Header */}
      <div className="flex flex-col gap-4 mb-6 md:flex-row md:items-center md:justify-between">
        <div className="flex items-center gap-2 md:gap-4">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => window.history.back()}
            data-testid="button-back"
          >
            <ArrowLeft className="w-4 h-4" />
          </Button>
          <div className="flex-1 min-w-0">
            <h1 className="text-xl md:text-2xl font-bold truncate">
              {isNew ? (
                "Nowa Matryca Produktów"
              ) : (
                <>
                  <span className="hidden md:inline">Edytuj Matrycę: </span>
                  {name}
                </>
              )}
            </h1>
            <p className="text-sm text-muted-foreground hidden md:block">
              {isNew
                ? "Skonfiguruj parametry matryce do generowania wariantów produktów"
                : "Modyfikuj parametry matryce produktów"}
            </p>
            {!isNew && (
              <div className="flex items-center gap-1 md:gap-2 mt-1 md:mt-2 flex-wrap">
                <span className="text-xs md:text-sm font-medium text-foreground hidden md:inline">
                  Symulowane warianty:
                </span>
                <span className="text-xs md:text-sm font-bold text-primary">
                  {lengths.length} <span className="hidden md:inline">rozmiarów</span> × {colors.length} <span className="hidden md:inline">kolorów</span> = {lengths.length * colors.length} <span className="hidden md:inline">wariantów</span>
                </span>
              </div>
            )}
          </div>
        </div>
        <div className="flex items-center gap-1 md:gap-2 flex-wrap">
          {!isNew && (
            <>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setIsDeleteDialogOpen(true)}
                data-testid="button-delete"
              >
                <Trash2 className="w-4 h-4 md:mr-2" />
                <span className="hidden md:inline">Usuń</span>
              </Button>
              <Button
                size="sm"
                onClick={handleGenerate}
                disabled={isGenerating}
                data-testid="button-generate"
                className="whitespace-nowrap"
              >
                {isGenerating ? (
                  <Loader2 className="w-4 h-4 md:mr-2 animate-spin" />
                ) : (
                  <Play className="w-4 h-4 md:mr-2" />
                )}
                <span className="hidden md:inline">Generuj produkty</span>
              </Button>
            </>
          )}
          <Button
            size="sm"
            onClick={handleSave}
            disabled={saveMutation.isPending}
            data-testid="button-save"
          >
            {saveMutation.isPending ? (
              <Loader2 className="w-4 h-4 md:mr-2 animate-spin" />
            ) : (
              <Save className="w-4 h-4 md:mr-2" />
            )}
            <span className="hidden md:inline">Zapisz</span>
          </Button>
        </div>
      </div>

      {/* Tabs */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="grid w-full grid-cols-5">
          <TabsTrigger value="podstawowe">Podstawowe</TabsTrigger>
          <TabsTrigger value="rozmiary">Rozmiary</TabsTrigger>
          <TabsTrigger value="kolory">Kolory</TabsTrigger>
          <TabsTrigger value="opis">Opis</TabsTrigger>
          <TabsTrigger value="podsumowanie">Podsumowanie</TabsTrigger>
        </TabsList>

        {/* PODSTAWOWE TAB */}
        <TabsContent value="podstawowe" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Podstawowe informacje</CardTitle>
              <CardDescription>
                Nazwa, prefix, suffix oraz parametry słownikowe
              </CardDescription>
            </CardHeader>
            <CardContent>
              {/* Main 2-column layout */}
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                {/* LEFT COLUMN */}
                <div className="space-y-4">
                  {/* Nazwa - wyróżniona */}
                  <div className="space-y-2 p-4 rounded-lg bg-accent/20 border border-accent/40">
                    <Label htmlFor="name" className="font-semibold">Nazwa matryce *</Label>
                    <Input
                      id="name"
                      value={name}
                      onChange={(e) => setName(e.target.value)}
                      placeholder="np. Szafki SUPRA"
                      data-testid="input-name"
                      className="bg-background"
                    />
                  </div>

                  {/* Prefix i Suffix - 2 kolumny */}
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="prefix">Prefix nazwy</Label>
                      <Input
                        id="prefix"
                        value={namePrefix}
                        onChange={(e) => setNamePrefix(e.target.value)}
                        placeholder="np. Komoda"
                        data-testid="input-prefix"
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="suffix">Suffix nazwy</Label>
                      <Input
                        id="suffix"
                        value={nameSuffix}
                        onChange={(e) => setNameSuffix(e.target.value)}
                        placeholder="np. Premium"
                        data-testid="input-suffix"
                      />
                    </div>
                  </div>

                  {/* Cena bazowa i Materiał - 2 kolumny */}
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="defaultPrice">Cena bazowa (PLN)</Label>
                      <Input
                        id="defaultPrice"
                        type="number"
                        step="0.01"
                        value={defaultPrice}
                        onChange={(e) => setDefaultPrice(e.target.value)}
                        placeholder="np. 999.99"
                        data-testid="input-default-price"
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="material">Materiał</Label>
                      <Select value={material} onValueChange={setMaterial}>
                        <SelectTrigger id="material" data-testid="select-material">
                          <SelectValue placeholder="Wybierz materiał" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="none">Brak</SelectItem>
                          {materialOptions?.filter(opt => opt.isActive).map((opt) => (
                            <SelectItem key={opt.id} value={opt.name || opt.code}>
                              {opt.readableName || opt.name || opt.code}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                  </div>
                </div>

                {/* RIGHT COLUMN */}
                <div className="space-y-4">
                  {/* Typ produktu - cała szerokość */}
                  <div className="space-y-2 p-3 bg-accent/10 rounded-md border border-accent/30">
                    <Label htmlFor="productType" className="text-sm font-medium">
                      Typ produktu <span className="text-destructive">*</span>
                    </Label>
                    <DictionaryComboboxWithAdd
                      items={productTypes || []}
                      value={productType === "none" ? "" : (productType || "")}
                      onChange={(value) => setProductType(value || "none")}
                      dictionaryType="product_type"
                      placeholder="Wybierz typ produktu"
                      searchPlaceholder="Szukaj typu produktu..."
                      emptyText="Nie znaleziono typu produktu"
                      testId="select-product-type"
                      valueField="code"
                      displayField="name"
                    />
                  </div>

                  {/* Drzwi i Nóżki - 2 kolumny */}
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="doors">Drzwi</Label>
                      <Select value={doors} onValueChange={setDoors}>
                        <SelectTrigger id="doors" data-testid="select-doors">
                          <SelectValue placeholder="Wybierz drzwi" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="none">Brak</SelectItem>
                          {doorsOptions?.filter(opt => opt.isActive).map((opt) => (
                            <SelectItem key={opt.id} value={opt.name || opt.code}>
                              {opt.readableName || opt.name || opt.code}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="legs">Nóżki</Label>
                      <Select value={legs} onValueChange={setLegs}>
                        <SelectTrigger id="legs" data-testid="select-legs">
                          <SelectValue placeholder="Wybierz nóżki" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="none">Brak</SelectItem>
                          {legsOptions?.filter(opt => opt.isActive).map((opt) => (
                            <SelectItem key={opt.id} value={opt.name || opt.code}>
                              {opt.readableName || opt.name || opt.code}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* ROZMIARY TAB */}
        <TabsContent value="rozmiary" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Szybkie dodawanie</CardTitle>
              <CardDescription>
                Dodaj wszystkie kombinacje predefiniowanych wymiarów dla wybranej wysokości:
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-2">
              <div className="flex flex-wrap gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => addPredefinedSizes(450)}
                  data-testid="button-add-h450"
                >
                  <Plus className="w-4 h-4 md:mr-2" />
                  <span className="hidden md:inline">Wszystkie kombinacje dla </span>H=450mm
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => addPredefinedSizes(820)}
                  data-testid="button-add-h820"
                >
                  <Plus className="w-4 h-4 md:mr-2" />
                  <span className="hidden md:inline">Wszystkie kombinacje dla </span>H=820mm
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => addPredefinedSizes(1240)}
                  data-testid="button-add-h1240"
                >
                  <Plus className="w-4 h-4 md:mr-2" />
                  <span className="hidden md:inline">Wszystkie kombinacje dla </span>H=1240mm
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => addPredefinedSizes(2050)}
                  data-testid="button-add-h2050"
                >
                  <Plus className="w-4 h-4 md:mr-2" />
                  <span className="hidden md:inline">Wszystkie kombinacje dla </span>H=2050mm
                </Button>
              </div>
              <p className="text-xs text-muted-foreground">
                • Długość: 500, 600, 800, 1000mm<br />
                • Szerokość: 300, 360mm<br />
                • Każdy przycisk doda 8 kombinacji
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Kopiowanie ustawień</CardTitle>
              <CardDescription>
                Skopiuj grupę produktu i modyfikator ceny z pierwszego rekordu do wszystkich pozostałych
              </CardDescription>
            </CardHeader>
            <CardContent>
              <Button
                variant="outline"
                size="sm"
                onClick={copySettingsFromFirstRecord}
                disabled={Math.max(lengths.length, widths.length, heights.length) === 0}
                data-testid="button-copy-from-first"
              >
                <Copy className="w-4 h-4 md:mr-2" />
                <span className="hidden md:inline">Kopiuj z pierwszego rekordu</span>
                <span className="md:hidden">Kopiuj z 1.</span>
              </Button>
              <p className="text-xs text-muted-foreground mt-2">
                {(() => {
                  const recordCount = Math.max(lengths.length, widths.length, heights.length);
                  return recordCount > 0 
                    ? `Skopiuje grupę produktu, modyfikator ceny i typ modyfikatora z pierwszego wiersza do wszystkich pozostałych (${recordCount - 1} rekordów)`
                    : "Dodaj najpierw rozmiary do tabeli, aby użyć tej funkcji";
                })()}
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Lub dodaj pojedynczy rozmiar ręcznie</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 md:grid-cols-4 gap-4 items-end">
                <div className="space-y-2">
                  <Label htmlFor="newLength">Długość (mm)</Label>
                  <Select value={newLengthInput} onValueChange={setNewLengthInput}>
                    <SelectTrigger id="newLength" data-testid="select-new-length">
                      <SelectValue placeholder="Wybierz" />
                    </SelectTrigger>
                    <SelectContent>
                      {getDimensionValues(dimensionLengths).map((val) => (
                        <SelectItem key={val} value={val}>{val}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="newWidth">Szerokość (mm)</Label>
                  <Select value={newWidthInput} onValueChange={setNewWidthInput}>
                    <SelectTrigger id="newWidth" data-testid="select-new-width">
                      <SelectValue placeholder="Wybierz" />
                    </SelectTrigger>
                    <SelectContent>
                      {getDimensionValues(dimensionWidths).map((val) => (
                        <SelectItem key={val} value={val}>{val}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="newHeight">Wysokość (mm)</Label>
                  <Select value={newHeightInput} onValueChange={setNewHeightInput}>
                    <SelectTrigger id="newHeight" data-testid="select-new-height">
                      <SelectValue placeholder="Wybierz" />
                    </SelectTrigger>
                    <SelectContent>
                      {getDimensionValues(dimensionHeights).map((val) => (
                        <SelectItem key={val} value={val}>{val}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <Button onClick={addDimensionRow} data-testid="button-add-dimension">
                  <Plus className="w-4 h-4 mr-2" />
                  Dodaj
                </Button>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Lista rozmiarów ({maxDimensions})</CardTitle>
                  <CardDescription>
                    Każdy rozmiar może mieć swoją grupę produktu i modyfikator ceny
                  </CardDescription>
                </div>
                {selectedSizes.length > 0 && (
                  <Button
                    variant="destructive"
                    size="sm"
                    onClick={removeSelectedSizes}
                    data-testid="button-delete-selected-sizes"
                  >
                    <Trash2 className="w-4 h-4 md:mr-2" />
                    <span className="hidden md:inline">Usuń zaznaczone</span> ({selectedSizes.length})
                  </Button>
                )}
              </div>
            </CardHeader>
            <CardContent>
              <div className="border rounded-md overflow-x-auto">
                <Table className="min-w-[700px]">
                  <TableHeader>
                    <TableRow>
                      <TableHead className="w-12">
                        <Checkbox
                          checked={selectedSizes.length === maxDimensions && maxDimensions > 0}
                          onCheckedChange={toggleAllSizes}
                          data-testid="checkbox-select-all-sizes"
                        />
                      </TableHead>
                      <TableHead className="min-w-[100px]">Długość</TableHead>
                      <TableHead className="min-w-[100px]">Szerokość</TableHead>
                      <TableHead className="min-w-[100px]">Wysokość</TableHead>
                      <TableHead className="min-w-[120px]">
                        <span className="hidden md:inline">Grupa produktu</span>
                        <span className="md:hidden">Grupa</span>
                      </TableHead>
                      <TableHead className="min-w-[180px]">
                        <span className="hidden md:inline">Modyfikator ceny</span>
                        <span className="md:hidden">Mod. ceny</span>
                      </TableHead>
                      <TableHead className="w-24">Akcje</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {maxDimensions === 0 || (lengths.length === 0 && widths.length === 0 && heights.length === 0) ? (
                      <TableRow>
                        <TableCell colSpan={7} className="text-center text-muted-foreground py-8">
                          Brak rozmiarów. Dodaj nowy rozmiar powyżej.
                        </TableCell>
                      </TableRow>
                    ) : (
                      Array.from({ length: maxDimensions }).map((_, index) => (
                        <TableRow key={index} data-testid={`row-dimension-${index}`}>
                          <TableCell>
                            <Checkbox
                              checked={selectedSizes.includes(index)}
                              onCheckedChange={() => toggleSizeSelection(index)}
                              data-testid={`checkbox-size-${index}`}
                            />
                          </TableCell>
                          <TableCell>
                            <Select
                              value={lengths[index] || ""}
                              onValueChange={(val) => updateDimension(index, 'length', val)}
                            >
                              <SelectTrigger data-testid={`select-length-${index}`}>
                                <SelectValue placeholder="Wybierz" />
                              </SelectTrigger>
                              <SelectContent>
                                {getDimensionValues(dimensionLengths).map((val) => (
                                  <SelectItem key={val} value={val}>{val}</SelectItem>
                                ))}
                              </SelectContent>
                            </Select>
                          </TableCell>
                          <TableCell>
                            <Select
                              value={widths[index] || ""}
                              onValueChange={(val) => updateDimension(index, 'width', val)}
                            >
                              <SelectTrigger data-testid={`select-width-${index}`}>
                                <SelectValue placeholder="Wybierz" />
                              </SelectTrigger>
                              <SelectContent>
                                {getDimensionValues(dimensionWidths).map((val) => (
                                  <SelectItem key={val} value={val}>{val}</SelectItem>
                                ))}
                              </SelectContent>
                            </Select>
                          </TableCell>
                          <TableCell>
                            <Select
                              value={heights[index] || ""}
                              onValueChange={(val) => updateDimension(index, 'height', val)}
                            >
                              <SelectTrigger data-testid={`select-height-${index}`}>
                                <SelectValue placeholder="Wybierz" />
                              </SelectTrigger>
                              <SelectContent>
                                {getDimensionValues(dimensionHeights).map((val) => (
                                  <SelectItem key={val} value={val}>{val}</SelectItem>
                                ))}
                              </SelectContent>
                            </Select>
                          </TableCell>
                          <TableCell>
                            <Popover 
                              open={productGroupOpenRows.has(index)} 
                              onOpenChange={(open) => {
                                const newSet = new Set(productGroupOpenRows);
                                if (open) {
                                  newSet.add(index);
                                } else {
                                  newSet.delete(index);
                                }
                                setProductGroupOpenRows(newSet);
                              }}
                            >
                              <PopoverTrigger asChild>
                                <Button
                                  variant="outline"
                                  role="combobox"
                                  aria-expanded={productGroupOpenRows.has(index)}
                                  className="w-full justify-between"
                                  data-testid={`select-group-${index}`}
                                >
                                  {productGroupsArray[index] 
                                    ? productGroups?.find(opt => (opt.name || opt.code) === productGroupsArray[index])?.readableName || productGroupsArray[index]
                                    : "Wybierz grupę"}
                                  <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                                </Button>
                              </PopoverTrigger>
                              <PopoverContent className="w-[200px] p-0">
                                <Command>
                                  <CommandInput placeholder="Szukaj grupy..." />
                                  <CommandList>
                                    <CommandEmpty>
                                      <div className="text-center py-2">
                                        <p className="text-sm text-muted-foreground mb-2">Nie znaleziono</p>
                                        <Button
                                          size="sm"
                                          onClick={() => {
                                            setIsNewGroupDialogOpen(true);
                                            const newSet = new Set(productGroupOpenRows);
                                            newSet.delete(index);
                                            setProductGroupOpenRows(newSet);
                                          }}
                                          data-testid="button-add-new-group"
                                        >
                                          <Plus className="h-4 w-4 mr-1" />
                                          Dodaj nową
                                        </Button>
                                      </div>
                                    </CommandEmpty>
                                    <CommandGroup>
                                      <CommandItem
                                        value="none"
                                        onSelect={() => {
                                          updateDimension(index, 'productGroup', '');
                                          const newSet = new Set(productGroupOpenRows);
                                          newSet.delete(index);
                                          setProductGroupOpenRows(newSet);
                                        }}
                                      >
                                        <Check
                                          className={`mr-2 h-4 w-4 ${productGroupsArray[index] === '' || !productGroupsArray[index] ? "opacity-100" : "opacity-0"}`}
                                        />
                                        Brak
                                      </CommandItem>
                                      {productGroups?.filter(opt => opt.isActive).map((opt) => (
                                        <CommandItem
                                          key={opt.id}
                                          value={opt.name || opt.code}
                                          onSelect={(currentValue) => {
                                            updateDimension(index, 'productGroup', currentValue);
                                            const newSet = new Set(productGroupOpenRows);
                                            newSet.delete(index);
                                            setProductGroupOpenRows(newSet);
                                          }}
                                        >
                                          <Check
                                            className={`mr-2 h-4 w-4 ${productGroupsArray[index] === (opt.name || opt.code) ? "opacity-100" : "opacity-0"}`}
                                          />
                                          {opt.readableName || opt.name || opt.code}
                                        </CommandItem>
                                      ))}
                                    </CommandGroup>
                                  </CommandList>
                                </Command>
                              </PopoverContent>
                            </Popover>
                          </TableCell>
                          <TableCell>
                            <div className="flex gap-1">
                              <Input
                                type="number"
                                step="0.01"
                                value={priceModifiersArray[index] || "0"}
                                onChange={(e) => updateDimension(index, 'priceModifier', e.target.value)}
                                placeholder="0"
                                className="w-20"
                                data-testid={`input-modifier-${index}`}
                              />
                              <Select
                                value={priceModifierTypesArray[index] || "fixed"}
                                onValueChange={(value) => updateDimension(index, 'priceModifierType', value)}
                              >
                                <SelectTrigger className="w-24" data-testid={`select-modifier-type-${index}`}>
                                  <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                  <SelectItem value="percent">%</SelectItem>
                                  <SelectItem value="fixed">PLN</SelectItem>
                                </SelectContent>
                              </Select>
                            </div>
                          </TableCell>
                          <TableCell>
                            <div className="flex items-center gap-1">
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => copyDimensionRow(index)}
                                data-testid={`button-copy-${index}`}
                              >
                                <Copy className="w-4 h-4" />
                              </Button>
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => removeDimensionRow(index)}
                                data-testid={`button-remove-${index}`}
                              >
                                <X className="w-4 h-4" />
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              </div>
              <p className="text-xs text-muted-foreground mt-2">
                Modyfikator ceny: wybierz typ (% lub PLN) i wartość. Wartość dodatnia zwiększa cenę, ujemna obniża.
              </p>
            </CardContent>
          </Card>
        </TabsContent>

        {/* KOLORY TAB */}
        <TabsContent value="kolory" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Kolory</CardTitle>
              <CardDescription>
                Tabela kolorów - każdy kolor stanie się wariantem produktu. Możesz dodać zdjęcia dla każdego koloru.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center gap-2 flex-wrap">
                <Button size="sm" onClick={addColor} data-testid="button-add-color">
                  <Plus className="w-4 h-4 mr-2" />
                  <span>Dodaj kolor</span>
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={loadAllColors}
                  data-testid="button-load-all-colors"
                >
                  <span className="hidden md:inline">Załaduj wszystkie ze słownika</span>
                  <span className="md:hidden">Załaduj wszystkie</span>
                </Button>
                <Button
                  variant="secondary"
                  size="sm"
                  onClick={cleanupSelectedColors}
                  data-testid="button-cleanup-selected-colors"
                  title="Usuwa zaznaczone kolory które już nie istnieją w tabeli kolorów"
                >
                  <span className="hidden md:inline">Wyczyść stare zaznaczenia</span>
                  <span className="md:hidden">Wyczyść stare</span>
                </Button>
                {selectedColorsForDelete.length > 0 && (
                  <Button
                    variant="destructive"
                    size="sm"
                    onClick={removeSelectedColors}
                    data-testid="button-delete-selected-colors"
                  >
                    <Trash2 className="w-4 h-4 md:mr-2" />
                    <span className="hidden md:inline">Usuń zaznaczone</span> ({selectedColorsForDelete.length})
                  </Button>
                )}
              </div>

              <div className="border rounded-lg overflow-x-auto">
                <Table className="min-w-[700px]">
                  <TableHeader>
                    <TableRow>
                      <TableHead className="w-10 py-1 px-2 h-8">
                        <Checkbox
                          checked={selectedColorsForDelete.length === colors.length && colors.length > 0}
                          onCheckedChange={toggleAllColors}
                          data-testid="checkbox-select-all-colors"
                        />
                      </TableHead>
                      <TableHead className="w-10 py-1 px-2 h-8">#</TableHead>
                      <TableHead className="min-w-[120px] py-1 px-1.5 h-8 text-xs">Kolor</TableHead>
                      <TableHead className="min-w-[130px] py-1 px-2 text-xs h-8">Zdjęcia</TableHead>
                      <TableHead className="w-16 py-1 px-1 h-8">
                        <div className="flex items-center gap-1 justify-center whitespace-nowrap">
                          <span className="text-xs">Gen.</span>
                          <Checkbox
                            className="h-3.5 w-3.5"
                            checked={colorsForGeneration.length === colors.length && colors.length > 0}
                            onCheckedChange={toggleAllColorsForGeneration}
                            data-testid="checkbox-select-all-for-generation"
                          />
                        </div>
                      </TableHead>
                      <TableHead className="min-w-[80px] py-1 px-1 h-8">
                        <span className="text-xs">Ost. gen.</span>
                      </TableHead>
                      <TableHead className="min-w-[80px] py-1 px-2 text-xs h-8">Opcje kolor</TableHead>
                      <TableHead className="w-20 text-right py-1 px-2 text-xs h-8">Akcje</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {colors.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={8} className="text-center text-muted-foreground">
                          Brak kolorów. Kliknij "Dodaj kolor" aby rozpocząć.
                        </TableCell>
                      </TableRow>
                    ) : (
                      colors.map((color, index) => {
                        const colorDictEntry = sortedColorOptions.find(opt => (opt.name || opt.code) === color);
                        const isSelectedForGeneration = colorsForGeneration.includes(index);
                        const wasOriginallySelected = originalColorsForGenerationRef.current.includes(index);
                        const hasUnsavedChange = isSelectedForGeneration !== wasOriginallySelected;
                        
                        return (
                        <TableRow 
                          key={index}
                          className={
                            hasUnsavedChange 
                              ? 'bg-red-50 dark:bg-red-900/20 border-l-4 border-l-red-500' 
                              : isSelectedForGeneration 
                                ? 'bg-green-50 dark:bg-green-900/20' 
                                : ''
                          }
                        >
                          <TableCell className="py-1 px-2">
                            <Checkbox
                              checked={selectedColorsForDelete.includes(index)}
                              onCheckedChange={() => toggleColorSelection(index)}
                              data-testid={`checkbox-color-${index}`}
                            />
                          </TableCell>
                          <TableCell className="font-mono text-muted-foreground py-1 px-2">
                            {index + 1}
                          </TableCell>
                          <TableCell className="py-1 px-1.5">
                            <Popover 
                              open={colorOpenRows.has(index)} 
                              onOpenChange={(open) => {
                                if (open) {
                                  setColorOpenRows(prev => {
                                    const newSet = new Set(prev);
                                    newSet.add(index);
                                    return newSet;
                                  });
                                } else {
                                  // Clear search input
                                  setColorSearchInputs(prev => {
                                    const newMap = new Map(prev);
                                    newMap.delete(index);
                                    return newMap;
                                  });
                                  // Close popover
                                  setColorOpenRows(prev => {
                                    const newSet = new Set(prev);
                                    newSet.delete(index);
                                    return newSet;
                                  });
                                }
                              }}
                            >
                              <PopoverTrigger asChild>
                                <Button
                                  variant="outline"
                                  role="combobox"
                                  aria-expanded={colorOpenRows.has(index)}
                                  className="w-full justify-between"
                                  style={colorDictEntry?.color ? {
                                    backgroundColor: colorDictEntry.color,
                                    color: getContrastColor(colorDictEntry.color)
                                  } : undefined}
                                  data-testid={`select-color-${index}`}
                                >
                                  {color 
                                    ? colorDictEntry?.name || color
                                    : "Wybierz kolor"}
                                  <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                                </Button>
                              </PopoverTrigger>
                              <PopoverContent className="w-[300px] p-0">
                                <Command shouldFilter={false}>
                                  <CommandInput 
                                    placeholder="Szukaj koloru..." 
                                    value={colorSearchInputs.get(index) || ""}
                                    onValueChange={(value) => {
                                      const newMap = new Map(colorSearchInputs);
                                      newMap.set(index, value);
                                      setColorSearchInputs(newMap);
                                    }}
                                  />
                                  <CommandList>
                                    <CommandEmpty>Nie znaleziono koloru</CommandEmpty>
                                    <CommandGroup>
                                      {sortedColorOptions
                                        .filter((opt) => {
                                          const searchTerm = (colorSearchInputs.get(index) || "").toLowerCase();
                                          if (!searchTerm) return true;
                                          // Search in name, readableName, and code
                                          const name = (opt.name || "").toLowerCase();
                                          const readableName = (opt.readableName || "").toLowerCase();
                                          const code = (opt.code || "").toLowerCase();
                                          return name.includes(searchTerm) || 
                                                 readableName.includes(searchTerm) || 
                                                 code.includes(searchTerm);
                                        })
                                        .map((opt) => (
                                          <CommandItem
                                            key={opt.id}
                                            value={opt.name || opt.code}
                                            onSelect={() => {
                                              const newColors = [...colors];
                                              newColors[index] = opt.name || opt.code;
                                              setColors(newColors);
                                              // Close the popover after selection
                                              const newSet = new Set(colorOpenRows);
                                              newSet.delete(index);
                                              setColorOpenRows(newSet);
                                              // Clear search input
                                              const newMap = new Map(colorSearchInputs);
                                              newMap.delete(index);
                                              setColorSearchInputs(newMap);
                                            }}
                                          >
                                            <Check
                                              className={`mr-2 h-4 w-4 ${color === (opt.name || opt.code) ? "opacity-100" : "opacity-0"}`}
                                            />
                                            {opt.name || opt.code}
                                          </CommandItem>
                                        ))}
                                    </CommandGroup>
                                  </CommandList>
                                </Command>
                              </PopoverContent>
                            </Popover>
                          </TableCell>
                          <TableCell className="p-0">
                            <div className="flex items-center gap-1.5 flex-wrap pt-1 px-2">
                              {/* Miniaturki zdjęć */}
                              {(() => {
                                const images = colorImages[index.toString()] || [];
                                const isExpanded = expandedColorImages.has(index);
                                const visibleImages = isExpanded ? images : images.slice(0, 2);
                                const remainingCount = images.length - 2;
                                
                                return (
                                  <>
                                    {visibleImages.map((imageUrl, imgIndex) => {
                                      const actualIndex = isExpanded ? imgIndex : imgIndex;
                                      return (
                                        <div 
                                          key={imgIndex} 
                                          className="relative group cursor-move flex-shrink-0 w-12 h-12 rounded border overflow-hidden"
                                          draggable={!isNew}
                                          onDragStart={() => setDraggedImage({ colorIndex: index, index: actualIndex })}
                                          onDragOver={(e) => {
                                            e.preventDefault();
                                            e.currentTarget.classList.add('ring-2', 'ring-primary');
                                          }}
                                          onDragLeave={(e) => {
                                            e.currentTarget.classList.remove('ring-2', 'ring-primary');
                                          }}
                                          onDrop={(e) => {
                                            e.preventDefault();
                                            e.currentTarget.classList.remove('ring-2', 'ring-primary');
                                            // Only reorder if dragging within the same color
                                            if (draggedImage && draggedImage.colorIndex === index) {
                                              handleImageReorder(index, draggedImage.index, actualIndex);
                                            }
                                            // Always clear dragged state
                                            setDraggedImage(null);
                                          }}
                                          onDragEnd={() => {
                                            // Cleanup state and visual feedback when drag ends
                                            setDraggedImage(null);
                                            // Remove any lingering ring styles
                                            document.querySelectorAll('.ring-2.ring-primary').forEach(el => {
                                              el.classList.remove('ring-2', 'ring-primary');
                                            });
                                          }}
                                        >
                                          <img
                                            src={imageUrl}
                                            alt={`${color} ${actualIndex + 1}`}
                                            className="w-full h-full object-cover"
                                          />
                                          
                                          {/* Hover overlay z numerem zdjęcia */}
                                          <div className="absolute inset-0 bg-black/60 rounded opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center pointer-events-none">
                                            <span className="text-white text-2xl font-bold">
                                              {actualIndex + 1}
                                            </span>
                                          </div>
                                          
                                          {actualIndex === 0 && (
                                            <span className="absolute top-1 left-1 bg-primary text-primary-foreground text-xs px-1.5 py-0.5 rounded-sm font-medium z-10">
                                              1
                                            </span>
                                          )}
                                          <Button
                                            variant="destructive"
                                            size="icon"
                                            className="absolute bottom-6 left-1 w-5 h-5 opacity-0 group-hover:opacity-100 transition-opacity z-10"
                                            onClick={() => handleColorImageDelete(index, imageUrl)}
                                            data-testid={`button-delete-image-${index}-${actualIndex}`}
                                          >
                                            <X className="w-3 h-3" />
                                          </Button>
                                        </div>
                                      );
                                    })}
                                    
                                    {/* Show "more" button if there are more than 2 images */}
                                    {!isExpanded && remainingCount > 0 && (
                                      <Button
                                        variant="outline"
                                        size="sm"
                                        className="h-12 px-2 text-xs"
                                        onClick={() => {
                                          setExpandedColorImages(prev => {
                                            const newSet = new Set(prev);
                                            newSet.add(index);
                                            return newSet;
                                          });
                                        }}
                                        data-testid={`button-show-more-images-${index}`}
                                      >
                                        +{remainingCount}
                                      </Button>
                                    )}
                                    
                                    {/* Show "less" button if expanded */}
                                    {isExpanded && images.length > 2 && (
                                      <Button
                                        variant="outline"
                                        size="sm"
                                        className="h-12 px-2 text-xs"
                                        onClick={() => {
                                          setExpandedColorImages(prev => {
                                            const newSet = new Set(prev);
                                            newSet.delete(index);
                                            return newSet;
                                          });
                                        }}
                                        data-testid={`button-show-less-images-${index}`}
                                      >
                                        <Eye className="w-3 h-3" />
                                      </Button>
                                    )}
                                    
                                    {/* Drag & Drop Upload */}
                                    {!isNew && (
                                      <label
                                        className="w-36 h-12 border-2 border-dashed rounded flex items-center justify-center cursor-pointer hover:border-primary hover:bg-accent/50 transition-colors flex-shrink-0"
                                        data-testid={`label-upload-${index}`}
                                        onDragOver={(e) => {
                                          e.preventDefault();
                                          e.currentTarget.classList.add('border-primary', 'bg-accent');
                                        }}
                                        onDragLeave={(e) => {
                                          e.preventDefault();
                                          e.currentTarget.classList.remove('border-primary', 'bg-accent');
                                        }}
                                        onDrop={(e) => {
                                          e.preventDefault();
                                          e.currentTarget.classList.remove('border-primary', 'bg-accent');
                                          handleColorImageUpload(index, e.dataTransfer.files);
                                        }}
                                      >
                                        <Upload className="w-4 h-4 text-muted-foreground" />
                                        <input
                                          type="file"
                                          accept="image/*"
                                          multiple
                                          className="hidden"
                                          onChange={(e) => handleColorImageUpload(index, e.target.files)}
                                          data-testid={`input-upload-${index}`}
                                        />
                                      </label>
                                    )}
                                    {isNew && (
                                      <span className="text-xs text-muted-foreground">
                                        (Zapisz najpierw)
                                      </span>
                                    )}
                                  </>
                                );
                              })()}
                            </div>
                          </TableCell>
                          <TableCell className="py-1 px-1 text-center">
                            <Checkbox
                              className="h-3.5 w-3.5"
                              checked={colorsForGeneration.includes(index)}
                              onCheckedChange={() => toggleColorForGeneration(index)}
                              data-testid={`checkbox-generate-${index}`}
                            />
                          </TableCell>
                          <TableCell className="py-1 px-1">
                            {matrix?.colorLastGenerated?.[color] ? (
                              <div className="flex flex-col text-xs text-muted-foreground leading-tight">
                                <span>
                                  {new Date(matrix.colorLastGenerated[color]).toLocaleDateString('pl-PL', {
                                    year: 'numeric',
                                    month: '2-digit',
                                    day: '2-digit'
                                  })}
                                </span>
                                <span className="text-[10px]">
                                  {new Date(matrix.colorLastGenerated[color]).toLocaleTimeString('pl-PL', {
                                    hour: '2-digit',
                                    minute: '2-digit'
                                  })}
                                </span>
                              </div>
                            ) : (
                              <span className="text-xs text-muted-foreground italic">Nigdy</span>
                            )}
                          </TableCell>
                          <TableCell className="py-1 px-2">
                            <div className="flex flex-wrap gap-1">
                              {(colorOptionsMatrix[index.toString()] || []).map((option, optIdx) => (
                                <Badge 
                                  key={optIdx} 
                                  variant="secondary" 
                                  className="text-xs"
                                  data-testid={`badge-color-option-${index}-${optIdx}`}
                                >
                                  {option}
                                  <button
                                    className="ml-1 hover:text-destructive"
                                    onClick={() => {
                                      const newOptions = [...(colorOptionsMatrix[index.toString()] || [])];
                                      newOptions.splice(optIdx, 1);
                                      setColorOptionsMatrix({
                                        ...colorOptionsMatrix,
                                        [index.toString()]: newOptions
                                      });
                                    }}
                                    data-testid={`button-remove-option-${index}-${optIdx}`}
                                  >
                                    <X className="w-3 h-3" />
                                  </button>
                                </Badge>
                              ))}
                              <Popover
                                open={colorOptionOpenRows.has(index)}
                                onOpenChange={(open) => {
                                  if (open) {
                                    setColorOptionOpenRows(prev => {
                                      const newSet = new Set(prev);
                                      newSet.add(index);
                                      return newSet;
                                    });
                                  } else {
                                    setColorOptionSearchInputs(prev => {
                                      const newMap = new Map(prev);
                                      newMap.delete(index);
                                      return newMap;
                                    });
                                    setColorOptionOpenRows(prev => {
                                      const newSet = new Set(prev);
                                      newSet.delete(index);
                                      return newSet;
                                    });
                                  }
                                }}
                              >
                                <PopoverTrigger asChild>
                                  <Button
                                    variant="outline"
                                    size="icon"
                                    className="h-6 w-6"
                                    data-testid={`button-add-color-option-${index}`}
                                  >
                                    <Plus className="w-3 h-3" />
                                  </Button>
                                </PopoverTrigger>
                                <PopoverContent className="w-80 p-0" align="start">
                                  <Command shouldFilter={false}>
                                    <CommandInput 
                                      placeholder="Szukaj opcji (użyj ; dla AND)..." 
                                      value={colorOptionSearchInputs.get(index) || ""}
                                      onValueChange={(value) => {
                                        const newMap = new Map(colorOptionSearchInputs);
                                        newMap.set(index, value);
                                        setColorOptionSearchInputs(newMap);
                                      }}
                                    />
                                    <CommandEmpty>Brak wyników</CommandEmpty>
                                    <CommandList>
                                      <CommandGroup heading="Dostępne opcje COMPONENT-COLOR">
                                        {availableColorOptions
                                          .filter(opt => {
                                            // Already selected
                                            if ((colorOptionsMatrix[index.toString()] || []).includes(opt.value)) return false;
                                            
                                            // Semicolon search: "bia;drzw" means contains "bia" AND "drzw"
                                            const searchTerm = (colorOptionSearchInputs.get(index) || "").toLowerCase();
                                            if (!searchTerm) return true;
                                            
                                            const tokens = searchTerm.split(';').map(t => t.trim()).filter(t => t);
                                            if (tokens.length === 0) return true;
                                            
                                            const optValue = opt.value.toLowerCase();
                                            return tokens.every(token => optValue.includes(token));
                                          })
                                          .map((opt) => (
                                            <CommandItem
                                              key={opt.value}
                                              value={opt.value}
                                              onSelect={(selectedValue) => {
                                                const newOptions = [...(colorOptionsMatrix[index.toString()] || []), selectedValue];
                                                setColorOptionsMatrix({
                                                  ...colorOptionsMatrix,
                                                  [index.toString()]: newOptions
                                                });
                                                
                                                // Close popover after selection
                                                const newSet = new Set(colorOptionOpenRows);
                                                newSet.delete(index);
                                                setColorOptionOpenRows(newSet);
                                                
                                                // Clear search input
                                                const newMap = new Map(colorOptionSearchInputs);
                                                newMap.delete(index);
                                                setColorOptionSearchInputs(newMap);
                                              }}
                                            >
                                              {opt.label}
                                            </CommandItem>
                                          ))}
                                        {availableColorOptions.length === 0 && (
                                          <CommandItem disabled>
                                            <span className="text-muted-foreground text-xs italic">
                                              Ładowanie opcji...
                                            </span>
                                          </CommandItem>
                                        )}
                                      </CommandGroup>
                                    </CommandList>
                                  </Command>
                                </PopoverContent>
                              </Popover>
                            </div>
                          </TableCell>
                          <TableCell className="text-right py-1 px-2">
                            <div className="flex items-center justify-end gap-1">
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => duplicateColor(index)}
                                data-testid={`button-copy-color-${index}`}
                                title="Duplikuj kolor (z tą samą nazwą)"
                              >
                                <Copy className="w-4 h-4" />
                              </Button>
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() => removeColor(index)}
                                data-testid={`button-remove-color-${index}`}
                              >
                                <X className="w-4 h-4" />
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                        );
                      })
                    )}
                  </TableBody>
                </Table>
              </div>
              
              {/* Drugi przycisk "Dodaj kolor" na dole tabeli */}
              <div className="flex items-center gap-2 flex-wrap">
                <Button size="sm" onClick={addColor} data-testid="button-add-color-bottom">
                  <Plus className="w-4 h-4 mr-2" />
                  <span>Dodaj kolor</span>
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={loadAllColors}
                  data-testid="button-load-all-colors-bottom"
                >
                  <span className="hidden md:inline">Załaduj wszystkie ze słownika</span>
                  <span className="md:hidden">Załaduj wszystkie</span>
                </Button>
              </div>
              
              <p className="text-xs text-muted-foreground">
                Produkty zostaną wygenerowane dla każdej kombinacji: rozmiar × kolor. Przeciągnij i upuść zdjęcia lub kliknij ikonę + aby dodać.
              </p>
            </CardContent>
          </Card>
        </TabsContent>

        {/* OPIS TAB */}
        <TabsContent value="opis" className="space-y-4">
          {/* TEMPLATE SYSTEM CARD */}
          <Card>
            <CardHeader>
              <CardTitle>System szablonów</CardTitle>
              <CardDescription>
                Zaawansowany system opisów z szablonami, AI i akcesoriami
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="template-select">Szablon opisu</Label>
                <div className="flex gap-2">
                  <Select
                    value={templateId?.toString() || "none"}
                    onValueChange={(value) => setTemplateId(value === "none" ? null : parseInt(value))}
                  >
                    <SelectTrigger id="template-select" data-testid="select-template" className="flex-1">
                      <SelectValue placeholder="Wybierz szablon..." />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="none">Brak (użyj legacy)</SelectItem>
                      {templates?.map((template) => (
                        <SelectItem key={template.id} value={template.id.toString()}>
                          {template.name}
                          {template.categoryName && ` (${template.categoryName})`}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {templateId && (
                    <Button
                      variant="outline"
                      size="default"
                      onClick={() => navigate(`/template-editor/${templateId}`)}
                      data-testid="button-edit-template"
                      className="hover-elevate active-elevate-2"
                    >
                      <Eye className="h-4 w-4 mr-2" />
                      Podgląd/Edycja
                    </Button>
                  )}
                </div>
                <p className="text-xs text-muted-foreground">
                  Wybierz szablon, który będzie używany do generowania opisów produktów. Szablony obsługują zmienne, akcesoria i AI.
                </p>
              </div>

              {templateId && (
                <>
                  <div className="rounded-lg bg-muted p-4">
                    <p className="text-sm font-medium mb-2">Konfiguracja AI dla szablonu</p>
                    <p className="text-xs text-muted-foreground mb-4">
                      Skonfiguruj różne sekcje opisu AI. Każda sekcja może mieć własny prompt i parametry.
                    </p>
                    
                    <div className="space-y-3">
                      {['intro', 'features', 'safety', 'care', 'warranty'].map((section) => (
                        <div key={section} className="border rounded-lg p-3 bg-background">
                          <div className="flex items-center justify-between mb-2">
                            <div className="flex items-center space-x-2">
                              <Checkbox
                                id={`ai-section-${section}`}
                                checked={aiConfig[section]?.enabled || false}
                                onCheckedChange={(checked) => {
                                  setAiConfig(prev => ({
                                    ...prev,
                                    [section]: {
                                      ...prev[section],
                                      enabled: !!checked,
                                      prompt: prev[section]?.prompt || getDefaultAiPrompt(section),
                                    }
                                  }));
                                }}
                                data-testid={`checkbox-ai-${section}`}
                              />
                              <Label htmlFor={`ai-section-${section}`} className="font-medium capitalize">
                                {section === 'intro' && 'Wprowadzenie'}
                                {section === 'features' && 'Cechy'}
                                {section === 'safety' && 'Bezpieczeństwo'}
                                {section === 'care' && 'Pielęgnacja'}
                                {section === 'warranty' && 'Gwarancja'}
                              </Label>
                            </div>
                            <Badge variant="secondary" className="text-xs">
                              {`{{ai.${section}}}`}
                            </Badge>
                          </div>
                          
                          {aiConfig[section]?.enabled && (
                            <div className="mt-2 space-y-2">
                              <Input
                                placeholder={`Prompt dla sekcji "${section}"...`}
                                value={aiConfig[section]?.prompt || ''}
                                onChange={(e) => {
                                  setAiConfig(prev => ({
                                    ...prev,
                                    [section]: {
                                      ...prev[section],
                                      enabled: true,
                                      prompt: e.target.value,
                                    }
                                  }));
                                }}
                                data-testid={`input-ai-prompt-${section}`}
                              />
                              <p className="text-xs text-muted-foreground">
                                Zmienne: {'{length}'}, {'{width}'}, {'{height}'}, {'{color}'}, {'{productType}'}, {'{name}'}
                              </p>
                            </div>
                          )}
                        </div>
                      ))}
                    </div>
                  </div>

                  {!isNew && generatedProducts.length > 0 && (
                    <div className="flex items-center justify-between p-4 border rounded-lg bg-blue-50 dark:bg-blue-950">
                      <div className="flex-1">
                        <p className="font-medium text-sm">Regeneruj opisy produktów</p>
                        <p className="text-xs text-muted-foreground">
                          Zaktualizuj opisy wszystkich {generatedProducts.length} produktów używając aktualnego szablonu i akcesoriów
                        </p>
                      </div>
                      <Button
                        type="button"
                        variant="default"
                        onClick={async () => {
                          try {
                            // Generate unique session ID for log streaming
                            const regenerateSessionId = `regenerate-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
                            setSessionId(regenerateSessionId);
                            setGenerationLogs([]); // Clear previous logs
                            setIsGenerating(true);
                            
                            toast({
                              title: "Regeneracja rozpoczęta",
                              description: `Regeneruję opisy dla ${generatedProducts.length} produktów...`,
                            });
                            
                            const result = await apiRequest("POST", `/api/product-matrices/${id}/regenerate-descriptions`, { sessionId: regenerateSessionId });
                            const data = await result.json();
                            
                            queryClient.invalidateQueries({ queryKey: [`/api/product-matrices/${id}/products`] });
                            
                            toast({
                              title: "Regeneracja zakończona",
                              description: `Zaktualizowano ${data.productsUpdated}/${data.totalProducts} produktów`,
                            });
                          } catch (error) {
                            toast({
                              title: "Błąd regeneracji",
                              description: "Nie udało się zregenerować opisów",
                              variant: "destructive",
                            });
                          } finally {
                            setIsGenerating(false);
                          }
                        }}
                        disabled={isGenerating}
                        data-testid="button-regenerate-descriptions"
                      >
                        {isGenerating ? (
                          <>
                            <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                            Regeneruję...
                          </>
                        ) : (
                          <>
                            <Sparkles className="w-4 h-4 mr-2" />
                            Regeneruj opisy
                          </>
                        )}
                      </Button>
                    </div>
                  )}
                </>
              )}
            </CardContent>
          </Card>

          {/* LEGACY DESCRIPTION CARD - only show if no template is selected */}
          {!templateId && (
            <Card>
              <CardHeader>
                <CardTitle>Opis produktu (Legacy)</CardTitle>
                <CardDescription>
                  Szablon opisu dla wygenerowanych produktów (możesz używać zmiennych)
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
              <div className="space-y-2">
                <div className="flex items-center justify-between">
                  <Label>Opis (Rich Text Editor)</Label>
                  <Button
                    type="button"
                    className="bg-green-600 hover:bg-green-700 text-white font-semibold shadow-lg"
                    size="default"
                    onClick={async () => {
                      try {
                        setIsGenerating(true);
                        
                        // Prepare sample product data
                        const sampleLength = lengths[0] || "100";
                        const sampleWidth = widths[0] || "50";
                        const sampleHeight = heights[0] || "200";
                        const sampleColor = colors[0] || "Dąb Sonoma";
                        
                        const sampleProductData = {
                          name: name || "Przykładowy produkt",
                          title: `${namePrefix || ""} ${productType || ""} ${sampleLength}x${sampleWidth}x${sampleHeight} ${sampleColor} ${nameSuffix || ""}`.trim(),
                          color: sampleColor,
                          material: material,
                          length: sampleLength,
                          width: sampleWidth,
                          height: sampleHeight,
                          productType: productType,
                          doors: doors,
                          legs: legs,
                        };
                        
                        const rawResponse = await apiRequest("POST", "/api/ai/generate-description", {
                          productData: sampleProductData
                        });
                        
                        const response = await rawResponse.json() as AIGeneratedDescriptionResponse;
                        
                        if (response.descriptionHtml) {
                          setDescriptionHtml(response.descriptionHtml);
                          // Note: descriptionDoc will be updated by the editor
                          toast({
                            title: "Opis wygenerowany",
                            description: `Użyto ${response.tokens.total} tokenów (koszt: ${response.cost.toFixed(4)} PLN)`,
                          });
                        }
                      } catch (error) {
                        console.error("Error generating description:", error);
                        toast({
                          title: "Błąd generowania",
                          description: "Nie udało się wygenerować opisu AI",
                          variant: "destructive",
                        });
                      } finally {
                        setIsGenerating(false);
                      }
                    }}
                    disabled={isGenerating}
                    data-testid="button-generate-ai-description"
                  >
                    {isGenerating ? (
                      <>
                        <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                        Generowanie...
                      </>
                    ) : (
                      <>
                        <Sparkles className="w-4 h-4 mr-2" />
                        Generuj opis AI
                      </>
                    )}
                  </Button>
                </div>
                <RichTextEditor
                  content={descriptionHtml || ''}
                  onContentChange={(html: string, json: any) => {
                    setDescriptionHtml(html);
                    setDescriptionDoc(json);
                  }}
                  placeholder="Wprowadź szablon opisu produktu..."
                />
                <p className="text-xs text-muted-foreground">
                  <Sparkles className="w-3 h-3 inline mr-1" />
                  Użyj przycisku "Generuj opis AI" aby wygenerować przykładowy opis na podstawie pierwszego wariantu produktu
                </p>
              </div>

              <div className="rounded-lg bg-muted p-4">
                <p className="text-sm font-medium mb-2">Dostępne zmienne:</p>
                <div className="grid grid-cols-2 md:grid-cols-4 gap-2 text-xs">
                  <code className="bg-background px-2 py-1 rounded">{"{name}"}</code>
                  <code className="bg-background px-2 py-1 rounded">{"{productType}"}</code>
                  <code className="bg-background px-2 py-1 rounded">{"{color}"}</code>
                  <code className="bg-background px-2 py-1 rounded">{"{size}"}</code>
                  <code className="bg-background px-2 py-1 rounded">{"{length}"}</code>
                  <code className="bg-background px-2 py-1 rounded">{"{width}"}</code>
                  <code className="bg-background px-2 py-1 rounded">{"{height}"}</code>
                  <code className="bg-background px-2 py-1 rounded">{"{productGroup}"}</code>
                  <code className="bg-background px-2 py-1 rounded">{"{doors}"}</code>
                  <code className="bg-background px-2 py-1 rounded">{"{legs}"}</code>
                </div>
              </div>

              <div className="flex items-center space-x-2 p-4 border rounded-lg">
                <Switch
                  checked={useAiGeneration}
                  onCheckedChange={setUseAiGeneration}
                  id="useAi"
                  data-testid="switch-use-ai"
                />
                <div className="flex-1">
                  <Label htmlFor="useAi" className="font-medium">
                    Użyj AI do generowania opisów
                  </Label>
                  <p className="text-xs text-muted-foreground">
                    AI wygeneruje unikalne opisy dla każdego wariantu produktu
                  </p>
                </div>
              </div>

              {useAiGeneration && (
                <div className="space-y-2">
                  <Label htmlFor="aiPrompt">
                    <Sparkles className="w-4 h-4 inline mr-1" />
                    Własny prompt AI (opcjonalny)
                  </Label>
                  <Input
                    id="aiPrompt"
                    value={aiPrompt}
                    onChange={(e) => setAiPrompt(e.target.value)}
                    placeholder="np. Napisz krótki, marketingowy opis podkreślający jakość wykonania..."
                    data-testid="input-ai-prompt"
                  />
                  <p className="text-xs text-muted-foreground">
                    Możesz używać tych samych zmiennych co w opisie. Pozostaw puste aby użyć globalnych ustawień AI.
                  </p>
                </div>
              )}
            </CardContent>
          </Card>
          )}
        </TabsContent>

        {/* PODSUMOWANIE TAB */}
        <TabsContent value="podsumowanie" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Podsumowanie generowania</CardTitle>
              <CardDescription>
                Podgląd produktów które zostaną wygenerowane
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              {/* Statystyki */}
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div className="rounded-lg border p-4">
                  <div className="text-sm text-muted-foreground">Rozmiarów</div>
                  <div className="text-2xl font-bold">{
                    Math.max(
                      lengths.length,
                      widths.length,
                      heights.length,
                      1
                    )
                  }</div>
                </div>
                <div className="rounded-lg border p-4">
                  <div className="text-sm text-muted-foreground">Kolorów</div>
                  <div className="text-2xl font-bold">{colors.length || 1}</div>
                </div>
                <div className="rounded-lg border p-4">
                  <div className="text-sm text-muted-foreground">Razem produktów</div>
                  <div className="text-2xl font-bold text-primary">{
                    Math.max(
                      lengths.length,
                      widths.length,
                      heights.length,
                      1
                    ) * (colors.length || 1)
                  }</div>
                </div>
              </div>

              {/* Przycisk generowania */}
              <div className="flex justify-center gap-4">
                <Button
                  size="lg"
                  variant="outline"
                  onClick={handlePreviewGeneration}
                  disabled={isNew || isLoadingPreview || !templateId}
                  data-testid="button-preview-generation"
                  className="gap-2"
                >
                  {isLoadingPreview ? (
                    <>
                      <Loader2 className="w-4 h-4 animate-spin" />
                      Ładowanie...
                    </>
                  ) : (
                    <>
                      <Eye className="w-4 h-4" />
                      Podgląd Generowania
                    </>
                  )}
                </Button>
                <Button
                  size="lg"
                  onClick={handleGenerate}
                  disabled={isNew || isGenerating || generationProgress.isGenerating}
                  data-testid="button-generate-products"
                  className="gap-2"
                >
                  {generationProgress.isGenerating ? (
                    <>
                      <Loader2 className="w-4 h-4 animate-spin" />
                      {generationProgress.currentStep}
                    </>
                  ) : (
                    <>
                      <Play className="w-4 h-4" />
                      Generuj produkty
                    </>
                  )}
                </Button>
              </div>

              {/* Podgląd produktów */}
              <div className="border rounded-lg overflow-hidden">
                {/* Mobile Card View */}
                <div className="md:hidden p-4 space-y-3">
                  {(() => {
                    const maxDimensions = Math.max(lengths.length, widths.length, heights.length, 1);
                    const colorsToUse = colors.length > 0 ? colors : [""];
                    const basePrice = parseFloat(defaultPrice) || 0;
                    let productIndex = 0;
                    const cards = [];
                    const productsByComboKey = new Map(generatedProducts.map(p => [p.combination_key, p]));

                    for (let sizeIndex = 0; sizeIndex < maxDimensions; sizeIndex++) {
                      const length = lengths[sizeIndex] || "";
                      const width = widths[sizeIndex] || "";
                      const height = heights[sizeIndex] || "";
                      const productGroup = productGroupsArray[sizeIndex] || "";
                      const modifier = parseFloat(priceModifiersArray[sizeIndex] || "0");
                      const modifierType = priceModifierTypesArray[sizeIndex] || "fixed";

                      let finalPrice = basePrice;
                      if (basePrice > 0 && modifier !== 0) {
                        if (modifierType === "percent") {
                          finalPrice = basePrice + (basePrice * modifier / 100);
                        } else {
                          finalPrice = basePrice + modifier;
                        }
                      }

                      const formatCm = (mm: string) => {
                        if (!mm) return "";
                        const numMm = parseFloat(mm);
                        if (isNaN(numMm)) return "";
                        const cm = numMm / 10;
                        return cm % 1 === 0 ? cm.toFixed(0) : cm.toFixed(1);
                      };

                      const sizeParts = [length, width, height].filter(Boolean);
                      const sizeStr = sizeParts.join("×") || "-";
                      const sizePartsCm = [length, width, height].map(formatCm).filter(Boolean);
                      const sizeStrCm = sizePartsCm.length > 0 ? `${sizePartsCm.join('×')} cm` : '';

                      for (let colorIndex = 0; colorIndex < colorsToUse.length; colorIndex++) {
                        const color = colorsToUse[colorIndex];
                        productIndex++;
                        const nameParts = [namePrefix, productType, sizeStrCm, nameSuffix, color].filter(Boolean);
                        const productName = nameParts.join(" ") || "Produkt bez nazwy";

                        const normalizeValue = (val: string): string => {
                          if (!val || val.trim() === '') return 'null';
                          const num = parseFloat(val.trim());
                          return isNaN(num) ? 'null' : num.toString();
                        };

                        const comboParts = [normalizeValue(length), normalizeValue(width), normalizeValue(height)];
                        const combinationKey = `${comboParts.join('×')}:${color || 'no-color'}:${colorIndex}`;
                        const existingProduct = productsByComboKey.get(combinationKey);

                        // Get color options for this color
                        const colorOptionsForColor = colorOptionsMatrix[colorIndex.toString()] || [];

                        cards.push(
                          <Card key={productIndex} className="hover-elevate">
                            <CardContent className="p-3">
                              <div className="flex items-start justify-between mb-2">
                                <div className="text-xs text-muted-foreground font-mono">#{productIndex}</div>
                                {existingProduct && (
                                  <Link 
                                    href={`/catalog-products/${existingProduct.id}`}
                                    className="text-primary hover:underline text-xs"
                                  >
                                    {existingProduct.sku}
                                  </Link>
                                )}
                              </div>
                              <div className="font-medium text-sm mb-2">{productName}</div>
                              <div className="flex flex-wrap gap-1.5 mb-2">
                                {color && <Badge variant="secondary" className="text-xs">{color}</Badge>}
                                {productGroup && <Badge variant="outline" className="text-xs">{productGroup}</Badge>}
                                {colorOptionsForColor.map((option, idx) => (
                                  <Badge key={idx} variant="secondary" className="text-xs opacity-70">
                                    {option}
                                  </Badge>
                                ))}
                              </div>
                              <div className="text-xs text-muted-foreground mb-2">{sizeStr}</div>
                              <div className="font-medium text-sm">
                                {finalPrice > 0 ? `${finalPrice.toFixed(2)} PLN` : "-"}
                              </div>
                            </CardContent>
                          </Card>
                        );
                      }
                    }

                    return cards.length > 0 ? cards : (
                      <div className="text-center text-muted-foreground py-8">
                        Brak danych do wygenerowania produktów. Dodaj rozmiary i kolory.
                      </div>
                    );
                  })()}
                </div>

                {/* Desktop Table View */}
                <Table className="hidden md:table">
                  <TableHeader>
                    <TableRow>
                      <TableHead className="w-12">#</TableHead>
                      <TableHead>Nazwa produktu</TableHead>
                      <TableHead>Rozmiar (D×S×W)</TableHead>
                      <TableHead>Kolor</TableHead>
                      <TableHead>Opcje</TableHead>
                      <TableHead>Grupa</TableHead>
                      <TableHead className="text-right">Cena (PLN)</TableHead>
                      <TableHead className="w-32">Link do produktu</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {(() => {
                      const maxDimensions = Math.max(
                        lengths.length,
                        widths.length,
                        heights.length,
                        1
                      );
                      const colorsToUse = colors.length > 0 ? colors : [""];
                      const basePrice = parseFloat(defaultPrice) || 0;
                      let productIndex = 0;
                      const previewProducts = [];

                      // Create map of generated products by combination_key
                      const productsByComboKey = new Map(
                        generatedProducts.map(p => [p.combination_key, p])
                      );

                      for (let sizeIndex = 0; sizeIndex < maxDimensions; sizeIndex++) {
                        const length = lengths[sizeIndex] || "";
                        const width = widths[sizeIndex] || "";
                        const height = heights[sizeIndex] || "";
                        const productGroup = productGroupsArray[sizeIndex] || "";
                        const modifier = parseFloat(priceModifiersArray[sizeIndex] || "0");
                        const modifierType = priceModifierTypesArray[sizeIndex] || "fixed";

                        // Calculate price based on modifier type
                        let finalPrice = basePrice;
                        if (basePrice > 0 && modifier !== 0) {
                          if (modifierType === "percent") {
                            finalPrice = basePrice + (basePrice * modifier / 100);
                          } else {
                            finalPrice = basePrice + modifier;
                          }
                        }

                        // Build size string for display in cm (convert from mm to cm)
                        const formatCm = (mm: string) => {
                          if (!mm) return "";
                          const numMm = parseFloat(mm);
                          if (isNaN(numMm)) return "";
                          const cm = numMm / 10;
                          return cm % 1 === 0 ? cm.toFixed(0) : cm.toFixed(1);
                        };
                        
                        const sizeParts = [];
                        if (length) sizeParts.push(length);
                        if (width) sizeParts.push(width);
                        if (height) sizeParts.push(height);
                        const sizeStr = sizeParts.join("×") || "-";
                        
                        // Build size string in cm for product name
                        const sizePartsCm = [];
                        if (length) sizePartsCm.push(formatCm(length));
                        if (width) sizePartsCm.push(formatCm(width));
                        if (height) sizePartsCm.push(formatCm(height));
                        const sizeStrCm = sizePartsCm.length > 0 ? `${sizePartsCm.join('×')} cm` : '';

                        for (let colorIndex = 0; colorIndex < colorsToUse.length; colorIndex++) {
                          const color = colorsToUse[colorIndex];
                          productIndex++;
                          
                          // 🔧 FIX: Get color options BEFORE using them in combinationKey
                          const colorOptionsForColor = colorOptionsMatrix[colorIndex.toString()] || [];
                          
                          // Build product name: [prefix] [type] [size in cm] [suffix] [color]
                          const nameParts = [];
                          if (namePrefix) nameParts.push(namePrefix);
                          if (productType) nameParts.push(productType);
                          if (sizeStrCm) nameParts.push(sizeStrCm);
                          if (nameSuffix) nameParts.push(nameSuffix);
                          if (color) nameParts.push(color);
                          const productName = nameParts.filter(Boolean).join(" ") || "Produkt bez nazwy";

                          // 🔧 FIX: Build combination_key to match EXACT backend format
                          // Backend format: "productType:length×width×height:color" (or with :options)
                          const normalizeValue = (val: string): string => {
                            if (!val || val.trim() === '') return 'null';
                            const num = parseFloat(val.trim());
                            return isNaN(num) ? 'null' : num.toString();
                          };
                          
                          const comboParts = [
                            normalizeValue(length),
                            normalizeValue(width),
                            normalizeValue(height),
                          ];
                          
                          // Match backend: productType:dimensions:color[:options]
                          let combinationKey = `${productType || 'NO_TYPE'}:${comboParts.join('×')}:${color || 'no-color'}`;
                          
                          // Add color options if they exist (sorted for consistency)
                          if (colorOptionsForColor.length > 0) {
                            const sortedOptions = [...colorOptionsForColor].sort().join(',');
                            combinationKey += `:${sortedOptions}`;
                          }
                          
                          // Find existing product
                          const existingProduct = productsByComboKey.get(combinationKey);

                          previewProducts.push(
                            <TableRow key={productIndex}>
                              <TableCell className="font-mono text-muted-foreground">
                                {productIndex}
                              </TableCell>
                              <TableCell className="font-medium">{productName}</TableCell>
                              <TableCell>{sizeStr}</TableCell>
                              <TableCell>{color || "-"}</TableCell>
                              <TableCell>
                                {colorOptionsForColor.length > 0 ? (
                                  <div className="flex flex-wrap gap-1">
                                    {colorOptionsForColor.map((option, idx) => (
                                      <Badge key={idx} variant="secondary" className="text-xs">
                                        {option}
                                      </Badge>
                                    ))}
                                  </div>
                                ) : (
                                  <span className="text-muted-foreground">-</span>
                                )}
                              </TableCell>
                              <TableCell>{productGroup || "-"}</TableCell>
                              <TableCell className="text-right font-mono">
                                {finalPrice > 0 ? finalPrice.toFixed(2) : "-"}
                              </TableCell>
                              <TableCell>
                                {existingProduct ? (
                                  <Link 
                                    href={`/catalog-products/${existingProduct.id}`}
                                    className="text-primary hover:underline text-sm"
                                    data-testid={`link-product-${existingProduct.id}`}
                                  >
                                    {existingProduct.sku}
                                  </Link>
                                ) : (
                                  <span className="text-muted-foreground text-sm">-</span>
                                )}
                              </TableCell>
                            </TableRow>
                          );
                        }
                      }

                      return previewProducts.length > 0 ? previewProducts : (
                        <TableRow>
                          <TableCell colSpan={8} className="text-center text-muted-foreground">
                            Brak danych do wygenerowania produktów. Dodaj rozmiary i kolory.
                          </TableCell>
                        </TableRow>
                      );
                    })()}
                  </TableBody>
                </Table>
              </div>

              <div className="rounded-lg bg-muted p-4 text-sm">
                <p className="font-medium mb-2">Informacje:</p>
                <ul className="space-y-1 text-muted-foreground">
                  <li>• Produkty zostaną wygenerowane po kliknięciu "Generuj produkty"</li>
                  <li>• Każdy produkt będzie miał unikalny SKU i można go edytować indywidualnie</li>
                  {useAiGeneration && (
                    <li>• AI wygeneruje unikalne opisy dla każdego wariantu</li>
                  )}
                </ul>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      {/* Add New Product Group Dialog */}
      <Dialog open={isNewGroupDialogOpen} onOpenChange={setIsNewGroupDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Dodaj nową grupę produktu</DialogTitle>
            <DialogDescription>
              Utwórz nową grupę produktu, która będzie dostępna w słowniku.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="new-group-code">Kod *</Label>
              <Input
                id="new-group-code"
                value={newGroupCode}
                onChange={(e) => setNewGroupCode(e.target.value)}
                placeholder="np. CABINET"
                data-testid="input-new-group-code"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="new-group-name">Nazwa *</Label>
              <Input
                id="new-group-name"
                value={newGroupName}
                onChange={(e) => setNewGroupName(e.target.value)}
                placeholder="np. Szafka"
                data-testid="input-new-group-name"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="new-group-readable">Czytelna nazwa (opcjonalna)</Label>
              <Input
                id="new-group-readable"
                value={newGroupReadableName}
                onChange={(e) => setNewGroupReadableName(e.target.value)}
                placeholder="np. Szafka łazienkowa"
                data-testid="input-new-group-readable"
              />
            </div>
          </div>
          <div className="flex justify-end gap-2">
            <Button
              variant="outline"
              onClick={() => {
                setIsNewGroupDialogOpen(false);
                setNewGroupCode("");
                setNewGroupName("");
                setNewGroupReadableName("");
              }}
              data-testid="button-cancel-new-group"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => {
                if (!newGroupCode.trim() || !newGroupName.trim()) {
                  toast({
                    title: "Błąd",
                    description: "Kod i nazwa są wymagane.",
                    variant: "destructive",
                  });
                  return;
                }
                createGroupMutation.mutate({
                  code: newGroupCode.trim(),
                  name: newGroupName.trim(),
                  readableName: newGroupReadableName.trim() || null,
                });
              }}
              disabled={createGroupMutation.isPending}
              data-testid="button-save-new-group"
            >
              {createGroupMutation.isPending ? (
                <>
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  Zapisywanie...
                </>
              ) : (
                "Zapisz"
              )}
            </Button>
          </div>
        </DialogContent>
      </Dialog>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Czy na pewno chcesz usunąć tę matrycę?</AlertDialogTitle>
            <AlertDialogDescription>
              Ta akcja nie może być cofnięta. Spowoduje to trwałe usunięcie matryce
              z bazy danych.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDelete}
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              data-testid="button-confirm-delete"
            >
              Usuń
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      {/* Preview Generation Dialog */}
      <Dialog open={isPreviewOpen} onOpenChange={setIsPreviewOpen}>
        <DialogContent className="max-w-5xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Podgląd generowania produktu</DialogTitle>
            <DialogDescription>
              Szczegółowy raport generowania pierwszego wariantu produktu
            </DialogDescription>
          </DialogHeader>
          {previewData && (
            <div className="space-y-6">
              {/* Product Info */}
              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Dane produktu</CardTitle>
                </CardHeader>
                <CardContent className="space-y-2 text-sm">
                  <div className="grid grid-cols-2 gap-2">
                    <div><span className="font-medium">Nazwa:</span> {previewData.preview.productName}</div>
                    <div><span className="font-medium">Wymiary:</span> {previewData.preview.dimensions.sizeStr || 'Brak'}</div>
                    <div><span className="font-medium">Kolor:</span> {previewData.preview.color || 'Brak'}</div>
                    <div><span className="font-medium">Cena:</span> {previewData.preview.price ? `${previewData.preview.price} PLN` : 'Brak'}</div>
                    {previewData.preview.productGroup && (
                      <div><span className="font-medium">Grupa:</span> {previewData.preview.productGroup}</div>
                    )}
                    {previewData.preview.priceModifier && (
                      <div><span className="font-medium">Modyfikator ceny:</span> {previewData.preview.priceModifier.value} ({previewData.preview.priceModifier.type})</div>
                    )}
                  </div>
                </CardContent>
              </Card>

              {/* AI Prompts */}
              {previewData.aiPrompts && Object.keys(previewData.aiPrompts).length > 0 && (
                <Card>
                  <CardHeader>
                    <CardTitle className="text-base">Konfiguracja AI</CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-3">
                    {Object.entries(previewData.aiPrompts).map(([section, config]: [string, any]) => (
                      config.enabled && (
                        <div key={section} className="border-l-2 border-primary pl-3 space-y-1">
                          <div className="font-medium text-sm capitalize">{section}</div>
                          <div className="text-xs text-muted-foreground">
                            {config.prompt.substring(0, 150)}...
                          </div>
                          <div className="text-xs text-muted-foreground">
                            Ton: {config.tone || 'professional'} | Język: {config.language || 'pl'} | Max: {config.maxLength || 500} znaków
                          </div>
                        </div>
                      )
                    ))}
                  </CardContent>
                </Card>
              )}

              {/* Accessories */}
              {previewData.accessories && previewData.accessories.length > 0 && (
                <Card>
                  <CardHeader>
                    <CardTitle className="text-base">Akcesoria ({previewData.accessories.length})</CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    {previewData.accessories.map((group: any, idx: number) => (
                      <div key={idx} className="space-y-2">
                        <div className="font-medium text-sm">
                          {group.groupName} ({group.groupCode}) - {group.itemCount} pozycji
                        </div>
                        <div className="grid grid-cols-2 md:grid-cols-3 gap-2 text-xs">
                          {group.items.map((item: any, itemIdx: number) => (
                            <div key={itemIdx} className="border rounded p-2 space-y-1">
                              <div className="font-medium">{item.name}</div>
                              {item.code && <div className="text-muted-foreground">({item.code})</div>}
                              {item.description && <div className="text-muted-foreground">{item.description}</div>}
                              <div className="text-muted-foreground">{item.hasImage ? '🖼️ Ze zdjęciem' : '📄 Bez zdjęcia'}</div>
                            </div>
                          ))}
                        </div>
                      </div>
                    ))}
                  </CardContent>
                </Card>
              )}

              {/* Images */}
              {previewData.images && previewData.images.length > 0 && (
                <Card>
                  <CardHeader>
                    <CardTitle className="text-base">Zdjęcia produktu ({previewData.images.length})</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                      {previewData.images.map((img: any, idx: number) => (
                        <div key={idx} className="space-y-2">
                          <img src={img.url} alt={`Zdjęcie ${img.index}`} className="w-full h-32 object-cover rounded border" />
                          <div className="text-xs text-center">
                            <div className="font-medium">Zdjęcie {img.index}</div>
                            <code className="text-muted-foreground">{img.tag}</code>
                          </div>
                        </div>
                      ))}
                    </div>
                  </CardContent>
                </Card>
              )}

              {/* Rendered HTML */}
              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Wynik renderowania</CardTitle>
                  <CardDescription>Podgląd wygenerowanego opisu HTML</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="border rounded-lg p-4 bg-muted/20 max-h-96 overflow-y-auto">
                    <div dangerouslySetInnerHTML={{ __html: previewData.renderedHtml }} />
                  </div>
                </CardContent>
              </Card>

              {/* Template Info */}
              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Szablon</CardTitle>
                </CardHeader>
                <CardContent className="text-sm">
                  <div className="grid grid-cols-2 gap-2">
                    <div><span className="font-medium">Nazwa:</span> {previewData.template.name}</div>
                    <div><span className="font-medium">Kategoria:</span> {previewData.template.categoryName || 'Brak'}</div>
                  </div>
                </CardContent>
              </Card>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
