import { useParams, useLocation, Link } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { useState, useEffect, useRef, useMemo } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { DictionaryCombobox } from "@/components/dictionary-combobox";
import { ComponentTemplateThumbnail } from "@/components/component-template-thumbnail";
import { useToast } from "@/hooks/use-toast";
import { ArrowLeft, ArrowRight, Save, Loader2, Upload, X, Edit2, Star, Plus, Trash2, Package, FileDown, FileText, FileSpreadsheet, Wrench, ChevronDown, ChevronRight, HelpCircle, AlertCircle, ExternalLink, Link as LinkIcon, Search, ShoppingCart } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { insertProductSchema } from "@shared/schema";
import type { Product, ProductPlatformData, ProductPlatformDataWithMarketplace, ProductImage, ProductAddonAssignment, ProductBOMResponse, ProductComponentWithMetrics } from "@shared/schema";
import { AIGenerationDialog } from "@/components/ai-generation-dialog";
import { AIGenerationHistory } from "@/components/ai-generation-history";
import { TemplatePickerDialog } from "@/components/template-picker-dialog";
import { RichTextEditor } from "@/components/rich-text-editor";
import { TagInput } from "@/components/tag-input";
import { Eye, Code2 } from "lucide-react";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from "@/components/ui/command";

// Platform data schema
const platformDataSchema = z.object({
  customTitle: z.string().optional(),
  categoryId: z.string().optional(),
  categoryName: z.string().optional(),
  description: z.string().optional(),
  customPrice: z.coerce.number().optional(),
  isPublished: z.boolean().default(false),
}).optional();

// Form schema extending the shared insert schema
const productFormSchema = insertProductSchema.extend({
  // Convert string inputs to numbers for numeric fields
  length: z.coerce.number().optional(),
  width: z.coerce.number().optional(),
  height: z.coerce.number().optional(),
  weight: z.coerce.number().optional(),
  basePrice: z.coerce.number().optional(),
  longDescriptionHtml: z.string().optional(),
  
  // Platform-specific data
  allegro: platformDataSchema,
  shoper: platformDataSchema,
  odoo: platformDataSchema,
});

type ProductFormData = z.infer<typeof productFormSchema>;

// Helper function to determine text color based on background luminance
function getTextColorForBackground(hexColor: string | null | undefined): string {
  if (!hexColor || !hexColor.startsWith('#')) return 'black';
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  // Calculate luminance using standard formula
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  // Return white text for dark backgrounds, black for light backgrounds
  return luminance > 128 ? 'black' : 'white';
}

// Helper function to determine if color needs visible border
function needsVisibleBorder(hexColor: string | null | undefined): boolean {
  if (!hexColor || !hexColor.startsWith('#')) return false;
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance < 30 || luminance > 230;
}

export default function CatalogProductEditor() {
  const { id } = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const isNew = id === "new";

  // Fetch product data if editing
  const { data: product, isLoading } = useQuery<Product & {
    platformData?: ProductPlatformDataWithMarketplace[];
    images?: ProductImage[];
    addons?: ProductAddonAssignment[];
  }>({
    queryKey: [`/api/catalog-products/${id}`],
    enabled: !isNew,
  });

  // Fetch matrix info if product has matrixId
  interface ProductMatrix {
    id: number;
    name: string;
  }
  
  const { data: matrix } = useQuery<ProductMatrix>({
    queryKey: [`/api/product-matrices/${product?.matrixId}`],
    enabled: !!product?.matrixId,
  });

  // Fetch product sets where this product is used
  interface ProductSet {
    id: number;
    sku: string;
    title: string;
    setMatrixId: number | null;
    quantity: number;
    componentType: string;
  }
  
  const { data: productSets = [] } = useQuery<ProductSet[]>({
    queryKey: [`/api/catalog-products/${id}/sets`],
    enabled: !isNew && !!id,
  });

  // Fetch BOM data
  const { data: bomData, refetch: refetchBom } = useQuery<ProductBOMResponse | null>({
    queryKey: [`/api/catalog-products/${id}/bom`],
    enabled: !isNew && !!id,
    queryFn: async () => {
      const res = await fetch(`/api/catalog-products/${id}/bom`, {
        credentials: 'include',
      });
      
      // Jeśli 404, to znaczy że BOM nie istnieje - zwróć null
      if (res.status === 404) {
        return null;
      }
      
      if (!res.ok) {
        const text = (await res.text()) || res.statusText;
        throw new Error(`${res.status}: ${text}`);
      }
      
      return await res.json();
    },
  });

  // Fetch BOM matrices for this product's furniture type
  interface BomMatrix {
    cz1: string;
    templateCount: number;
    sampleId: number;
    furnitureType: string;
  }
  
  const { data: bomMatrices = [] } = useQuery<BomMatrix[]>({
    queryKey: [`/api/bom/matrices/${product?.productType}`],
    enabled: !isNew && !!product?.productType,
  });

  // Fetch Odoo URL
  const { data: odooConfig } = useQuery<{ url: string | null; userUrl: string | null }>({
    queryKey: ['/api/odoo/url'],
  });

  // Fetch dictionaries
  interface ProductCreatorDictionary {
    id: number;
    dictionaryType: string;
    code: string;
    name: string;
    readableName: string | null;
    isActive: boolean;
  }

  const { data: colors } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=color"],
  });
  const { data: materials } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=material"],
  });
  const { data: dimensionLengths } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_length"],
  });
  const { data: dimensionWidths } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_width"],
  });
  const { data: dimensionHeights } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_height"],
  });
  const { data: productTypes } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=product_type"],
  });
  const { data: productGroups } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=product_group"],
  });
  const { data: doorsOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=door"],
  });
  const { data: legsOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=leg"],
  });

  // Marketplace linking state
  const [linkingDialogOpen, setLinkingDialogOpen] = useState(false);
  const [manageLinkDialogOpen, setManageLinkDialogOpen] = useState(false);
  const [searchSku, setSearchSku] = useState("");
  const [selectedMarketplaceProduct, setSelectedMarketplaceProduct] = useState<{
    id: number;
    externalId: string;
    platform: string;
    sku: string;
    name: string;
  } | null>(null);
  
  // Get marketplace links from platformData
  const marketplaceLinks = useMemo(() => {
    if (!product?.platformData) return [];
    return product.platformData.filter(pd => pd.marketplaceProductId);
  }, [product]);

  // Query for marketplace product suggestions when searching
  const { data: marketplaceSuggestions = [], isLoading: suggestionsLoading } = useQuery<Array<{
    id: number;
    external_id: string;
    platform: string;
    sku: string;
    name: string;
  }>>({
    queryKey: ["/api/commerce/marketplace-products/search", searchSku],
    queryFn: async () => {
      if (!searchSku || searchSku.length < 2) return [];
      const res = await fetch(`/api/commerce/marketplace-products?search=${encodeURIComponent(searchSku)}`);
      if (!res.ok) throw new Error('Failed to fetch suggestions');
      return res.json();
    },
    enabled: !!searchSku && searchSku.length >= 2 && linkingDialogOpen,
  });

  // Link marketplace product mutation
  const linkMarketplaceMutation = useMutation({
    mutationFn: async ({ marketplaceExternalId, marketplacePlatform }: {
      marketplaceExternalId: string;
      marketplacePlatform: string;
    }) => {
      return apiRequest("POST", '/api/catalog/products/link-marketplace', {
        catalogProductId: parseInt(id!),
        externalId: marketplaceExternalId,
        platform: marketplacePlatform,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${id}`] });
      setLinkingDialogOpen(false);
      setSearchSku("");
      setSelectedMarketplaceProduct(null);
      toast({
        title: "Produkt połączony",
        description: "Produkt marketplace został połączony z produktem katalogowym.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd połączenia",
        description: error.message || "Nie udało się połączyć produktu.",
        variant: "destructive",
      });
    },
  });

  const { data: componentCz1Options } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=component_cz1"],
  });

  // Generate COMPONENT_CZ1-COLOR combinations for colorOptions
  const availableColorOptions = useMemo(() => {
    if (!componentCz1Options || !colors) return [];
    const combinations: Array<{ value: string; label: string }> = [];
    componentCz1Options.forEach(component => {
      colors.forEach(color => {
        combinations.push({
          value: `${component.code}-${color.code}`,
          label: `${component.code}-${color.code}`
        });
      });
    });
    return combinations;
  }, [componentCz1Options, colors]);

  // State for color options (editable)
  const [colorOptionsState, setColorOptionsState] = useState<string[]>([]);

  // Initialize form
  const form = useForm<ProductFormData>({
    resolver: zodResolver(productFormSchema),
    defaultValues: {
      sku: "",
      title: "",
      shortDescription: "",
      longDescriptionHtml: "",
      length: undefined,
      width: undefined,
      height: undefined,
      weight: undefined,
      color: "",
      material: "",
      productType: "",
      productGroup: "",
      doors: "",
      legs: "",
      basePrice: undefined,
      currency: "PLN",
      isActive: true,
      allegro: {
        customTitle: "",
        categoryId: "",
        categoryName: "",
        description: "",
        customPrice: undefined,
        isPublished: false,
      },
      shoper: {
        customTitle: "",
        categoryId: "",
        categoryName: "",
        description: "",
        customPrice: undefined,
        isPublished: false,
      },
      odoo: {
        customTitle: "",
        categoryId: "",
        categoryName: "",
        description: "",
        customPrice: undefined,
        isPublished: false,
      },
    },
  });

  // Template Picker Dialog State
  const [isTemplatePickerOpen, setIsTemplatePickerOpen] = useState(false);
  
  // Description Editor Mode: 'code' for HTML textarea, 'editor' for WYSIWYG
  const [descriptionMode, setDescriptionMode] = useState<'code' | 'editor'>('editor');
  
  // Description Preview Dialog State
  const [isPreviewDialogOpen, setIsPreviewDialogOpen] = useState(false);

  // Update color options state when product data loads
  useEffect(() => {
    if (product && !isNew && product.colorOptions) {
      setColorOptionsState(product.colorOptions);
    }
  }, [product, isNew]);

  // Update form when product data loads
  useEffect(() => {
    if (product && !isNew) {
      const allegroPlatform = product.platformData?.find(p => p.platform === "allegro");
      const shoperPlatform = product.platformData?.find(p => p.platform === "shoper");
      const odooPlatform = product.platformData?.find(p => p.platform === "odoo");

      form.reset({
        sku: product.sku,
        title: product.title,
        shortDescription: product.shortDescription ?? "",
        longDescriptionHtml: product.longDescriptionHtml ?? "",
        length: product.length ? Number(product.length) : undefined,
        width: product.width ? Number(product.width) : undefined,
        height: product.height ? Number(product.height) : undefined,
        weight: product.weight ? Number(product.weight) : undefined,
        color: product.color ?? "",
        material: product.material ?? "",
        productType: product.productType ?? "",
        productGroup: product.productGroup ?? "",
        doors: product.doors ?? "",
        legs: product.legs ?? "",
        basePrice: product.basePrice ? Number(product.basePrice) : undefined,
        currency: product.currency ?? "PLN",
        isActive: product.isActive ?? true,
        allegro: allegroPlatform ? {
          customTitle: allegroPlatform.customTitle ?? "",
          categoryId: allegroPlatform.categoryId ?? "",
          categoryName: allegroPlatform.categoryName ?? "",
          description: allegroPlatform.description ?? "",
          customPrice: allegroPlatform.customPrice ? Number(allegroPlatform.customPrice) : undefined,
          isPublished: allegroPlatform.isPublished ?? false,
        } : undefined,
        shoper: shoperPlatform ? {
          customTitle: shoperPlatform.customTitle ?? "",
          categoryId: shoperPlatform.categoryId ?? "",
          categoryName: shoperPlatform.categoryName ?? "",
          description: shoperPlatform.description ?? "",
          customPrice: shoperPlatform.customPrice ? Number(shoperPlatform.customPrice) : undefined,
          isPublished: shoperPlatform.isPublished ?? false,
        } : undefined,
        odoo: odooPlatform ? {
          customTitle: odooPlatform.customTitle ?? "",
          categoryId: odooPlatform.categoryId ?? "",
          categoryName: odooPlatform.categoryName ?? "",
          description: odooPlatform.description ?? "",
          customPrice: odooPlatform.customPrice ? Number(odooPlatform.customPrice) : undefined,
          isPublished: odooPlatform.isPublished ?? false,
        } : undefined,
      });
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [product]);

  // Generate BOM mutation
  const generateBomMutation = useMutation({
    mutationFn: async () => {
      const response = await fetch(`/api/catalog-products/${id}/generate-bom`, {
        method: 'POST',
        credentials: 'include',
      });
      if (!response.ok) {
        const error = await response.json();
        const err: any = new Error(error.error || 'Failed to generate BOM');
        err.code = error.code; // Przekaż kod błędu (np. GENERATION_IN_PROGRESS)
        throw err;
      }
      return response.json();
    },
    onSuccess: (data: any) => {
      toast({
        title: "BOM wygenerowany",
        description: `Wygenerowano ${data.componentsGenerated} komponentów`,
      });
      refetchBom();
      queryClient.invalidateQueries({ queryKey: ['/api/boms'] });
    },
    onError: (error: any) => {
      // Obsługa błędu równoległego generowania
      if (error.code === "GENERATION_IN_PROGRESS") {
        toast({
          title: "Generowanie w toku",
          description: "BOM jest już generowany dla tego produktu. Poczekaj na zakończenie poprzedniej operacji.",
          variant: "destructive",
        });
      } else {
        toast({
          title: "Błąd generowania BOM",
          description: error.message,
          variant: "destructive",
        });
      }
    },
  });

  // Export BOM to Odoo mutation
  const exportBomMutation = useMutation({
    mutationFn: async () => {
      const response = await fetch(`/api/catalog-products/${id}/export-bom-to-odoo`, {
        method: 'POST',
        credentials: 'include',
      });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to export BOM to Odoo');
      }
      return response.json();
    },
    onSuccess: (data: any) => {
      toast({
        title: "BOM wyeksportowany do Odoo",
        description: `Odoo BOM ID: ${data.odoo_bom_id}`,
      });
      refetchBom();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd eksportu BOM do Odoo",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Export Product to Odoo mutation
  const exportProductMutation = useMutation({
    mutationFn: async () => {
      const response = await fetch(`/api/catalog-products/${id}/export-to-odoo`, {
        method: 'POST',
        credentials: 'include',
      });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || error.details || 'Failed to export product to Odoo');
      }
      return response.json();
    },
    onSuccess: (data: any) => {
      toast({
        title: "✅ Produkt wyeksportowany do Odoo",
        description: data.message || `Odoo Product ID: ${data.odoo_product_id}`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "❌ Błąd eksportu produktu do Odoo",
        description: error.message,
        variant: "destructive",
        duration: 10000,
      });
    },
  });

  // Update color options mutation
  const updateColorOptionsMutation = useMutation({
    mutationFn: async (options: string[]) => {
      const response = await fetch(`/api/catalog-products/${id}/color-options`, {
        method: 'PUT',
        credentials: 'include',
        body: JSON.stringify({ colorOptions: options }),
        headers: {
          'Content-Type': 'application/json',
        },
      });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to update color options');
      }
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Opcje kolorystyczne zapisane",
      });
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${id}`] });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd zapisu opcji",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Delete BOM mutation
  const deleteBomMutation = useMutation({
    mutationFn: async () => {
      const response = await fetch(`/api/catalog-products/${id}/bom`, {
        method: 'DELETE',
        credentials: 'include',
      });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to delete BOM');
      }
      return response.json();
    },
    onSuccess: async () => {
      // Najpierw ustaw BOM na null w cache
      queryClient.setQueryData([`/api/catalog-products/${id}/bom`], null);
      
      // Następnie invalidate aby refetch (który zwróci null dla 404)
      await queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${id}/bom`] });
      
      // Also invalidate product data to refresh
      await queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${id}`] });
      
      toast({
        title: "BOM usunięty",
        description: "Bill of Materials został pomyślnie usunięty",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania BOM",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Save mutation
  const saveMutation = useMutation({
    mutationFn: async (data: ProductFormData) => {
      const response = isNew
        ? await apiRequest("POST", "/api/catalog-products", data)
        : await apiRequest("PUT", `/api/catalog-products/${id}`, data);
      
      // Parse the response to get the product data
      return await response.json();
    },
    onSuccess: (savedProduct: any) => {
      toast({
        title: "Sukces",
        description: isNew ? "Produkt został utworzony" : "Produkt został zaktualizowany",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/catalog-products"] });
      
      if (isNew && savedProduct?.id) {
        // For new products, navigate to edit page with the new ID
        navigate(`/catalog-products/${savedProduct.id}`);
      } else if (!isNew) {
        // For existing products, stay on the page and refresh data
        queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${id}`] });
      }
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: ProductFormData) => {
    // Jeśli BOM istnieje, pola są disabled i React Hook Form nie wysyła ich wartości
    // Musimy ręcznie dodać wartości z oryginalnego produktu (z bazy danych)
    if (bomData?.components?.length > 0 && product) {
      // Pola tekstowe (text)
      data.productType = product.productType || undefined;
      data.color = product.color || undefined;
      
      // Pola tekstowe słownikowe (text) - mogą być null
      if (product.productGroup) data.productGroup = product.productGroup;
      if (product.doors) data.doors = product.doors;
      if (product.legs) data.legs = product.legs;
      
      // Pola numeryczne (decimal) - konwertuj string na number
      if (product.length) data.length = parseFloat(product.length);
      if (product.width) data.width = parseFloat(product.width);
      if (product.height) data.height = parseFloat(product.height);
    }
    
    saveMutation.mutate(data);
  };

  // Copy shared data to platform-specific fields
  const copySharedToPlatform = (platform: "allegro" | "shoper" | "odoo") => {
    const sharedData = form.getValues();
    
    form.setValue(`${platform}.customTitle`, sharedData.title || "");
    form.setValue(`${platform}.description`, sharedData.longDescriptionHtml || "");
    form.setValue(`${platform}.customPrice`, sharedData.basePrice);
    
    toast({
      title: "Skopiowano dane",
      description: `Dane wspólne zostały skopiowane do zakładki ${platform.toUpperCase()}`,
    });
  };

  // Export functions
  const handleExport = async (format: 'html' | 'csv' | 'pdf') => {
    if (isNew) {
      toast({
        title: "Zapisz produkt",
        description: "Najpierw zapisz produkt, aby go wyeksportować",
        variant: "destructive",
      });
      return;
    }

    try {
      const response = await fetch(`/api/catalog-products/${id}/export?format=${format}`, {
        method: 'GET',
      });

      if (!response.ok) {
        throw new Error('Export failed');
      }

      // Get filename from Content-Disposition header or use default
      const contentDisposition = response.headers.get('Content-Disposition');
      let filename = `product-${product?.sku || id}.${format}`;
      if (contentDisposition) {
        const filenameMatch = contentDisposition.match(/filename="?(.+)"?/i);
        if (filenameMatch) {
          filename = filenameMatch[1];
        }
      }

      // Download file
      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);

      toast({
        title: "Eksport zakończony",
        description: `Plik ${filename} został pobrany`,
      });
    } catch (error) {
      console.error('Export error:', error);
      toast({
        title: "Błąd eksportu",
        description: "Nie udało się wyeksportować produktu",
        variant: "destructive",
      });
    }
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  return (
    <div className="flex flex-col h-full w-full max-w-full overflow-x-hidden">
      {/* Header */}
      <Card className="rounded-none border-x-0 border-t-0">
        <CardHeader className="min-h-10 px-3 py-2">
          <div className="flex flex-col sm:flex-row gap-2 justify-between items-start sm:items-center w-full">
            <div className="flex items-center gap-2 flex-wrap min-w-0">
              <Link href="/catalog-products">
                <Button variant="ghost" size="icon" data-testid="button-back" className="h-8 w-8 flex-shrink-0">
                  <ArrowLeft className="h-4 w-4" />
                </Button>
              </Link>
              <CardTitle className="text-base truncate" data-testid="title-product">
                {isNew ? "Nowy produkt" : `Edycja: ${product?.title}`}
              </CardTitle>
              {!isNew && product && (
                <Badge variant="secondary" className="h-5 text-xs flex-shrink-0">
                  {product.sku}
                </Badge>
              )}
            </div>
            <div className="flex items-center gap-1 flex-wrap">
              {matrix && (
                <Link href={`/product-matrices/${matrix.id}/edit`}>
                  <Button variant="outline" size="sm" data-testid="link-matrix" className="h-8">
                    <Package className="h-3.5 w-3.5 mr-1" />
                    <span className="hidden sm:inline text-xs">{matrix.name}</span>
                  </Button>
                </Link>
              )}
              {!isNew && (
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => generateBomMutation.mutate()}
                  disabled={generateBomMutation.isPending}
                  data-testid="button-generate-bom"
                  className="h-8"
                >
                  {generateBomMutation.isPending ? (
                    <Loader2 className="h-3.5 w-3.5 animate-spin" />
                  ) : (
                    <Package className="h-3.5 w-3.5" />
                  )}
                  <span className="hidden lg:inline text-xs ml-1">BOM</span>
                </Button>
              )}
              {!isNew && (
                <DropdownMenu>
                  <DropdownMenuTrigger asChild>
                    <Button variant="outline" size="sm" data-testid="button-export" aria-label="Eksport produktu" className="h-8">
                      <FileDown className="h-3.5 w-3.5" />
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent align="end">
                    <DropdownMenuItem onClick={() => handleExport('html')} data-testid="export-html">
                      <FileText className="h-4 w-4 mr-2" />
                      Pobierz HTML
                    </DropdownMenuItem>
                    <DropdownMenuItem onClick={() => handleExport('csv')} data-testid="export-csv">
                      <FileSpreadsheet className="h-4 w-4 mr-2" />
                      Pobierz CSV
                    </DropdownMenuItem>
                    <DropdownMenuItem onClick={() => handleExport('pdf')} data-testid="export-pdf">
                      <FileText className="h-4 w-4 mr-2" />
                      Pobierz PDF
                    </DropdownMenuItem>
                  </DropdownMenuContent>
                </DropdownMenu>
              )}
              <Button
                onClick={form.handleSubmit(onSubmit)}
                disabled={saveMutation.isPending}
                size="sm"
                data-testid="button-save"
                className="h-8"
              >
                {saveMutation.isPending ? (
                  <Loader2 className="h-3.5 w-3.5 mr-1 animate-spin" />
                ) : (
                  <Save className="h-3.5 w-3.5 mr-1" />
                )}
                <span className="hidden sm:inline text-xs">Zapisz</span>
              </Button>
            </div>
          </div>
        </CardHeader>
      </Card>

      {/* Content */}
      <div className="flex-1 overflow-x-hidden overflow-y-auto w-full">
        <div className="w-full px-2 sm:px-4 lg:px-6 py-2 sm:py-4 lg:py-6">
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4 sm:space-y-6 w-full max-w-full">
              {/* Zestawy, w których występuje produkt */}
              {!isNew && productSets.length > 0 && (
                <Card className="border-l-4 border-l-blue-500 bg-blue-50/50 dark:bg-blue-950/20">
                  <CardHeader>
                    <CardTitle className="text-base flex items-center gap-2">
                      <Package className="h-5 w-5" />
                      Występowanie w zestawach ({productSets.length})
                    </CardTitle>
                    <CardDescription>
                      Ten produkt jest komponentem w następujących zestawach:
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <ol className="list-decimal list-inside space-y-2">
                      {productSets.map((set, index) => (
                        <li key={set.id} className="text-sm">
                          <Link href={`/catalog-sets/${set.id}`} className="text-primary hover:underline font-medium">
                            {set.title}
                          </Link>
                          <span className="text-muted-foreground ml-2">
                            (SKU: {set.sku}, Ilość: {set.quantity})
                          </span>
                        </li>
                      ))}
                    </ol>
                  </CardContent>
                </Card>
              )}


              {/* Marketplace Linking Dialog */}
              <Dialog open={linkingDialogOpen} onOpenChange={setLinkingDialogOpen}>
                <DialogContent className="max-w-2xl">
                  <DialogHeader>
                    <DialogTitle>Powiązania marketplace ({marketplaceLinks.length})</DialogTitle>
                    <DialogDescription>
                      Zarządzaj połączeniami z ofertami marketplace (Allegro/Shoper)
                    </DialogDescription>
                  </DialogHeader>
                  <div className="space-y-4">
                    {/* Istniejące połączenia */}
                    {marketplaceLinks.length > 0 && (
                      <div className="border rounded-lg p-4 bg-muted/30">
                        <h4 className="text-sm font-semibold mb-3">Aktualne połączenia:</h4>
                        <div className="space-y-2">
                          {marketplaceLinks.map((link) => (
                            <div 
                              key={`${link.platform}-${link.externalId}`}
                              className="flex items-center gap-2 p-2 bg-background rounded border"
                              data-testid={`marketplace-link-${link.platform}`}
                            >
                              <Badge variant="secondary" className="shrink-0">
                                {link.platform === 'allegro' ? 'Allegro' : link.platform === 'shoper' ? 'Shoper' : link.platform}
                              </Badge>
                              <div className="flex-1 text-sm">
                                <span className="font-medium">SKU: {link.marketplaceSku || link.externalId}</span>
                                {link.marketplaceName && (
                                  <span className="text-muted-foreground ml-2">- {link.marketplaceName}</span>
                                )}
                              </div>
                            </div>
                          ))}
                        </div>
                      </div>
                    )}

                    <div className="border-t pt-4">
                      <label className="text-sm font-medium mb-2 block">Szukaj i dodaj nowe połączenie</label>
                      <Input
                        placeholder="Wpisz SKU lub nazwę produktu marketplace..."
                        value={searchSku}
                        onChange={(e) => setSearchSku(e.target.value)}
                        data-testid="input-search-marketplace"
                      />
                    </div>

                    {suggestionsLoading && (
                      <div className="flex justify-center p-4">
                        <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
                      </div>
                    )}

                    {!suggestionsLoading && searchSku && marketplaceSuggestions.length > 0 && (
                      <div className="space-y-2 max-h-64 overflow-y-auto">
                        <div className="text-sm font-medium mb-2">Wyniki wyszukiwania:</div>
                        {marketplaceSuggestions.map((suggestion) => (
                          <div
                            key={`${suggestion.platform}-${suggestion.external_id}`}
                            className={`p-3 border rounded cursor-pointer hover-elevate ${
                              selectedMarketplaceProduct?.id === suggestion.id
                                ? 'border-primary bg-primary/5'
                                : ''
                            }`}
                            onClick={() => setSelectedMarketplaceProduct({
                              id: suggestion.id,
                              externalId: suggestion.external_id,
                              platform: suggestion.platform,
                              sku: suggestion.sku,
                              name: suggestion.name,
                            })}
                            data-testid={`marketplace-suggestion-${suggestion.id}`}
                          >
                            <div className="flex items-start gap-2 mb-1">
                              <Badge variant="secondary" className="mt-0.5">
                                {suggestion.platform === 'allegro' ? 'Allegro' : suggestion.platform === 'shoper' ? 'Shoper' : suggestion.platform}
                              </Badge>
                              <div className="flex-1">
                                <div className="font-medium text-sm line-clamp-2 mb-1">
                                  {suggestion.name}
                                </div>
                                <button
                                  onClick={(e) => {
                                    e.stopPropagation();
                                    setSearchSku(suggestion.sku);
                                  }}
                                  className="text-xs text-muted-foreground hover:text-primary hover:underline"
                                >
                                  SKU: {suggestion.sku}
                                </button>
                              </div>
                            </div>
                          </div>
                        ))}
                      </div>
                    )}

                    {!suggestionsLoading && searchSku && marketplaceSuggestions.length === 0 && searchSku.length >= 2 && (
                      <div className="text-center p-6 text-muted-foreground">
                        <Search className="h-12 w-12 mx-auto mb-2 opacity-50" />
                        <p>Nie znaleziono produktów marketplace</p>
                        <p className="text-sm mt-2">Spróbuj innej frazy wyszukiwania</p>
                      </div>
                    )}

                    <div className="flex justify-end gap-2 pt-4 border-t">
                      <Button
                        variant="outline"
                        onClick={() => {
                          setLinkingDialogOpen(false);
                          setSearchSku("");
                          setSelectedMarketplaceProduct(null);
                        }}
                        data-testid="button-cancel-linking"
                      >
                        Anuluj
                      </Button>
                      <Button
                        onClick={() => {
                          if (selectedMarketplaceProduct) {
                            linkMarketplaceMutation.mutate({
                              marketplaceExternalId: selectedMarketplaceProduct.externalId,
                              marketplacePlatform: selectedMarketplaceProduct.platform.toLowerCase(),
                            });
                          }
                        }}
                        disabled={!selectedMarketplaceProduct || linkMarketplaceMutation.isPending}
                        data-testid="button-confirm-linking"
                      >
                        {linkMarketplaceMutation.isPending && (
                          <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                        )}
                        <LinkIcon className="mr-2 h-4 w-4" />
                        Połącz
                      </Button>
                    </div>
                  </div>
                </DialogContent>
              </Dialog>
              
              <Tabs defaultValue="shared" className="w-full max-w-full overflow-hidden">
              <div className="w-full overflow-x-auto overflow-y-hidden pb-2">
                <TabsList className="flex sm:grid w-max sm:w-full sm:grid-cols-6 gap-1 min-w-0">
                  <TabsTrigger value="shared" data-testid="tab-shared" className="whitespace-nowrap">
                    Wspólne
                  </TabsTrigger>
                  <TabsTrigger value="allegro" data-testid="tab-allegro" className="whitespace-nowrap">
                    Allegro
                  </TabsTrigger>
                  <TabsTrigger value="shoper" data-testid="tab-shoper" className="whitespace-nowrap">
                    Shoper
                  </TabsTrigger>
                  <TabsTrigger value="odoo" data-testid="tab-odoo" className="whitespace-nowrap">
                    Odoo
                  </TabsTrigger>
                  <TabsTrigger value="gallery" data-testid="tab-gallery" className="whitespace-nowrap">
                    Galeria
                  </TabsTrigger>
                  <TabsTrigger value="bom" data-testid="tab-bom" className="whitespace-nowrap">
                    BOM {bomData?.components?.length ? `(${bomData.components.length})` : ''}
                  </TabsTrigger>
                </TabsList>
              </div>

              {/* Zakładka: Wspólne */}
              <TabsContent value="shared" className="space-y-4">
                <Card className="overflow-hidden">
                  <CardHeader className="flex flex-row items-start justify-between gap-4 space-y-0 pb-2">
                    <div>
                      <CardTitle>Podstawowe informacje</CardTitle>
                      <CardDescription>
                        Dane wspólne dla wszystkich platform
                      </CardDescription>
                    </div>
                    {!isNew && product && (
                      <div className="flex items-center gap-2">
                        {marketplaceLinks.length > 0 ? (
                          <Card className="px-4 py-2.5 bg-green-50 dark:bg-green-950/20 border-green-200 dark:border-green-800">
                            <div className="flex items-center gap-3">
                              <div className="flex items-center gap-2">
                                <div className="relative flex h-2 w-2">
                                  <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-green-400 opacity-75"></span>
                                  <span className="relative inline-flex rounded-full h-2 w-2 bg-green-500"></span>
                                </div>
                                <span className="text-sm font-medium text-green-700 dark:text-green-300">
                                  Połączono ({marketplaceLinks.length})
                                </span>
                              </div>
                              <div className="h-4 w-px bg-green-200 dark:bg-green-800" />
                              {marketplaceLinks.slice(0, 2).map((link, idx) => (
                                <button
                                  key={idx}
                                  onClick={() => navigate(`/product/${link.externalId}`)}
                                  className="inline-flex items-center gap-1.5 px-2.5 py-1 rounded-md text-xs font-medium bg-white dark:bg-gray-900 border border-green-200 dark:border-green-800 text-green-700 dark:text-green-300 hover:bg-green-50 dark:hover:bg-green-900/30 transition-colors cursor-pointer"
                                  title={`Przejdź do produktu marketplace: ${link.marketplaceName || link.externalId}`}
                                >
                                  {link.platform === 'allegro' ? '🔴' : '🟢'} {link.externalId}
                                </button>
                              ))}
                              <Button
                                size="sm"
                                variant="ghost"
                                onClick={() => setLinkingDialogOpen(true)}
                                data-testid="button-view-marketplace-links"
                                title="Zobacz wszystkie połączenia"
                                className="h-auto px-2 py-1 text-green-700 dark:text-green-300 hover:text-green-800 dark:hover:text-green-200 hover:bg-green-100 dark:hover:bg-green-900/20 gap-1"
                              >
                                <ExternalLink className="h-3.5 w-3.5" />
                                Zobacz
                              </Button>
                            </div>
                          </Card>
                        ) : (
                          <Button
                            size="default"
                            variant="outline"
                            onClick={() => setLinkingDialogOpen(true)}
                            data-testid="button-link-marketplace"
                            title="Połącz z produktem marketplace"
                            className="bg-blue-600 hover:bg-blue-700 border-blue-600 text-white gap-2"
                          >
                            <ShoppingCart className="h-4 w-4" />
                            Połącz z katalogiem
                          </Button>
                        )}
                        {product.odooProductId && odooConfig?.userUrl && (
                          <Button
                            size="icon"
                            variant="outline"
                            onClick={() => window.open(`${odooConfig.userUrl}/products/${product.odooProductId}`, '_blank')}
                            data-testid="button-odoo-link-shared"
                            title="Otwórz produkt w Odoo"
                            className="bg-purple-600 hover:bg-purple-700 border-purple-600 text-white"
                          >
                            <ExternalLink className="h-4 w-4" />
                          </Button>
                        )}
                        <Button
                          size="icon"
                          variant={product.odooProductId ? "default" : "destructive"}
                          onClick={() => exportProductMutation.mutate()}
                          disabled={exportProductMutation.isPending}
                          data-testid="button-export-odoo-shared"
                          title={
                            product.odooExportError 
                              ? `Błąd eksportu: ${product.odooExportError}` 
                              : product.odooProductId 
                                ? `Wyeksportowano: ${product.odooExportedAt ? new Date(product.odooExportedAt).toLocaleString('pl-PL') : ''}` 
                                : "Nie wyeksportowano - kliknij aby wysłać do Odoo"
                          }
                          className={
                            product.odooExportError 
                              ? "bg-yellow-500 hover:bg-yellow-600 relative" 
                              : product.odooProductId 
                                ? "bg-green-600 hover:bg-green-700" 
                                : ""
                          }
                        >
                          {exportProductMutation.isPending ? (
                            <Loader2 className="h-4 w-4 animate-spin" />
                          ) : (
                            <Upload className="h-4 w-4" />
                          )}
                          {product.odooExportError && (
                            <span className="absolute -top-1 -right-1 flex h-3 w-3">
                              <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-red-400 opacity-75"></span>
                              <span className="relative inline-flex rounded-full h-3 w-3 bg-red-500"></span>
                            </span>
                          )}
                        </Button>
                      </div>
                    )}
                  </CardHeader>
                  <CardContent className="space-y-4 overflow-x-auto">
                    {/* Tytuł - większy na górze */}
                    <FormField
                      control={form.control}
                      name="title"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel className="text-lg font-semibold">Tytuł produktu *</FormLabel>
                          <FormControl>
                            <Input {...field} data-testid="input-title" className="text-lg" />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    {/* SKU z wyróżnionym tłem */}
                    <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                      {!isNew && product && (
                        <FormItem>
                          <FormLabel className="text-sm font-semibold">ID</FormLabel>
                          <div className="bg-muted/30 p-3 border border-muted-foreground/10">
                            <div className="font-mono text-sm text-muted-foreground" data-testid="text-product-id">
                              {product.id}
                            </div>
                          </div>
                        </FormItem>
                      )}
                      <FormField
                        control={form.control}
                        name="sku"
                        render={({ field }) => (
                          <FormItem className={isNew ? "md:col-span-4" : "md:col-span-3"}>
                            <FormLabel className="text-sm font-semibold">SKU (identyfikator) *</FormLabel>
                            <FormControl>
                              <div className="bg-muted/50 p-3 border border-muted-foreground/20">
                                <Input {...field} data-testid="input-sku" className="bg-background" />
                              </div>
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </div>

                    {/* Belka z parametrami produktu - WIDOCZNA */}
                    {!isNew && product && (
                      <div className="bg-accent/10 p-4 border border-accent/30" data-testid="product-parameters-bar">
                        <div className="space-y-3">
                          <div className="flex flex-wrap items-center gap-2">
                            {/* Rodzaj szafki - nazwa z product_type */}
                            {product.productType && (
                              <Badge 
                                variant="secondary" 
                                className="text-sm font-semibold px-3 py-1.5 bg-primary/10 text-primary border-primary/30"
                                data-testid="param-furniture-type"
                              >
                                {productTypes?.find(pt => pt.code === product.productType)?.name || product.productType}
                              </Badge>
                            )}
                            
                            {/* Długość */}
                            {product.length && (
                              <Badge 
                                variant="outline" 
                                className="text-sm px-3 py-1.5 bg-background"
                                data-testid="param-length"
                              >
                                Długość: {product.length}
                              </Badge>
                            )}
                            
                            {/* Szerokość */}
                            {product.width && (
                              <Badge 
                                variant="outline" 
                                className="text-sm px-3 py-1.5 bg-background"
                                data-testid="param-width"
                              >
                                Szerokość: {product.width}
                              </Badge>
                            )}
                            
                            {/* Wysokość */}
                            {product.height && (
                              <Badge 
                                variant="outline" 
                                className="text-sm px-3 py-1.5 bg-background"
                                data-testid="param-height"
                              >
                                Wysokość: {product.height}
                              </Badge>
                            )}
                            
                            {/* Drzwi */}
                            {product.doors && (
                              <Badge 
                                variant="outline" 
                                className="text-sm px-3 py-1.5 bg-purple-500/10 text-purple-700 dark:text-purple-300 border-purple-500/30"
                                data-testid="param-doors"
                              >
                                Drzwi: {product.doors}
                              </Badge>
                            )}
                            
                            {/* Nogi */}
                            {product.legs && (
                              <Badge 
                                variant="outline" 
                                className="text-sm px-3 py-1.5 bg-purple-500/10 text-purple-700 dark:text-purple-300 border-purple-500/30"
                                data-testid="param-legs"
                              >
                                Nogi: {product.legs}
                              </Badge>
                            )}
                            
                            {/* Kolor */}
                            {product.color && (
                              <Badge 
                                variant="outline" 
                                className="text-sm px-3 py-1.5 bg-background"
                                data-testid="param-color"
                              >
                                Kolor: {colors?.find(c => c.code === product.color)?.name || product.color}
                              </Badge>
                            )}
                            
                            {/* Materiał */}
                            {product.material && (
                              <Badge 
                                variant="outline" 
                                className="text-sm px-3 py-1.5 bg-background"
                                data-testid="param-material"
                              >
                                Płyta meblowa: {materials?.find(m => m.code === product.material)?.name || product.material}
                              </Badge>
                            )}
                            
                            {/* Opcje kolorystyczne */}
                            {product.colorOptions && product.colorOptions.length > 0 && (
                              <div className="flex flex-wrap gap-1.5">
                                <span className="text-sm text-muted-foreground">Opcje:</span>
                                {product.colorOptions.map((option, idx) => (
                                  <Badge 
                                    key={idx}
                                    variant="secondary" 
                                    className="text-sm px-3 py-1.5"
                                    data-testid={`param-option-${idx}`}
                                  >
                                    {option}
                                  </Badge>
                                ))}
                              </div>
                            )}
                          </div>
                          
                          {/* Informacja o blokowaniu edycji jeśli BOM istnieje */}
                          {bomData?.components?.length > 0 && (
                            <div className="flex items-center gap-2 text-sm text-amber-600 dark:text-amber-400 bg-amber-500/10 px-3 py-2 rounded-md border border-amber-500/30">
                              <Package className="w-4 h-4 flex-shrink-0" />
                              <span className="font-medium" data-testid="bom-lock-message">
                                Parametry zablokowane - BOM utworzony ({bomData.components.length} komponentów). Usuń BOM aby edytować parametry.
                              </span>
                            </div>
                          )}
                        </div>
                      </div>
                    )}

                    {/* Layout: Zdjęcie + Parametry */}
                    <div className="flex flex-col lg:flex-row gap-4">
                      {/* Pierwsze zdjęcie produktu */}
                      <div className="w-full lg:w-64 flex-shrink-0">
                        <div className="space-y-3">
                          <p className="text-sm font-medium">Zdjęcie podglądowe</p>
                          {product?.images && product.images.length > 0 ? (
                            <div className="aspect-square rounded-lg border bg-muted overflow-hidden max-w-[280px] mx-auto lg:mx-0">
                              <img
                                src={product.images[0].url}
                                alt={product.images[0].altText || product.title || 'Zdjęcie produktu'}
                                className="w-full h-full object-cover"
                              />
                            </div>
                          ) : (
                            <div className="aspect-square rounded-lg border bg-muted flex items-center justify-center text-muted-foreground max-w-[280px] mx-auto lg:mx-0">
                              <div className="text-center p-4">
                                <Package className="h-12 w-12 mx-auto mb-2 opacity-50" />
                                <p className="text-sm">Brak zdjęcia</p>
                                <p className="text-xs mt-1">Dodaj w zakładce Galeria</p>
                              </div>
                            </div>
                          )}
                          
                          {/* Metadata: daty utworzenia i aktualizacji */}
                          {!isNew && product && product.createdAt && product.updatedAt && (
                            <div className="space-y-2 p-3 bg-muted/30 rounded-md border text-xs max-w-[280px] mx-auto lg:mx-0" data-testid="product-metadata">
                              <div className="space-y-1">
                                <p className="font-medium text-muted-foreground">Utworzono:</p>
                                <p className="font-mono" data-testid="created-at">
                                  {new Date(product.createdAt).toLocaleString('pl-PL', {
                                    year: 'numeric',
                                    month: '2-digit',
                                    day: '2-digit',
                                    hour: '2-digit',
                                    minute: '2-digit',
                                  })}
                                </p>
                              </div>
                              <div className="space-y-1">
                                <p className="font-medium text-muted-foreground">Zaktualizowano:</p>
                                <p className="font-mono" data-testid="updated-at">
                                  {new Date(product.updatedAt).toLocaleString('pl-PL', {
                                    year: 'numeric',
                                    month: '2-digit',
                                    day: '2-digit',
                                    hour: '2-digit',
                                    minute: '2-digit',
                                  })}
                                </p>
                              </div>
                            </div>
                          )}
                        </div>
                      </div>

                      {/* Parametry produktu */}
                      <div className="flex-1 space-y-2">
                        {/* Długość + Szerokość */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                          <FormField
                            control={form.control}
                            name="length"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Długość</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={dimensionLengths}
                                    value={field.value?.toString() || ""}
                                    onChange={(value) => field.onChange(value ? parseFloat(value) : undefined)}
                                    placeholder="Wybierz"
                                    searchPlaceholder="Szukaj długości..."
                                    testId="select-length"
                                    valueField="code"
                                    disabled={bomData?.components?.length > 0}
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <FormField
                            control={form.control}
                            name="width"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Szerokość</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={dimensionWidths}
                                    value={field.value?.toString() || ""}
                                    onChange={(value) => field.onChange(value ? parseFloat(value) : undefined)}
                                    placeholder="Wybierz"
                                    searchPlaceholder="Szukaj szerokości..."
                                    testId="select-width"
                                    valueField="code"
                                    disabled={bomData?.components?.length > 0}
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>

                        {/* Wysokość + Waga */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                          <FormField
                            control={form.control}
                            name="height"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Wysokość</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={dimensionHeights}
                                    value={field.value?.toString() || ""}
                                    onChange={(value) => field.onChange(value ? parseFloat(value) : undefined)}
                                    placeholder="Wybierz"
                                    searchPlaceholder="Szukaj wysokości..."
                                    testId="select-height"
                                    valueField="code"
                                    disabled={bomData?.components?.length > 0}
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <FormField
                            control={form.control}
                            name="weight"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Waga (kg)</FormLabel>
                                <FormControl>
                                  <Input
                                    {...field}
                                    type="number"
                                    step="0.01"
                                    placeholder="0.00"
                                    className="h-8"
                                    data-testid="input-weight"
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>

                        {/* Kolor + Materiał (obowiązkowe) */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                          <FormField
                            control={form.control}
                            name="color"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs font-semibold">Kolor *</FormLabel>
                                <div className="bg-muted/30 rounded-md">
                                  <FormControl>
                                    <DictionaryCombobox
                                      items={colors}
                                      value={field.value || ""}
                                      onChange={field.onChange}
                                      placeholder="Wybierz kolor"
                                      searchPlaceholder="Szukaj koloru..."
                                      testId="select-color"
                                      displayField="name"
                                      disabled={bomData?.components?.length > 0}
                                    />
                                  </FormControl>
                                </div>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <FormField
                            control={form.control}
                            name="material"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs font-semibold">Materiał *</FormLabel>
                                <div className="bg-muted/30 rounded-md">
                                  <FormControl>
                                    <DictionaryCombobox
                                      items={materials}
                                      value={field.value || ""}
                                      onChange={field.onChange}
                                      placeholder="Wybierz materiał"
                                      searchPlaceholder="Szukaj materiału..."
                                      testId="select-material"
                                    />
                                  </FormControl>
                                </div>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>

                        {/* Rodzaj + Grupa produktu */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                          <FormField
                            control={form.control}
                            name="productType"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Rodzaj mebla</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={productTypes}
                                    value={field.value || ""}
                                    onChange={field.onChange}
                                    placeholder="Wybierz rodzaj"
                                    searchPlaceholder="Szukaj rodzaju..."
                                    testId="select-product-type"
                                    disabled={bomData?.components?.length > 0}
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <FormField
                            control={form.control}
                            name="productGroup"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Grupa produktu</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={productGroups}
                                    value={field.value || ""}
                                    onChange={field.onChange}
                                    placeholder="Wybierz grupę"
                                    searchPlaceholder="Szukaj grupy..."
                                    testId="select-product-group"
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>

                        {/* Drzwi + Nóżki */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                          <FormField
                            control={form.control}
                            name="doors"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Drzwi</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={doorsOptions}
                                    value={field.value || ""}
                                    onChange={field.onChange}
                                    placeholder="Wybierz drzwi"
                                    searchPlaceholder="Szukaj drzwi..."
                                    testId="select-doors"
                                    disabled={bomData?.components?.length > 0}
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />

                          <FormField
                            control={form.control}
                            name="legs"
                            render={({ field }) => (
                              <FormItem>
                                <FormLabel className="text-xs">Nóżki</FormLabel>
                                <FormControl>
                                  <DictionaryCombobox
                                    items={legsOptions}
                                    value={field.value || ""}
                                    onChange={field.onChange}
                                    placeholder="Wybierz nóżki"
                                    searchPlaceholder="Szukaj nóżek..."
                                    testId="select-legs"
                                    disabled={bomData?.components?.length > 0}
                                  />
                                </FormControl>
                                <FormMessage />
                              </FormItem>
                            )}
                          />
                        </div>

                        {/* Opcje kolorystyczne (edytowalne) */}
                        {!isNew && (() => {
                          const originalOptions = product?.colorOptions || [];
                          const hasUnsavedChanges = JSON.stringify(colorOptionsState.sort()) !== JSON.stringify(originalOptions.sort());
                          
                          return (
                            <div 
                              className={`space-y-2 p-3 rounded-md border ${
                                hasUnsavedChanges 
                                  ? 'bg-destructive/10 border-destructive/50' 
                                  : 'bg-muted/30'
                              }`}
                              data-testid="color-options-section"
                            >
                              <div className="flex items-center justify-between">
                                <p className="font-medium text-sm">Opcje kolorystyczne:</p>
                                <Button
                                  size="sm"
                                  variant="outline"
                                  onClick={() => updateColorOptionsMutation.mutate(colorOptionsState)}
                                  disabled={updateColorOptionsMutation.isPending}
                                  data-testid="button-save-color-options"
                                >
                                  {updateColorOptionsMutation.isPending ? (
                                    <><Loader2 className="w-4 h-4 mr-1 animate-spin" /> Zapisywanie</>
                                  ) : (
                                    <><Save className="w-4 h-4 mr-1" /> Zapisz</>
                                  )}
                                </Button>
                              </div>
                              <div className="flex flex-wrap gap-1.5">
                                {colorOptionsState.map((option, optIdx) => (
                                  <Badge 
                                    key={optIdx} 
                                    variant="secondary"
                                    data-testid={`badge-color-option-${optIdx}`}
                                  >
                                    {option}
                                    <button
                                      className="ml-1 hover:text-destructive"
                                      onClick={() => {
                                        const newOptions = [...colorOptionsState];
                                        newOptions.splice(optIdx, 1);
                                        setColorOptionsState(newOptions);
                                      }}
                                      data-testid={`button-remove-option-${optIdx}`}
                                    >
                                      <X className="w-3 h-3" />
                                    </button>
                                  </Badge>
                                ))}
                                <Popover>
                                  <PopoverTrigger asChild>
                                    <Button
                                      variant="outline"
                                      size="icon"
                                      className="h-7 w-7"
                                      data-testid="button-add-color-option"
                                    >
                                      <Plus className="w-4 h-4" />
                                    </Button>
                                  </PopoverTrigger>
                                  <PopoverContent className="w-80 p-0" align="start">
                                    <Command>
                                      <CommandInput placeholder="Szukaj opcji..." />
                                      <CommandEmpty>Brak wyników</CommandEmpty>
                                      <CommandList>
                                        <CommandGroup heading="Dostępne opcje UNIT-COLOR">
                                          {availableColorOptions
                                            .filter(opt => !colorOptionsState.includes(opt.value))
                                            .map((opt) => (
                                              <CommandItem
                                                key={opt.value}
                                                value={opt.value}
                                                onSelect={(selectedValue) => {
                                                  setColorOptionsState([...colorOptionsState, selectedValue]);
                                                }}
                                              >
                                                {opt.label}
                                              </CommandItem>
                                            ))}
                                          {availableColorOptions.length === 0 && (
                                            <CommandItem disabled>
                                              <span className="text-muted-foreground text-xs italic">
                                                Ładowanie opcji...
                                              </span>
                                            </CommandItem>
                                          )}
                                        </CommandGroup>
                                      </CommandList>
                                    </Command>
                                  </PopoverContent>
                                </Popover>
                              </div>
                              {hasUnsavedChanges && (
                                <p className="text-xs text-destructive font-medium flex items-center gap-1">
                                  <AlertCircle className="w-3 h-3" />
                                  Niezapisane zmiany - kliknij "Zapisz" aby zachować
                                </p>
                              )}
                              <p className="text-xs text-muted-foreground italic">
                                Kliknij "+" aby dodać opcję, "X" aby usunąć, następnie "Zapisz"
                              </p>
                            </div>
                          );
                        })()}
                      </div>
                    </div>

                    {/* Cena + Waluta + Aktywny */}
                    <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-2">
                      <FormField
                        control={form.control}
                        name="basePrice"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel className="text-xs">Cena bazowa</FormLabel>
                            <FormControl>
                              <Input
                                {...field}
                                type="number"
                                step="0.01"
                                className="h-8"
                                data-testid="input-base-price"
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="currency"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel className="text-xs">Waluta</FormLabel>
                            <FormControl>
                              <Input {...field} value={field.value ?? ""} disabled className="h-8" data-testid="input-currency" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="isActive"
                        render={({ field }) => (
                          <FormItem className="flex flex-row items-center justify-between rounded-lg border py-1 px-2">
                            <div className="space-y-0">
                              <FormLabel className="text-xs">Aktywny</FormLabel>
                            </div>
                            <FormControl>
                              <Switch
                                checked={field.value ?? false}
                                onCheckedChange={field.onChange}
                                data-testid="switch-active"
                              />
                            </FormControl>
                          </FormItem>
                        )}
                      />
                    </div>

                    {/* Opisy - na końcu po wszystkich parametrach */}
                    <FormField
                      control={form.control}
                      name="shortDescription"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Krótki opis</FormLabel>
                          <FormControl>
                            <Textarea
                              {...field}
                              value={field.value ?? ""}
                              rows={3}
                              data-testid="textarea-short-description"
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="longDescriptionHtml"
                      render={({ field }) => (
                        <FormItem>
                          <div className="flex flex-col sm:flex-row items-start sm:items-center gap-2 sm:justify-between mb-2">
                            <FormLabel className="flex-shrink-0">Długi opis</FormLabel>
                            <div className="flex flex-wrap gap-1.5 sm:gap-2">
                              {/* Mode Toggle */}
                              <div className="flex gap-1 border rounded-md p-1">
                                <Button
                                  type="button"
                                  variant={descriptionMode === 'editor' ? 'default' : 'ghost'}
                                  size="sm"
                                  onClick={() => setDescriptionMode('editor')}
                                  data-testid="button-mode-editor"
                                  className="h-7 px-2"
                                >
                                  <Eye className="h-3 w-3 mr-1" />
                                  Edytor
                                </Button>
                                <Button
                                  type="button"
                                  variant={descriptionMode === 'code' ? 'default' : 'ghost'}
                                  size="sm"
                                  onClick={() => setDescriptionMode('code')}
                                  data-testid="button-mode-code"
                                  className="h-7 px-2"
                                >
                                  <Code2 className="h-3 w-3 mr-1" />
                                  Kod
                                </Button>
                              </div>
                              
                              <Button
                                type="button"
                                variant="outline"
                                size="sm"
                                onClick={() => setIsTemplatePickerOpen(true)}
                                data-testid="button-use-template"
                              >
                                <Package className="h-4 w-4 mr-2" />
                                Użyj szablonu
                              </Button>
                              <Button
                                type="button"
                                variant="outline"
                                size="sm"
                                onClick={() => setIsPreviewDialogOpen(true)}
                                disabled={!form.watch('longDescriptionHtml')}
                                data-testid="button-preview-description"
                              >
                                <Eye className="h-4 w-4 mr-2" />
                                Podgląd opisu
                              </Button>
                              {!isNew && (
                                <>
                                  <AIGenerationHistory
                                    productId={parseInt(id!)}
                                    onReuse={(description) => {
                                      form.setValue("longDescriptionHtml", description);
                                    }}
                                  />
                                  <AIGenerationDialog
                                    productId={parseInt(id!)}
                                    onGenerated={(description) => {
                                      form.setValue("longDescriptionHtml", description);
                                    }}
                                  />
                                </>
                              )}
                            </div>
                          </div>
                          <FormDescription>
                            {descriptionMode === 'editor' 
                              ? 'Edytor WYSIWYG z możliwością formatowania tekstu, dodawania tabel i obrazów'
                              : 'Tryb kodu HTML - bezpośrednia edycja znaczników HTML'}
                          </FormDescription>
                          <FormControl>
                            {descriptionMode === 'code' ? (
                              <Textarea
                                {...field}
                                rows={10}
                                placeholder="<p>Wprowadź szczegółowy opis produktu...</p>"
                                data-testid="textarea-long-description"
                                className="font-mono text-sm"
                              />
                            ) : (
                              <RichTextEditor
                                content={field.value || ''}
                                onContentChange={(html, json) => {
                                  field.onChange(html);
                                }}
                                placeholder="Wprowadź szczegółowy opis produktu..."
                                enableGalleryPicker={!isNew}
                                enableAddonPicker={!isNew}
                              />
                            )}
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </CardContent>
                </Card>
              </TabsContent>

              {/* Zakładka: Allegro */}
              <TabsContent value="allegro" className="space-y-4">
                <Card>
                  <CardHeader className="flex flex-row items-center justify-between gap-2 sm:gap-4 space-y-0 pb-2">
                    <div>
                      <CardTitle>Allegro - dane specyficzne</CardTitle>
                      <CardDescription>
                        Konfiguracja produktu dla platformy Allegro
                      </CardDescription>
                    </div>
                    <Button
                      type="button"
                      variant="outline"
                      size="sm"
                      onClick={() => copySharedToPlatform("allegro")}
                      data-testid="button-copy-to-allegro"
                    >
                      Kopiuj z Wspólne
                    </Button>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <FormField
                      control={form.control}
                      name="allegro.customTitle"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Tytuł (nadpisany)</FormLabel>
                          <FormControl>
                            <Input {...field} data-testid="input-allegro-title" />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć tytułu wspólnego
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      <FormField
                        control={form.control}
                        name="allegro.categoryId"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>ID kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-allegro-category-id" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="allegro.categoryName"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Nazwa kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-allegro-category-name" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </div>

                    <FormField
                      control={form.control}
                      name="allegro.description"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Opis</FormLabel>
                          <FormControl>
                            <Textarea
                              {...field}
                              rows={10}
                              data-testid="textarea-allegro-description"
                            />
                          </FormControl>
                          <FormDescription>
                            HTML z osadzonymi obrazami i renderami
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="allegro.customPrice"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Cena (nadpisana)</FormLabel>
                          <FormControl>
                            <Input
                              {...field}
                              type="number"
                              step="0.01"
                              data-testid="input-allegro-price"
                            />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć ceny bazowej
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="allegro.isPublished"
                      render={({ field }) => (
                        <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                          <div className="space-y-0.5">
                            <FormLabel className="text-base">Opublikowany</FormLabel>
                            <FormDescription>
                              Czy produkt jest opublikowany na Allegro
                            </FormDescription>
                          </div>
                          <FormControl>
                            <Switch
                              checked={field.value}
                              onCheckedChange={field.onChange}
                              data-testid="switch-allegro-published"
                            />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  </CardContent>
                </Card>
              </TabsContent>

              {/* Zakładka: Shoper */}
              <TabsContent value="shoper" className="space-y-4">
                <Card>
                  <CardHeader className="flex flex-row items-center justify-between gap-2 sm:gap-4 space-y-0 pb-2">
                    <div>
                      <CardTitle>Shoper - dane specyficzne</CardTitle>
                      <CardDescription>
                        Konfiguracja produktu dla platformy Shoper
                      </CardDescription>
                    </div>
                    <Button
                      type="button"
                      variant="outline"
                      size="sm"
                      onClick={() => copySharedToPlatform("shoper")}
                      data-testid="button-copy-to-shoper"
                    >
                      Kopiuj z Wspólne
                    </Button>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <FormField
                      control={form.control}
                      name="shoper.customTitle"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Tytuł (nadpisany)</FormLabel>
                          <FormControl>
                            <Input {...field} data-testid="input-shoper-title" />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć tytułu wspólnego
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      <FormField
                        control={form.control}
                        name="shoper.categoryId"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>ID kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-shoper-category-id" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="shoper.categoryName"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Nazwa kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-shoper-category-name" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </div>

                    <FormField
                      control={form.control}
                      name="shoper.description"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Opis</FormLabel>
                          <FormControl>
                            <Textarea
                              {...field}
                              rows={10}
                              data-testid="textarea-shoper-description"
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="shoper.customPrice"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Cena (nadpisana)</FormLabel>
                          <FormControl>
                            <Input
                              {...field}
                              type="number"
                              step="0.01"
                              data-testid="input-shoper-price"
                            />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć ceny bazowej
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="shoper.isPublished"
                      render={({ field }) => (
                        <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                          <div className="space-y-0.5">
                            <FormLabel className="text-base">Opublikowany</FormLabel>
                            <FormDescription>
                              Czy produkt jest opublikowany w Shoper
                            </FormDescription>
                          </div>
                          <FormControl>
                            <Switch
                              checked={field.value}
                              onCheckedChange={field.onChange}
                              data-testid="switch-shoper-published"
                            />
                          </FormControl>
                        </FormItem>
                      )}
                    />
                  </CardContent>
                </Card>
              </TabsContent>

              {/* Zakładka: Odoo */}
              <TabsContent value="odoo" className="space-y-4">
                <Card>
                  <CardHeader className="flex flex-row items-start justify-between gap-2 sm:gap-4 space-y-0 pb-2">
                    <div>
                      <CardTitle>Odoo - dane specyficzne</CardTitle>
                      <CardDescription>
                        Konfiguracja produktu dla systemu Odoo
                      </CardDescription>
                    </div>
                    <div className="flex items-center gap-2">
                      <Button
                        type="button"
                        variant="outline"
                        size="sm"
                        onClick={() => copySharedToPlatform("odoo")}
                        data-testid="button-copy-to-odoo"
                      >
                        Kopiuj z Wspólne
                      </Button>
                      {!isNew && product && (
                        <>
                          {product.odooProductId && odooConfig?.userUrl && (
                            <Button
                              size="icon"
                              variant="outline"
                              onClick={() => window.open(`${odooConfig.userUrl}/products/${product.odooProductId}`, '_blank')}
                              data-testid="button-odoo-link-tab"
                              title="Otwórz produkt w Odoo"
                              className="bg-purple-600 hover:bg-purple-700 border-purple-600 text-white"
                            >
                              <ExternalLink className="h-4 w-4" />
                            </Button>
                          )}
                          <Button
                            size="icon"
                            variant={product.odooProductId ? "default" : "destructive"}
                            onClick={() => exportProductMutation.mutate()}
                            disabled={exportProductMutation.isPending}
                            data-testid="button-export-odoo-tab"
                            title={
                              product.odooExportError 
                                ? `Błąd eksportu: ${product.odooExportError}` 
                                : product.odooProductId 
                                  ? `Wyeksportowano: ${product.odooExportedAt ? new Date(product.odooExportedAt).toLocaleString('pl-PL') : ''}` 
                                  : "Nie wyeksportowano - kliknij aby wysłać do Odoo"
                            }
                            className={
                              product.odooExportError 
                                ? "bg-yellow-500 hover:bg-yellow-600 relative" 
                                : product.odooProductId 
                                  ? "bg-green-600 hover:bg-green-700" 
                                  : ""
                            }
                          >
                            {exportProductMutation.isPending ? (
                              <Loader2 className="h-4 w-4 animate-spin" />
                            ) : (
                              <Upload className="h-4 w-4" />
                            )}
                            {product.odooExportError && (
                              <span className="absolute -top-1 -right-1 flex h-3 w-3">
                                <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-red-400 opacity-75"></span>
                                <span className="relative inline-flex rounded-full h-3 w-3 bg-red-500"></span>
                              </span>
                            )}
                          </Button>
                        </>
                      )}
                    </div>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    {/* Informacja o statusie eksportu */}
                    {!isNew && product && (
                      <div className={`rounded-lg border p-4 ${
                        product.odooExportError 
                          ? 'bg-yellow-50 dark:bg-yellow-950/20 border-yellow-500' 
                          : product.odooProductId 
                            ? 'bg-green-50 dark:bg-green-950/20 border-green-500' 
                            : 'bg-muted/50'
                      }`}>
                        <div className="flex items-start justify-between gap-4">
                          <div className="flex-1">
                            {product.odooExportError ? (
                              <>
                                <div className="font-medium mb-1 text-yellow-700 dark:text-yellow-400">
                                  ⚠️ Błąd eksportu do Odoo
                                </div>
                                <div className="text-sm text-yellow-600 dark:text-yellow-500">
                                  {product.odooExportError}
                                </div>
                                <div className="text-xs text-muted-foreground mt-2">
                                  Kliknij przycisk "Wyślij do Odoo" aby spróbować ponownie lub otwórz produkt w Odoo i popraw ręcznie.
                                </div>
                              </>
                            ) : product.odooProductId ? (
                              <>
                                <div className="font-medium mb-1 text-green-700 dark:text-green-400">
                                  ✅ Produkt wyeksportowany do Odoo
                                </div>
                                <div className="text-sm text-muted-foreground">
                                  ID: {product.odooProductId}
                                  {product.odooExportedAt && (
                                    <span className="ml-2">
                                      • Ostatni eksport: {new Date(product.odooExportedAt).toLocaleString('pl-PL')}
                                    </span>
                                  )}
                                </div>
                              </>
                            ) : (
                              <>
                                <div className="font-medium mb-1">Produkt nie jest zsynchronizowany z Odoo</div>
                                <div className="text-sm text-muted-foreground">
                                  Kliknij przycisk "Wyślij do Odoo" aby wyeksportować produkt.
                                </div>
                              </>
                            )}
                          </div>
                        </div>
                      </div>
                    )}

                    <FormField
                      control={form.control}
                      name="odoo.customTitle"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Nazwa produktu (nadpisana)</FormLabel>
                          <FormControl>
                            <Input {...field} data-testid="input-odoo-title" />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć tytułu wspólnego
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      <FormField
                        control={form.control}
                        name="odoo.categoryId"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>ID kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-odoo-category-id" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="odoo.categoryName"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Nazwa kategorii</FormLabel>
                            <FormControl>
                              <Input {...field} data-testid="input-odoo-category-name" />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </div>

                    <FormField
                      control={form.control}
                      name="odoo.description"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Opis</FormLabel>
                          <FormControl>
                            <Textarea
                              {...field}
                              rows={5}
                              data-testid="textarea-odoo-description"
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="odoo.customPrice"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Cena (nadpisana)</FormLabel>
                          <FormControl>
                            <Input
                              {...field}
                              type="number"
                              step="0.01"
                              data-testid="input-odoo-price"
                            />
                          </FormControl>
                          <FormDescription>
                            Zostaw puste aby użyć ceny bazowej
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="odoo.isPublished"
                      render={({ field }) => (
                        <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                          <div className="space-y-0.5">
                            <FormLabel className="text-base">Zsynchronizowany</FormLabel>
                            <FormDescription>
                              Czy produkt jest zsynchronizowany z Odoo
                            </FormDescription>
                          </div>
                          <FormControl>
                            <Switch
                              checked={field.value}
                              onCheckedChange={field.onChange}
                              data-testid="switch-odoo-published"
                            />
                          </FormControl>
                        </FormItem>
                      )}
                    />

                  </CardContent>
                </Card>
              </TabsContent>

              {/* Zakładka: Galeria */}
              <TabsContent value="gallery" className="space-y-4">
                {!isNew && <ProductGallery productId={parseInt(id!)} />}
                {isNew && (
                  <Card>
                    <CardContent className="pt-6">
                      <div className="text-center text-muted-foreground py-12">
                        <p>Zapisz najpierw produkt, aby dodać zdjęcia</p>
                      </div>
                    </CardContent>
                  </Card>
                )}
              </TabsContent>

              {/* Zakładka: BOM */}
              <TabsContent value="bom" className="space-y-4">
                {!isNew ? (
                  <div className="space-y-4">
                    <Card>
                      <CardHeader>
                        <div className="flex items-center justify-between gap-2">
                          <div>
                            <CardTitle>Bill of Materials</CardTitle>
                            <CardDescription>
                              Lista komponentów wymaganych do produkcji tego produktu
                            </CardDescription>
                          </div>
                          <div className="flex gap-2 flex-wrap">
                            {product?.productType && (
                              <Link href={`/matrix/${product.productType.split('-')[0]}`}>
                                <Button
                                  variant="outline"
                                  size="sm"
                                  data-testid="button-bom-matrices"
                                  className="flex-shrink-0"
                                >
                                  <Package className="h-4 w-4 mr-2" />
                                  Matryce BOM
                                </Button>
                              </Link>
                            )}
                            <Dialog>
                              <DialogTrigger asChild>
                                <Button
                                  variant="outline"
                                  size="icon"
                                  data-testid="button-bom-info"
                                  className="flex-shrink-0"
                                >
                                  <HelpCircle className="h-4 w-4" />
                                </Button>
                              </DialogTrigger>
                              <DialogContent className="max-w-3xl max-h-[80vh] overflow-y-auto">
                                <DialogHeader>
                                  <DialogTitle className="flex items-center gap-2">
                                    <Package className="h-5 w-5" />
                                    Jak działa generowanie BOM?
                                  </DialogTitle>
                                  <DialogDescription>
                                    Szczegółowe wyjaśnienie logiki generowania formatek w systemie matrycowym
                                  </DialogDescription>
                                </DialogHeader>

                                <div className="space-y-6 text-sm">
                                  {/* Krok 1: Parametry produktu */}
                                  <div className="space-y-2">
                                    <h3 className="font-semibold text-base flex items-center gap-2">
                                      <span className="flex items-center justify-center w-6 h-6 rounded-full bg-primary text-primary-foreground text-xs">1</span>
                                      Parametry produktu
                                    </h3>
                                    <p className="text-muted-foreground">
                                      System analizuje parametry produktu z zakładki "Podstawowe":
                                    </p>
                                    <ul className="list-disc list-inside space-y-1 ml-4 text-muted-foreground">
                                      <li><strong>Rodzaj mebla</strong> (productType) - np. "Szafka TRES"</li>
                                      <li><strong>Wymiary</strong> - Długość, Szerokość, Wysokość (w cm)</li>
                                      <li><strong>Kolor</strong> - np. "BIAŁY", "SZARY"</li>
                                      <li><strong>Drzwi</strong> (doors) - np. "D1", "D2" (opcjonalnie)</li>
                                      <li><strong>Nóżki</strong> (legs) - np. "N1", "N2" (opcjonalnie)</li>
                                    </ul>
                                  </div>

                                  {/* Krok 2: Mechanizm matrycowy */}
                                  <div className="space-y-2">
                                    <h3 className="font-semibold text-base flex items-center gap-2">
                                      <span className="flex items-center justify-center w-6 h-6 rounded-full bg-primary text-primary-foreground text-xs">2</span>
                                      Mechanizm matrycowy - ekstrakcja D1/N1 z nazwy
                                    </h3>
                                    <p className="text-muted-foreground">
                                      System automatycznie wydobywa parametry D1 i N1 z tytułu produktu:
                                    </p>
                                    <div className="bg-muted/50 p-4 rounded-lg space-y-2">
                                      <p className="font-mono text-xs">
                                        <strong>Przykład:</strong> "Szafka TRES na buty z szufladą TRES 50×30×82 cm 3 półki i szuflada BIAŁY"
                                      </p>
                                      <p className="text-muted-foreground">
                                        → System rozpoznaje wzorce:
                                      </p>
                                      <ul className="list-disc list-inside space-y-1 ml-4 text-xs">
                                        <li><code>50×30×82</code> lub <code>50x30x82</code> → wymiary w cm</li>
                                        <li><code>3 półki</code> → N1 = "N3"</li>
                                        <li><code>i szuflada</code> → D1 = "D1"</li>
                                      </ul>
                                    </div>
                                  </div>

                                  {/* Krok 3: Wyszukiwanie szablonów */}
                                  <div className="space-y-2">
                                    <h3 className="font-semibold text-base flex items-center gap-2">
                                      <span className="flex items-center justify-center w-6 h-6 rounded-full bg-primary text-primary-foreground text-xs">3</span>
                                      Wyszukiwanie szablonów formatek
                                    </h3>
                                    <p className="text-muted-foreground">
                                      System przeszukuje bazę szablonów formatek (bom.component_templates) według kryteriów:
                                    </p>
                                    <div className="bg-blue-500/10 border border-blue-500/30 p-4 rounded-lg space-y-2">
                                      <p className="font-semibold text-blue-700 dark:text-blue-300">Wymagane dopasowania:</p>
                                      <ul className="list-disc list-inside space-y-1 ml-4">
                                        <li><strong>furniture_type</strong> = rodzaj mebla z produktu</li>
                                        <li><strong>cz1</strong> = CZ1 rozpoznane z nazwy (np. BOCZKI-L, BOK-P)</li>
                                        <li><strong>cz2</strong> = CZ2 z nazwy lub dodatkowe kryteria</li>
                                        <li><strong>color</strong> = kolor produktu (opcjonalnie)</li>
                                        <li><strong>door</strong> = D1 z parametrów (jeśli szablon tego wymaga)</li>
                                        <li><strong>leg</strong> = N1 z parametrów (jeśli szablon tego wymaga)</li>
                                      </ul>
                                    </div>
                                  </div>

                                  {/* Krok 4: Kalkulacja wymiarów */}
                                  <div className="space-y-2">
                                    <h3 className="font-semibold text-base flex items-center gap-2">
                                      <span className="flex items-center justify-center w-6 h-6 rounded-full bg-primary text-primary-foreground text-xs">4</span>
                                      Kalkulacja wymiarów formatki
                                    </h3>
                                    <p className="text-muted-foreground">
                                      Dla każdego dopasowanego szablonu system oblicza finalne wymiary:
                                    </p>
                                    <div className="bg-muted/50 p-4 rounded-lg space-y-2">
                                      <p className="font-semibold">Formuły kalkulacji:</p>
                                      <ul className="list-disc list-inside space-y-1 ml-4 font-mono text-xs">
                                        <li>calculated_length = base_length + (length_offset × Długość produktu)</li>
                                        <li>calculated_width = base_width + (width_offset × Szerokość produktu)</li>
                                      </ul>
                                      <p className="text-muted-foreground text-xs mt-2">
                                        <strong>Przykład:</strong> Jeśli szablon ma base_length=500mm i length_offset=0, 
                                        to formatka będzie miała 500mm niezależnie od wymiarów produktu.
                                      </p>
                                    </div>
                                  </div>

                                  {/* Krok 5: Generowanie nazwy */}
                                  <div className="space-y-2">
                                    <h3 className="font-semibold text-base flex items-center gap-2">
                                      <span className="flex items-center justify-center w-6 h-6 rounded-full bg-primary text-primary-foreground text-xs">5</span>
                                      Generowanie nazwy komponentu
                                    </h3>
                                    <p className="text-muted-foreground">
                                      Każdy komponent otrzymuje unikalną nazwę według wzorca:
                                    </p>
                                    <div className="bg-purple-500/10 border border-purple-500/30 p-4 rounded-lg">
                                      <p className="font-mono text-sm font-semibold">
                                        CZ1-CZ2-długość-szerokość-rodzajMebla-kolor
                                      </p>
                                      <p className="text-muted-foreground text-xs mt-2">
                                        <strong>Przykład:</strong> BOCZKI-L-SZUFLADA-760-280-TRES-BIALY
                                      </p>
                                    </div>
                                  </div>

                                  {/* Krok 6: Ilość i zapis */}
                                  <div className="space-y-2">
                                    <h3 className="font-semibold text-base flex items-center gap-2">
                                      <span className="flex items-center justify-center w-6 h-6 rounded-full bg-primary text-primary-foreground text-xs">6</span>
                                      Ilość i zapis do bazy
                                    </h3>
                                    <p className="text-muted-foreground">
                                      System zapisuje każdy komponent z jego parametrami:
                                    </p>
                                    <ul className="list-disc list-inside space-y-1 ml-4 text-muted-foreground">
                                      <li><strong>component_template_id</strong> - referencja do szablonu źródłowego</li>
                                      <li><strong>quantity</strong> - ilość sztuk (domyślnie 1, chyba że szablon określa inaczej)</li>
                                      <li><strong>calculated_length</strong> i <strong>calculated_width</strong> - finalne wymiary</li>
                                      <li><strong>generated_name</strong> - wygenerowana nazwa według wzorca</li>
                                      <li>Parametry oklejenia: edge1, edge2, edge3, edge4</li>
                                      <li>Materiał płyty, kolor, grubość</li>
                                    </ul>
                                  </div>

                                  {/* Dlaczego tak działa? */}
                                  <div className="space-y-2 border-t pt-4">
                                    <h3 className="font-semibold text-base">💡 Dlaczego system matrycowy?</h3>
                                    <ul className="list-disc list-inside space-y-1 ml-4 text-muted-foreground">
                                      <li><strong>Skalowalność</strong> - jeden szablon obsługuje wiele wariantów produktu</li>
                                      <li><strong>Konsystencja</strong> - wszystkie szafki tego samego typu mają takie same komponenty</li>
                                      <li><strong>Łatwość aktualizacji</strong> - zmiana szablonu propaguje się na wszystkie produkty</li>
                                      <li><strong>Automatyzacja</strong> - nie trzeba ręcznie definiować BOM dla każdego produktu</li>
                                    </ul>
                                  </div>

                                  {/* Przykład kompletny */}
                                  <div className="space-y-2 bg-green-500/10 border border-green-500/30 p-4 rounded-lg">
                                    <h3 className="font-semibold">📋 Przykład kompletny:</h3>
                                    <p className="text-xs text-muted-foreground">
                                      Produkt: "Szafka TRES 50×30×82 cm 3 półki i szuflada BIAŁY"
                                    </p>
                                    <div className="space-y-1 text-xs">
                                      <p>1. Rozpoznane parametry: Rodzaj=TRES, Długość=50, Szerokość=30, Wysokość=82, D1=D1, N1=N3, Kolor=BIAŁY</p>
                                      <p>2. Znalezione szablony: BOCZKI-L, BOK-P, DRZWI-L, FRONT, POLKA-1, POLKA-2, itd.</p>
                                      <p>3. Każdy szablon ma swoje wymiary bazowe i offsety</p>
                                      <p>4. System generuje 23 komponenty z wyliczonymi wymiarami</p>
                                      <p>5. Wszystkie komponenty są zapisywane w tabeli bom.product_components</p>
                                    </div>
                                  </div>
                                </div>
                              </DialogContent>
                            </Dialog>

                            <Button
                              onClick={() => generateBomMutation.mutate()}
                              disabled={generateBomMutation.isPending}
                              data-testid="button-generate-bom-tab"
                              size="default"
                              className="bg-green-600 hover:bg-green-700 text-white"
                            >
                              {generateBomMutation.isPending ? (
                                <>
                                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                                  Generowanie...
                                </>
                              ) : (
                                <>
                                  <Package className="h-4 w-4 mr-2" />
                                  {bomData ? "Regeneruj BOM" : "Generuj BOM"}
                                </>
                              )}
                            </Button>
                            {bomData && (
                              <>
                                {product?.odooProductId && odooConfig?.userUrl && (
                                  <Button
                                    size="default"
                                    variant="outline"
                                    onClick={() => window.open(`${odooConfig.userUrl}/products/${product.odooProductId}`, '_blank')}
                                    data-testid="button-odoo-product-link-bom"
                                    title="Otwórz produkt w Odoo"
                                    className="bg-purple-600 hover:bg-purple-700 border-purple-600 text-white"
                                  >
                                    <ExternalLink className="h-4 w-4" />
                                  </Button>
                                )}
                                <Button
                                  size="default"
                                  variant={(bomData.bom as any)?.odoo_bom_id ? "default" : "destructive"}
                                  onClick={() => exportBomMutation.mutate()}
                                  disabled={exportBomMutation.isPending || !bomData}
                                  data-testid="button-export-bom-odoo"
                                  title={
                                    (bomData.bom as any)?.odoo_sync_error
                                      ? `Błąd eksportu BOM: ${(bomData.bom as any).odoo_sync_error}`
                                      : (bomData.bom as any)?.odoo_bom_id
                                        ? `BOM wyeksportowany do Odoo (ID: ${(bomData.bom as any).odoo_bom_id})`
                                        : "Nie wyeksportowano - kliknij aby wysłać do Odoo"
                                  }
                                  className={
                                    (bomData.bom as any)?.odoo_sync_error
                                      ? "bg-yellow-500 hover:bg-yellow-600 relative"
                                      : (bomData.bom as any)?.odoo_bom_id
                                        ? "bg-green-600 hover:bg-green-700"
                                        : ""
                                  }
                                >
                                  {exportBomMutation.isPending ? (
                                    <Loader2 className="h-4 w-4 animate-spin" />
                                  ) : (
                                    <Upload className="h-4 w-4" />
                                  )}
                                  {(bomData.bom as any)?.odoo_sync_error && (
                                    <span className="absolute -top-1 -right-1 flex h-3 w-3">
                                      <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-red-400 opacity-75"></span>
                                      <span className="relative inline-flex rounded-full h-3 w-3 bg-red-500"></span>
                                    </span>
                                  )}
                                </Button>
                                <Button
                                  onClick={() => {
                                    if (window.confirm('Czy na pewno chcesz usunąć BOM? Ta operacja jest nieodwracalna.')) {
                                      deleteBomMutation.mutate();
                                    }
                                  }}
                                  disabled={deleteBomMutation.isPending || !bomData}
                                  data-testid="button-delete-bom"
                                  size="default"
                                  variant="destructive"
                                >
                                  {deleteBomMutation.isPending ? (
                                    <>
                                      <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                                      Usuwanie...
                                    </>
                                  ) : (
                                    <>
                                      <Trash2 className="h-4 w-4 mr-2" />
                                      Usuń BOM
                                    </>
                                  )}
                                </Button>
                              </>
                            )}
                          </div>
                        </div>
                      </CardHeader>
                      {bomData && (
                        <CardContent>
                          <div className="space-y-4">
                            {/* Informacja o statusie eksportu BOM */}
                            <div className={`rounded-lg border p-4 ${
                              (bomData.bom as any)?.odoo_sync_error 
                                ? 'bg-yellow-50 dark:bg-yellow-950/20 border-yellow-500' 
                                : (bomData.bom as any)?.odoo_bom_id 
                                  ? 'bg-green-50 dark:bg-green-950/20 border-green-500' 
                                  : 'bg-muted/50'
                            }`}>
                              <div className="flex items-start justify-between gap-4">
                                <div className="flex-1">
                                  {(bomData.bom as any)?.odoo_sync_error ? (
                                    <>
                                      <div className="font-medium mb-1 text-yellow-700 dark:text-yellow-400">
                                        ⚠️ Błąd eksportu BOM do Odoo
                                      </div>
                                      <div className="text-sm text-yellow-600 dark:text-yellow-500">
                                        {(bomData.bom as any).odoo_sync_error}
                                      </div>
                                      <div className="text-xs text-muted-foreground mt-2">
                                        Kliknij przycisk "Eksportuj do Odoo" aby spróbować ponownie lub dodaj brakujące wartości do pola "Rodzaj formatki" w Odoo.
                                      </div>
                                    </>
                                  ) : (bomData.bom as any)?.odoo_bom_id ? (
                                    <>
                                      <div className="font-medium mb-1 text-green-700 dark:text-green-400">
                                        ✅ BOM wyeksportowany do Odoo
                                      </div>
                                      <div className="text-sm text-muted-foreground">
                                        ID: {(bomData.bom as any).odoo_bom_id}
                                        {(bomData.bom as any)?.odoo_synced_at && (
                                          <span className="ml-2">
                                            • Ostatni eksport: {new Date((bomData.bom as any).odoo_synced_at).toLocaleString('pl-PL')}
                                          </span>
                                        )}
                                      </div>
                                    </>
                                  ) : (
                                    <>
                                      <div className="font-medium mb-1">BOM nie jest zsynchronizowany z Odoo</div>
                                      <div className="text-sm text-muted-foreground">
                                        Kliknij przycisk "Wyślij do Odoo" aby wyeksportować BOM.
                                      </div>
                                    </>
                                  )}
                                </div>
                              </div>
                            </div>
                            
                            <div className="flex items-center gap-4 text-sm">
                              <Badge variant="outline">
                                Wersja: {bomData.bom?.version || "1.0"}
                              </Badge>
                              {(bomData.bom as any)?.is_approved && (
                                <Badge variant="default">Zatwierdzony</Badge>
                              )}
                              {(bomData.bom as any)?.odoo_bom_id && (
                                <Badge variant="secondary">
                                  Odoo ID: {(bomData.bom as any).odoo_bom_id}
                                </Badge>
                              )}
                            </div>

                            {/* BOM Groups - Kompaktowy widok */}
                            <div className="space-y-3">
                              {/* Formatki Group */}
                              <Collapsible defaultOpen={true}>
                                <div className="border rounded-lg">
                                  <CollapsibleTrigger className="w-full">
                                    <div className="flex items-center justify-between p-3 hover-elevate">
                                      <div className="flex items-center gap-3">
                                        <Package className="w-4 h-4 text-primary flex-shrink-0" />
                                        <h4 className="font-medium text-sm">Formatki</h4>
                                        <Badge variant="outline" className="text-xs">
                                          {bomData.components?.length || 0}
                                        </Badge>
                                      </div>
                                      <ChevronDown className="w-4 h-4 text-muted-foreground transition-transform ui-state-open:rotate-180" />
                                    </div>
                                  </CollapsibleTrigger>
                                  
                                  <CollapsibleContent>
                                    {bomData.components && bomData.components.length > 0 ? (
                                      <div className="border-t overflow-x-auto w-full">
                                        <Table>
                                          <TableHeader>
                                            <TableRow>
                                              <TableHead className="sticky left-0 z-20 bg-background border-r w-16 text-center">#</TableHead>
                                              <TableHead className="sticky left-16 z-20 bg-background border-r w-20">ID</TableHead>
                                              <TableHead className="w-20">Zdjęcie</TableHead>
                                              <TableHead className="min-w-[200px]">Nazwa</TableHead>
                                              <TableHead className="w-24">Kolor</TableHead>
                                              <TableHead className="w-20 text-right">CZ1</TableHead>
                                              <TableHead className="w-20 text-right">CZ2</TableHead>
                                              <TableHead className="w-20 text-right">L</TableHead>
                                              <TableHead className="w-20 text-right">W</TableHead>
                                              <TableHead className="w-20 text-right">H</TableHead>
                                              <TableHead className="w-24">Typ</TableHead>
                                              <TableHead className="min-w-[150px]">Płyta</TableHead>
                                              <TableHead className="min-w-[150px]">Obrzeże</TableHead>
                                              <TableHead className="w-24 text-center">Krawędzie</TableHead>
                                              <TableHead className="w-24 text-right">Koszt</TableHead>
                                              <TableHead className="min-w-[150px]">Produkt</TableHead>
                                              <TableHead className="w-20">Zdjęcie prod.</TableHead>
                                              <TableHead className="w-32">Utworzono</TableHead>
                                              <TableHead className="w-32">Zaktualizowano</TableHead>
                                            </TableRow>
                                          </TableHeader>
                                          <TableBody>
                                            {bomData.components.map((component: ProductComponentWithMetrics, index: number) => (
                                              <TableRow
                                                key={component.id || index}
                                                className={component.id ? "cursor-pointer hover-elevate" : "opacity-50"}
                                                onClick={() => {
                                                  if (component.id) {
                                                    navigate(`/bom-components/${component.id}`);
                                                  }
                                                }}
                                                data-testid={`row-bom-component-${index}`}
                                              >
                                                {/* # - Sticky */}
                                                <TableCell className="sticky left-0 z-10 bg-background border-r w-16 text-center font-bold text-muted-foreground">
                                                  {component.positionInBom ?? index + 1}
                                                </TableCell>

                                                {/* ID - Sticky */}
                                                <TableCell className="sticky left-16 z-10 bg-background border-r w-20">
                                                  {component.id && (
                                                    <Badge variant="secondary" className="text-xs font-mono">
                                                      {component.id}
                                                    </Badge>
                                                  )}
                                                </TableCell>

                                                {/* Zdjęcie - ComponentTemplateThumbnail */}
                                                <TableCell>
                                                  <ComponentTemplateThumbnail
                                                    baseLength={(component as any).base_length ?? 100}
                                                    baseWidth={(component as any).base_width ?? 100}
                                                    edge1={(component as any).edge1 ?? false}
                                                    edge2={(component as any).edge2 ?? false}
                                                    edge3={(component as any).edge3 ?? false}
                                                    edge4={(component as any).edge4 ?? false}
                                                    size="xs"
                                                  />
                                                </TableCell>

                                                {/* Nazwa */}
                                                <TableCell className="font-mono text-sm">
                                                  <div className="flex items-center gap-2">
                                                    <span className="font-medium">{(component as any).generated_name || '-'}</span>
                                                    {(component as any).odoo_product_id && odooConfig?.userUrl && (
                                                      <a
                                                        href={`${odooConfig.userUrl.replace(/\/$/, '')}/products/${(component as any).odoo_product_id}`}
                                                        target="_blank"
                                                        rel="noopener noreferrer"
                                                        onClick={(e) => e.stopPropagation()}
                                                        className="text-purple-600 dark:text-purple-400 hover:text-purple-700 dark:hover:text-purple-300"
                                                        data-testid={`link-formatka-odoo-${index}`}
                                                      >
                                                        <ExternalLink className="w-3.5 h-3.5" />
                                                      </a>
                                                    )}
                                                  </div>
                                                </TableCell>

                                                {/* Kolor */}
                                                <TableCell>
                                                  {(component as any).color ? (() => {
                                                    const bgColor = (component as any).color_hex || '#94A3B8';
                                                    const textColor = getTextColorForBackground(bgColor);
                                                    const hasBorder = needsVisibleBorder(bgColor);
                                                    return (
                                                      <Badge 
                                                        variant="outline" 
                                                        className={`text-xs h-5 px-1.5 ${hasBorder ? '' : 'border-transparent'}`}
                                                        style={{ backgroundColor: bgColor, color: textColor }}
                                                      >
                                                        {(component as any).color}
                                                      </Badge>
                                                    );
                                                  })() : '-'}
                                                </TableCell>

                                                {/* CZ1 */}
                                                <TableCell className="text-right font-mono text-sm">
                                                  {(component as any).cz1 || '-'}
                                                </TableCell>

                                                {/* CZ2 */}
                                                <TableCell className="text-right font-mono text-sm">
                                                  {(component as any).cz2 || '-'}
                                                </TableCell>

                                                {/* L - Długość */}
                                                <TableCell className="text-right font-mono text-sm">
                                                  {(component as any).base_length ?? '-'}
                                                </TableCell>

                                                {/* W - Szerokość */}
                                                <TableCell className="text-right font-mono text-sm">
                                                  {(component as any).base_width ?? '-'}
                                                </TableCell>

                                                {/* H - Grubość */}
                                                <TableCell className="text-right font-mono text-sm">
                                                  {(component as any).thickness ?? '-'}
                                                </TableCell>

                                                {/* Typ */}
                                                <TableCell className="text-sm">
                                                  {(component as any).component_type || '-'}
                                                </TableCell>

                                                {/* Płyta - plate_name */}
                                                <TableCell className="text-sm">
                                                  {(component as any).plate_name ? (
                                                    <span className="text-primary">{(component as any).plate_name}</span>
                                                  ) : (
                                                    <span className="text-muted-foreground">-</span>
                                                  )}
                                                </TableCell>

                                                {/* Obrzeże - edge_name */}
                                                <TableCell className="text-sm">
                                                  {(component as any).edge_name ? (
                                                    <span className="text-primary">{(component as any).edge_name}</span>
                                                  ) : (
                                                    <span className="text-muted-foreground">-</span>
                                                  )}
                                                </TableCell>

                                                {/* Krawędzie - edge1-4 visualization */}
                                                <TableCell className="text-center">
                                                  {((component as any).edge1 || (component as any).edge2 || (component as any).edge3 || (component as any).edge4) ? (
                                                    <Badge variant="secondary" className="text-xs font-mono">
                                                      {[
                                                        (component as any).edge1 && '1',
                                                        (component as any).edge2 && '2',
                                                        (component as any).edge3 && '3',
                                                        (component as any).edge4 && '4'
                                                      ].filter(Boolean).join(',')}
                                                    </Badge>
                                                  ) : '-'}
                                                </TableCell>

                                                {/* Koszt - total_material_cost */}
                                                <TableCell className="text-right font-mono text-sm">
                                                  {(component as any).total_material_cost != null ? (
                                                    <span className="text-green-600 dark:text-green-400 font-medium">
                                                      {(component as any).total_material_cost.toFixed(2)} zł
                                                    </span>
                                                  ) : (
                                                    <span className="text-muted-foreground">-</span>
                                                  )}
                                                </TableCell>

                                                {/* Produkt - parent product title */}
                                                <TableCell className="text-sm">
                                                  {bomData.product?.title ? (
                                                    <Link
                                                      href={`/catalog-products/${bomData.product.id}`}
                                                      onClick={(e) => e.stopPropagation()}
                                                      className="text-primary hover:underline"
                                                    >
                                                      {bomData.product.title}
                                                    </Link>
                                                  ) : (
                                                    <span className="text-muted-foreground">-</span>
                                                  )}
                                                </TableCell>

                                                {/* Zdjęcie prod. - parent product image */}
                                                <TableCell>
                                                  {bomData.product?.imageUrl ? (
                                                    <img
                                                      src={bomData.product.imageUrl}
                                                      alt={bomData.product.title || 'Product'}
                                                      className="w-12 h-12 object-cover rounded border"
                                                    />
                                                  ) : (
                                                    <div className="w-12 h-12 bg-muted rounded border flex items-center justify-center">
                                                      <Package className="w-4 h-4 text-muted-foreground" />
                                                    </div>
                                                  )}
                                                </TableCell>

                                                {/* Created At */}
                                                <TableCell className="text-xs text-muted-foreground">
                                                  {(component as any).created_at ? new Date((component as any).created_at).toLocaleDateString('pl-PL', { 
                                                    year: '2-digit', month: '2-digit', day: '2-digit', 
                                                    hour: '2-digit', minute: '2-digit' 
                                                  }) : '-'}
                                                </TableCell>

                                                {/* Updated At */}
                                                <TableCell className="text-xs text-muted-foreground">
                                                  {(component as any).updated_at ? new Date((component as any).updated_at).toLocaleDateString('pl-PL', { 
                                                    year: '2-digit', month: '2-digit', day: '2-digit', 
                                                    hour: '2-digit', minute: '2-digit' 
                                                  }) : '-'}
                                                </TableCell>
                                              </TableRow>
                                            ))}
                                          </TableBody>
                                        </Table>
                                      </div>
                                    ) : (
                                      <div className="p-4 text-sm text-muted-foreground text-center border-t">
                                        Brak formatek
                                      </div>
                                    )}
                                  </CollapsibleContent>
                                </div>
                              </Collapsible>

                              {/* Placeholder groups - collapsed by default */}
                              <Collapsible defaultOpen={false}>
                                <div className="border rounded-lg">
                                  <CollapsibleTrigger className="w-full">
                                    <div className="flex items-center justify-between p-3 hover-elevate">
                                      <div className="flex items-center gap-3">
                                        <Wrench className="w-4 h-4 text-muted-foreground flex-shrink-0" />
                                        <h4 className="font-medium text-sm text-muted-foreground">Okucia</h4>
                                        <Badge variant="outline" className="text-xs">0</Badge>
                                      </div>
                                      <ChevronDown className="w-4 h-4 text-muted-foreground transition-transform ui-state-open:rotate-180" />
                                    </div>
                                  </CollapsibleTrigger>
                                  <CollapsibleContent>
                                    <div className="p-4 text-sm text-muted-foreground text-center border-t">
                                      Wkrótce
                                    </div>
                                  </CollapsibleContent>
                                </div>
                              </Collapsible>

                              <Collapsible defaultOpen={false}>
                                <div className="border rounded-lg">
                                  <CollapsibleTrigger className="w-full">
                                    <div className="flex items-center justify-between p-3 hover-elevate">
                                      <div className="flex items-center gap-3">
                                        <Package className="w-4 h-4 text-muted-foreground flex-shrink-0" />
                                        <h4 className="font-medium text-sm text-muted-foreground">Opakowania</h4>
                                        <Badge variant="outline" className="text-xs">0</Badge>
                                      </div>
                                      <ChevronDown className="w-4 h-4 text-muted-foreground transition-transform ui-state-open:rotate-180" />
                                    </div>
                                  </CollapsibleTrigger>
                                  <CollapsibleContent>
                                    <div className="p-4 text-sm text-muted-foreground text-center border-t">
                                      Wkrótce
                                    </div>
                                  </CollapsibleContent>
                                </div>
                              </Collapsible>
                            </div>
                          </div>
                        </CardContent>
                      )}
                      {!bomData && (
                        <CardContent className="pt-6">
                          <div className="space-y-6">
                            {/* Check if product has furniture type */}
                            {!product?.productType ? (
                              <div className="text-center text-muted-foreground py-8">
                                <div className="rounded-lg border border-yellow-200 dark:border-yellow-800 bg-yellow-50 dark:bg-yellow-950/20 p-6">
                                  <p className="text-sm font-medium text-yellow-800 dark:text-yellow-200 mb-2">
                                    ⚠️ Brak rodzaju mebla
                                  </p>
                                  <p className="text-sm text-yellow-700 dark:text-yellow-300">
                                    Aby wygenerować BOM, najpierw zdefiniuj <strong>Rodzaj mebla</strong> w zakładce "Podstawowe".
                                  </p>
                                </div>
                              </div>
                            ) : (
                              <div className="space-y-4">
                                <div className="text-center">
                                  <p className="text-sm text-muted-foreground mb-4">
                                    Brak wygenerowanego BOM dla tego produktu.
                                  </p>
                                  <p className="text-xs text-muted-foreground">
                                    Kliknij przycisk <strong>Matryce BOM</strong> powyżej aby przejść do matryc szablonów.
                                  </p>
                                </div>
                              </div>
                            )}
                          </div>
                        </CardContent>
                      )}
                    </Card>
                  </div>
                ) : (
                  <Card>
                    <CardContent className="pt-6">
                      <div className="text-center text-muted-foreground py-12">
                        Zapisz produkt, aby móc wygenerować BOM
                      </div>
                    </CardContent>
                  </Card>
                )}
              </TabsContent>
            </Tabs>
          </form>
        </Form>

        {/* Template Picker Dialog */}
        <TemplatePickerDialog
          open={isTemplatePickerOpen}
          onOpenChange={setIsTemplatePickerOpen}
          onSelect={(template) => {
            form.setValue("longDescriptionHtml", template.htmlContent);
            toast({
              title: "Szablon zastosowany",
              description: `Treść szablonu "${template.name}" została skopiowana do opisu produktu.`,
            });
          }}
        />

        {/* Description Preview Dialog */}
        <Dialog open={isPreviewDialogOpen} onOpenChange={setIsPreviewDialogOpen}>
          <DialogContent className="max-w-[95vw] sm:max-w-4xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Podgląd opisu produktu</DialogTitle>
              <DialogDescription>
                Podgląd jak opis będzie wyglądał na platformach sprzedażowych (Allegro/Shoper)
              </DialogDescription>
            </DialogHeader>
            <div className="mt-4">
              {/* Preview container with Allegro/Shoper-like styling */}
              <div 
                className="prose prose-sm max-w-none bg-white dark:bg-gray-900 text-gray-900 dark:text-gray-300 p-4 sm:p-6 rounded-lg border"
                style={{
                  fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif',
                  fontSize: '14px',
                  lineHeight: '1.6',
                }}
                dangerouslySetInnerHTML={{ 
                  __html: form.watch('longDescriptionHtml') || '<p class="text-muted-foreground">Brak opisu</p>' 
                }}
              />
            </div>
            <div className="flex justify-end gap-2 mt-4">
              <Button
                type="button"
                variant="outline"
                onClick={() => setIsPreviewDialogOpen(false)}
                data-testid="button-close-preview"
              >
                Zamknij
              </Button>
            </div>
          </DialogContent>
        </Dialog>

        </div>
      </div>
    </div>
  );
}

// Product Gallery Component
function ProductGallery({ productId }: { productId: number }) {
  const { toast } = useToast();
  const fileInputRef = useRef<HTMLInputElement>(null);
  const [editingImage, setEditingImage] = useState<ProductImage | null>(null);
  const [editAltText, setEditAltText] = useState("");
  const [editImageType, setEditImageType] = useState<string>("packshot");
  const [editTags, setEditTags] = useState<string[]>([]);
  const [draggedImage, setDraggedImage] = useState<ProductImage | null>(null);
  const [isDraggingFiles, setIsDraggingFiles] = useState(false);
  const [uploadingCount, setUploadingCount] = useState(0);

  // Fetch images
  const { data: images, isLoading } = useQuery<ProductImage[]>({
    queryKey: [`/api/catalog-products/${productId}/images`],
    enabled: !!productId,
  });

  // Fetch color dictionary for tag suggestions
  interface ProductCreatorDictionary {
    id: number;
    dictionaryType: string;
    code: string;
    name: string;
    readableName: string | null;
    isActive: boolean;
  }
  
  const { data: colors } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=color"],
  });

  // Upload mutation
  const uploadMutation = useMutation({
    mutationFn: async (formData: FormData) => {
      const response = await fetch(`/api/catalog-products/${productId}/images`, {
        method: "POST",
        body: formData,
        credentials: "include",
      });
      if (!response.ok) throw new Error("Upload failed");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/images`] });
      queryClient.invalidateQueries({ predicate: (query) => 
        Array.isArray(query.queryKey) && 
        typeof query.queryKey[0] === 'string' && 
        query.queryKey[0].startsWith('/api/catalog-products')
      });
      toast({ title: "Zdjęcie dodane" });
      if (fileInputRef.current) {
        fileInputRef.current.value = "";
      }
    },
    onError: () => {
      toast({ title: "Błąd uploadu", variant: "destructive" });
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async (imageId: number) => {
      await apiRequest("DELETE", `/api/catalog-products/${productId}/images/${imageId}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/images`] });
      queryClient.invalidateQueries({ predicate: (query) => 
        Array.isArray(query.queryKey) && 
        typeof query.queryKey[0] === 'string' && 
        query.queryKey[0].startsWith('/api/catalog-products')
      });
      toast({ title: "Zdjęcie usunięte" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania", variant: "destructive" });
    },
  });

  // Update mutation
  const updateMutation = useMutation({
    mutationFn: async ({ imageId, data }: { imageId: number; data: any }) => {
      await apiRequest("PATCH", `/api/catalog-products/${productId}/images/${imageId}`, data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/images`] });
      queryClient.invalidateQueries({ predicate: (query) => 
        Array.isArray(query.queryKey) && 
        typeof query.queryKey[0] === 'string' && 
        query.queryKey[0].startsWith('/api/catalog-products')
      });
      toast({ title: "Zdjęcie zaktualizowane" });
      setEditingImage(null);
    },
    onError: () => {
      toast({ title: "Błąd aktualizacji", variant: "destructive" });
    },
  });

  const uploadFiles = async (files: FileList | File[]) => {
    const fileArray = Array.from(files);
    if (fileArray.length === 0) return;

    setUploadingCount(fileArray.length);
    let successCount = 0;
    let failedCount = 0;

    try {
      for (const file of fileArray) {
        try {
          const formData = new FormData();
          formData.append("image", file);
          formData.append("imageType", "packshot");
          formData.append("altText", "");
          formData.append("isPrimary", "false");

          const response = await fetch(`/api/catalog-products/${productId}/images`, {
            method: "POST",
            body: formData,
            credentials: "include",
          });

          if (response.ok) {
            successCount++;
          } else {
            failedCount++;
          }
        } catch (error) {
          failedCount++;
        }
      }

      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/images`] });
      queryClient.invalidateQueries({ predicate: (query) => 
        Array.isArray(query.queryKey) && 
        typeof query.queryKey[0] === 'string' && 
        query.queryKey[0].startsWith('/api/catalog-products')
      });
      
      if (successCount > 0 && failedCount === 0) {
        toast({ title: `${successCount} zdjęć dodanych` });
      } else if (successCount > 0 && failedCount > 0) {
        toast({ 
          title: `${successCount} zdjęć dodanych, ${failedCount} nieudanych`,
          variant: "default"
        });
      } else {
        toast({ title: "Wszystkie uploady nieudane", variant: "destructive" });
      }

      if (fileInputRef.current) {
        fileInputRef.current.value = "";
      }
    } finally {
      setUploadingCount(0);
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files) {
      uploadFiles(e.target.files);
    }
  };

  const handleFileDrop = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDraggingFiles(false);

    const files = e.dataTransfer.files;
    if (files.length > 0) {
      uploadFiles(files);
    }
  };

  const handleDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDraggingFiles(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    
    // Only set to false if we're leaving the dropzone entirely
    const rect = e.currentTarget.getBoundingClientRect();
    const x = e.clientX;
    const y = e.clientY;
    
    if (x <= rect.left || x >= rect.right || y <= rect.top || y >= rect.bottom) {
      setIsDraggingFiles(false);
    }
  };

  const handleDragOverFiles = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  const handleEdit = (image: ProductImage) => {
    setEditingImage(image);
    setEditAltText(image.altText || "");
    setEditImageType(image.imageType || "packshot");
    setEditTags(image.tags || []);
  };

  const handleSaveEdit = () => {
    if (!editingImage) return;
    updateMutation.mutate({
      imageId: editingImage.id,
      data: {
        altText: editAltText,
        imageType: editImageType,
        tags: editTags,
      },
    });
  };

  const handleSetPrimary = (imageId: number) => {
    updateMutation.mutate({
      imageId,
      data: { isPrimary: true },
    });
  };

  const handleDragStart = (image: ProductImage) => {
    setDraggedImage(image);
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
  };

  const handleDrop = async (targetImage: ProductImage) => {
    if (!draggedImage || draggedImage.id === targetImage.id) {
      setDraggedImage(null);
      return;
    }

    const sortedImages = [...(images || [])].sort((a, b) => (a.sortOrder ?? 0) - (b.sortOrder ?? 0));
    const draggedIndex = sortedImages.findIndex(img => img.id === draggedImage.id);
    const targetIndex = sortedImages.findIndex(img => img.id === targetImage.id);

    const reordered = [...sortedImages];
    const [removed] = reordered.splice(draggedIndex, 1);
    reordered.splice(targetIndex, 0, removed);

    try {
      // Update sort order and set first image as primary
      await Promise.all(
        reordered.map((img, index) =>
          apiRequest("PATCH", `/api/catalog-products/${productId}/images/${img.id}`, {
            sortOrder: index,
            isPrimary: index === 0, // First image is always primary
          })
        )
      );
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/images`] });
      queryClient.invalidateQueries({ predicate: (query) => 
        Array.isArray(query.queryKey) && 
        typeof query.queryKey[0] === 'string' && 
        query.queryKey[0].startsWith('/api/catalog-products')
      });
      toast({ title: "Kolejność zaktualizowana" });
    } catch (error) {
      toast({ title: "Błąd zmiany kolejności", variant: "destructive" });
    }

    setDraggedImage(null);
  };

  if (isLoading) {
    return (
      <Card>
        <CardContent className="pt-6">
          <div className="flex justify-center py-12">
            <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle>Galeria zdjęć</CardTitle>
        <CardDescription>
          Zarządzanie zdjęciami produktu (packshot, rendery, wizualizacje)
        </CardDescription>
      </CardHeader>
      <CardContent className="space-y-4">
        {/* Drag & Drop Zone */}
        <div
          onDrop={handleFileDrop}
          onDragOver={handleDragOverFiles}
          onDragEnter={handleDragEnter}
          onDragLeave={handleDragLeave}
          className={`border-2 border-dashed rounded-lg p-8 text-center transition-colors ${
            isDraggingFiles
              ? "border-primary bg-primary/5"
              : "border-muted-foreground/25 hover:border-muted-foreground/50"
          }`}
          data-testid="dropzone"
        >
          <input
            ref={fileInputRef}
            type="file"
            accept="image/*"
            multiple
            className="hidden"
            onChange={handleFileChange}
            data-testid="input-image-upload"
          />
          
          {uploadingCount > 0 ? (
            <div className="space-y-2">
              <Loader2 className="h-8 w-8 animate-spin mx-auto text-primary" />
              <p className="text-sm font-medium">Wgrywanie {uploadingCount} zdjęć...</p>
            </div>
          ) : (
            <>
              <Upload className="h-12 w-12 mx-auto mb-4 text-muted-foreground" />
              <p className="text-lg font-medium mb-2">
                Przeciągnij i upuść zdjęcia tutaj
              </p>
              <p className="text-sm text-muted-foreground mb-4">
                lub
              </p>
              <Button
                type="button"
                variant="outline"
                onClick={() => fileInputRef.current?.click()}
                disabled={uploadingCount > 0}
                data-testid="button-upload-image"
              >
                <Upload className="mr-2 h-4 w-4" />
                Wybierz pliki
              </Button>
              <p className="text-xs text-muted-foreground mt-2">
                Obsługiwane formaty: JPEG, PNG, WebP (max 5MB każde)
              </p>
            </>
          )}
        </div>

        {/* Images grid */}
        {!images || images.length === 0 ? (
          <div className="text-center text-muted-foreground py-8">
            <p className="text-sm">Brak zdjęć. Użyj powyższej strefy aby dodać zdjęcia.</p>
          </div>
        ) : (
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
            {[...images].sort((a, b) => (a.sortOrder ?? 0) - (b.sortOrder ?? 0)).map((image, index) => (
              <div
                key={image.id}
                draggable
                onDragStart={() => handleDragStart(image)}
                onDragOver={handleDragOver}
                onDrop={() => handleDrop(image)}
                className={`relative group rounded-lg overflow-hidden hover-elevate cursor-move ${
                  draggedImage?.id === image.id ? "opacity-50" : ""
                } ${index === 0 ? "border-4 border-primary shadow-lg" : "border"}`}
                data-testid={`image-card-${image.id}`}
              >
                <div className="aspect-square bg-muted relative">
                  {/* Image number badge */}
                  <div className="absolute top-2 left-2 bg-background/90 backdrop-blur-sm text-foreground px-2.5 py-1.5 rounded-md text-sm font-bold shadow-md z-10">
                    #{index + 1}
                  </div>
                  
                  <img
                    src={image.url}
                    alt={image.altText || "Zdjęcie produktu"}
                    className="w-full h-full object-cover"
                    data-testid={`img-product-${image.id}`}
                  />
                  {index === 0 && (
                    <div className="absolute top-2 right-2 bg-primary text-primary-foreground px-3 py-2 rounded-md text-sm font-semibold flex items-center gap-1.5 shadow-lg">
                      <Star className="h-4 w-4 fill-current" />
                      GŁÓWNE
                    </div>
                  )}
                  <div className="absolute bottom-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity flex gap-1">
                    <Button
                      size="icon"
                      variant="secondary"
                      onClick={() => handleEdit(image)}
                      data-testid={`button-edit-${image.id}`}
                    >
                      <Edit2 className="h-4 w-4" />
                    </Button>
                    <Button
                      size="icon"
                      variant="destructive"
                      onClick={() => deleteMutation.mutate(image.id)}
                      disabled={deleteMutation.isPending}
                      data-testid={`button-delete-${image.id}`}
                    >
                      <X className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
                <div className="p-2 space-y-1">
                  <div className="text-xs font-medium capitalize">{image.imageType}</div>
                  {image.altText && (
                    <div className="text-xs text-muted-foreground truncate">{image.altText}</div>
                  )}
                </div>
              </div>
            ))}
          </div>
        )}

        {/* Edit dialog */}
        {editingImage && (
          <div className="fixed inset-0 bg-black/50 flex items-center justify-center p-4 z-50">
            <Card className="w-full max-w-md">
              <CardHeader>
                <CardTitle>Edytuj zdjęcie</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <label className="text-sm font-medium">Typ zdjęcia</label>
                  <Select value={editImageType} onValueChange={setEditImageType}>
                    <SelectTrigger data-testid="select-image-type">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="packshot">Packshot</SelectItem>
                      <SelectItem value="render">Render</SelectItem>
                      <SelectItem value="visualization">Wizualizacja</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <label className="text-sm font-medium">ALT Text (SEO)</label>
                  <Input
                    value={editAltText}
                    onChange={(e) => setEditAltText(e.target.value)}
                    placeholder="Opis zdjęcia dla wyszukiwarek"
                    data-testid="input-alt-text"
                  />
                </div>
                <div>
                  <label className="text-sm font-medium">Tagi</label>
                  <TagInput
                    tags={editTags}
                    onChange={setEditTags}
                    suggestions={colors?.filter(c => c.isActive).map(c => c.readableName || c.name) || []}
                    placeholder="Dodaj tag..."
                  />
                </div>
                <div className="flex gap-2">
                  <Button
                    onClick={handleSaveEdit}
                    disabled={updateMutation.isPending}
                    data-testid="button-save-edit"
                  >
                    {updateMutation.isPending ? (
                      <>
                        <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                        Zapisywanie...
                      </>
                    ) : (
                      "Zapisz"
                    )}
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => setEditingImage(null)}
                    data-testid="button-cancel-edit"
                  >
                    Anuluj
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        )}
      </CardContent>
    </Card>
  );
}

// Product Addons Component
function ProductAddons({ productId }: { productId: number }) {
  const { toast } = useToast();
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [selectedAddonId, setSelectedAddonId] = useState<number | null>(null);

  // Fetch assigned addons
  const { data: assignments, isLoading: loadingAssignments } = useQuery<any[]>({
    queryKey: [`/api/catalog-products/${productId}/addons`],
    enabled: !!productId,
  });

  // Fetch all available addons
  const { data: allAddons } = useQuery<any[]>({
    queryKey: ["/api/product-addons"],
  });

  // Assign addon mutation
  const assignMutation = useMutation({
    mutationFn: async (addonId: number) => {
      await apiRequest("POST", `/api/catalog-products/${productId}/addons`, {
        addonId,
        sortOrder: (assignments?.length || 0),
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/addons`] });
      toast({ title: "Dodatek przypisany" });
      setIsDialogOpen(false);
      setSelectedAddonId(null);
    },
    onError: () => {
      toast({ title: "Błąd przypisywania dodatku", variant: "destructive" });
    },
  });

  // Unassign addon mutation
  const unassignMutation = useMutation({
    mutationFn: async (assignmentId: number) => {
      await apiRequest("DELETE", `/api/catalog-products/${productId}/addons/${assignmentId}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/addons`] });
      toast({ title: "Dodatek usunięty" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania dodatku", variant: "destructive" });
    },
  });

  const handleAssign = () => {
    if (selectedAddonId) {
      assignMutation.mutate(selectedAddonId);
    }
  };

  const handleUnassign = (assignmentId: number, addonName: string) => {
    if (window.confirm(`Czy na pewno chcesz usunąć dodatek "${addonName}" z tego produktu?`)) {
      unassignMutation.mutate(assignmentId);
    }
  };

  // Get available addons (not yet assigned)
  const assignedAddonIds = new Set(assignments?.map((a) => a.addonId) || []);
  const availableAddons = allAddons?.filter((addon) => !assignedAddonIds.has(addon.id)) || [];

  const ADDON_TYPE_LABELS: Record<string, string> = {
    fabric: "Tkanina",
    board: "Płyta",
    certificate: "Certyfikat",
    accessory: "Akcesorium",
    component: "Komponent",
  };

  if (loadingAssignments) {
    return (
      <Card>
        <CardContent className="pt-6">
          <div className="flex justify-center py-12">
            <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <div>
            <CardTitle>Dodatki produktu</CardTitle>
            <CardDescription>
              Przypisane tkaniny, płyty, certyfikaty i akcesoria
            </CardDescription>
          </div>
          <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
            <DialogTrigger asChild>
              <Button data-testid="button-add-addon-assignment">
                <Plus className="mr-2 h-4 w-4" />
                Przypisz dodatek
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Przypisz dodatek</DialogTitle>
                <DialogDescription>
                  Wybierz dodatek do przypisania do tego produktu
                </DialogDescription>
              </DialogHeader>
              <div className="space-y-4">
                {availableAddons.length === 0 ? (
                  <div className="text-center text-muted-foreground py-8">
                    <p>Wszystkie dodatki są już przypisane</p>
                    <p className="text-sm mt-2">
                      Możesz utworzyć nowe dodatki w sekcji "Dodatki"
                    </p>
                  </div>
                ) : (
                  <>
                    <Select value={selectedAddonId?.toString()} onValueChange={(val) => setSelectedAddonId(parseInt(val))}>
                      <SelectTrigger data-testid="select-addon">
                        <SelectValue placeholder="Wybierz dodatek" />
                      </SelectTrigger>
                      <SelectContent>
                        {availableAddons.map((addon) => (
                          <SelectItem key={addon.id} value={addon.id.toString()}>
                            {ADDON_TYPE_LABELS[addon.addonType] || addon.addonType}: {addon.name}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <div className="flex justify-end gap-2">
                      <Button
                        variant="outline"
                        onClick={() => {
                          setIsDialogOpen(false);
                          setSelectedAddonId(null);
                        }}
                      >
                        Anuluj
                      </Button>
                      <Button
                        onClick={handleAssign}
                        disabled={!selectedAddonId || assignMutation.isPending}
                        data-testid="button-confirm-assign"
                      >
                        {assignMutation.isPending && (
                          <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                        )}
                        Przypisz
                      </Button>
                    </div>
                  </>
                )}
              </div>
            </DialogContent>
          </Dialog>
        </div>
      </CardHeader>
      <CardContent>
        {!assignments || assignments.length === 0 ? (
          <div className="text-center text-muted-foreground py-12">
            <Package className="h-12 w-12 mx-auto mb-4 text-muted-foreground" />
            <p>Brak przypisanych dodatków</p>
            <p className="text-sm mt-2">
              Kliknij "Przypisz dodatek" aby dodać pierwszy dodatek
            </p>
          </div>
        ) : (
          <div className="space-y-3">
            {assignments.map((assignment) => (
              <div
                key={assignment.id}
                className="flex items-center justify-between p-4 border rounded-lg hover-elevate"
                data-testid={`addon-assignment-${assignment.id}`}
              >
                <div className="flex items-center gap-4">
                  <Package className="h-5 w-5 text-muted-foreground" />
                  <div>
                    <div className="flex items-center gap-2">
                      <span className="font-medium">{assignment.name}</span>
                      <Badge variant="outline">
                        {ADDON_TYPE_LABELS[assignment.addonType] || assignment.addonType}
                      </Badge>
                    </div>
                    {assignment.description && (
                      <p className="text-sm text-muted-foreground mt-1">
                        {assignment.description}
                      </p>
                    )}
                    {assignment.code && (
                      <p className="text-xs text-muted-foreground mt-1">
                        Kod: {assignment.code}
                      </p>
                    )}
                  </div>
                </div>
                <Button
                  size="icon"
                  variant="ghost"
                  onClick={() => handleUnassign(assignment.id, assignment.name)}
                  disabled={unassignMutation.isPending}
                  data-testid={`button-unassign-${assignment.id}`}
                >
                  <Trash2 className="h-4 w-4" />
                </Button>
              </div>
            ))}
          </div>
        )}
      </CardContent>
    </Card>
  );
}

