import { useParams, useLocation, Link } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ArrowLeft, ExternalLink, Package, Ruler, Calculator, RefreshCw } from "lucide-react";
import { Skeleton } from "@/components/ui/skeleton";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

interface BomComponentDetail {
  id: number;
  product_bom_id: number;
  component_template_id: number;
  generated_name: string;
  quantity: number;
  component_type: string;
  cz1: string;
  cz2: string;
  furniture_type: string;
  template_base_length: number;
  template_base_width: number;
  thickness: number;
  edge1: string | null;
  edge2: string | null;
  edge3: string | null;
  edge4: string | null;
  calculated_length: number;
  calculated_width: number;
  calculated_color: string;
  color_hex: string | null;
  product_id: number;
  product_sku: string;
  product_title: string;
  product_type: string;
  product_color: string;
  product_length: number;
  product_width: number;
  product_height: number;
  product_doors: string | null;
  product_legs: string | null;
  product_images: Array<{ id: number; image_url: string; is_primary: boolean }> | null;
  plate_id: number | null;
  plate_name: string | null;
  edge_id: number | null;
  edge_name: string | null;
}

interface MaterialBreakdown {
  id: number;
  component_id: number;
  board_area_m2: number | null;
  board_waste_percentage: number | null;
  board_total_m2: number | null;
  edging_length_mb: number | null;
  edging_overage_cm: number | null;
  edging_total_mb: number | null;
  adhesive_grams_per_mb: number | null;
  adhesive_total_g: number | null;
  board_unit_price: number | null;
  edging_unit_price: number | null;
  adhesive_unit_price: number | null;
  board_cost: number | null;
  edging_cost: number | null;
  adhesive_cost: number | null;
  total_cost: number | null;
  calculated_at: string;
}

// Helper function to determine text color based on background luminance
function getTextColorForBackground(hexColor: string | null | undefined): string {
  if (!hexColor || !hexColor.startsWith('#')) return 'black';
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance > 128 ? 'black' : 'white';
}

export default function BomComponentDetail() {
  const { id } = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();

  const { data: component, isLoading } = useQuery<BomComponentDetail>({
    queryKey: [`/api/bom-components/${id}`],
  });

  // Query for material breakdown - handle 404 as "no data" not error
  const { data: breakdown, isLoading: breakdownLoading, error: breakdownError } = useQuery<MaterialBreakdown>({
    queryKey: [`/api/production/boms/components/${id}/material-breakdown`],
    enabled: !!id,
    retry: false, // Don't retry 404
  });

  // Check if error is 404 (no breakdown exists yet)
  const is404 = breakdownError && (breakdownError as any).message?.includes('404');

  // Mutation to save breakdown (first time)
  const saveMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest('POST', `/api/production/boms/components/${id}/material-breakdown/save`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/boms/components/${id}/material-breakdown`] });
      toast({
        title: "Sukces",
        description: "Rozk\u0142ad materia\u0142owy zosta\u0142 obliczony i zapisany",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "B\u0142\u0105d",
        description: error.message || "Nie uda\u0142o si\u0119 obliczy\u0107 rozk\u0142adu",
        variant: "destructive",
      });
    },
  });

  // Mutation to recalculate breakdown
  const recalculateMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest('POST', `/api/production/boms/components/${id}/material-breakdown/recalculate`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/boms/components/${id}/material-breakdown`] });
      toast({
        title: "Sukces",
        description: "Rozk\u0142ad materia\u0142owy zosta\u0142 przeliczony",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "B\u0142\u0105d",
        description: error.message || "Nie uda\u0142o si\u0119 przeliczy\u0107 rozk\u0142adu",
        variant: "destructive",
      });
    },
  });

  if (isLoading) {
    return (
      <div className="w-full h-full overflow-auto">
        <div className="px-6 py-8 space-y-6 max-w-7xl">
          <Skeleton className="h-10 w-full" />
          <Skeleton className="h-96 w-full" />
        </div>
      </div>
    );
  }

  if (!component) {
    return (
      <div className="w-full h-full overflow-auto">
        <div className="px-6 py-8">
          <div className="text-center py-12">
            <Package className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
            <h3 className="text-lg font-medium mb-2">Formatka nie znaleziona</h3>
            <Button onClick={() => navigate("/bom-components")}>
              <ArrowLeft className="h-4 w-4 mr-2" />
              Powrót do listy
            </Button>
          </div>
        </div>
      </div>
    );
  }

  const primaryImage = component.product_images?.find(img => img.is_primary)?.image_url;
  const textColor = getTextColorForBackground(component.color_hex);

  // Fixed SVG canvas size for consistent visualization
  const CANVAS_SIZE = 300;
  const svgWidth = CANVAS_SIZE;
  const svgHeight = CANVAS_SIZE;
  
  // Fixed margins for labels (gutters around the formatka)
  const MARGIN_TOP = 40;
  const MARGIN_RIGHT = 50;
  const MARGIN_BOTTOM = 40;
  const MARGIN_LEFT = 50;
  
  // Drawable area for the formatka rectangle
  const drawableWidth = CANVAS_SIZE - MARGIN_LEFT - MARGIN_RIGHT;
  const drawableHeight = CANVAS_SIZE - MARGIN_TOP - MARGIN_BOTTOM;
  
  // Component actual dimensions in mm
  const componentLength = component.calculated_length;
  const componentWidth = component.calculated_width;
  const aspectRatio = componentLength / componentWidth;
  
  // Scale the formatka to fit within drawable area while maintaining aspect ratio
  // Try to enforce minimum height for visibility, but don't exceed canvas
  const MIN_SHORT_EDGE = 80; // Minimum for the shorter dimension
  
  let rectWidth: number;
  let rectHeight: number;
  
  // First, scale to fit drawable area
  if (aspectRatio > (drawableWidth / drawableHeight)) {
    // Limited by width (long horizontal formatka)
    rectWidth = drawableWidth;
    rectHeight = drawableWidth / aspectRatio;
  } else {
    // Limited by height (tall vertical formatka)
    rectHeight = drawableHeight;
    rectWidth = drawableHeight * aspectRatio;
  }
  
  // For very narrow formatki, try to increase the short edge
  // but ensure both dimensions still fit within drawable area
  const shortEdge = Math.min(rectWidth, rectHeight);
  if (shortEdge < MIN_SHORT_EDGE) {
    const scaleFactor = MIN_SHORT_EDGE / shortEdge;
    const newWidth = rectWidth * scaleFactor;
    const newHeight = rectHeight * scaleFactor;
    
    // Only apply if both dimensions fit
    if (newWidth <= drawableWidth && newHeight <= drawableHeight) {
      rectWidth = newWidth;
      rectHeight = newHeight;
    }
  }
  
  // Center the formatka within the canvas
  const rectX = (CANVAS_SIZE - rectWidth) / 2;
  const rectY = (CANVAS_SIZE - rectHeight) / 2;

  // Edge labels - dashed lines 10px from rectangle, text 3px from lines
  const edgeOffset = 10;
  const labelOffset = 3; // Distance from dashed line to label
  const edgeLabelFontSize = 9;
  const dimensionLabelFontSize = 11;
  
  // Precise spacing constants (approximate font height for each size)
  const edgeLabelHeight = edgeLabelFontSize * 1.2; // ~10.8px
  const dimensionLabelHeight = dimensionLabelFontSize * 1.2; // ~13.2px
  
  const edges = [
    { active: !!component.edge1, label: "Krawędź 1 (góra)", x1: rectX, y1: rectY - edgeOffset, x2: rectX + rectWidth, y2: rectY - edgeOffset },
    { active: !!component.edge2, label: "Krawędź 2 (prawo)", x1: rectX + rectWidth + edgeOffset, y1: rectY, x2: rectX + rectWidth + edgeOffset, y2: rectY + rectHeight },
    { active: !!component.edge3, label: "Krawędź 3 (dół)", x1: rectX, y1: rectY + rectHeight + edgeOffset, x2: rectX + rectWidth, y2: rectY + rectHeight + edgeOffset },
    { active: !!component.edge4, label: "Krawędź 4 (lewo)", x1: rectX - edgeOffset, y1: rectY, x2: rectX - edgeOffset, y2: rectY + rectHeight },
  ];

  return (
    <div className="w-full h-full overflow-auto">
      <div className="px-6 py-8 space-y-6 max-w-7xl">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-3">
            <Button 
              variant="outline" 
              size="icon"
              onClick={() => navigate("/bom-components")}
              data-testid="button-back"
            >
              <ArrowLeft className="h-4 w-4" />
            </Button>
            <div>
              <h1 className="text-3xl font-bold">Szczegóły formatki</h1>
              <p className="text-sm text-muted-foreground font-mono">
                ID: {component.id}
              </p>
            </div>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Main Info */}
          <div className="lg:col-span-2 space-y-6">
            {/* Visualization */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Ruler className="w-5 h-5" />
                  Wizualizacja formatki
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="bg-muted/20 rounded-lg p-4 flex items-center justify-center">
                  <svg
                    width={svgWidth}
                    height={svgHeight}
                    viewBox={`0 0 ${svgWidth} ${svgHeight}`}
                    className="mx-auto"
                  >
                    {/* Main rectangle */}
                    <rect
                      x={rectX}
                      y={rectY}
                      width={rectWidth}
                      height={rectHeight}
                      fill="hsl(var(--card))"
                      stroke="hsl(var(--foreground))"
                      strokeWidth="1.5"
                      opacity="0.3"
                      data-testid="svg-rectangle"
                    />

                    {/* Dimension labels - 3px from edge labels (or 3px from rectangle if no edge) */}
                    {/* Top dimension label */}
                    <text
                      x={rectX + rectWidth / 2}
                      y={component.edge1 
                        ? (rectY - edgeOffset - labelOffset - edgeLabelHeight / 2 - labelOffset - dimensionLabelHeight / 2)
                        : (rectY - labelOffset - dimensionLabelHeight / 2)
                      }
                      textAnchor="middle"
                      alignmentBaseline="middle"
                      fontSize={dimensionLabelFontSize}
                      fill="hsl(var(--foreground))"
                      fontWeight="600"
                    >
                      {component.calculated_length} mm
                    </text>
                    {/* Right dimension label */}
                    <text
                      x={component.edge2 
                        ? (rectX + rectWidth + edgeOffset + labelOffset + edgeLabelHeight / 2 + labelOffset + dimensionLabelHeight / 2)
                        : (rectX + rectWidth + labelOffset + dimensionLabelHeight / 2)
                      }
                      y={rectY + rectHeight / 2}
                      textAnchor="middle"
                      alignmentBaseline="middle"
                      fontSize={dimensionLabelFontSize}
                      fill="hsl(var(--foreground))"
                      fontWeight="600"
                      transform={`rotate(90, ${component.edge2 ? (rectX + rectWidth + edgeOffset + labelOffset + edgeLabelHeight / 2 + labelOffset + dimensionLabelHeight / 2) : (rectX + rectWidth + labelOffset + dimensionLabelHeight / 2)}, ${rectY + rectHeight / 2})`}
                    >
                      {component.calculated_width} mm
                    </text>

                    {/* Edging lines */}
                    {edges.map((edge, index) => {
                      if (!edge.active) return null;
                      
                      // Determine if this is a vertical edge (edge2=right, edge4=left)
                      const isVerticalEdge = index === 1 || index === 3;
                      const centerX = (edge.x1 + edge.x2) / 2;
                      const centerY = (edge.y1 + edge.y2) / 2;
                      
                      return (
                        <g key={index}>
                          {/* Dashed line for edge */}
                          <line
                            x1={edge.x1}
                            y1={edge.y1}
                            x2={edge.x2}
                            y2={edge.y2}
                            stroke="hsl(var(--primary))"
                            strokeWidth="2"
                            strokeDasharray="5,5"
                            data-testid={`svg-edge-${index + 1}`}
                          />
                          {/* Label - exactly 3px from dashed line on all sides */}
                          {isVerticalEdge ? (
                            // Vertical edges (left/right) - rotate 90 degrees
                            // Distance from line: labelOffset + half of text height to center
                            <text
                              x={centerX + (index === 1 ? (labelOffset + edgeLabelHeight / 2) : -(labelOffset + edgeLabelHeight / 2))}
                              y={centerY}
                              textAnchor="middle"
                              alignmentBaseline="middle"
                              fontSize={edgeLabelFontSize}
                              fill="hsl(var(--primary))"
                              fontWeight="500"
                              transform={`rotate(90, ${centerX + (index === 1 ? (labelOffset + edgeLabelHeight / 2) : -(labelOffset + edgeLabelHeight / 2))}, ${centerY})`}
                            >
                              {component.edge_name || "Obrzeże"}
                            </text>
                          ) : (
                            // Horizontal edges (top/bottom)
                            // Distance from line: labelOffset + half of text height to center
                            <text
                              x={centerX}
                              y={centerY + (index === 0 ? -(labelOffset + edgeLabelHeight / 2) : (labelOffset + edgeLabelHeight / 2))}
                              textAnchor="middle"
                              alignmentBaseline="middle"
                              fontSize={edgeLabelFontSize}
                              fill="hsl(var(--primary))"
                              fontWeight="500"
                            >
                              {component.edge_name || "Obrzeże"}
                            </text>
                          )}
                        </g>
                      );
                    })}

                    {/* Center label */}
                    <text
                      x={rectX + rectWidth / 2}
                      y={rectY + rectHeight / 2}
                      textAnchor="middle"
                      fontSize="12"
                      fill="hsl(var(--muted-foreground))"
                      fontWeight="500"
                    >
                      {component.cz1} {component.cz2}
                    </text>
                  </svg>
                </div>

                {/* Edge legend */}
                <div className="mt-4 grid grid-cols-2 gap-2">
                  {edges.map((edge, index) => (
                    <div key={index} className="flex items-center gap-2 text-sm">
                      <div className={`w-3 h-3 rounded-sm ${edge.active ? 'bg-primary' : 'bg-muted'}`} />
                      <span className={edge.active ? 'text-foreground' : 'text-muted-foreground'}>
                        {edge.label}
                      </span>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            {/* Basic Information */}
            <Card>
              <CardHeader>
                <CardTitle>Informacje podstawowe</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Nazwa generowana</label>
                  <div className="mt-1 font-mono text-lg font-bold">
                    {component.generated_name}
                  </div>
                </div>

                <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">CZ1</label>
                    <div className="mt-1 font-mono font-bold">{component.cz1}</div>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">CZ2</label>
                    <div className="mt-1 font-mono font-bold">{component.cz2}</div>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Typ mebla</label>
                    <div className="mt-1">{component.furniture_type || '-'}</div>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Ilość</label>
                    <div className="mt-1 font-bold">{component.quantity} szt.</div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Dimensions */}
            <Card>
              <CardHeader>
                <CardTitle>Wymiary</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                  <div className="bg-muted/50 rounded p-3">
                    <label className="text-xs text-muted-foreground">Długość</label>
                    <div className="mt-1 font-mono text-2xl font-bold">
                      {component.calculated_length} <span className="text-sm font-normal">mm</span>
                    </div>
                  </div>
                  <div className="bg-muted/50 rounded p-3">
                    <label className="text-xs text-muted-foreground">Szerokość</label>
                    <div className="mt-1 font-mono text-2xl font-bold">
                      {component.calculated_width} <span className="text-sm font-normal">mm</span>
                    </div>
                  </div>
                  <div className="bg-muted/50 rounded p-3">
                    <label className="text-xs text-muted-foreground">Grubość</label>
                    <div className="mt-1 font-mono text-2xl font-bold">
                      {component.thickness} <span className="text-sm font-normal">mm</span>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Color and Materials */}
            <Card>
              <CardHeader>
                <CardTitle>Kolor i materiały</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Kolor</label>
                  <div className="mt-2">
                    <div 
                      className="inline-flex items-center gap-2 rounded px-4 py-3 border" 
                      style={{
                        backgroundColor: component.color_hex || '#e5e7eb',
                      }}
                    >
                      <span 
                        className="font-medium text-lg"
                        style={{ color: textColor }}
                      >
                        {component.calculated_color}
                      </span>
                      <span 
                        className="text-xs font-mono"
                        style={{ color: textColor, opacity: 0.7 }}
                      >
                        {component.color_hex}
                      </span>
                    </div>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Płyta</label>
                    <div className="mt-1">
                      {component.plate_id ? (
                        <Link href={`/warehouse/plyty/${component.plate_id}`}>
                          <Badge variant="secondary" className="text-sm hover:bg-secondary/80">
                            {component.plate_name}
                            <ExternalLink className="ml-2 h-3 w-3" />
                          </Badge>
                        </Link>
                      ) : (
                        <span className="text-muted-foreground">Nie znaleziono</span>
                      )}
                    </div>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Obrzeże (0.8mm)</label>
                    <div className="mt-1">
                      {component.edge_id ? (
                        <Link href={`/warehouse/obrzeza/${component.edge_id}`}>
                          <Badge variant="secondary" className="text-sm hover:bg-secondary/80">
                            {component.edge_name}
                            <ExternalLink className="ml-2 h-3 w-3" />
                          </Badge>
                        </Link>
                      ) : (
                        <span className="text-muted-foreground">Nie znaleziono</span>
                      )}
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Edges Configuration */}
            {(component.edge1 || component.edge2 || component.edge3 || component.edge4) && (
              <Card>
                <CardHeader>
                  <CardTitle>Konfiguracja obrzeży</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                    <div>
                      <label className="text-xs text-muted-foreground">Krawędź 1</label>
                      <div className="mt-1 font-medium">{component.edge1 || '-'}</div>
                    </div>
                    <div>
                      <label className="text-xs text-muted-foreground">Krawędź 2</label>
                      <div className="mt-1 font-medium">{component.edge2 || '-'}</div>
                    </div>
                    <div>
                      <label className="text-xs text-muted-foreground">Krawędź 3</label>
                      <div className="mt-1 font-medium">{component.edge3 || '-'}</div>
                    </div>
                    <div>
                      <label className="text-xs text-muted-foreground">Krawędź 4</label>
                      <div className="mt-1 font-medium">{component.edge4 || '-'}</div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            )}
          </div>

          {/* Sidebar - Product Info */}
          <div className="space-y-6">
            {/* Product Image */}
            {primaryImage && (
              <Card>
                <CardContent className="pt-6">
                  <img 
                    src={primaryImage} 
                    alt={component.product_title}
                    className="w-full h-auto rounded-lg"
                  />
                </CardContent>
              </Card>
            )}

            {/* Product Info */}
            <Card>
              <CardHeader>
                <CardTitle>Produkt źródłowy</CardTitle>
                <CardDescription>Z którego pochodzi formatka</CardDescription>
              </CardHeader>
              <CardContent className="space-y-3">
                <div>
                  <label className="text-xs text-muted-foreground">SKU</label>
                  <Link href={`/catalog-products/${component.product_id}`}>
                    <div className="mt-1 font-mono font-bold text-primary hover:underline flex items-center gap-1">
                      {component.product_sku}
                      <ExternalLink className="h-3 w-3" />
                    </div>
                  </Link>
                </div>

                <div>
                  <label className="text-xs text-muted-foreground">Nazwa</label>
                  <div className="mt-1 font-medium">{component.product_title}</div>
                </div>

                {component.product_type && (
                  <div>
                    <label className="text-xs text-muted-foreground">Typ</label>
                    <div className="mt-1">
                      <Badge variant="outline">{component.product_type}</Badge>
                    </div>
                  </div>
                )}

                {(component.product_length || component.product_width || component.product_height) && (
                  <div>
                    <label className="text-xs text-muted-foreground">Wymiary produktu</label>
                    <div className="mt-1 font-mono text-sm">
                      {component.product_length && `${component.product_length}mm`}
                      {component.product_length && component.product_width && ' × '}
                      {component.product_width && `${component.product_width}mm`}
                      {(component.product_length || component.product_width) && component.product_height && ' × '}
                      {component.product_height && `${component.product_height}mm`}
                    </div>
                  </div>
                )}

                {component.product_doors && (
                  <div>
                    <label className="text-xs text-muted-foreground">Drzwi</label>
                    <div className="mt-1">
                      <Badge variant="outline" className="text-xs">{component.product_doors}</Badge>
                    </div>
                  </div>
                )}

                {component.product_legs && (
                  <div>
                    <label className="text-xs text-muted-foreground">Nogi</label>
                    <div className="mt-1">
                      <Badge variant="outline" className="text-xs">{component.product_legs}</Badge>
                    </div>
                  </div>
                )}

                <div className="pt-3">
                  <Link href={`/catalog-products/${component.product_id}`}>
                    <Button variant="outline" size="sm" className="w-full">
                      Zobacz produkt
                      <ExternalLink className="ml-2 h-3 w-3" />
                    </Button>
                  </Link>
                </div>
              </CardContent>
            </Card>

            {/* Technical Info */}
            <Card>
              <CardHeader>
                <CardTitle>Informacje techniczne</CardTitle>
              </CardHeader>
              <CardContent className="space-y-2 text-sm">
                <div className="flex justify-between">
                  <span className="text-muted-foreground">ID formatki:</span>
                  <span className="font-mono">{component.id}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">ID BOM:</span>
                  <span className="font-mono">{component.product_bom_id}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">ID szablonu:</span>
                  <span className="font-mono">{component.component_template_id}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Typ komponentu:</span>
                  <span className="font-medium">{component.component_type}</span>
                </div>
              </CardContent>
            </Card>

            {/* Material Breakdown */}
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <CardTitle className="flex items-center gap-2">
                    <Calculator className="w-5 h-5" />
                    Rozk\u0142ad materia\u0142owy
                  </CardTitle>
                  {breakdown && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => recalculateMutation.mutate()}
                      disabled={recalculateMutation.isPending}
                      data-testid="button-recalculate-breakdown"
                    >
                      <RefreshCw className={`h-3 w-3 mr-2 ${recalculateMutation.isPending ? 'animate-spin' : ''}`} />
                      Przelicz
                    </Button>
                  )}
                </div>
                <CardDescription>
                  Zużycie płyty, obrzeży i kleju z cenami z magazynu
                </CardDescription>
              </CardHeader>
              <CardContent>
                {breakdownLoading ? (
                  <div className="space-y-2">
                    <Skeleton className="h-4 w-full" />
                    <Skeleton className="h-4 w-full" />
                    <Skeleton className="h-4 w-full" />
                  </div>
                ) : breakdown ? (
                  <div className="space-y-4">
                    {/* Płyta */}
                    <div className="border-b pb-3">
                      <h4 className="font-medium mb-2">Płyta meblowa</h4>
                      <div className="space-y-1 text-sm">
                        <div className="flex justify-between">
                          <span className="text-muted-foreground">Powierzchnia:</span>
                          <span className="font-mono">{breakdown.board_area_m2?.toFixed(4) || '0.0000'} m²</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-muted-foreground">Odpad ({breakdown.board_waste_percentage || 0}%):</span>
                          <span className="font-mono">{breakdown.board_total_m2?.toFixed(4) || '0.0000'} m²</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-muted-foreground">Cena jednostkowa:</span>
                          <span className="font-mono">{breakdown.board_unit_price?.toFixed(2) || '0.00'} zł/m²</span>
                        </div>
                        <div className="flex justify-between font-medium">
                          <span>Koszt płyty:</span>
                          <span className="font-mono text-primary">{breakdown.board_cost?.toFixed(2) || '0.00'} zł</span>
                        </div>
                      </div>
                    </div>

                    {/* Obrzeże */}
                    <div className="border-b pb-3">
                      <h4 className="font-medium mb-2">Obrzeże</h4>
                      <div className="space-y-1 text-sm">
                        <div className="flex justify-between">
                          <span className="text-muted-foreground">Długość:</span>
                          <span className="font-mono">{breakdown.edging_length_mb?.toFixed(2) || '0.00'} mb</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-muted-foreground">Naddatek (+{breakdown.edging_overage_cm || 0}cm):</span>
                          <span className="font-mono">{breakdown.edging_total_mb?.toFixed(2) || '0.00'} mb</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-muted-foreground">Cena jednostkowa:</span>
                          <span className="font-mono">{breakdown.edging_unit_price?.toFixed(2) || '0.00'} zł/mb</span>
                        </div>
                        <div className="flex justify-between font-medium">
                          <span>Koszt obrzeży:</span>
                          <span className="font-mono text-primary">{breakdown.edging_cost?.toFixed(2) || '0.00'} zł</span>
                        </div>
                      </div>
                    </div>

                    {/* Klej */}
                    <div className="border-b pb-3">
                      <h4 className="font-medium mb-2">Klej</h4>
                      <div className="space-y-1 text-sm">
                        <div className="flex justify-between">
                          <span className="text-muted-foreground">Zużycie:</span>
                          <span className="font-mono">{breakdown.adhesive_total_g?.toFixed(2) || '0.00'} g ({breakdown.adhesive_grams_per_mb || 0}g/mb)</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-muted-foreground">Cena jednostkowa:</span>
                          <span className="font-mono">{breakdown.adhesive_unit_price?.toFixed(3) || '0.000'} zł/g</span>
                        </div>
                        <div className="flex justify-between font-medium">
                          <span>Koszt kleju:</span>
                          <span className="font-mono text-primary">{breakdown.adhesive_cost?.toFixed(2) || '0.00'} zł</span>
                        </div>
                      </div>
                    </div>

                    {/* Całkowity koszt */}
                    <div className="bg-primary/5 rounded-lg p-3">
                      <div className="flex justify-between items-center">
                        <span className="font-semibold text-lg">CAŁKOWITY KOSZT:</span>
                        <span className="font-bold text-2xl text-primary font-mono" data-testid="text-total-cost">
                          {breakdown.total_cost?.toFixed(2) || '0.00'} zł
                        </span>
                      </div>
                      <p className="text-xs text-muted-foreground mt-1">
                        Przeliczono: {new Date(breakdown.calculated_at).toLocaleString('pl-PL')}
                      </p>
                    </div>
                  </div>
                ) : (
                  <div className="text-center py-6 text-muted-foreground">
                    <p className="mb-3">Brak rozkładu materiałowego</p>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => saveMutation.mutate()}
                      disabled={saveMutation.isPending || breakdownLoading}
                      data-testid="button-calculate-breakdown"
                    >
                      {saveMutation.isPending ? (
                        <RefreshCw className="h-3 w-3 mr-2 animate-spin" />
                      ) : (
                        <Calculator className="h-3 w-3 mr-2" />
                      )}
                      {saveMutation.isPending ? 'Obliczam...' : 'Oblicz teraz'}
                    </Button>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
}
