import { useMemo } from 'react';
import { Badge } from '@/components/ui/badge';
import { useFilterTabs, FilterTab } from './use-filter-tabs';

export interface FilterTabsBarProps {
  planId: string;
  storageKeyPrefix: string;
  searchFilter: string;
  onSearchChange: (value: string) => void;
  testIdPrefix?: string;
}

export interface FilterTabsData {
  persistentTabs: FilterTab[];
  addTab: (label: string) => void;
  removeTab: (tabId: string) => void;
  toggleTab: (tabId: string) => void;
  reorderTabs: (newTabs: FilterTab[]) => void;
}

export function useFilterTabsBar({ 
  planId, 
  storageKeyPrefix,
  searchFilter, 
  onSearchChange,
  testIdPrefix = 'filter-tab'
}: FilterTabsBarProps) {
  const { tabs: persistentTabs, addTab, toggleTab, removeTab, reorderTabs } = useFilterTabs({
    planId,
    storageKeyPrefix
  });

  // Create temporary tabs from searchFilter (tokens not in persistent tabs)
  const temporaryTabs = useMemo(() => {
    const filterTokens = searchFilter
      ? searchFilter.split(';').map(s => s.trim().toUpperCase()).filter(Boolean)
      : [];
    
    const persistentTokens = persistentTabs.map(t => t.token);
    const uniqueTokens = filterTokens.filter(token => !persistentTokens.includes(token));
    
    return uniqueTokens.map(token => ({
      id: `temp-${token}`,
      label: token,
      token: token,
      isActive: true,
      isTemporary: true
    }));
  }, [searchFilter, persistentTabs]);

  // Combine persistent and temporary tabs
  const allTabs = useMemo(() => {
    const persistentWithFlags = persistentTabs.map(tab => ({
      ...tab,
      isTemporary: false
    }));
    return [...persistentWithFlags, ...temporaryTabs];
  }, [persistentTabs, temporaryTabs]);

  const handleAddTab = (label: string) => {
    const sanitizedToken = label.trim().toUpperCase();
    addTab(sanitizedToken);
    
    // Add to search filter
    const existingFilterParts = searchFilter 
      ? searchFilter.split(';').map(s => s.trim().toUpperCase()).filter(Boolean) 
      : [];
    
    if (!existingFilterParts.includes(sanitizedToken)) {
      const newFilter = existingFilterParts.length > 0 
        ? `${searchFilter};${sanitizedToken}`
        : sanitizedToken;
      onSearchChange(newFilter);
    }
  };

  const handleToggleTab = (tab: FilterTab & { isTemporary: boolean }) => {
    if (tab.isTemporary) {
      // Temporary tab - remove from search filter
      const filterParts = searchFilter 
        ? searchFilter.split(';').map(s => s.trim().toUpperCase()).filter(Boolean) 
        : [];
      const newParts = filterParts.filter(part => part !== tab.token);
      onSearchChange(newParts.join(';'));
    } else {
      // Persistent tab - toggle active state and sync to search
      toggleTab(tab.id);
      
      const filterParts = searchFilter 
        ? searchFilter.split(';').map(s => s.trim().toUpperCase()).filter(Boolean) 
        : [];
      
      if (tab.isActive) {
        // Currently active, will be deactivated - remove from search
        const newParts = filterParts.filter(part => part !== tab.token);
        onSearchChange(newParts.join(';'));
      } else {
        // Currently inactive, will be activated - add to search
        if (!filterParts.includes(tab.token)) {
          const newFilter = filterParts.length > 0 
            ? [...filterParts, tab.token].join(';')
            : tab.token;
          onSearchChange(newFilter);
        }
      }
    }
  };

  const handleRemoveTab = (tabId: string) => {
    const tab = persistentTabs.find(t => t.id === tabId);
    if (!tab) return;
    
    // Remove from persistent tabs
    removeTab(tabId);
    
    // Remove from search filter
    const filterParts = searchFilter 
      ? searchFilter.split(';').map(s => s.trim().toUpperCase()).filter(Boolean) 
      : [];
    const newParts = filterParts.filter(part => part !== tab.token);
    onSearchChange(newParts.join(';'));
  };

  const handleManageToggleTab = (tabId: string) => {
    const tab = persistentTabs.find(t => t.id === tabId);
    if (!tab) return;
    
    toggleTab(tabId);
    
    const filterParts = searchFilter 
      ? searchFilter.split(';').map(s => s.trim().toUpperCase()).filter(Boolean) 
      : [];
    
    if (tab.isActive) {
      // Currently active, will be deactivated - remove from search
      const newParts = filterParts.filter(part => part !== tab.token);
      onSearchChange(newParts.join(';'));
    } else {
      // Currently inactive, will be activated - add to search
      if (!filterParts.includes(tab.token)) {
        const newFilter = filterParts.length > 0 
          ? [...filterParts, tab.token].join(';')
          : tab.token;
        onSearchChange(newFilter);
      }
    }
  };

  // Return tabs data for use by ManageFiltersDialog at parent level
  const tabsData: FilterTabsData = {
    persistentTabs,
    addTab: handleAddTab,
    removeTab: handleRemoveTab,
    toggleTab: handleManageToggleTab,
    reorderTabs
  };

  return {
    tabs: (
      <div className="flex items-center gap-2 flex-wrap" style={{ minHeight: '25px' }}>
        {allTabs.map(tab => (
          <Badge
            key={tab.id}
            variant={tab.isActive ? 'default' : 'outline'}
            className="cursor-pointer hover-elevate active-elevate-2"
            onClick={() => handleToggleTab(tab)}
            data-testid={`badge-${testIdPrefix}-${tab.token.toLowerCase()}`}
            title={tab.isTemporary ? 'Kliknij aby usunąć z filtra' : 'Kliknij aby przełączyć'}
          >
            {tab.label}
          </Badge>
        ))}
      </div>
    ),
    tabsData
  };
}

// Component version for backward compatibility (renders only tabs, no search/settings)
export function FilterTabsBar(props: FilterTabsBarProps) {
  const { tabs } = useFilterTabsBar(props);
  return tabs;
}
