import { type Express } from "express";
import { pool } from "../../postgres";
import { z } from "zod";

// Validation schemas
const createGroupSchema = z.object({
  name: z.string().min(1).max(255),
  color_hex: z.string().regex(/^#[0-9A-Fa-f]{6}$/),
  is_active: z.boolean().default(true),
});

const updateGroupSchema = createGroupSchema.partial();

export function registerProductGroupRoutes(app: Express) {
  // GET /api/production/product-groups - List all groups
  app.get("/api/production/product-groups", async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT * FROM production.production_product_groups
        ORDER BY created_at DESC
      `);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching product groups:", error);
      res.status(500).json({ message: "Failed to fetch product groups" });
    }
  });

  // GET /api/production/product-groups/:id - Get single group with items
  app.get("/api/production/product-groups/:id", async (req, res) => {
    try {
      const groupId = parseInt(req.params.id);
      
      const groupResult = await pool.query(
        `SELECT * FROM production.production_product_groups WHERE id = $1`,
        [groupId]
      );

      if (groupResult.rows.length === 0) {
        return res.status(404).json({ message: "Product group not found" });
      }

      const itemsResult = await pool.query(
        `SELECT * FROM production.production_product_group_items WHERE group_id = $1`,
        [groupId]
      );

      res.json({ ...groupResult.rows[0], items: itemsResult.rows });
    } catch (error) {
      console.error("Error fetching product group:", error);
      res.status(500).json({ message: "Failed to fetch product group" });
    }
  });

  // POST /api/production/product-groups - Create new group
  app.post("/api/production/product-groups", async (req, res) => {
    try {
      const data = createGroupSchema.parse(req.body);
      
      const result = await pool.query(
        `INSERT INTO production.production_product_groups (name, color_hex, is_active)
         VALUES ($1, $2, $3)
         RETURNING *`,
        [data.name, data.color_hex, data.is_active]
      );

      res.status(201).json(result.rows[0]);
    } catch (error: any) {
      console.error("Error creating product group:", error);
      
      if (error.name === "ZodError") {
        return res.status(400).json({ message: "Invalid data", errors: error.errors });
      }
      
      if (error.code === "23505") {
        return res.status(409).json({ message: "Product group with this name already exists" });
      }
      
      res.status(500).json({ message: "Failed to create product group" });
    }
  });

  // PATCH /api/production/product-groups/:id - Update group
  app.patch("/api/production/product-groups/:id", async (req, res) => {
    try {
      const groupId = parseInt(req.params.id);
      const data = updateGroupSchema.parse(req.body);
      
      const updates: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;

      if (data.name !== undefined) {
        updates.push(`name = $${paramIndex++}`);
        values.push(data.name);
      }
      if (data.color_hex !== undefined) {
        updates.push(`color_hex = $${paramIndex++}`);
        values.push(data.color_hex);
      }
      if (data.is_active !== undefined) {
        updates.push(`is_active = $${paramIndex++}`);
        values.push(data.is_active);
      }

      if (updates.length === 0) {
        return res.status(400).json({ message: "No fields to update" });
      }

      values.push(groupId);
      
      const result = await pool.query(
        `UPDATE production.production_product_groups
         SET ${updates.join(', ')}
         WHERE id = $${paramIndex}
         RETURNING *`,
        values
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Product group not found" });
      }

      res.json(result.rows[0]);
    } catch (error: any) {
      console.error("Error updating product group:", error);
      
      if (error.name === "ZodError") {
        return res.status(400).json({ message: "Invalid data", errors: error.errors });
      }
      
      if (error.code === "23505") {
        return res.status(409).json({ message: "Product group with this name already exists" });
      }
      
      res.status(500).json({ message: "Failed to update product group" });
    }
  });

  // DELETE /api/production/product-groups/:id - Delete group
  app.delete("/api/production/product-groups/:id", async (req, res) => {
    try {
      const groupId = parseInt(req.params.id);
      
      // Check if group has items
      const itemsCheck = await pool.query(
        `SELECT id FROM production.production_product_group_items WHERE group_id = $1 LIMIT 1`,
        [groupId]
      );

      if (itemsCheck.rows.length > 0) {
        return res.status(409).json({ 
          message: "Cannot delete product group with assigned products. Remove all products first." 
        });
      }

      const result = await pool.query(
        `DELETE FROM production.production_product_groups WHERE id = $1 RETURNING *`,
        [groupId]
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Product group not found" });
      }

      res.json({ message: "Product group deleted successfully" });
    } catch (error) {
      console.error("Error deleting product group:", error);
      res.status(500).json({ message: "Failed to delete product group" });
    }
  });

  // POST /api/production/product-groups/:id/products/:productId - Add product to group
  app.post("/api/production/product-groups/:id/products/:productId", async (req, res) => {
    try {
      const groupId = parseInt(req.params.id);
      const productId = parseInt(req.params.productId);
      
      // Check if group exists
      const groupCheck = await pool.query(
        `SELECT id FROM production.production_product_groups WHERE id = $1`,
        [groupId]
      );

      if (groupCheck.rows.length === 0) {
        return res.status(404).json({ message: "Product group not found" });
      }

      const result = await pool.query(
        `INSERT INTO production.production_product_group_items (group_id, product_id)
         VALUES ($1, $2)
         RETURNING *`,
        [groupId, productId]
      );

      res.status(201).json(result.rows[0]);
    } catch (error: any) {
      console.error("Error adding product to group:", error);
      
      if (error.code === "23505") {
        return res.status(409).json({ message: "Product already in this group" });
      }
      
      if (error.code === "23503") {
        return res.status(404).json({ message: "Product not found" });
      }
      
      res.status(500).json({ message: "Failed to add product to group" });
    }
  });

  // DELETE /api/production/product-groups/:id/products/:productId - Remove product from group
  app.delete("/api/production/product-groups/:id/products/:productId", async (req, res) => {
    try {
      const groupId = parseInt(req.params.id);
      const productId = parseInt(req.params.productId);
      
      const result = await pool.query(
        `DELETE FROM production.production_product_group_items 
         WHERE group_id = $1 AND product_id = $2
         RETURNING *`,
        [groupId, productId]
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Product not found in this group" });
      }

      res.json({ message: "Product removed from group successfully" });
    } catch (error) {
      console.error("Error removing product from group:", error);
      res.status(500).json({ message: "Failed to remove product from group" });
    }
  });
}
