import { useQuery, useMutation } from "@tanstack/react-query";
import { useRoute, useLocation } from "wouter";
import { ArrowLeft, Edit, Play, CheckCircle, XCircle, Loader2, Calendar, Package, User, MapPin, Image, Filter, Grid3x3, List, AlertTriangle } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import { apiRequest } from "@/lib/queryClient";
import type { ProductionOrder } from "@shared/schema";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { useState, useMemo } from "react";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";

const statusColors: Record<string, string> = {
  draft: "bg-gray-500",
  confirmed: "bg-blue-500",
  planned: "bg-indigo-500",
  in_progress: "bg-yellow-500",
  paused: "bg-orange-500",
  done: "bg-green-500",
  cancelled: "bg-red-500",
};

const statusLabels: Record<string, string> = {
  draft: "Projekt",
  confirmed: "Potwierdzone",
  planned: "Zaplanowane",
  in_progress: "W realizacji",
  paused: "Wstrzymane",
  done: "Zakończone",
  cancelled: "Anulowane",
};

const priorityColors: Record<string, string> = {
  low: "bg-gray-400",
  normal: "bg-blue-400",
  high: "bg-orange-400",
  urgent: "bg-red-500",
};

const priorityLabels: Record<string, string> = {
  low: "Niski",
  normal: "Normalny",
  high: "Wysoki",
  urgent: "Pilny",
};

const workflowStageColors: Record<string, string> = {
  magazine: "bg-gray-500",
  cutting: "bg-blue-500",
  edging: "bg-indigo-500",
  drilling: "bg-purple-500",
  upholstering: "bg-pink-500",
  picking: "bg-yellow-500",
  packing: "bg-orange-500",
  strapping: "bg-amber-500",
  ready: "bg-green-500",
  shipped: "bg-emerald-600",
};

const workflowStageLabels: Record<string, string> = {
  magazine: "Magazyn",
  cutting: "Cięcie",
  edging: "Oklejanie",
  drilling: "Wiercenie",
  upholstering: "Tapicerowanie",
  picking: "Kompletowanie",
  packing: "Pakowanie",
  strapping: "Bandowanie",
  ready: "Gotowe",
  shipped: "Wysłane",
};

export default function ProductionOrderDetailPage() {
  const [, params] = useRoute("/production/orders/:id");
  const [, setLocation] = useLocation();
  const { toast } = useToast();

  const orderId = params?.id ? parseInt(params.id) : null;
  
  // Formatki filters state
  const [viewMode, setViewMode] = useState<'list' | 'gallery'>('list');
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedColor, setSelectedColor] = useState<string>('all');
  const [selectedEdging, setSelectedEdging] = useState<string>('all');
  const [selectedDrilling, setSelectedDrilling] = useState<string>('all');
  const [selectedBoardType, setSelectedBoardType] = useState<string>('all');
  const [showDamagedOnly, setShowDamagedOnly] = useState(false);
  const [selectedFormatka, setSelectedFormatka] = useState<any | null>(null);

  // Fetch order
  const { data: order, isLoading } = useQuery<ProductionOrder>({
    queryKey: ["/api/production/orders", orderId],
    queryFn: async () => {
      const response = await fetch(`/api/production/orders/${orderId}`, {
        credentials: "include",
      });
      if (!response.ok) {
        throw new Error("Failed to fetch order");
      }
      return await response.json();
    },
    enabled: !!orderId,
  });

  // Status transition mutations
  const startMutation = useMutation({
    mutationFn: async () => {
      const res = await apiRequest("POST", `/api/production/orders/${orderId}/start`);
      return await res.json();
    },
    onSuccess: () => {
      // Invalidate all queries related to orders (including filtered views)
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Zlecenie rozpoczęte",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się rozpocząć zlecenia",
      });
    },
  });

  const completeMutation = useMutation({
    mutationFn: async () => {
      const res = await apiRequest("POST", `/api/production/orders/${orderId}/complete`);
      return await res.json();
    },
    onSuccess: () => {
      // Invalidate all queries related to orders (including filtered views)
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Zlecenie zakończone",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zakończyć zlecenia",
      });
    },
  });

  const cancelMutation = useMutation({
    mutationFn: async () => {
      const res = await apiRequest("POST", `/api/production/orders/${orderId}/cancel`);
      return await res.json();
    },
    onSuccess: () => {
      // Invalidate all queries related to orders (including filtered views)
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Zlecenie anulowane",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się anulować zlecenia",
      });
    },
  });

  // Update workflow stage mutation
  const updateWorkflowStageMutation = useMutation({
    mutationFn: async (workflowStage: string) => {
      const res = await apiRequest("PATCH", `/api/production/orders/${orderId}/workflow-stage`, { workflowStage });
      return await res.json();
    },
    onSuccess: () => {
      // Invalidate all queries related to orders (including filtered views)
      queryClient.invalidateQueries({
        predicate: (query) => {
          const firstKey = query.queryKey[0];
          return firstKey === "/api/production/orders";
        },
      });
      toast({
        title: "Sukces",
        description: "Etap workflow zaktualizowany",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować etapu workflow",
      });
    },
  });

  const formatDate = (date: Date | null | undefined) => {
    if (!date) return "-";
    return format(new Date(date), "dd.MM.yyyy HH:mm", { locale: pl });
  };

  const formatDateShort = (date: Date | null | undefined) => {
    if (!date) return "-";
    return format(new Date(date), "dd.MM.yyyy", { locale: pl });
  };

  // Filter and group components
  const filteredComponents = useMemo(() => {
    if (!order || !(order as any).components) return [];
    
    const components = (order as any).components;
    
    return components.filter((comp: any) => {
      // Search filter
      if (searchQuery && !comp.generatedName?.toLowerCase().includes(searchQuery.toLowerCase())) {
        return false;
      }
      
      // Color filter
      if (selectedColor !== 'all' && comp.color !== selectedColor) {
        return false;
      }
      
      // Edging filter
      if (selectedEdging !== 'all' && comp.edgingPattern !== selectedEdging) {
        return false;
      }
      
      // Drilling filter
      if (selectedDrilling !== 'all' && comp.drillingPattern !== selectedDrilling) {
        return false;
      }
      
      // Board type filter
      if (selectedBoardType !== 'all' && comp.boardType !== selectedBoardType) {
        return false;
      }
      
      // Damaged only filter
      if (showDamagedOnly && !comp.isDamaged) {
        return false;
      }
      
      return true;
    });
  }, [order, searchQuery, selectedColor, selectedEdging, selectedDrilling, selectedBoardType, showDamagedOnly]);

  // Extract unique values for filters
  const filterOptions = useMemo(() => {
    if (!order || !(order as any).components) return { colors: [], edgingPatterns: [], drillingPatterns: [], boardTypes: [] };
    
    const components = (order as any).components;
    
    return {
      colors: Array.from(new Set(components.map((c: any) => c.color).filter(Boolean))),
      edgingPatterns: Array.from(new Set(components.map((c: any) => c.edgingPattern).filter(Boolean))),
      drillingPatterns: Array.from(new Set(components.map((c: any) => c.drillingPattern).filter(Boolean))),
      boardTypes: Array.from(new Set(components.map((c: any) => c.boardType).filter(Boolean))),
    };
  }, [order]);

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
      </div>
    );
  }

  if (!order) {
    return (
      <div className="flex flex-col items-center justify-center h-full">
        <Package className="h-16 w-16 text-muted-foreground mb-4" />
        <p className="text-muted-foreground">Nie znaleziono zlecenia produkcyjnego</p>
        <Button
          variant="outline"
          className="mt-4"
          onClick={() => setLocation("/production/orders")}
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Powrót do listy
        </Button>
      </div>
    );
  }

  return (
    <div className="flex flex-col h-full">
      {/* Header */}
      <div className="border-b p-4">
        <div className="flex items-center gap-4 mb-4">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => setLocation("/production/orders")}
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div className="flex-1">
            <h1 className="text-2xl font-bold font-mono" data-testid="text-title">
              {order.orderNumber}
            </h1>
            <p className="text-sm text-muted-foreground">Zlecenie Produkcyjne</p>
          </div>
          <div className="flex items-center gap-2">
            <Button
              size="sm"
              variant="outline"
              onClick={() => {/* TODO: Open edit dialog */}}
              data-testid="button-edit"
            >
              <Edit className="h-4 w-4 mr-2" />
              Edytuj
            </Button>
            {order.status === 'draft' || order.status === 'confirmed' || order.status === 'planned' ? (
              <Button
                size="sm"
                onClick={() => startMutation.mutate()}
                disabled={startMutation.isPending}
                data-testid="button-start"
              >
                <Play className="h-4 w-4 mr-2" />
                Rozpocznij
              </Button>
            ) : null}
            {order.status === 'in_progress' ? (
              <Button
                size="sm"
                onClick={() => completeMutation.mutate()}
                disabled={completeMutation.isPending}
                data-testid="button-complete"
              >
                <CheckCircle className="h-4 w-4 mr-2" />
                Zakończ
              </Button>
            ) : null}
            {order.status !== 'done' && order.status !== 'cancelled' ? (
              <Button
                size="sm"
                variant="destructive"
                onClick={() => cancelMutation.mutate()}
                disabled={cancelMutation.isPending}
                data-testid="button-cancel"
              >
                <XCircle className="h-4 w-4 mr-2" />
                Anuluj
              </Button>
            ) : null}
          </div>
        </div>

        {/* Status and Workflow badges */}
        <div className="flex items-center gap-4">
          <div className="flex items-center gap-2">
            <span className="text-sm text-muted-foreground">Status:</span>
            <Badge className={statusColors[order.status]}>
              {statusLabels[order.status]}
            </Badge>
          </div>
          
          <div className="flex items-center gap-2">
            <span className="text-sm text-muted-foreground">Etap:</span>
            {order.workflowStage ? (
              <select
                value={order.workflowStage}
                onChange={(e) => updateWorkflowStageMutation.mutate(e.target.value)}
                disabled={updateWorkflowStageMutation.isPending || order.status === 'done' || order.status === 'cancelled'}
                className="h-8 px-3 rounded-md border bg-background text-sm font-medium"
                data-testid="select-workflow-stage"
              >
                {Object.entries(workflowStageLabels).map(([value, label]) => (
                  <option key={value} value={value}>{label}</option>
                ))}
              </select>
            ) : (
              <select
                onChange={(e) => updateWorkflowStageMutation.mutate(e.target.value)}
                disabled={updateWorkflowStageMutation.isPending || order.status === 'done' || order.status === 'cancelled'}
                className="h-8 px-3 rounded-md border bg-background text-sm text-muted-foreground"
                data-testid="select-workflow-stage"
                value=""
              >
                <option value="">Ustaw etap...</option>
                {Object.entries(workflowStageLabels).map(([value, label]) => (
                  <option key={value} value={value}>{label}</option>
                ))}
              </select>
            )}
          </div>

          <div className="flex items-center gap-2">
            <span className="text-sm text-muted-foreground">Priorytet:</span>
            <Badge variant="outline" className={priorityColors[order.priority]}>
              {priorityLabels[order.priority]}
            </Badge>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="flex-1 overflow-auto p-4">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
          {/* Main Info */}
          <Card>
            <CardHeader>
              <CardTitle>Informacje Podstawowe</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <p className="text-sm text-muted-foreground">Numer Zlecenia</p>
                <p className="font-mono font-semibold">{order.orderNumber}</p>
              </div>

              <Separator />

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-muted-foreground">Ilość Planowana</p>
                  <p className="font-semibold">{order.quantityPlanned} {order.unitOfMeasure}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">Ilość Wyprodukowana</p>
                  <p className="font-semibold">{order.quantityProduced ?? 0} {order.unitOfMeasure}</p>
                </div>
              </div>

              {typeof order.quantityScrap === 'number' && order.quantityScrap > 0 && (
                <div>
                  <p className="text-sm text-muted-foreground">Ilość Braków</p>
                  <p className="font-semibold text-destructive">{order.quantityScrap} {order.unitOfMeasure}</p>
                </div>
              )}

              <Separator />

              <div>
                <p className="text-sm text-muted-foreground">ID Produktu</p>
                <p>{order.productId}</p>
              </div>

              {order.bomId && (
                <div>
                  <p className="text-sm text-muted-foreground">ID BOM</p>
                  <p>{order.bomId}</p>
                </div>
              )}

              {order.routingId && (
                <div>
                  <p className="text-sm text-muted-foreground">ID Marszruty</p>
                  <p>{order.routingId}</p>
                </div>
              )}

              {order.sourceOrderNumber && (
                <div>
                  <p className="text-sm text-muted-foreground">Źródłowe Zamówienie</p>
                  <p className="font-mono">{order.sourceOrderNumber}</p>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Dates */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Calendar className="h-5 w-5" />
                Harmonogram
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-muted-foreground">Planowany Start</p>
                  <p>{formatDateShort(order.plannedStartDate)}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">Planowany Koniec</p>
                  <p>{formatDateShort(order.plannedEndDate)}</p>
                </div>
              </div>

              {(order.actualStartDate || order.actualEndDate) && (
                <>
                  <Separator />
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-sm text-muted-foreground">Rzeczywisty Start</p>
                      <p className="font-semibold">{formatDate(order.actualStartDate)}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Rzeczywisty Koniec</p>
                      <p className="font-semibold">{formatDate(order.actualEndDate)}</p>
                    </div>
                  </div>
                </>
              )}

              <Separator />

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-muted-foreground">Utworzone</p>
                  <p className="text-sm">{formatDate(order.createdAt)}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">Ostatnia Aktualizacja</p>
                  <p className="text-sm">{formatDate(order.updatedAt)}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Additional Info */}
          <Card className="lg:col-span-2">
            <CardHeader>
              <CardTitle>Dodatkowe Informacje</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                {order.responsibleUserId && (
                  <div className="flex items-start gap-2">
                    <User className="h-5 w-5 text-muted-foreground mt-0.5" />
                    <div>
                      <p className="text-sm text-muted-foreground">Odpowiedzialny</p>
                      <p>ID: {order.responsibleUserId}</p>
                    </div>
                  </div>
                )}

                {order.locationId && (
                  <div className="flex items-start gap-2">
                    <MapPin className="h-5 w-5 text-muted-foreground mt-0.5" />
                    <div>
                      <p className="text-sm text-muted-foreground">Lokalizacja</p>
                      <p>ID: {order.locationId}</p>
                    </div>
                  </div>
                )}
              </div>

              {order.notes && (
                <>
                  <Separator />
                  <div>
                    <p className="text-sm text-muted-foreground mb-2">Uwagi</p>
                    <p className="text-sm whitespace-pre-wrap">{order.notes}</p>
                  </div>
                </>
              )}
            </CardContent>
          </Card>

          {/* Komponenty i Formatki */}
          {(order as any).components && (order as any).components.length > 0 && (
            <Card className="lg:col-span-2">
              <CardHeader>
                <div className="flex items-center justify-between">
                  <CardTitle className="flex items-center gap-2">
                    <Package className="h-5 w-5" />
                    Komponenty i Formatki ({filteredComponents.length}/{(order as any).components.length})
                  </CardTitle>
                  <div className="flex items-center gap-2">
                    <Button
                      size="sm"
                      variant={viewMode === 'list' ? 'default' : 'outline'}
                      onClick={() => setViewMode('list')}
                      data-testid="button-view-list"
                    >
                      <List className="h-4 w-4" />
                    </Button>
                    <Button
                      size="sm"
                      variant={viewMode === 'gallery' ? 'default' : 'outline'}
                      onClick={() => setViewMode('gallery')}
                      data-testid="button-view-gallery"
                    >
                      <Grid3x3 className="h-4 w-4" />
                    </Button>
                  </div>
                </div>

                {/* Filters */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-3 mt-4">
                  <Input
                    placeholder="Szukaj komponentu..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    data-testid="input-search-component"
                  />
                  
                  <Select value={selectedColor} onValueChange={setSelectedColor}>
                    <SelectTrigger data-testid="select-color-filter">
                      <SelectValue placeholder="Wszystkie kolory" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Wszystkie kolory</SelectItem>
                      {(filterOptions.colors as string[]).map((color: string) => (
                        <SelectItem key={color} value={color}>{color}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  
                  <Select value={selectedEdging} onValueChange={setSelectedEdging}>
                    <SelectTrigger data-testid="select-edging-filter">
                      <SelectValue placeholder="Wszystkie oklejania" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Wszystkie oklejania</SelectItem>
                      {(filterOptions.edgingPatterns as string[]).map((pattern: string) => (
                        <SelectItem key={pattern} value={pattern}>{pattern}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  
                  <Select value={selectedDrilling} onValueChange={setSelectedDrilling}>
                    <SelectTrigger data-testid="select-drilling-filter">
                      <SelectValue placeholder="Wszystkie wiercenia" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Wszystkie wiercenia</SelectItem>
                      {(filterOptions.drillingPatterns as string[]).map((pattern: string) => (
                        <SelectItem key={pattern} value={pattern}>{pattern}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              </CardHeader>

              <CardContent>
                {viewMode === 'list' ? (
                  <div className="space-y-2">
                    {filteredComponents.map((comp: any) => (
                      <div
                        key={comp.id}
                        className="flex items-center gap-4 p-3 border rounded-md hover-elevate cursor-pointer"
                        onClick={() => setSelectedFormatka(comp)}
                        data-testid={`component-item-${comp.id}`}
                      >
                        {comp.visualizationUrl ? (
                          <img
                            src={comp.visualizationUrl}
                            alt={comp.generatedName}
                            className="w-20 h-20 object-contain border rounded"
                          />
                        ) : (
                          <div className="w-20 h-20 bg-muted flex items-center justify-center border rounded">
                            <Image className="h-8 w-8 text-muted-foreground" />
                          </div>
                        )}
                        <div className="flex-1 min-w-0">
                          <div className="flex items-center gap-2">
                            <p className="font-medium truncate">{comp.generatedName}</p>
                            {comp.isDamaged && (
                              <Badge variant="destructive" className="shrink-0">
                                <AlertTriangle className="h-3 w-3 mr-1" />
                                Uszkodzony
                              </Badge>
                            )}
                          </div>
                          <div className="flex items-center gap-3 mt-1 text-sm text-muted-foreground">
                            {comp.color && <span>Kolor: {comp.color}</span>}
                            {comp.edgingPattern && <span>Okl.: {comp.edgingPattern}</span>}
                            {comp.drillingPattern && <span>Wierc.: {comp.drillingPattern}</span>}
                            {comp.calculatedLength && comp.calculatedWidth && (
                              <span>{comp.calculatedLength} × {comp.calculatedWidth} mm</span>
                            )}
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                    {filteredComponents.map((comp: any) => (
                      <div
                        key={comp.id}
                        className="border rounded-md p-3 hover-elevate cursor-pointer"
                        onClick={() => setSelectedFormatka(comp)}
                        data-testid={`component-card-${comp.id}`}
                      >
                        {comp.visualizationUrl ? (
                          <img
                            src={comp.visualizationUrl}
                            alt={comp.generatedName}
                            className="w-full aspect-square object-contain border rounded mb-2"
                          />
                        ) : (
                          <div className="w-full aspect-square bg-muted flex items-center justify-center border rounded mb-2">
                            <Image className="h-12 w-12 text-muted-foreground" />
                          </div>
                        )}
                        <p className="text-sm font-medium truncate">{comp.generatedName}</p>
                        {comp.isDamaged && (
                          <Badge variant="destructive" className="mt-2">Uszkodzony</Badge>
                        )}
                      </div>
                    ))}
                  </div>
                )}

                {filteredComponents.length === 0 && (
                  <div className="text-center py-8 text-muted-foreground">
                    Brak komponentów spełniających kryteria filtrowania
                  </div>
                )}
              </CardContent>
            </Card>
          )}
        </div>
      </div>

      {/* Formatka Detail Modal */}
      <Dialog open={!!selectedFormatka} onOpenChange={(open) => !open && setSelectedFormatka(null)}>
        <DialogContent className="max-w-3xl">
          <DialogHeader>
            <DialogTitle>{selectedFormatka?.generatedName}</DialogTitle>
            <DialogDescription>
              Szczegóły komponentu
            </DialogDescription>
          </DialogHeader>
          {selectedFormatka && (
            <div className="space-y-4">
              {selectedFormatka.visualizationUrl && (
                <img
                  src={selectedFormatka.visualizationUrl}
                  alt={selectedFormatka.generatedName}
                  className="w-full max-h-96 object-contain border rounded"
                />
              )}
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <p className="text-muted-foreground">Typ</p>
                  <p className="font-medium">{selectedFormatka.componentType}</p>
                </div>
                {selectedFormatka.color && (
                  <div>
                    <p className="text-muted-foreground">Kolor</p>
                    <p className="font-medium">{selectedFormatka.color}</p>
                  </div>
                )}
                {selectedFormatka.boardType && (
                  <div>
                    <p className="text-muted-foreground">Typ płyty</p>
                    <p className="font-medium">{selectedFormatka.boardType}</p>
                  </div>
                )}
                {selectedFormatka.edgingPattern && (
                  <div>
                    <p className="text-muted-foreground">Oklejanie</p>
                    <p className="font-medium">{selectedFormatka.edgingPattern}</p>
                  </div>
                )}
                {selectedFormatka.drillingPattern && (
                  <div>
                    <p className="text-muted-foreground">Wiercenie</p>
                    <p className="font-medium">{selectedFormatka.drillingPattern}</p>
                  </div>
                )}
                {selectedFormatka.calculatedLength && selectedFormatka.calculatedWidth && (
                  <div>
                    <p className="text-muted-foreground">Wymiary</p>
                    <p className="font-medium">{selectedFormatka.calculatedLength} × {selectedFormatka.calculatedWidth} mm</p>
                  </div>
                )}
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
