import { useRef, useEffect } from "react";
import { useRoute, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Loader2, ArrowLeft, X, Image as ImageIcon } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { apiRequest, queryClient } from "@/lib/queryClient";
import type { ProductCreatorDictionary, DictionaryType } from "@shared/schema";

const dictionaryFormSchema = z.object({
  dictionaryType: z.string().min(1, "Typ słownika jest wymagany"),
  code: z.string().min(1, "Kod jest wymagany"),
  name: z.string().min(1, "Nazwa jest wymagana"),
  readableName: z.string().optional(),
  shortName: z.string().optional(),
  description: z.string().optional(),
  color: z.string().regex(/^#[0-9A-Fa-f]{6}$/, "Nieprawidłowy format koloru HEX").optional().or(z.literal("")),
  category: z.string().optional(),
  sortOrder: z.coerce.number().default(0),
  isActive: z.boolean().default(true),
});

type DictionaryFormData = z.infer<typeof dictionaryFormSchema>;

const DICTIONARY_LABELS: Record<DictionaryType, { label: string; description: string }> = {
  product_type: { label: "Rodzaj produktu", description: "SUPRA, TRES, VB, BINI" },
  unit: { label: "Jednostka", description: "BOCZKI, BOK-L, BOK-P, POLKA, DRZWI" },
  color: { label: "Kolor", description: "BIAŁY, KASZMIR, WOTAN" },
  material: { label: "Materiał", description: "płyta meblowa, szkło, metal" },
  product_group: { label: "Grupa produktów", description: "A1, A2, A3, A4" },
  door: { label: "Drzwi", description: "D1, D2, D3, D4" },
  leg: { label: "Nóżki", description: "N1, N2, N3, N4" },
  fabric: { label: "Tkaniny", description: "ES1, ES2" },
  dimension_length: { label: "Wymiar - długość", description: "50-100 cm" },
  dimension_width: { label: "Wymiar - szerokość", description: "30-36 cm" },
  dimension_height: { label: "Wymiar - wysokość", description: "45/82/205 cm" },
  element_type: { label: "Typ elementu", description: "FORMATKA, OKUCIE, PAKOWANIE" },
  component_cz1: { label: "Komponent cz1", description: "BOCZKI, DNO, DRZWI" },
  component_cz2: { label: "Komponent cz2", description: "SZUFLADA, TRES, VB" },
};

export default function DictionaryEntryEditPage() {
  const [, params] = useRoute("/dictionaries/:type/:id/edit");
  const [, navigate] = useLocation();
  const dictionaryType = params?.type as DictionaryType;
  const entryId = params?.id ? parseInt(params.id) : null;
  const { toast } = useToast();
  const fileInputRef = useRef<HTMLInputElement>(null);

  // Fetch entry data
  const { data: entry, isLoading } = useQuery<ProductCreatorDictionary>({
    queryKey: [`/api/dictionaries/${entryId}`],
    enabled: !!entryId,
  });

  // Images query
  const { data: dictionaryImages = [] } = useQuery<any[]>({
    queryKey: [`/api/dictionaries/${entryId}/images`],
    enabled: !!entryId,
  });

  // Form
  const form = useForm<DictionaryFormData>({
    resolver: zodResolver(dictionaryFormSchema),
    defaultValues: {
      dictionaryType: dictionaryType || "",
      code: "",
      name: "",
      readableName: "",
      shortName: "",
      description: "",
      color: "",
      category: "",
      sortOrder: 0,
      isActive: true,
    },
  });

  // Load entry data into form (use useEffect to avoid infinite loop)
  useEffect(() => {
    if (entry) {
      form.reset({
        dictionaryType: entry.dictionaryType,
        code: entry.code,
        name: entry.name,
        readableName: entry.readableName || "",
        shortName: entry.shortName || "",
        description: entry.description || "",
        color: entry.color || "",
        category: entry.category || "",
        sortOrder: entry.sortOrder ?? 0,
        isActive: entry.isActive ?? true,
      });
    }
  }, [entry, form]);

  // Update mutation
  const updateMutation = useMutation({
    mutationFn: async (data: Partial<DictionaryFormData>) => {
      if (!entryId) throw new Error("Brak ID pozycji");
      const response = await apiRequest("PATCH", `/api/dictionaries/${entryId}`, {
        ...data,
        code: data.code ? data.code.toUpperCase() : undefined,
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      queryClient.invalidateQueries({ queryKey: [`/api/dictionaries/${entryId}`] });
      toast({ title: "Pozycja zaktualizowana" });
      navigate(`/dictionaries/${dictionaryType}`);
    },
    onError: (error: Error) => {
      toast({ 
        title: "Błąd aktualizacji pozycji", 
        description: error.message,
        variant: "destructive" 
      });
    },
  });

  // Delete image mutation
  const deleteImageMutation = useMutation({
    mutationFn: async ({ dictionaryId, imageId }: { dictionaryId: number; imageId: number }) => {
      return await apiRequest("DELETE", `/api/dictionaries/${dictionaryId}/images/${imageId}`);
    },
    onSuccess: () => {
      if (entryId) {
        queryClient.invalidateQueries({ queryKey: [`/api/dictionaries/${entryId}/images`] });
      }
      toast({ title: "Obraz usunięty" });
    },
    onError: () => {
      toast({ title: "Błąd usuwania obrazu", variant: "destructive" });
    },
  });

  // File upload handler
  const handleFileUpload = async (files: FileList | null, dictionaryId: number) => {
    if (!files || files.length === 0) return;

    const formData = new FormData();
    Array.from(files).forEach(file => {
      formData.append('images', file);
    });

    try {
      const response = await fetch(`/api/dictionaries/${dictionaryId}/images`, {
        method: 'POST',
        credentials: 'include',
        body: formData,
      });

      if (!response.ok) {
        throw new Error('Upload failed');
      }

      queryClient.invalidateQueries({ queryKey: [`/api/dictionaries/${dictionaryId}/images`] });
      toast({ title: "Obrazy zostały dodane" });
    } catch (error) {
      toast({ 
        title: "Błąd uploadu", 
        description: error instanceof Error ? error.message : "Nieznany błąd",
        variant: "destructive" 
      });
    }
  };

  const handleSubmit = (data: DictionaryFormData) => {
    updateMutation.mutate(data);
  };

  if (!dictionaryType || !entryId) {
    return (
      <div className="w-full px-6 py-8">
        <Card>
          <CardHeader>
            <CardTitle>Błąd</CardTitle>
            <CardDescription>Nieprawidłowy URL</CardDescription>
          </CardHeader>
        </Card>
      </div>
    );
  }

  const typeInfo = DICTIONARY_LABELS[dictionaryType];

  if (isLoading) {
    return (
      <div className="w-full px-6 py-8">
        <div className="flex justify-center p-8">
          <Loader2 className="h-8 w-8 animate-spin" />
        </div>
      </div>
    );
  }

  return (
    <div className="w-full px-6 py-8 space-y-6">
      {/* Header */}
      <div className="flex items-center gap-4">
        <Button
          variant="ghost"
          size="icon"
          onClick={() => window.history.back()}
          data-testid="button-back"
        >
          <ArrowLeft className="h-5 w-5" />
        </Button>
        <div>
          <h1 className="text-3xl font-bold" data-testid="text-entry-title">
            Edytuj pozycję
          </h1>
          <p className="text-muted-foreground">{typeInfo.label}</p>
        </div>
      </div>

      {/* Main Form Card */}
      <Card>
        <CardHeader>
          <CardTitle>Edytuj pozycję słownika</CardTitle>
          <CardDescription>
            Edytuj pola poniżej i zapisz zmiany
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(handleSubmit)} className="space-y-6">
              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="code"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kod *</FormLabel>
                      <FormControl>
                        <Input {...field} placeholder="np. 800" data-testid="input-code" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Nazwa *</FormLabel>
                      <FormControl>
                        <Input {...field} placeholder="np. 800" data-testid="input-name" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="readableName"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa czytelna</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. 80cm" data-testid="input-readable-name" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="shortName"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa krótka (do budowania nazw)</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. SUPRA, D1, N2" data-testid="input-short-name" />
                    </FormControl>
                    <FormDescription>
                      Używana w automatycznym generowaniu nazw produktów. Jeśli puste, użyty zostanie kod.
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} rows={3} data-testid="input-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="color"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kolor (opcjonalny)</FormLabel>
                      <FormControl>
                        <div className="flex gap-2">
                          <Input 
                            {...field} 
                            placeholder="#6366F1" 
                            data-testid="input-color"
                            className="flex-1"
                          />
                          <div className="flex items-center gap-2">
                            <input
                              type="color"
                              value={field.value || "#6366F1"}
                              onChange={(e) => field.onChange(e.target.value)}
                              className="h-10 w-16 rounded border cursor-pointer"
                              data-testid="input-color-picker"
                            />
                            {field.value && (
                              <div 
                                className="h-10 w-10 rounded border flex items-center justify-center text-xs font-mono"
                                style={{ backgroundColor: field.value }}
                                title={field.value}
                              />
                            )}
                          </div>
                        </div>
                      </FormControl>
                      <FormDescription>
                        Kolor tła dla etykiet (np. #6366F1). Używany w katalog produktów i innych miejscach.
                      </FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="category"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kategoria</FormLabel>
                      <FormControl>
                        <Input {...field} data-testid="input-category" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="sortOrder"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kolejność</FormLabel>
                      <FormControl>
                        <Input type="number" {...field} data-testid="input-sort-order" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="isActive"
                render={({ field }) => (
                  <FormItem className="flex items-center justify-between rounded-lg border p-4">
                    <div className="space-y-0.5">
                      <FormLabel className="text-base">Aktywny</FormLabel>
                      <FormDescription>
                        Czy ta pozycja jest aktywna i widoczna
                      </FormDescription>
                    </div>
                    <FormControl>
                      <Switch
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        data-testid="switch-active"
                      />
                    </FormControl>
                  </FormItem>
                )}
              />

              <div className="flex justify-end gap-2 pt-4">
                <Button 
                  type="button" 
                  variant="outline" 
                  onClick={() => navigate(`/dictionaries/${dictionaryType}`)}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button 
                  type="submit" 
                  disabled={updateMutation.isPending}
                  data-testid="button-submit"
                >
                  {updateMutation.isPending && (
                    <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  )}
                  Zapisz
                </Button>
              </div>
            </form>
          </Form>
        </CardContent>
      </Card>

      {/* Images Card */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>Obrazy</CardTitle>
              <CardDescription>Zarządzaj obrazami tej pozycji</CardDescription>
            </div>
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={() => fileInputRef.current?.click()}
              data-testid="button-upload-images"
            >
              <ImageIcon className="h-4 w-4 mr-2" />
              Dodaj obrazy
            </Button>
            <input
              ref={fileInputRef}
              type="file"
              multiple
              accept="image/*"
              className="hidden"
              onChange={(e) => handleFileUpload(e.target.files, entryId)}
            />
          </div>
        </CardHeader>
        <CardContent>
          {dictionaryImages.length === 0 ? (
            <p className="text-muted-foreground text-center py-8">
              Brak obrazów dla tej pozycji
            </p>
          ) : (
            <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
              {dictionaryImages.map((img: any) => (
                <div key={img.id} className="relative group">
                  <img
                    src={img.url}
                    alt={img.alt || ""}
                    className="w-full h-32 object-cover rounded border"
                  />
                  <Button
                    type="button"
                    variant="destructive"
                    size="icon"
                    className="absolute top-1 right-1 h-6 w-6 opacity-0 group-hover:opacity-100 transition-opacity"
                    onClick={() => deleteImageMutation.mutate({ 
                      dictionaryId: entryId, 
                      imageId: img.id 
                    })}
                    data-testid={`button-delete-image-${img.id}`}
                  >
                    <X className="h-3 w-3" />
                  </Button>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
