import { useState, useEffect, useCallback } from 'react';

export interface OrderTab {
  id: string;
  label: string;
  token: string;
  isActive: boolean;
}

export interface UseOrdersTabsReturn {
  tabs: OrderTab[];
  activeTabs: OrderTab[];
  activeTokens: string[];
  addTab: (label: string) => void;
  toggleTab: (tabId: string) => void;
  removeTab: (tabId: string) => void;
  setTabsFromTokens: (tokens: string[]) => void;
}

const STORAGE_KEY_PREFIX = 'production-orders-tabs';

function getStorageKey(planId: string): string {
  return `${STORAGE_KEY_PREFIX}:${planId}`;
}

function sanitizeToken(input: string): string {
  return input.trim().toUpperCase();
}

function loadTabsFromStorage(planId: string): OrderTab[] {
  if (typeof window === 'undefined') return [];
  
  try {
    const storageKey = getStorageKey(planId);
    const stored = localStorage.getItem(storageKey);
    
    if (!stored) return [];
    
    const parsed = JSON.parse(stored);
    
    if (!Array.isArray(parsed)) return [];
    
    return parsed.filter(
      (item): item is OrderTab =>
        typeof item === 'object' &&
        item !== null &&
        typeof item.id === 'string' &&
        typeof item.label === 'string' &&
        typeof item.token === 'string' &&
        typeof item.isActive === 'boolean'
    );
  } catch (error) {
    console.error('Error loading orders tabs from storage:', error);
    return [];
  }
}

function saveTabsToStorage(planId: string, tabs: OrderTab[]): void {
  if (typeof window === 'undefined') return;
  
  try {
    const storageKey = getStorageKey(planId);
    localStorage.setItem(storageKey, JSON.stringify(tabs));
  } catch (error) {
    console.error('Error saving orders tabs to storage:', error);
  }
}

export function useOrdersTabs(planId: string): UseOrdersTabsReturn {
  const [tabs, setTabs] = useState<OrderTab[]>(() => loadTabsFromStorage(planId));

  useEffect(() => {
    saveTabsToStorage(planId, tabs);
  }, [planId, tabs]);

  const addTab = useCallback((label: string) => {
    const token = sanitizeToken(label);
    
    if (!token) return;
    
    setTabs(prev => {
      const exists = prev.some(tab => tab.token === token);
      if (exists) return prev;
      
      const newTab: OrderTab = {
        id: `tab-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        label: token,
        token: token,
        isActive: true
      };
      
      return [...prev, newTab];
    });
  }, []);

  const toggleTab = useCallback((tabId: string) => {
    setTabs(prev => prev.map(tab => 
      tab.id === tabId ? { ...tab, isActive: !tab.isActive } : tab
    ));
  }, []);

  const removeTab = useCallback((tabId: string) => {
    setTabs(prev => prev.filter(tab => tab.id !== tabId));
  }, []);

  const setTabsFromTokens = useCallback((tokens: string[]) => {
    const sanitizedTokens = tokens.map(t => sanitizeToken(t)).filter(Boolean);
    
    setTabs(prev => {
      const existingTokens = prev.map(t => t.token);
      const missingTokens = sanitizedTokens.filter(token => !existingTokens.includes(token));
      
      const newTabs = missingTokens.map(token => ({
        id: `tab-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        label: token,
        token: token,
        isActive: true
      }));
      
      const updatedTabs = prev.map(tab => ({
        ...tab,
        isActive: sanitizedTokens.includes(tab.token)
      }));
      
      const hasChanges = 
        newTabs.length > 0 || 
        updatedTabs.some((tab, idx) => tab.isActive !== prev[idx].isActive);
      
      if (!hasChanges) return prev;
      
      return [...updatedTabs, ...newTabs];
    });
  }, []);

  const activeTabs = tabs.filter(t => t.isActive);
  const activeTokens = activeTabs.map(t => t.token);

  return {
    tabs,
    activeTabs,
    activeTokens,
    addTab,
    toggleTab,
    removeTab,
    setTabsFromTokens
  };
}
