import type { Pool } from "pg";
import type { ProductionLocationGroup, InsertProductionLocationGroup } from "@shared/schema";

export async function getLocationGroups(pool: Pool): Promise<ProductionLocationGroup[]> {
  const result = await pool.query(`
    SELECT * FROM production.production_location_groups ORDER BY sort_order ASC, name ASC
  `);
  
  return result.rows.map(row => ({
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    locationType: row.location_type,
    allowsSublocations: row.allows_sublocations,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getLocationGroupById(pool: Pool, id: number): Promise<ProductionLocationGroup | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_location_groups WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    locationType: row.location_type,
    allowsSublocations: row.allows_sublocations,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function createLocationGroup(pool: Pool, data: InsertProductionLocationGroup): Promise<ProductionLocationGroup> {
  const result = await pool.query(`
    INSERT INTO production.production_location_groups 
    (code, name, description, location_type, allows_sublocations, is_active, sort_order)
    VALUES ($1, $2, $3, $4, $5, $6, $7)
    RETURNING *
  `, [
    data.code,
    data.name,
    data.description || null,
    data.locationType,
    data.allowsSublocations ?? true,
    data.isActive ?? true,
    data.sortOrder || 0,
  ]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    locationType: row.location_type,
    allowsSublocations: row.allows_sublocations,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function updateLocationGroup(pool: Pool, id: number, data: Partial<InsertProductionLocationGroup>): Promise<ProductionLocationGroup | null> {
  const updates: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.code !== undefined) {
    updates.push(`code = $${paramIndex++}`);
    values.push(data.code);
  }
  if (data.name !== undefined) {
    updates.push(`name = $${paramIndex++}`);
    values.push(data.name);
  }
  if (data.description !== undefined) {
    updates.push(`description = $${paramIndex++}`);
    values.push(data.description);
  }
  if (data.locationType !== undefined) {
    updates.push(`location_type = $${paramIndex++}`);
    values.push(data.locationType);
  }
  if (data.allowsSublocations !== undefined) {
    updates.push(`allows_sublocations = $${paramIndex++}`);
    values.push(data.allowsSublocations);
  }
  if (data.isActive !== undefined) {
    updates.push(`is_active = $${paramIndex++}`);
    values.push(data.isActive);
  }
  if (data.sortOrder !== undefined) {
    updates.push(`sort_order = $${paramIndex++}`);
    values.push(data.sortOrder);
  }

  if (updates.length === 0) return getLocationGroupById(pool, id);

  updates.push(`updated_at = NOW()`);
  values.push(id);

  const result = await pool.query(`
    UPDATE production.production_location_groups
    SET ${updates.join(', ')}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);

  if (result.rows.length === 0) return null;

  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    locationType: row.location_type,
    allowsSublocations: row.allows_sublocations,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function deleteLocationGroup(pool: Pool, id: number): Promise<boolean> {
  const result = await pool.query(`
    DELETE FROM production.production_location_groups WHERE id = $1
  `, [id]);
  
  return result.rowCount !== null && result.rowCount > 0;
}
