import type { Express } from "express";
import { requirePermission } from "../../auth";
import { pool } from "../../postgres";
import { insertProductionWorkCenterSchema } from "@shared/schema";
import {
  getWorkCenters,
  getWorkCenterById,
  createWorkCenter,
  updateWorkCenter,
  deleteWorkCenter,
} from "../../services/production/work-centers";

const partialWorkCenterSchema = insertProductionWorkCenterSchema.partial();

export function registerWorkCenterRoutes(app: Express) {
  // GET /api/production/work-centers - List all work centers
  app.get("/api/production/work-centers", requirePermission("view_production"), async (req, res) => {
    try {
      const workCenters = await getWorkCenters(pool);
      res.json(workCenters);
    } catch (error: any) {
      console.error("Error fetching work centers:", error);
      res.status(500).json({ error: "Failed to fetch work centers" });
    }
  });

  // GET /api/production/work-centers/:id - Get single work center
  app.get("/api/production/work-centers/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const workCenter = await getWorkCenterById(pool, id);
      
      if (!workCenter) {
        return res.status(404).json({ error: "Work center not found" });
      }
      
      res.json(workCenter);
    } catch (error: any) {
      console.error("Error fetching work center:", error);
      res.status(500).json({ error: "Failed to fetch work center" });
    }
  });

  // POST /api/production/work-centers - Create new work center
  app.post("/api/production/work-centers", requirePermission("manage_production"), async (req, res) => {
    try {
      const parsed = insertProductionWorkCenterSchema.safeParse(req.body);
      
      if (!parsed.success) {
        return res.status(400).json({ 
          error: "Invalid request data", 
          details: parsed.error.errors 
        });
      }
      
      const workCenter = await createWorkCenter(pool, parsed.data);
      res.status(201).json(workCenter);
    } catch (error: any) {
      console.error("Error creating work center:", error);
      if (error.code === "23505") { // Unique constraint violation
        return res.status(400).json({ error: "Work center with this code already exists" });
      }
      res.status(500).json({ error: "Failed to create work center" });
    }
  });

  // PATCH /api/production/work-centers/:id - Update work center
  app.patch("/api/production/work-centers/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const parsed = partialWorkCenterSchema.safeParse(req.body);
      
      if (!parsed.success) {
        return res.status(400).json({ 
          error: "Invalid request data", 
          details: parsed.error.errors 
        });
      }
      
      const workCenter = await updateWorkCenter(pool, id, parsed.data);
      
      if (!workCenter) {
        return res.status(404).json({ error: "Work center not found" });
      }
      
      res.json(workCenter);
    } catch (error: any) {
      console.error("Error updating work center:", error);
      if (error.code === "23505") { // Unique constraint violation
        return res.status(400).json({ error: "Work center with this code already exists" });
      }
      res.status(500).json({ error: "Failed to update work center" });
    }
  });

  // DELETE /api/production/work-centers/:id - Delete work center
  app.delete("/api/production/work-centers/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const deleted = await deleteWorkCenter(pool, id);
      
      if (!deleted) {
        return res.status(404).json({ error: "Work center not found" });
      }
      
      res.status(204).send();
    } catch (error: any) {
      console.error("Error deleting work center:", error);
      res.status(500).json({ error: "Failed to delete work center" });
    }
  });
}
