import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useAuth } from "@/hooks/use-auth";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Switch } from "@/components/ui/switch";
import { Key, Trash2, Plus, Copy, CheckCheck, FileText } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { useState } from "react";
import { useLocation } from "wouter";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface ApiToken {
  id: number;
  name: string;
  description?: string;
  token_prefix: string;
  is_active: boolean;
  last_used_at: string | null;
  expires_at: string | null;
  created_by: number;
  created_at: string;
}

const createTokenFormSchema = z.object({
  name: z.string().min(3, "Nazwa musi mieć minimum 3 znaki").max(255, "Nazwa może mieć maksymalnie 255 znaków"),
  description: z.string().optional(),
});

export default function ApiTokensPage() {
  const { user } = useAuth();
  const { toast } = useToast();
  const [, setLocation] = useLocation();
  const [deleteTokenId, setDeleteTokenId] = useState<number | null>(null);
  const [showCreateDialog, setShowCreateDialog] = useState(false);
  const [showTokenDialog, setShowTokenDialog] = useState(false);
  const [newTokenData, setNewTokenData] = useState<{ token: string; name: string } | null>(null);
  const [copied, setCopied] = useState(false);

  const form = useForm<z.infer<typeof createTokenFormSchema>>({
    resolver: zodResolver(createTokenFormSchema),
    defaultValues: {
      name: "",
      description: "",
    },
  });

  const { data: tokens, isLoading } = useQuery<ApiToken[]>({
    queryKey: ["/api/api-tokens"],
    enabled: user?.role === 'admin',
  });

  const createTokenMutation = useMutation({
    mutationFn: async (data: z.infer<typeof createTokenFormSchema>) => {
      const response = await apiRequest("POST", "/api/api-tokens", data);
      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/api-tokens"] });
      setNewTokenData({ token: data.token, name: form.getValues("name") });
      setShowCreateDialog(false);
      setShowTokenDialog(true);
      form.reset();
      toast({
        title: "Sukces",
        description: "Token API został utworzony",
      });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się utworzyć tokenu",
        variant: "destructive",
      });
    },
  });

  const updateTokenMutation = useMutation({
    mutationFn: async ({ tokenId, isActive }: { tokenId: number; isActive: boolean }) => {
      const response = await fetch(`/api/api-tokens/${tokenId}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ isActive }),
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to update token");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/api-tokens"] });
      toast({
        title: "Sukces",
        description: "Status tokenu został zaktualizowany",
      });
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się zaktualizować tokenu",
        variant: "destructive",
      });
    },
  });

  const deleteTokenMutation = useMutation({
    mutationFn: async (tokenId: number) => {
      const response = await fetch(`/api/api-tokens/${tokenId}`, {
        method: "DELETE",
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to delete token");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/api-tokens"] });
      toast({
        title: "Sukces",
        description: "Token został usunięty",
      });
      setDeleteTokenId(null);
    },
    onError: () => {
      toast({
        title: "Błąd",
        description: "Nie udało się usunąć tokenu",
        variant: "destructive",
      });
      setDeleteTokenId(null);
    },
  });

  const copyToClipboard = async (text: string) => {
    await navigator.clipboard.writeText(text);
    setCopied(true);
    setTimeout(() => setCopied(false), 2000);
    toast({
      title: "Skopiowano",
      description: "Token został skopiowany do schowka",
    });
  };

  if (user?.role !== 'admin') {
    return (
      <div className="flex items-center justify-center h-full">
        <Card>
          <CardHeader>
            <CardTitle>Brak dostępu</CardTitle>
            <CardDescription>
              Tylko administratorzy mają dostęp do zarządzania tokenami API.
            </CardDescription>
          </CardHeader>
        </Card>
      </div>
    );
  }

  const getStatusBadge = (isActive: boolean) => {
    return isActive ? (
      <Badge variant="default" data-testid="badge-status-active">Aktywny</Badge>
    ) : (
      <Badge variant="secondary" data-testid="badge-status-inactive">Nieaktywny</Badge>
    );
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Tokeny API</h1>
          <p className="text-muted-foreground">Zarządzaj tokenami dostępu do zewnętrznego API</p>
        </div>
        <div className="flex items-center gap-3">
          <Button
            variant="outline"
            onClick={() => setLocation("/api-tokens/logs")}
            data-testid="button-view-logs"
          >
            <FileText className="w-4 h-4 mr-2" />
            Logi requestów
          </Button>
          <Button
            onClick={() => setShowCreateDialog(true)}
            data-testid="button-create-token"
          >
            <Plus className="w-4 h-4 mr-2" />
            Nowy token
          </Button>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Lista tokenów</CardTitle>
          <CardDescription>
            Tokeny umożliwiają dostęp do API dla zewnętrznych systemów (np. Odoo)
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="text-center py-8 text-muted-foreground">Ładowanie...</div>
          ) : tokens && tokens.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Prefix</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Ostatnie użycie</TableHead>
                  <TableHead>Utworzono</TableHead>
                  <TableHead className="text-right">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {tokens.map((token) => (
                  <TableRow key={token.id} data-testid={`row-token-${token.id}`}>
                    <TableCell className="font-medium">{token.name}</TableCell>
                    <TableCell>
                      <code className="px-2 py-1 bg-muted rounded text-sm" data-testid={`text-prefix-${token.id}`}>
                        {token.token_prefix}...
                      </code>
                    </TableCell>
                    <TableCell>{getStatusBadge(token.is_active)}</TableCell>
                    <TableCell className="text-muted-foreground">
                      {token.last_used_at
                        ? format(new Date(token.last_used_at), "dd.MM.yyyy HH:mm", { locale: pl })
                        : "Nigdy"}
                    </TableCell>
                    <TableCell className="text-muted-foreground">
                      {format(new Date(token.created_at), "dd.MM.yyyy HH:mm", { locale: pl })}
                    </TableCell>
                    <TableCell className="text-right">
                      <div className="flex items-center justify-end gap-2">
                        <Switch
                          checked={token.is_active}
                          onCheckedChange={(checked) =>
                            updateTokenMutation.mutate({ tokenId: token.id, isActive: checked })
                          }
                          data-testid={`switch-active-${token.id}`}
                        />
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => setDeleteTokenId(token.id)}
                          data-testid={`button-delete-${token.id}`}
                        >
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          ) : (
            <div className="text-center py-8 text-muted-foreground">
              <Key className="w-12 h-12 mx-auto mb-4 opacity-50" />
              <p>Brak tokenów API</p>
              <p className="text-sm">Utwórz pierwszy token, aby umożliwić dostęp do API</p>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Create Token Dialog */}
      <Dialog open={showCreateDialog} onOpenChange={setShowCreateDialog}>
        <DialogContent data-testid="dialog-create-token">
          <DialogHeader>
            <DialogTitle>Nowy token API</DialogTitle>
            <DialogDescription>
              Utwórz nowy token dostępu do zewnętrznego API. Token zostanie wyświetlony tylko raz.
            </DialogDescription>
          </DialogHeader>
          <Form {...form}>
            <form onSubmit={form.handleSubmit((data) => createTokenMutation.mutate(data))} className="space-y-4 py-4">
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa tokenu</FormLabel>
                    <FormControl>
                      <Input
                        placeholder="np. Integracja z Odoo - produkcja"
                        data-testid="input-token-name"
                        {...field}
                      />
                    </FormControl>
                    <FormDescription>
                      Krótka nazwa identyfikująca token
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis (opcjonalny)</FormLabel>
                    <FormControl>
                      <Textarea
                        placeholder="Dodatkowy opis przeznaczenia tokenu..."
                        data-testid="input-token-description"
                        {...field}
                      />
                    </FormControl>
                    <FormDescription>
                      Podaj opis, który pomoże zidentyfikować przeznaczenie tokenu
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <DialogFooter>
                <Button variant="outline" type="button" onClick={() => setShowCreateDialog(false)}>
                  Anuluj
                </Button>
                <Button
                  type="submit"
                  disabled={createTokenMutation.isPending}
                  data-testid="button-confirm-create"
                >
                  {createTokenMutation.isPending ? "Tworzenie..." : "Utwórz token"}
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>

      {/* Show Token Dialog (only once) */}
      <Dialog open={showTokenDialog} onOpenChange={setShowTokenDialog}>
        <DialogContent className="max-w-2xl" data-testid="dialog-show-token">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <CheckCheck className="w-5 h-5 text-green-600" />
              Token został utworzony
            </DialogTitle>
            <DialogDescription>
              Skopiuj i zapisz token w bezpiecznym miejscu. <strong>Nie będzie możliwości ponownego wyświetlenia.</strong>
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>Nazwa tokenu</Label>
              <p className="text-sm" data-testid="text-created-name">{newTokenData?.name}</p>
            </div>
            <div className="space-y-2">
              <Label>Token dostępu</Label>
              <div className="flex gap-2">
                <Input
                  value={newTokenData?.token || ""}
                  readOnly
                  className="font-mono text-sm"
                  data-testid="input-new-token"
                />
                <Button
                  variant="outline"
                  onClick={() => copyToClipboard(newTokenData?.token || "")}
                  data-testid="button-copy-token"
                >
                  {copied ? <CheckCheck className="w-4 h-4" /> : <Copy className="w-4 h-4" />}
                </Button>
              </div>
            </div>
            <div className="bg-yellow-50 dark:bg-yellow-950/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4">
              <p className="text-sm text-yellow-800 dark:text-yellow-200">
                <strong>Uwaga:</strong> Token zostanie zaszyfrowany i zapisany w bazie danych. 
                Po zamknięciu tego okna nie będzie można go ponownie wyświetlić.
              </p>
            </div>
          </div>
          <DialogFooter>
            <Button onClick={() => setShowTokenDialog(false)} data-testid="button-close-token-dialog">
              Zamknij
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={deleteTokenId !== null} onOpenChange={() => setDeleteTokenId(null)}>
        <AlertDialogContent data-testid="dialog-delete-token">
          <AlertDialogHeader>
            <AlertDialogTitle>Czy na pewno usunąć token?</AlertDialogTitle>
            <AlertDialogDescription>
              Ta akcja jest nieodwracalna. Token zostanie natychmiast dezaktywowany i wszystkie 
              aplikacje używające tego tokenu stracą dostęp do API.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => deleteTokenId && deleteTokenMutation.mutate(deleteTokenId)}
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              data-testid="button-confirm-delete"
            >
              Usuń token
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
