import { UseFormReturn } from "react-hook-form";
import { UseMutationResult } from "@tanstack/react-query";
import { Loader2 } from "lucide-react";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Button } from "@/components/ui/button";
import type { ProductionWorkCenter } from "@shared/schema";
import { z } from "zod";
import { insertProductionRoutingSchema } from "@shared/schema";

export const routingFormSchema = insertProductionRoutingSchema.extend({
  code: z.string().min(1, "Kod jest wymagany"),
  name: z.string().min(1, "Nazwa jest wymagana"),
  defaultWorkCenterId: z.coerce.number().optional().nullable(),
});

export type RoutingFormData = z.infer<typeof routingFormSchema>;

interface ProductionRoutingFormProps {
  form: UseFormReturn<RoutingFormData>;
  onSubmit: (data: RoutingFormData) => void;
  mutation: UseMutationResult<any, any, any, any>;
  workCenters: ProductionWorkCenter[];
  mode: "create" | "edit";
  onCancel?: () => void;
}

export function ProductionRoutingForm({
  form,
  onSubmit,
  mutation,
  workCenters,
  mode,
  onCancel,
}: ProductionRoutingFormProps) {
  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="code"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Kod</FormLabel>
                <FormControl>
                  <Input 
                    {...field} 
                    placeholder="np. MRSZ-001"
                    data-testid="input-routing-code"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <FormField
            control={form.control}
            name="name"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Nazwa</FormLabel>
                <FormControl>
                  <Input 
                    {...field} 
                    placeholder="Nazwa marszruty"
                    data-testid="input-routing-name"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <FormField
          control={form.control}
          name="description"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Opis</FormLabel>
              <FormControl>
                <Textarea 
                  {...field} 
                  value={field.value ?? ""}
                  placeholder="Opis marszruty..."
                  data-testid="input-routing-description"
                />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="productType"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Typ produktu</FormLabel>
                <FormControl>
                  <Input 
                    {...field} 
                    value={field.value ?? ""}
                    placeholder="np. Komoda, Szafa"
                    data-testid="input-routing-product-type"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="defaultWorkCenterId"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Domyślne gniazdo robocze</FormLabel>
                <Select 
                  onValueChange={(value) => field.onChange(value === "none" ? null : parseInt(value))} 
                  value={field.value?.toString() ?? "none"}
                >
                  <FormControl>
                    <SelectTrigger data-testid="select-routing-work-center">
                      <SelectValue placeholder="Wybierz gniazdo" />
                    </SelectTrigger>
                  </FormControl>
                  <SelectContent>
                    <SelectItem value="none">Brak</SelectItem>
                    {workCenters.map((wc) => (
                      <SelectItem key={wc.id} value={wc.id.toString()}>
                        {wc.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <FormField
          control={form.control}
          name="version"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Wersja</FormLabel>
              <FormControl>
                <Input 
                  {...field} 
                  type="number"
                  value={field.value ?? 1}
                  onChange={(e) => field.onChange(parseInt(e.target.value) || 1)}
                  min={1}
                  data-testid="input-routing-version"
                />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="notes"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Notatki</FormLabel>
              <FormControl>
                <Textarea 
                  {...field} 
                  value={field.value ?? ""}
                  placeholder="Dodatkowe notatki..."
                  data-testid="input-routing-notes"
                />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="isActive"
          render={({ field }) => (
            <FormItem className="flex flex-row items-start space-x-3 space-y-0">
              <FormControl>
                <Checkbox
                  checked={field.value ?? true}
                  onCheckedChange={field.onChange}
                  data-testid="checkbox-routing-active"
                />
              </FormControl>
              <div className="space-y-1 leading-none">
                <FormLabel>Aktywna</FormLabel>
              </div>
            </FormItem>
          )}
        />

        <div className="flex gap-2 justify-end pt-4">
          {onCancel && (
            <Button
              type="button"
              variant="outline"
              onClick={onCancel}
              disabled={mutation.isPending}
              data-testid="button-cancel"
            >
              Anuluj
            </Button>
          )}
          <Button 
            type="submit" 
            disabled={mutation.isPending}
            data-testid="button-submit-routing"
          >
            {mutation.isPending && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
            {mode === "create" ? "Utwórz marszrutę" : "Zapisz zmiany"}
          </Button>
        </div>
      </form>
    </Form>
  );
}
