import { useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Loader2, AlertTriangle, CheckCircle2, Package, Ruler, Layers } from "lucide-react";

interface MatchedTemplate {
  templateId: number;
  cz1: string;
  cz2: string | null;
  furnitureType: string;
  baseLength: number;
  baseWidth: number;
  calculatedLength: number;
  calculatedWidth: number;
  thickness: number | null;
  plateType: string | null;
  color: string | null;
  edgingMaterial: string | null;
  edge1: boolean;
  edge2: boolean;
  edge3: boolean;
  edge4: boolean;
  quantity: number;
  positionInBom: number;
}

interface BOMGenerationDialogProps {
  productId: number;
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onSuccess: () => void;
}

export function BOMGenerationDialog({ productId, open, onOpenChange, onSuccess }: BOMGenerationDialogProps) {
  const { toast } = useToast();
  const [matchedTemplates, setMatchedTemplates] = useState<MatchedTemplate[]>([]);
  const [quantities, setQuantities] = useState<Record<number, number>>({});
  const [warnings, setWarnings] = useState<string[]>([]);

  // Generate BOM preview
  const generateMutation = useMutation({
    mutationFn: async () => {
      const response = await apiRequest('POST', `/api/catalog/products/${productId}/bom/generate`);
      return response;
    },
    onSuccess: (data: any) => {
      setMatchedTemplates(data.matchedTemplates || []);
      setWarnings(data.warnings || []);
      
      // Initialize quantities
      const initialQuantities: Record<number, number> = {};
      (data.matchedTemplates || []).forEach((t: MatchedTemplate) => {
        initialQuantities[t.templateId] = t.quantity;
      });
      setQuantities(initialQuantities);

      if (data.matchedTemplates?.length === 0) {
        toast({
          title: "Brak dopasowań",
          description: "Nie znaleziono formatek pasujących do tego produktu",
          variant: "destructive",
        });
      } else {
        toast({
          title: "BOM wygenerowany",
          description: `Znaleziono ${data.matchedTemplates.length} dopasowanych formatek`,
        });
      }
    },
    onError: (error: any) => {
      toast({
        title: "Błąd generowania BOM",
        description: error.message || "Nie udało się wygenerować BOM",
        variant: "destructive",
      });
    },
  });

  // Save BOM
  const saveMutation = useMutation({
    mutationFn: async () => {
      const components = matchedTemplates.map(t => ({
        templateId: t.templateId,
        quantity: quantities[t.templateId] || t.quantity,
        calculatedLength: t.calculatedLength,
        calculatedWidth: t.calculatedWidth,
        positionInBom: t.positionInBom,
      }));

      const response = await apiRequest('POST', `/api/catalog/products/${productId}/bom/save`, { components });
      return response;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${productId}/bom`] });
      toast({
        title: "BOM zapisany",
        description: "Lista materiałów została pomyślnie zapisana",
      });
      onSuccess();
      onOpenChange(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd zapisu BOM",
        description: error.message || "Nie udało się zapisać BOM",
        variant: "destructive",
      });
    },
  });

  const handleQuantityChange = (templateId: number, value: string) => {
    const qty = parseInt(value) || 1;
    setQuantities(prev => ({ ...prev, [templateId]: qty }));
  };

  const handleGenerate = () => {
    generateMutation.mutate();
  };

  const handleSave = () => {
    if (matchedTemplates.length === 0) {
      toast({
        title: "Brak komponentów",
        description: "Najpierw wygeneruj BOM",
        variant: "destructive",
      });
      return;
    }
    saveMutation.mutate();
  };

  const getEdgingDisplay = (t: MatchedTemplate) => {
    const edges = [];
    if (t.edge1) edges.push("K1");
    if (t.edge2) edges.push("K2");
    if (t.edge3) edges.push("K3");
    if (t.edge4) edges.push("K4");
    return edges.join(", ") || "Bez okleiny";
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-4xl max-h-[90vh]" data-testid="dialog-bom-generation">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Package className="h-5 w-5" />
            Generowanie BOM (Bill of Materials)
          </DialogTitle>
          <DialogDescription>
            Automatyczne dopasowanie formatek do wymiarów produktu
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-4">
          {/* Warnings */}
          {warnings.length > 0 && (
            <Alert variant="destructive">
              <AlertTriangle className="h-4 w-4" />
              <AlertDescription>
                <ul className="list-disc list-inside space-y-1">
                  {warnings.map((warning, idx) => (
                    <li key={idx}>{warning}</li>
                  ))}
                </ul>
              </AlertDescription>
            </Alert>
          )}

          {/* Generate Button */}
          {matchedTemplates.length === 0 && (
            <div className="flex justify-center py-8">
              <Button
                onClick={handleGenerate}
                disabled={generateMutation.isPending}
                size="lg"
                data-testid="button-generate-bom"
              >
                {generateMutation.isPending ? (
                  <>
                    <Loader2 className="h-4 w-4 animate-spin" />
                    Generowanie...
                  </>
                ) : (
                  <>
                    <Package className="h-4 w-4" />
                    Generuj BOM
                  </>
                )}
              </Button>
            </div>
          )}

          {/* Matched Templates List */}
          {matchedTemplates.length > 0 && (
            <>
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <CheckCircle2 className="h-5 w-5 text-green-600" />
                  <span className="font-medium">
                    Znaleziono {matchedTemplates.length} dopasowanych formatek
                  </span>
                </div>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={handleGenerate}
                  disabled={generateMutation.isPending}
                  data-testid="button-regenerate-bom"
                >
                  {generateMutation.isPending ? (
                    <>
                      <Loader2 className="h-4 w-4 animate-spin" />
                      Regeneruj
                    </>
                  ) : (
                    "Regeneruj"
                  )}
                </Button>
              </div>

              <ScrollArea className="h-[400px] pr-4">
                <div className="space-y-3">
                  {matchedTemplates.map((template) => (
                    <Card key={template.templateId} className="p-4" data-testid={`bom-component-${template.templateId}`}>
                      <div className="space-y-3">
                        {/* Header */}
                        <div className="flex items-start justify-between">
                          <div className="space-y-1">
                            <div className="flex items-center gap-2">
                              <span className="font-semibold text-lg">
                                {template.cz1}{template.cz2 ? `-${template.cz2}` : ''}
                              </span>
                              {template.color && (
                                <Badge variant="secondary">{template.color}</Badge>
                              )}
                            </div>
                            <div className="text-sm text-muted-foreground">
                              {template.furnitureType}
                            </div>
                          </div>
                          <div className="flex items-center gap-2">
                            <span className="text-sm text-muted-foreground">Ilość:</span>
                            <Input
                              type="number"
                              min="1"
                              value={quantities[template.templateId] || template.quantity}
                              onChange={(e) => handleQuantityChange(template.templateId, e.target.value)}
                              className="w-20"
                              data-testid={`input-quantity-${template.templateId}`}
                            />
                          </div>
                        </div>

                        {/* Dimensions */}
                        <div className="grid grid-cols-2 gap-4">
                          <div className="flex items-center gap-2">
                            <Ruler className="h-4 w-4 text-muted-foreground" />
                            <div className="space-y-0.5">
                              <div className="text-sm font-medium">Wymiary wyliczone</div>
                              <div className="text-sm text-muted-foreground">
                                {template.calculatedLength} × {template.calculatedWidth}
                                {template.thickness ? ` × ${template.thickness}mm` : ''}
                              </div>
                            </div>
                          </div>

                          <div className="flex items-center gap-2">
                            <Layers className="h-4 w-4 text-muted-foreground" />
                            <div className="space-y-0.5">
                              <div className="text-sm font-medium">Materiał</div>
                              <div className="text-sm text-muted-foreground">
                                {template.plateType || "N/A"}
                              </div>
                            </div>
                          </div>
                        </div>

                        {/* Edging */}
                        <div className="flex items-center gap-2">
                          <Badge variant="outline" className="text-xs">
                            {getEdgingDisplay(template)}
                          </Badge>
                          {template.edgingMaterial && (
                            <span className="text-xs text-muted-foreground">
                              Material: {template.edgingMaterial}
                            </span>
                          )}
                        </div>
                      </div>
                    </Card>
                  ))}
                </div>
              </ScrollArea>

              {/* Actions */}
              <div className="flex justify-end gap-2 pt-4 border-t">
                <Button
                  variant="outline"
                  onClick={() => onOpenChange(false)}
                  data-testid="button-cancel-bom"
                >
                  Anuluj
                </Button>
                <Button
                  onClick={handleSave}
                  disabled={saveMutation.isPending}
                  data-testid="button-save-bom"
                >
                  {saveMutation.isPending ? (
                    <>
                      <Loader2 className="h-4 w-4 animate-spin" />
                      Zapisywanie...
                    </>
                  ) : (
                    <>
                      <CheckCircle2 className="h-4 w-4" />
                      Zapisz BOM
                    </>
                  )}
                </Button>
              </div>
            </>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
}
