import { useState } from "react";
import { Check, ChevronsUpDown, Plus } from "lucide-react";
import { cn } from "@/lib/utils";
import { Button } from "@/components/ui/button";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { useToast } from "@/hooks/use-toast";
import { useMutation } from "@tanstack/react-query";
import { queryClient } from "@/lib/queryClient";

interface AccessoryGroup {
  id: number;
  name: string;
  code: string;
  category?: string | null;
  description?: string | null;
}

interface AccessoryGroupComboboxWithAddProps {
  groups: AccessoryGroup[];
  value: string;
  onChange: (value: string) => void;
  placeholder?: string;
  searchPlaceholder?: string;
  emptyText?: string;
  testId?: string;
}

export function AccessoryGroupComboboxWithAdd({
  groups,
  value,
  onChange,
  placeholder = "Wybierz grupę",
  searchPlaceholder = "Szukaj grupy...",
  emptyText = "Nie znaleziono grup.",
  testId,
}: AccessoryGroupComboboxWithAddProps) {
  const { toast } = useToast();
  const [open, setOpen] = useState(false);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [newName, setNewName] = useState("");
  const [newCode, setNewCode] = useState("");
  const [newCategory, setNewCategory] = useState("");
  const [newDescription, setNewDescription] = useState("");

  const sortedGroups = [...groups].sort((a, b) => 
    a.name.toLowerCase().localeCompare(b.name.toLowerCase())
  );

  const selectedGroup = sortedGroups.find(g => g.id.toString() === value);

  const createMutation = useMutation({
    mutationFn: async (data: { 
      name: string; 
      code: string; 
      category?: string; 
      description?: string;
    }) => {
      const response = await fetch("/api/accessory-groups", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({
          ...data,
          code: data.code.toUpperCase(),
          isActive: true,
          displayOrder: 0,
        }),
      });
      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(errorText || "Nie udało się dodać grupy");
      }
      return response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessory-groups"] });
      toast({
        title: "Sukces",
        description: "Nowa grupa została dodana",
      });
      // Select the newly created group
      onChange(data.id.toString());
      setDialogOpen(false);
      setNewName("");
      setNewCode("");
      setNewCategory("");
      setNewDescription("");
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleAddNew = () => {
    setOpen(false);
    setDialogOpen(true);
  };

  const handleSaveNew = () => {
    if (!newName.trim() || !newCode.trim()) {
      toast({
        title: "Błąd",
        description: "Nazwa i kod są wymagane",
        variant: "destructive",
      });
      return;
    }

    createMutation.mutate({
      name: newName.trim(),
      code: newCode.trim(),
      category: newCategory.trim() || undefined,
      description: newDescription.trim() || undefined,
    });
  };

  return (
    <>
      <Popover open={open} onOpenChange={setOpen}>
        <PopoverTrigger asChild>
          <Button
            variant="outline"
            role="combobox"
            aria-expanded={open}
            className="w-full justify-between"
            data-testid={testId}
          >
            {selectedGroup ? selectedGroup.name : placeholder}
            <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
          </Button>
        </PopoverTrigger>
        <PopoverContent className="w-[--radix-popover-trigger-width] p-0" align="start">
          <Command>
            <div className="flex items-center border-b px-3">
              <CommandInput 
                placeholder={searchPlaceholder} 
                className="flex-1 border-0 focus:ring-0"
              />
              <Button
                type="button"
                size="icon"
                variant="ghost"
                onClick={handleAddNew}
                className="h-8 w-8 shrink-0"
                data-testid="button-add-new-group"
              >
                <Plus className="h-4 w-4" />
              </Button>
            </div>
            <CommandList>
              <CommandEmpty>{emptyText}</CommandEmpty>
              <CommandGroup>
                <CommandItem
                  value="none"
                  onSelect={() => {
                    onChange("");
                    setOpen(false);
                  }}
                  data-testid="option-none"
                >
                  <Check
                    className={cn(
                      "mr-2 h-4 w-4",
                      value === "" ? "opacity-100" : "opacity-0"
                    )}
                  />
                  <span className="text-muted-foreground">Brak grupy</span>
                </CommandItem>
                {sortedGroups.map((group) => (
                  <CommandItem
                    key={group.id}
                    value={group.name}
                    onSelect={() => {
                      onChange(group.id.toString());
                      setOpen(false);
                    }}
                    data-testid={`option-group-${group.id}`}
                  >
                    <Check
                      className={cn(
                        "mr-2 h-4 w-4",
                        value === group.id.toString() ? "opacity-100" : "opacity-0"
                      )}
                    />
                    {group.name}
                  </CommandItem>
                ))}
              </CommandGroup>
            </CommandList>
          </Command>
        </PopoverContent>
      </Popover>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="w-[95vw] max-w-md" data-testid="dialog-add-group">
          <DialogHeader>
            <DialogTitle>Dodaj nową grupę akcesoriów</DialogTitle>
            <DialogDescription>
              Wypełnij poniższe pola aby dodać nową grupę akcesoriów do katalogu.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="name">
                Nazwa <span className="text-destructive">*</span>
              </Label>
              <Input
                id="name"
                value={newName}
                onChange={(e) => setNewName(e.target.value)}
                placeholder="np. Haki wieszaka"
                data-testid="input-new-name"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="code">
                Kod <span className="text-destructive">*</span>
              </Label>
              <Input
                id="code"
                value={newCode}
                onChange={(e) => setNewCode(e.target.value)}
                placeholder="np. HAKI_VB"
                data-testid="input-new-code"
                className="uppercase"
              />
              <p className="text-xs text-muted-foreground">
                Unikalny kod grupy (tylko duże litery, cyfry i podkreślenia)
              </p>
            </div>
            <div className="space-y-2">
              <Label htmlFor="category">Kategoria</Label>
              <Input
                id="category"
                value={newCategory}
                onChange={(e) => setNewCategory(e.target.value)}
                placeholder="np. Okucia meblowe"
                data-testid="input-new-category"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="description">Opis</Label>
              <Textarea
                id="description"
                value={newDescription}
                onChange={(e) => setNewDescription(e.target.value)}
                placeholder="Opcjonalny opis grupy..."
                rows={3}
                data-testid="textarea-new-description"
              />
            </div>
          </div>
          <DialogFooter className="flex-col-reverse sm:flex-row gap-2">
            <Button
              variant="outline"
              onClick={() => setDialogOpen(false)}
              disabled={createMutation.isPending}
              data-testid="button-cancel-add"
              className="w-full sm:w-auto"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleSaveNew}
              disabled={createMutation.isPending}
              data-testid="button-save-new"
              className="w-full sm:w-auto"
            >
              {createMutation.isPending ? "Dodawanie..." : "Dodaj grupę"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  );
}
