import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Calendar } from "@/components/ui/calendar";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Link } from "wouter";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { Calendar as CalendarIcon, ExternalLink, CheckCircle2, XCircle } from "lucide-react";
import { cn } from "@/lib/utils";

type Platform = "allegro" | "shoper";
type DateRangeType = "today" | "yesterday" | "last-7-days" | "last-30-days" | "custom";

interface RecentOrder {
  id: number;
  source: string;
  sourceOrderId: string;
  orderNumber: string;
  orderDate: string;
  marketplaceUpdatedAt: string;
  lastUpdatedAt: string;
  fulfillmentStatus: string;
  paymentStatus: string;
  buyerLogin: string;
  buyerEmail: string;
  totalAmount: number;
  currency: string;
  odooOrderId: number | null;
  inOdoo: boolean;
}

interface Summary {
  total: number;
  inOdoo: number;
  notInOdoo: number;
  paid: number;
  unpaid: number;
}

interface RecentUpdatesResponse {
  platform: string;
  dateRange: string;
  startDate: string;
  endDate: string;
  summary: Summary;
  orders: RecentOrder[];
}

export default function RecentUpdates() {
  const [platform, setPlatform] = useState<Platform>("allegro");
  const [dateRangeType, setDateRangeType] = useState<DateRangeType>("last-7-days");
  const [customDateFrom, setCustomDateFrom] = useState<Date | undefined>(undefined);
  const [customDateTo, setCustomDateTo] = useState<Date | undefined>(undefined);

  const queryParams = new URLSearchParams({
    platform,
    dateRange: dateRangeType,
    ...(dateRangeType === 'custom' && customDateFrom && customDateTo ? {
      startDate: customDateFrom.toISOString().split('T')[0],
      endDate: customDateTo.toISOString().split('T')[0],
    } : {})
  });

  const { data, isLoading, error} = useQuery<RecentUpdatesResponse>({
    queryKey: [`/api/orders/recent-updates?${queryParams.toString()}`],
    enabled: dateRangeType !== 'custom' || (!!customDateFrom && !!customDateTo),
  });

  const getPlatformOrderUrl = (platform: string, orderId: string) => {
    if (platform === 'ALLEGRO') {
      return `https://allegro.pl/moje-allegro/sprzedaz/zamowienia/${orderId}`;
    } else if (platform === 'SHOPER') {
      return `https://plywobox.shoparena.pl/admin/orders/${orderId}`;
    }
    return '#';
  };

  const getPaymentStatusLabel = (status: string) => {
    const labels: Record<string, string> = {
      'PAID': 'Opłacone',
      'UNPAID': 'Nieopłacone',
      'PENDING': 'Oczekujące',
      'PARTIAL': 'Częściowe'
    };
    return labels[status] || status;
  };

  const getPaymentStatusVariant = (status: string): "default" | "secondary" | "destructive" | "outline" => {
    if (status === 'PAID') return 'default';
    if (status === 'UNPAID') return 'destructive';
    if (status === 'PENDING') return 'secondary';
    return 'outline';
  };

  const summary = data?.summary;

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold" data-testid="heading-recent-updates">Ostatnie Aktualizacje</h1>
          <p className="text-muted-foreground mt-1" data-testid="text-description">
            Zamówienia zaktualizowane ostatnio w marketplace
          </p>
        </div>
      </div>

      {summary && (
        <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="text-sm font-medium text-muted-foreground">Łącznie</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold" data-testid="stat-total">{summary.total}</div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="text-sm font-medium text-muted-foreground">W Odoo</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-600" data-testid="stat-in-odoo">{summary.inOdoo}</div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="text-sm font-medium text-muted-foreground">Brak w Odoo</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-orange-600" data-testid="stat-not-in-odoo">{summary.notInOdoo}</div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="text-sm font-medium text-muted-foreground">Opłacone</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-600" data-testid="stat-paid">{summary.paid}</div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="text-sm font-medium text-muted-foreground">Nieopłacone</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-red-600" data-testid="stat-unpaid">{summary.unpaid}</div>
            </CardContent>
          </Card>
        </div>
      )}

      <Card>
        <CardHeader>
          <CardTitle data-testid="card-title">Weryfikacja Zamówień</CardTitle>
          <CardDescription data-testid="text-card-description">
            Wybierz platformę i zakres dat aby sprawdzić ostatnie aktualizacje
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Tabs value={platform} onValueChange={(v) => setPlatform(v as Platform)}>
            <div className="flex items-center justify-between mb-4 flex-wrap gap-3">
              <TabsList data-testid="tabs-platform">
                <TabsTrigger value="allegro" data-testid="tab-allegro">Allegro</TabsTrigger>
                <TabsTrigger value="shoper" data-testid="tab-shoper">Shoper</TabsTrigger>
              </TabsList>

              <div className="flex gap-2 flex-wrap">
                <Button
                  size="sm"
                  variant={dateRangeType === 'today' ? 'default' : 'outline'}
                  onClick={() => setDateRangeType('today')}
                  data-testid="button-today"
                >
                  Dzisiaj
                </Button>
                <Button
                  size="sm"
                  variant={dateRangeType === 'yesterday' ? 'default' : 'outline'}
                  onClick={() => setDateRangeType('yesterday')}
                  data-testid="button-yesterday"
                >
                  Wczoraj
                </Button>
                <Button
                  size="sm"
                  variant={dateRangeType === 'last-7-days' ? 'default' : 'outline'}
                  onClick={() => setDateRangeType('last-7-days')}
                  data-testid="button-last-7-days"
                >
                  Ostatnie 7 dni
                </Button>
                <Button
                  size="sm"
                  variant={dateRangeType === 'last-30-days' ? 'default' : 'outline'}
                  onClick={() => setDateRangeType('last-30-days')}
                  data-testid="button-last-30-days"
                >
                  Ostatnie 30 dni
                </Button>
                
                <Popover>
                  <PopoverTrigger asChild>
                    <Button
                      size="sm"
                      variant={dateRangeType === 'custom' ? 'default' : 'outline'}
                      data-testid="button-custom-range"
                      className={cn(
                        "justify-start text-left font-normal",
                        !customDateFrom && !customDateTo && "text-muted-foreground"
                      )}
                    >
                      <CalendarIcon className="mr-2 h-4 w-4" />
                      {customDateFrom && customDateTo ? (
                        <>
                          {format(customDateFrom, "dd.MM.yy", { locale: pl })} - {format(customDateTo, "dd.MM.yy", { locale: pl })}
                        </>
                      ) : (
                        <span>Zakres dat</span>
                      )}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-0" align="end">
                    <div className="p-3 space-y-3">
                      <div className="space-y-2">
                        <div className="text-sm font-medium">Data od:</div>
                        <Calendar
                          mode="single"
                          selected={customDateFrom}
                          onSelect={(date) => {
                            setCustomDateFrom(date);
                            setDateRangeType('custom');
                          }}
                          locale={pl}
                          disabled={(date) => date > new Date()}
                        />
                      </div>
                      <div className="space-y-2">
                        <div className="text-sm font-medium">Data do:</div>
                        <Calendar
                          mode="single"
                          selected={customDateTo}
                          onSelect={(date) => {
                            setCustomDateTo(date);
                            setDateRangeType('custom');
                          }}
                          locale={pl}
                          disabled={(date) => date > new Date() || (customDateFrom ? date < customDateFrom : false)}
                        />
                      </div>
                    </div>
                  </PopoverContent>
                </Popover>
              </div>
            </div>

            <TabsContent value={platform} className="mt-0">
              {isLoading ? (
                <div className="text-center py-8 text-muted-foreground" data-testid="text-loading">
                  Ładowanie...
                </div>
              ) : error ? (
                <div className="text-center py-8 text-destructive" data-testid="text-error">
                  Błąd: {error instanceof Error ? error.message : 'Nieznany błąd'}
                </div>
              ) : data?.orders && data.orders.length > 0 ? (
                <div className="border rounded-md">
                  <Table>
                    <TableHeader>
                      <TableRow className="bg-muted/50">
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-order-id">ID Zamówienia</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-updated">Zaktualizowane</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-oms-code">Kod OMS</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-payment">Płatność</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-odoo">Status Odoo</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-buyer">Kupujący</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold text-right" data-testid="header-amount">Kwota</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {data.orders.map((order, idx) => {
                        const platformUrl = getPlatformOrderUrl(order.source, order.sourceOrderId);

                        return (
                          <TableRow key={idx} data-testid={`row-order-${idx}`} className="h-10 hover:bg-muted/30">
                            <TableCell className="py-1 px-2 text-xs font-mono" data-testid={`cell-order-id-${idx}`}>
                              <a
                                href={platformUrl}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="text-primary hover:underline inline-flex items-center gap-1"
                                data-testid={`link-platform-${idx}`}
                              >
                                {order.sourceOrderId}
                                <ExternalLink className="w-3 h-3" />
                              </a>
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs whitespace-nowrap" data-testid={`cell-updated-${idx}`}>
                              {order.marketplaceUpdatedAt && !isNaN(new Date(order.marketplaceUpdatedAt).getTime()) 
                                ? format(new Date(order.marketplaceUpdatedAt), "dd.MM.yy HH:mm", { locale: pl })
                                : '-'
                              }
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs" data-testid={`cell-oms-code-${idx}`}>
                              <Link href={`/order/${order.orderNumber}`} data-testid={`link-oms-${idx}`}>
                                <Badge 
                                  variant="outline" 
                                  className="hover-elevate cursor-pointer h-5 text-xs px-1.5"
                                  data-testid={`badge-oms-${idx}`}
                                >
                                  {order.orderNumber}
                                </Badge>
                              </Link>
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs" data-testid={`cell-payment-${idx}`}>
                              <Badge 
                                variant={getPaymentStatusVariant(order.paymentStatus)}
                                className="h-5 text-xs px-1.5"
                                data-testid={`badge-payment-${idx}`}
                              >
                                {getPaymentStatusLabel(order.paymentStatus)}
                              </Badge>
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs" data-testid={`cell-odoo-${idx}`}>
                              {order.inOdoo ? (
                                <div className="flex items-center gap-1 text-green-600">
                                  <CheckCircle2 className="w-4 h-4" />
                                  <span>W Odoo</span>
                                </div>
                              ) : (
                                <div className="flex items-center gap-1 text-orange-600">
                                  <XCircle className="w-4 h-4" />
                                  <span>Brak</span>
                                </div>
                              )}
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs" data-testid={`cell-buyer-${idx}`}>
                              {order.buyerLogin}
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs text-right font-semibold" data-testid={`cell-amount-${idx}`}>
                              {order.totalAmount?.toFixed(2)} {order.currency}
                            </TableCell>
                          </TableRow>
                        );
                      })}
                    </TableBody>
                  </Table>
                </div>
              ) : (
                <div className="text-center py-8 text-muted-foreground" data-testid="text-no-data">
                  Brak zamówień w wybranym zakresie dat
                </div>
              )}
            </TabsContent>
          </Tabs>
        </CardContent>
      </Card>
    </div>
  );
}
