import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Plus, Edit2, Trash2, Grid3x3, List, Loader2, Copy, Table, MoreHorizontal, ExternalLink, Circle, AlignJustify, LayoutGrid, MoreVertical, Files } from "lucide-react";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
  DropdownMenuSeparator,
} from "@/components/ui/dropdown-menu";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useState, useEffect } from "react";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { CompactListItem } from "@/components/compact-list-item";

type AccessoryGroup = {
  id: number;
  name: string;
  code: string;
  category: string | null;
  description: string | null;
  displayOrder: number;
  isActive: boolean;
};

type Accessory = {
  id: number;
  groupId: number | null;
  warehouseMaterialId: number | null;
  name: string;
  code: string;
  alpmaCode: string | null;
  description: string | null;
  imageUrl: string | null;
  displayOrder: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
};

type WarehouseMaterial = {
  id: number;
  name: string;
  internalCode: string | null;
  category: string;
  groupName: string | null;
};

const STORAGE_KEY_VIEW_MODE = 'accessories-view-mode';

export default function AccessoryGroupPage() {
  const params = useParams<{ groupCode: string }>();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  
  // Load view mode from localStorage, default to "list"
  const [viewMode, setViewMode] = useState<"list" | "grid" | "compact">(() => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY_VIEW_MODE);
      return (saved as "list" | "grid" | "compact") || "list";
    } catch (e) {
      console.error('Failed to load view mode:', e);
      return "list";
    }
  });

  const [showInactive, setShowInactive] = useState(false);
  const [selectedAccessoryIds, setSelectedAccessoryIds] = useState<Set<number>>(new Set());
  
  const [duplicateDialogOpen, setDuplicateDialogOpen] = useState(false);
  const [accessoryToDuplicate, setAccessoryToDuplicate] = useState<Accessory | null>(null);
  const [duplicateTargetGroupId, setDuplicateTargetGroupId] = useState<string>("");

  // Save view mode to localStorage
  useEffect(() => {
    try {
      localStorage.setItem(STORAGE_KEY_VIEW_MODE, viewMode);
    } catch (e) {
      console.error('Failed to save view mode:', e);
    }
  }, [viewMode]);

  const groupCode = params.groupCode;

  // Fetch all groups to find current one
  const { data: groups = [], isLoading: groupsLoading } = useQuery<AccessoryGroup[]>({
    queryKey: ["/api/accessory-groups"],
  });

  // Find current group
  const currentGroup = groups.find(g => g.code === groupCode);

  // Fetch accessories
  const { data: accessories = [], isLoading: accessoriesLoading } = useQuery<Accessory[]>({
    queryKey: ["/api/accessories", { showInactive }],
  });

  // Fetch warehouse materials for linking info
  const { data: materials = [] } = useQuery<WarehouseMaterial[]>({
    queryKey: ["/api/warehouse/materials/all"],
  });

  // Filter accessories for current group
  const groupAccessories = accessories
    .filter(a => a.groupId === currentGroup?.id)
    .sort((a, b) => a.displayOrder - b.displayOrder || a.name.localeCompare(b.name));

  const isLoading = groupsLoading || accessoriesLoading;

  // Mutation: Duplicate accessory
  const duplicateMutation = useMutation({
    mutationFn: async ({ accessoryId, targetGroupId }: { accessoryId: number; targetGroupId: number }) => {
      return await apiRequest("POST", `/api/accessories/${accessoryId}/duplicate`, {
        targetGroupId,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      toast({
        title: "Sukces",
        description: "Akcesorium zostało zduplikowane",
      });
      setDuplicateDialogOpen(false);
      setAccessoryToDuplicate(null);
      setDuplicateTargetGroupId("");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zduplikować akcesorium",
        variant: "destructive",
      });
    },
  });

  const handleDuplicateClick = (accessory: Accessory) => {
    setAccessoryToDuplicate(accessory);
    setDuplicateTargetGroupId(accessory.groupId?.toString() || "");
    setDuplicateDialogOpen(true);
  };

  const handleDuplicateConfirm = () => {
    if (!accessoryToDuplicate || !duplicateTargetGroupId) {
      toast({
        title: "Błąd",
        description: "Wybierz grupę docelową",
        variant: "destructive",
      });
      return;
    }

    duplicateMutation.mutate({
      accessoryId: accessoryToDuplicate.id,
      targetGroupId: parseInt(duplicateTargetGroupId),
    });
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  if (!currentGroup) {
    return (
      <div className="p-2 sm:p-4 md:p-8">
        <Card>
          <CardContent className="p-8">
            <div className="text-center text-muted-foreground">
              Nie znaleziono grupy akcesoriów
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="p-2 sm:p-4 md:p-8 space-y-4 md:space-y-6">
      {/* Header */}
      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3">
        <div className="min-w-0 flex-1">
          <div className="flex flex-wrap items-center gap-2">
            <h1 className="text-2xl md:text-3xl font-bold">{currentGroup.name}</h1>
            <Badge variant="outline" className="h-6 text-xs">
              {groupAccessories.length} {groupAccessories.length === 1 ? 'akcesorium' : 'akcesoriów'}
            </Badge>
            {currentGroup.category && (
              <Badge variant="secondary" className="h-6 text-xs">
                {currentGroup.category}
              </Badge>
            )}
          </div>
          {currentGroup.description && (
            <p className="text-muted-foreground mt-1 text-sm">{currentGroup.description}</p>
          )}
        </div>

        <div className="flex gap-2 shrink-0">
          {/* View toggle */}
          {/* View mode buttons */}
          <div className="flex gap-1">
            <Button
              variant="outline"
              size="sm"
              onClick={() => setViewMode('list')}
              data-testid="button-view-list"
              className={viewMode === 'list' ? 'bg-accent' : ''}
            >
              <List className="w-4 h-4" />
            </Button>
            <Button
              variant="outline"
              size="sm"
              onClick={() => setViewMode('compact')}
              data-testid="button-view-compact"
              className={viewMode === 'compact' ? 'bg-accent' : ''}
            >
              <AlignJustify className="w-4 h-4" />
            </Button>
            <Button
              variant="outline"
              size="sm"
              onClick={() => setViewMode('grid')}
              data-testid="button-view-grid"
              className={viewMode === 'grid' ? 'bg-accent' : ''}
            >
              <LayoutGrid className="w-4 h-4" />
            </Button>
          </div>

          {/* Group actions menu */}
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="outline" size="sm" data-testid="button-group-actions">
                <MoreHorizontal className="h-4 w-4" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end">
              <DropdownMenuItem onClick={() => toast({ title: "Edycja grupy", description: "Funkcjonalność będzie wkrótce" })}>
                <Edit2 className="h-4 w-4 mr-2" />
                Edytuj grupę
              </DropdownMenuItem>
              <DropdownMenuItem onClick={() => toast({ title: "Duplikowanie grupy", description: "Funkcjonalność będzie wkrótce" })}>
                <Copy className="h-4 w-4 mr-2" />
                Duplikuj grupę
              </DropdownMenuItem>
              <DropdownMenuItem 
                className="text-destructive"
                onClick={() => {
                  if (confirm(`Czy na pewno chcesz usunąć grupę "${currentGroup.name}"?`)) {
                    toast({ title: "Usuwanie grupy", description: "Funkcjonalność będzie wkrótce" });
                  }
                }}
              >
                <Trash2 className="h-4 w-4 mr-2" />
                Usuń grupę
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>

          <Button size="sm" data-testid="button-add-accessory" onClick={() => navigate(`/accessories/new?groupId=${groupCode}`)}>
            <Plus className="h-4 w-4 md:mr-2" />
            <span className="hidden md:inline">Dodaj</span>
          </Button>
        </div>
      </div>

      {/* Show inactive toggle */}
      <Card>
        <CardContent className="py-3">
          <label className="flex items-center gap-2 cursor-pointer">
            <input
              type="checkbox"
              checked={showInactive}
              onChange={(e) => setShowInactive(e.target.checked)}
              className="rounded"
              data-testid="checkbox-show-inactive"
            />
            <span className="text-sm">Pokaż nieaktywne akcesoria</span>
          </label>
        </CardContent>
      </Card>

      {/* Accessories List/Compact/Grid */}
      <Card>
        <CardContent className="p-0">
          {groupAccessories.length === 0 ? (
            <div className="text-center text-muted-foreground py-8">
              {showInactive 
                ? "Brak akcesoriów w tej grupie"
                : "Brak aktywnych akcesoriów w tej grupie"}
            </div>
          ) : (
            <div className="space-y-2 p-2">
              {/* List View */}
              {viewMode === 'list' && groupAccessories.map((accessory) => {
                const linkedMaterial = accessory.warehouseMaterialId 
                  ? materials.find(m => m.id === accessory.warehouseMaterialId)
                  : null;
                
                return (
                  <div
                    key={accessory.id}
                    className="flex items-start gap-3 p-2 border-b hover-elevate cursor-pointer"
                    data-testid={`accessory-${accessory.id}`}
                    onClick={() => navigate(`/accessories/${accessory.id}/edit`)}
                  >
                    <div className="w-12 h-12 flex-shrink-0 border border-dashed border-muted-foreground/30 flex items-center justify-center overflow-hidden">
                      {accessory.imageUrl ? (
                        <img 
                          src={accessory.imageUrl} 
                          alt={accessory.name}
                          className="w-full h-full object-cover"
                        />
                      ) : (
                        <span className="text-muted-foreground text-[10px]">Brak</span>
                      )}
                    </div>
                    
                    <div className="flex-1">
                      <h3 className="text-sm font-semibold">{accessory.name}</h3>
                      <p className="text-xs text-muted-foreground">
                        Kod: {accessory.code}
                        {accessory.alpmaCode && ` | Alpma: ${accessory.alpmaCode}`}
                      </p>
                      {linkedMaterial ? (
                        <div 
                          className="flex items-center gap-1 text-xs text-primary mt-0.5 hover:underline"
                          onClick={(e) => {
                            e.stopPropagation();
                            navigate(`/warehouse/${linkedMaterial.category}?materialId=${linkedMaterial.id}`);
                          }}
                          data-testid={`link-material-${linkedMaterial.id}`}
                        >
                          <ExternalLink className="h-3 w-3" />
                          <span>{linkedMaterial.name}</span>
                        </div>
                      ) : (
                        <div className="flex items-center gap-1 text-xs text-destructive mt-0.5">
                          <Circle className="h-2 w-2 fill-current" />
                          <span>Brak połączenia z magazynem</span>
                        </div>
                      )}
                      {accessory.description && (
                        <p className="text-sm mt-1">{accessory.description}</p>
                      )}
                    </div>
                    <div className="flex flex-col items-end gap-1">
                      {!accessory.isActive && (
                        <Badge variant="secondary" className="text-[10px]">
                          Nieaktywny
                        </Badge>
                      )}
                    </div>
                    <div onClick={(e) => e.stopPropagation()}>
                      <DropdownMenu>
                        <DropdownMenuTrigger asChild>
                          <Button
                            variant="ghost"
                            size="icon"
                            data-testid={`button-menu-accessory-${accessory.id}`}
                            className="flex-shrink-0"
                          >
                            <MoreVertical className="w-4 h-4" />
                          </Button>
                        </DropdownMenuTrigger>
                        <DropdownMenuContent align="end">
                          <DropdownMenuItem onClick={() => navigate(`/accessories/${accessory.id}/edit`)}>
                            <Edit2 className="w-4 h-4 mr-2" />
                            Edytuj
                          </DropdownMenuItem>
                          <DropdownMenuSeparator />
                          <DropdownMenuItem onClick={() => handleDuplicateClick(accessory)}>
                            <Files className="w-4 h-4 mr-2" />
                            Duplikuj
                          </DropdownMenuItem>
                        </DropdownMenuContent>
                      </DropdownMenu>
                    </div>
                  </div>
                );
              })}

              {/* Compact View */}
              {viewMode === 'compact' && groupAccessories.map((accessory, idx) => {
                const linkedMaterial = accessory.warehouseMaterialId 
                  ? materials.find(m => m.id === accessory.warehouseMaterialId)
                  : null;
                
                return (
                  <CompactListItem
                    key={accessory.id}
                    index={idx}
                    imageUrl={accessory.imageUrl}
                    primaryText={accessory.name}
                    secondaryText={accessory.code}
                    onClick={() => navigate(`/accessories/${accessory.id}/edit`)}
                    testId={`accessory-${accessory.id}`}
                    badge={
                      <>
                        {!accessory.isActive && (
                          <Badge variant="secondary" className="text-[10px] h-4">
                            Nieaktywny
                          </Badge>
                        )}
                      </>
                    }
                    menuItems={
                      <>
                        <DropdownMenuItem onClick={() => navigate(`/accessories/${accessory.id}/edit`)}>
                          <Edit2 className="w-4 h-4 mr-2" />
                          Edytuj
                        </DropdownMenuItem>
                        <DropdownMenuSeparator />
                        <DropdownMenuItem onClick={() => handleDuplicateClick(accessory)}>
                          <Files className="w-4 h-4 mr-2" />
                          Duplikuj
                        </DropdownMenuItem>
                      </>
                    }
                  />
                );
              })}

              {/* Grid View */}
              {viewMode === 'grid' && (
                <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-2">
                  {groupAccessories.map((accessory) => {
                    const linkedMaterial = accessory.warehouseMaterialId 
                      ? materials.find(m => m.id === accessory.warehouseMaterialId)
                      : null;
                    
                    return (
                      <div
                        key={accessory.id}
                        className="border overflow-hidden hover-elevate cursor-pointer"
                        data-testid={`accessory-${accessory.id}`}
                        onClick={() => navigate(`/accessories/${accessory.id}/edit`)}
                      >
                        <div className="aspect-square bg-muted flex items-center justify-center">
                          {accessory.imageUrl ? (
                            <img 
                              src={accessory.imageUrl} 
                              alt={accessory.name}
                              className="w-full h-full object-cover"
                            />
                          ) : (
                            <span className="text-muted-foreground text-xs">Brak zdjęcia</span>
                          )}
                        </div>
                        <div className="p-2">
                          <div className="flex items-start gap-1 mb-1">
                            <div className="flex-1 min-w-0">
                              <h3 className="font-semibold text-xs truncate">{accessory.name}</h3>
                              <p className="text-xs text-muted-foreground truncate">{accessory.code}</p>
                              {linkedMaterial ? (
                                <div 
                                  className="flex items-center gap-1 text-[10px] text-primary mt-0.5 hover:underline"
                                  onClick={(e) => {
                                    e.stopPropagation();
                                    navigate(`/warehouse/${linkedMaterial.category}?materialId=${linkedMaterial.id}`);
                                  }}
                                  data-testid={`link-material-${linkedMaterial.id}`}
                                >
                                  <ExternalLink className="h-2.5 w-2.5" />
                                  <span className="line-clamp-1">{linkedMaterial.name}</span>
                                </div>
                              ) : (
                                <div className="flex items-center gap-1 text-[10px] text-destructive mt-0.5">
                                  <Circle className="h-1.5 w-1.5 fill-current" />
                                </div>
                              )}
                            </div>
                            <div onClick={(e) => e.stopPropagation()}>
                              <DropdownMenu>
                                <DropdownMenuTrigger asChild>
                                  <Button
                                    variant="ghost"
                                    size="icon"
                                    className="h-6 w-6"
                                    data-testid={`button-menu-accessory-${accessory.id}`}
                                  >
                                    <MoreVertical className="w-3 h-3" />
                                  </Button>
                                </DropdownMenuTrigger>
                                <DropdownMenuContent align="end">
                                  <DropdownMenuItem onClick={() => navigate(`/accessories/${accessory.id}/edit`)}>
                                    <Edit2 className="w-4 h-4 mr-2" />
                                    Edytuj
                                  </DropdownMenuItem>
                                  <DropdownMenuSeparator />
                                  <DropdownMenuItem onClick={() => handleDuplicateClick(accessory)}>
                                    <Files className="w-4 h-4 mr-2" />
                                    Duplikuj
                                  </DropdownMenuItem>
                                </DropdownMenuContent>
                              </DropdownMenu>
                            </div>
                          </div>
                          {!accessory.isActive && (
                            <Badge variant="secondary" className="text-[10px] w-full justify-center">
                              Nieaktywny
                            </Badge>
                          )}
                        </div>
                      </div>
                    );
                  })}
                </div>
              )}
            </div>
          )}
        </CardContent>
      </Card>

      {/* Add button at bottom - green */}
      <div className="flex justify-center pt-6">
        <Button 
          size="lg" 
          className="bg-green-600 hover:bg-green-700 text-white"
          onClick={() => navigate(`/accessories/new?groupId=${groupCode}`)}
          data-testid="button-add-accessory-bottom"
        >
          <Plus className="h-5 w-5 mr-2" />
          Dodaj nowe akcesorium
        </Button>
      </div>

      {/* Duplicate Dialog */}
      <Dialog open={duplicateDialogOpen} onOpenChange={setDuplicateDialogOpen}>
        <DialogContent data-testid="dialog-duplicate-accessory">
          <DialogHeader>
            <DialogTitle>Duplikuj akcesorium</DialogTitle>
            <DialogDescription>
              {accessoryToDuplicate && `Duplikuj: ${accessoryToDuplicate.name}`}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="target-group">Grupa docelowa</Label>
              <Select value={duplicateTargetGroupId} onValueChange={setDuplicateTargetGroupId}>
                <SelectTrigger id="target-group" data-testid="select-target-group">
                  <SelectValue placeholder="Wybierz grupę" />
                </SelectTrigger>
                <SelectContent>
                  {groups.map((group) => (
                    <SelectItem key={group.id} value={group.id.toString()}>
                      {group.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <p className="text-xs text-muted-foreground">
                Kod akcesorium zostanie automatycznie dostosowany jeśli będzie konflikt
              </p>
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setDuplicateDialogOpen(false)}
              data-testid="button-cancel-duplicate"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleDuplicateConfirm}
              disabled={duplicateMutation.isPending || !duplicateTargetGroupId}
              data-testid="button-confirm-duplicate"
            >
              {duplicateMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Duplikowanie...
                </>
              ) : (
                <>
                  <Files className="w-4 h-4 mr-2" />
                  Duplikuj
                </>
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
