import { sql } from "drizzle-orm";
import { pgTable, pgSchema, text, varchar, timestamp, decimal, jsonb, boolean, serial, index, integer, uniqueIndex } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

export const allegroConnections = pgTable("allegro_connections", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  clientId: text("client_id").notNull(),
  clientSecret: text("client_secret").notNull(),
  accessToken: text("access_token"),
  refreshToken: text("refresh_token"),
  tokenExpiresAt: timestamp("token_expires_at"),
  isActive: boolean("is_active").default(false),
  connectionError: text("connection_error"),
  lastErrorAt: timestamp("last_error_at"),
  requiresReauth: boolean("requires_reauth").default(false),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// System Error Logs - centralized error tracking
export const errorLogs = pgTable("error_logs", {
  id: serial("id").primaryKey(),
  type: varchar("type", { length: 50 }).notNull(), // allegro, shoper, odoo, database, api, webhook, sync, etc.
  message: text("message").notNull(),
  stackTrace: text("stack_trace"),
  context: jsonb("context"), // {orderId, userId, endpoint, etc.}
  severity: varchar("severity", { length: 20 }).notNull().default("error"), // error, warning, info
  timestamp: timestamp("timestamp").notNull().defaultNow(),
  resolvedAt: timestamp("resolved_at"),
  resolvedBy: integer("resolved_by"), // user id who marked as resolved
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_error_logs_type").on(table.type),
  index("idx_error_logs_severity").on(table.severity),
  index("idx_error_logs_timestamp").on(table.timestamp),
  index("idx_error_logs_resolved").on(table.resolvedAt),
]);

export const allegroOrders = pgTable("allegro_orders", {
  id: varchar("id").primaryKey(),
  allegroOrderId: text("allegro_order_id").notNull().unique(),
  buyerLogin: text("buyer_login").notNull(),
  buyerEmail: text("buyer_email"),
  totalAmount: decimal("total_amount", { precision: 10, scale: 2 }).notNull(),
  currency: text("currency").default("PLN"),
  paymentStatus: text("payment_status").notNull(),
  fulfillmentStatus: text("fulfillment_status"),
  itemsCount: text("items_count").notNull(),
  orderDate: timestamp("order_date").notNull(),
  paymentDate: timestamp("payment_date"),
  rawData: jsonb("raw_data"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const syncSettings = pgTable("sync_settings", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  autoRefreshEnabled: boolean("auto_refresh_enabled").default(true),
  refreshIntervalMinutes: text("refresh_interval_minutes").default("3"),
  lastSyncAt: timestamp("last_sync_at"),
  fullSyncEnabled: boolean("full_sync_enabled").default(true),
  fullSyncWindowHours: text("full_sync_window_hours").default("2"),
  shoperAutoRefreshEnabled: boolean("shoper_auto_refresh_enabled").default(true),
  shoperRefreshIntervalMinutes: text("shoper_refresh_interval_minutes").default("5"),
  lastShoperSyncAt: timestamp("last_shoper_sync_at"),
  shoperFullSyncEnabled: boolean("shoper_full_sync_enabled").default(true),
  shoperFullSyncWindowHours: text("shoper_full_sync_window_hours").default("2"),
  nightlySyncEnabled: boolean("nightly_sync_enabled").default(true),
  nightlySyncHour: integer("nightly_sync_hour").default(2),
  nightlySyncWindowDays: integer("nightly_sync_window_days").default(7),
  lastNightlySyncAt: timestamp("last_nightly_sync_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const orderChanges = pgTable("order_changes", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  orderId: varchar("order_id").notNull(),
  orderNumber: text("order_number"),
  source: text("source").notNull(),
  changeType: text("change_type").notNull(),
  fieldChanged: text("field_changed"),
  oldValue: text("old_value"),
  newValue: text("new_value"),
  detectedAt: timestamp("detected_at").defaultNow(),
  createdAt: timestamp("created_at").defaultNow(),
});

// Allegro Fees tables
export const allegroFees = pgTable("allegro_fees", {
  id: serial("id").primaryKey(),
  entryId: text("entry_id").unique(), // Unikalny ID z Allegro API
  typeId: varchar("type_id", { length: 50 }).notNull(), // SUC, LIS, NSP, DTR, etc.
  typeName: text("type_name").notNull(),
  amount: decimal("amount", { precision: 10, scale: 2 }).notNull(),
  currency: varchar("currency", { length: 10 }).default("PLN"),
  occurredAt: timestamp("occurred_at").notNull(),
  orderId: text("order_id"), // ID zamówienia jeśli dotyczy
  offerId: text("offer_id"), // ID oferty jeśli dotyczy
  rawData: jsonb("raw_data"), // Pełne dane z API
  category: varchar("category", { length: 50 }), // commission, ads, delivery, refund, other
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_allegro_fees_occurred_at").on(table.occurredAt),
  index("idx_allegro_fees_type_id").on(table.typeId),
  index("idx_allegro_fees_category").on(table.category),
]);

export const allegroFeeSummaries = pgTable("allegro_fee_summaries", {
  id: serial("id").primaryKey(),
  summaryDate: timestamp("summary_date").notNull(), // Data podsumowania (dzień)
  commissionTotal: decimal("commission_total", { precision: 10, scale: 2 }).default("0"),
  commissionCount: integer("commission_count").default(0),
  listingTotal: decimal("listing_total", { precision: 10, scale: 2 }).default("0"),
  listingCount: integer("listing_count").default(0),
  adsTotal: decimal("ads_total", { precision: 10, scale: 2 }).default("0"),
  adsCount: integer("ads_count").default(0),
  deliveryTotal: decimal("delivery_total", { precision: 10, scale: 2 }).default("0"),
  deliveryCount: integer("delivery_count").default(0),
  refundsTotal: decimal("refunds_total", { precision: 10, scale: 2 }).default("0"),
  refundsCount: integer("refunds_count").default(0),
  otherTotal: decimal("other_total", { precision: 10, scale: 2 }).default("0"),
  otherCount: integer("other_count").default(0),
  grandTotal: decimal("grand_total", { precision: 10, scale: 2 }).default("0"),
  currency: varchar("currency", { length: 10 }).default("PLN"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  uniqueIndex("idx_allegro_fee_summaries_date").on(table.summaryDate),
]);

export const insertAllegroConnectionSchema = createInsertSchema(allegroConnections).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertErrorLogSchema = createInsertSchema(errorLogs).omit({
  id: true,
  createdAt: true,
});

export const insertAllegroOrderSchema = createInsertSchema(allegroOrders).omit({
  createdAt: true,
  updatedAt: true,
});

export const insertSyncSettingsSchema = createInsertSchema(syncSettings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertOrderChangeSchema = createInsertSchema(orderChanges).omit({
  id: true,
  createdAt: true,
});

export const insertAllegroFeeSchema = createInsertSchema(allegroFees).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertAllegroFeeSummarySchema = createInsertSchema(allegroFeeSummaries).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type AllegroConnection = typeof allegroConnections.$inferSelect;
export type InsertAllegroConnection = z.infer<typeof insertAllegroConnectionSchema>;

export type ErrorLog = typeof errorLogs.$inferSelect;
export type InsertErrorLog = z.infer<typeof insertErrorLogSchema>;

export type AllegroOrder = typeof allegroOrders.$inferSelect & {
  orderNumber?: number;
  source?: string;
};
export type InsertAllegroOrder = z.infer<typeof insertAllegroOrderSchema>;

export type SyncSettings = typeof syncSettings.$inferSelect;
export type InsertSyncSettings = z.infer<typeof insertSyncSettingsSchema>;

export type OrderChange = typeof orderChanges.$inferSelect;
export type InsertOrderChange = z.infer<typeof insertOrderChangeSchema>;

export type AllegroFee = typeof allegroFees.$inferSelect;
export type InsertAllegroFee = z.infer<typeof insertAllegroFeeSchema>;

export type AllegroFeeSummary = typeof allegroFeeSummaries.$inferSelect;
export type InsertAllegroFeeSummary = z.infer<typeof insertAllegroFeeSummarySchema>;

export const orderStatusSchema = z.enum([
  "PAID",
  "UNPAID",
  "PENDING",
  "CANCELLED",
]);

export const fulfillmentStatusSchema = z.enum([
  "NEW",
  "PROCESSING",
  "READY_FOR_SHIPMENT",
  "SENT",
  "DELIVERED",
  "CANCELLED",
]);

export type OrderStatus = z.infer<typeof orderStatusSchema>;
export type FulfillmentStatus = z.infer<typeof fulfillmentStatusSchema>;

// Auth tables
export const sessions = pgTable(
  "sessions",
  {
    sid: varchar("sid").primaryKey(),
    sess: jsonb("sess").notNull(),
    expire: timestamp("expire").notNull(),
  },
  (table) => [index("IDX_session_expire").on(table.expire)],
);

export const users = pgTable("users", {
  id: serial("id").primaryKey(),
  username: varchar("username", { length: 255 }).notNull().unique(),
  email: varchar("email", { length: 255 }).notNull().unique(),
  password: text("password").notNull(),
  firstName: varchar("first_name", { length: 255 }),
  lastName: varchar("last_name", { length: 255 }),
  role: varchar("role", { length: 50 }).notNull().default("user"),
  permissions: jsonb("permissions").default(sql`'[]'::jsonb`),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const passwordResetTokens = pgTable("password_reset_tokens", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id),
  token: varchar("token", { length: 255 }).notNull().unique(),
  expiresAt: timestamp("expires_at").notNull(),
  used: boolean("used").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertUserSchema = createInsertSchema(users, {
  username: z.string().min(3, "Username musi mieć minimum 3 znaki").max(255),
  email: z.string().email("Nieprawidłowy adres email"),
  password: z.string().min(6, "Hasło musi mieć minimum 6 znaków"),
}).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertPasswordResetTokenSchema = createInsertSchema(passwordResetTokens).omit({
  id: true,
  createdAt: true,
});

export const userRoleSchema = z.enum(["admin", "manager", "user"]);
export type UserRole = z.infer<typeof userRoleSchema>;

export const permissionSchema = z.enum([
  "orders.view",
  "orders.edit",
  "orders.delete",
  "settings.view",
  "settings.edit",
  "users.view",
  "users.edit",
  "sync.run",
]);
export type Permission = z.infer<typeof permissionSchema>;

export type User = typeof users.$inferSelect & {
  role: UserRole;
  permissions: Permission[];
};
export type InsertUser = z.infer<typeof insertUserSchema>;
export type PasswordResetToken = typeof passwordResetTokens.$inferSelect;
export type InsertPasswordResetToken = z.infer<typeof insertPasswordResetTokenSchema>;

// API Tokens for external integrations (e.g., Odoo)
export const apiTokens = pgTable("api_tokens", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  token: text("token").notNull().unique(),
  tokenPrefix: varchar("token_prefix", { length: 16 }),
  permissions: jsonb("permissions").default(sql`'[]'::jsonb`),
  isActive: boolean("is_active").default(true),
  lastUsedAt: timestamp("last_used_at"),
  expiresAt: timestamp("expires_at"),
  createdByUserId: integer("created_by_user_id").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Webhook configurations for pushing updates to external systems
export const webhookConfigs = pgTable("webhook_configs", {
  id: serial("id").primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  url: text("url").notNull(),
  secret: text("secret"),
  events: jsonb("events").default(sql`'[]'::jsonb`),
  isActive: boolean("is_active").default(true),
  retryAttempts: integer("retry_attempts").default(3),
  lastTriggeredAt: timestamp("last_triggered_at"),
  createdBy: integer("created_by").references(() => users.id),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// API request logs for audit trail
export const apiRequestLogs = pgTable("api_request_logs", {
  id: serial("id").primaryKey(),
  tokenId: integer("token_id").references(() => apiTokens.id),
  method: varchar("method", { length: 10 }).notNull(),
  path: text("path").notNull(),
  statusCode: integer("status_code"),
  responseTime: integer("response_time"),
  ipAddress: varchar("ip_address", { length: 45 }),
  userAgent: text("user_agent"),
  createdAt: timestamp("created_at").defaultNow(),
});

// Webhook delivery logs
export const webhookLogs = pgTable("webhook_logs", {
  id: serial("id").primaryKey(),
  webhookId: integer("webhook_id").references(() => webhookConfigs.id),
  event: varchar("event", { length: 100 }).notNull(),
  payload: jsonb("payload"),
  statusCode: integer("status_code"),
  responseBody: text("response_body"),
  attemptNumber: integer("attempt_number").default(1),
  success: boolean("success").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertApiTokenSchema = createInsertSchema(apiTokens).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertWebhookConfigSchema = createInsertSchema(webhookConfigs).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type ApiToken = typeof apiTokens.$inferSelect;
export type InsertApiToken = z.infer<typeof insertApiTokenSchema>;
export type WebhookConfig = typeof webhookConfigs.$inferSelect;
export type InsertWebhookConfig = z.infer<typeof insertWebhookConfigSchema>;
export type ApiRequestLog = typeof apiRequestLogs.$inferSelect;
export type WebhookLog = typeof webhookLogs.$inferSelect;

// Shoper Configuration
export const shoperConfig = pgTable("shoper_config", {
  id: serial("id").primaryKey(),
  ordersUrl: text("orders_url").notNull().default("https://alpmeb.pl/admin/orders/"),
  productsUrl: text("products_url").notNull().default("https://sklep378098.shoparena.pl/admin/products/edit/id/"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Odoo Integration tables
export const odooConfig = pgTable("odoo_config", {
  id: serial("id").primaryKey(),
  url: text("url").notNull(), // URL dla synchronizacji (API) - np. https://alpma.app/odoo
  userUrl: text("user_url"), // URL dla linków klikanych przez użytkowników - np. http://100.99.76.111:8070/odoo
  omsDomain: text("oms_domain").default("alp-oms.replit.app"), // Domena OMS do generowania linków zwrotnych
  database: varchar("database", { length: 255 }).notNull(),
  username: varchar("username", { length: 255 }).notNull(),
  isActive: boolean("is_active").default(true),
  autoConfirmOrders: boolean("auto_confirm_orders").default(false),
  syncIntervalMinutes: integer("sync_interval_minutes").default(3),
  lastSyncAt: timestamp("last_sync_at"),
  lastSyncStatus: varchar("last_sync_status", { length: 50 }),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const odooSyncQueue = pgTable("odoo_sync_queue", {
  id: serial("id").primaryKey(),
  orderNumber: integer("order_number").notNull(), // UNIQUE constraint already exists in DB
  source: varchar("source", { length: 20 }).notNull(),
  operation: varchar("operation", { length: 20 }).notNull().default("create"),
  status: varchar("status", { length: 50 }).notNull().default("pending"),
  attempts: integer("attempts").default(0),
  maxAttempts: integer("max_attempts").default(3),
  lastError: text("last_error"),
  odooOrderId: integer("odoo_order_id"),
  scheduledAt: timestamp("scheduled_at").defaultNow(),
  processedAt: timestamp("processed_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const odooSyncLogs = pgTable("odoo_sync_logs", {
  id: serial("id").primaryKey(),
  queueId: integer("queue_id").references(() => odooSyncQueue.id),
  orderNumber: integer("order_number"),
  operation: varchar("operation", { length: 20 }).notNull(),
  status: varchar("status", { length: 50 }).notNull(),
  message: text("message"),
  errorDetails: jsonb("error_details"),
  requestPayload: jsonb("request_payload"),
  responseData: jsonb("response_data"),
  duration: integer("duration"),
  odooOrderId: integer("odoo_order_id"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const insertShoperConfigSchema = createInsertSchema(shoperConfig).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertOdooConfigSchema = createInsertSchema(odooConfig).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertOdooSyncQueueSchema = createInsertSchema(odooSyncQueue).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertOdooSyncLogSchema = createInsertSchema(odooSyncLogs).omit({
  id: true,
  createdAt: true,
});

export type ShoperConfig = typeof shoperConfig.$inferSelect;
export type InsertShoperConfig = z.infer<typeof insertShoperConfigSchema>;
export type OdooConfig = typeof odooConfig.$inferSelect;
export type InsertOdooConfig = z.infer<typeof insertOdooConfigSchema>;
export type OdooSyncQueue = typeof odooSyncQueue.$inferSelect;
export type InsertOdooSyncQueue = z.infer<typeof insertOdooSyncQueueSchema>;
export type OdooSyncLog = typeof odooSyncLogs.$inferSelect;
export type InsertOdooSyncLog = z.infer<typeof insertOdooSyncLogSchema>;

// ==================== Catalog Schema ====================
// Dedykowany schemat PostgreSQL dla katalogu produktów
export const catalogSchema = pgSchema("catalog");

// Product Catalog Tables
export const products = catalogSchema.table("products", {
  id: serial("id").primaryKey(),
  sku: varchar("sku", { length: 100 }).notNull().unique(),
  
  // Wspólne pola (synchronizowane między platformami)
  title: text("title").notNull(),
  shortDescription: text("short_description"),
  longDescriptionHtml: text("long_description_html"),
  longDescriptionDoc: jsonb("long_description_doc"),
  gallery: jsonb("gallery"), // Array of image URLs
  
  // Wymiary i waga
  length: decimal("length", { precision: 10, scale: 2 }),
  width: decimal("width", { precision: 10, scale: 2 }),
  height: decimal("height", { precision: 10, scale: 2 }),
  weight: decimal("weight", { precision: 10, scale: 2 }),
  
  // Kolory i materiały
  color: text("color"),
  colorOptions: text("color_options").array().notNull().default(sql`ARRAY[]::text[]`), // Opcje kolorystyczne: ["LISTWA-P-WOTAN", "DRZWI-CZARNY"]
  material: text("material"),
  
  // Rodzaj mebla (ze słownika product_type)
  productType: text("product_type"),
  
  // Grupa produktu (ze słownika product_group: A1, A2, A3, A4)
  productGroup: text("product_group"),
  
  // Drzwi (ze słownika door: D1, D2, D3, D4)
  doors: text("doors"),
  
  // Nóżki (ze słownika leg: N1, N2, N3, N4)
  legs: text("legs"),
  
  // Cena bazowa
  basePrice: decimal("base_price", { precision: 10, scale: 2 }),
  currency: varchar("currency", { length: 3 }).default("PLN"),
  
  // Status
  isActive: boolean("is_active").default(true),
  
  // Matrix generation
  matrixId: integer("matrix_id").references(() => productMatrices.id, { onDelete: "set null" }),
  combinationKey: text("combination_key"), // Unique key: matrixId:lengthIdx:widthIdx:heightIdx:colorIdx
  generatedFromMatrix: boolean("generated_from_matrix").default(false),
  
  // Metadata
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  uniqueMatrixCombination: uniqueIndex("unique_matrix_combination").on(table.matrixId, table.combinationKey),
}));

export const productPlatformData = catalogSchema.table("product_platform_data", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }).notNull(),
  platform: varchar("platform", { length: 20 }).notNull(), // 'allegro', 'shoper', 'odoo'
  
  // Tytuł specyficzny dla platformy (może być edytowany)
  customTitle: text("custom_title"),
  
  // Kategoria specyficzna dla platformy
  categoryId: text("category_id"),
  categoryName: text("category_name"),
  
  // Opis specyficzny dla platformy
  description: text("description"),
  inheritsSharedDescription: boolean("inherits_shared_description").default(true),
  
  // Cena specyficzna dla platformy
  customPrice: decimal("custom_price", { precision: 10, scale: 2 }),
  
  // Parametry specyficzne dla platformy (JSON)
  customAttributes: jsonb("custom_attributes"),
  
  // Status publikacji
  isPublished: boolean("is_published").default(false),
  publishedAt: timestamp("published_at"),
  
  // ID w systemie zewnętrznym
  externalId: text("external_id"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const productImages = catalogSchema.table("product_images", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }).notNull(),
  
  // Ścieżka do pliku
  url: text("url").notNull(),
  filename: text("filename").notNull(),
  
  // Typ zdjęcia
  imageType: varchar("image_type", { length: 30 }).notNull(), // 'packshot', 'render', 'visualization'
  
  // Atrybuty SEO
  altText: text("alt_text"),
  title: text("title"),
  
  // Tagi (wielokrotny wybór)
  tags: text("tags").array().default(sql`ARRAY[]::text[]`),
  
  // Kolejność wyświetlania
  sortOrder: integer("sort_order").default(0),
  
  // Czy to główne zdjęcie (packshot na listę)
  isPrimary: boolean("is_primary").default(false),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const productAddons = catalogSchema.table("product_addons", {
  id: serial("id").primaryKey(),
  
  // Typ dodatku
  addonType: varchar("addon_type", { length: 50 }).notNull(), // 'fabric', 'board', 'certificate', 'accessory', 'component'
  
  // Nazwa dodatku
  name: text("name").notNull(),
  
  // Kod/SKU dodatku
  code: varchar("code", { length: 100 }),
  
  // Opis dodatku
  description: text("description"),
  
  // Szablon HTML do wstawienia w opis produktu
  htmlTemplate: text("html_template"),
  
  // Parametry dodatku (JSON)
  attributes: jsonb("attributes"),
  
  // Status
  isActive: boolean("is_active").default(true),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const addonImages = catalogSchema.table("addon_images", {
  id: serial("id").primaryKey(),
  addonId: integer("addon_id").references(() => productAddons.id, { onDelete: "cascade" }).notNull(),
  
  // Ścieżka do pliku
  url: text("url").notNull(),
  filename: text("filename").notNull(),
  
  // Atrybuty SEO
  altText: text("alt_text"),
  title: text("title"),
  
  // Kolejność wyświetlania
  sortOrder: integer("sort_order").default(0),
  
  createdAt: timestamp("created_at").defaultNow(),
});

export const productAddonAssignments = catalogSchema.table("product_addon_assignments", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }).notNull(),
  addonId: integer("addon_id").references(() => productAddons.id, { onDelete: "cascade" }).notNull(),
  
  // Kolejność w opisie produktu
  sortOrder: integer("sort_order").default(0),
  
  // Parametry nadpisane dla tego przypisania
  customAttributes: jsonb("custom_attributes"),
  
  createdAt: timestamp("created_at").defaultNow(),
});

// Template Categories for organizing description templates
export const descriptionTemplateCategories = catalogSchema.table("description_template_categories", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  description: text("description"),
  sortOrder: integer("sort_order").default(0),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Description Templates with Tiptap support and variables
export const descriptionTemplates = catalogSchema.table("description_templates", {
  id: serial("id").primaryKey(),
  
  // Nazwa szablonu
  name: text("name").notNull(),
  
  // Kategoria szablonu (FK)
  categoryId: integer("category_id").references(() => descriptionTemplateCategories.id, { onDelete: "set null" }),
  
  // Typ szablonu (deprecated - użyj categoryId zamiast tego)
  templateType: varchar("template_type", { length: 50 }).notNull(), // 'banner', 'certificate', 'footer', 'custom'
  
  // Treść szablonu jako dokument Tiptap (JSON)
  contentDoc: jsonb("content_doc"),
  
  // Treść szablonu HTML (generowany z contentDoc)
  htmlContent: text("html_content").notNull(),
  
  // Metadata o zmiennych używanych w szablonie
  variables: jsonb("variables"), // { "color": { type: "string", required: true }, "addons": { type: "array", required: false } }
  
  // Twórca szablonu
  createdBy: integer("created_by").references(() => users.id, { onDelete: "set null" }),
  
  // Czy szablon jest globalny (stosowany automatycznie)
  isGlobal: boolean("is_global").default(false),
  
  // Status
  isActive: boolean("is_active").default(true),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Junction table: Products ↔ Templates
export const productDescriptionTemplates = catalogSchema.table("product_description_templates", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }).notNull(),
  templateId: integer("template_id").references(() => descriptionTemplates.id, { onDelete: "cascade" }).notNull(),
  
  // Konfiguracja renderowania (np. mapowanie zmiennych)
  renderConfig: jsonb("render_config"),
  
  createdAt: timestamp("created_at").defaultNow(),
});

// AI Generation Requests tracking
export const aiGenerationRequests = catalogSchema.table("ai_generation_requests", {
  id: serial("id").primaryKey(),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }).notNull(),
  templateId: integer("template_id").references(() => descriptionTemplates.id, { onDelete: "set null" }),
  
  // Status: 'pending', 'processing', 'completed', 'failed'
  status: varchar("status", { length: 20 }).notNull().default("pending"),
  
  // Prompt wysłany do AI
  prompt: text("prompt"),
  
  // Odpowiedź od AI
  response: text("response"),
  
  // Metadata o kosztach (tokens, cost, model)
  costMetadata: jsonb("cost_metadata"),
  
  // Twórca requestu
  createdBy: integer("created_by").references(() => users.id, { onDelete: "set null" }),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Insert schemas
export const insertProductSchema = createInsertSchema(products).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductPlatformDataSchema = createInsertSchema(productPlatformData).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductImageSchema = createInsertSchema(productImages).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductAddonSchema = createInsertSchema(productAddons).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertAddonImageSchema = createInsertSchema(addonImages).omit({
  id: true,
  createdAt: true,
});

export const insertProductAddonAssignmentSchema = createInsertSchema(productAddonAssignments).omit({
  id: true,
  createdAt: true,
});

export const insertDescriptionTemplateSchema = createInsertSchema(descriptionTemplates).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertDescriptionTemplateCategorySchema = createInsertSchema(descriptionTemplateCategories).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductDescriptionTemplateSchema = createInsertSchema(productDescriptionTemplates).omit({
  id: true,
  createdAt: true,
});

export const insertAiGenerationRequestSchema = createInsertSchema(aiGenerationRequests).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// AI Prompt Settings (global configuration for AI description generation)
export const aiPromptSettings = pgTable("ai_prompt_settings", {
  id: serial("id").primaryKey(),
  systemInstructions: text("system_instructions").default(""),
  productMaterialInfo: text("product_material_info").default("płyta meblowa 18mm renomowanych polskich firm"),
  edgeBandInfo: text("edge_band_info").default("obrzeże ABS 0.8mm"),
  includeProductTable: boolean("include_product_table").default(true),
  productTableHeader: text("product_table_header").default("Wymiary produktu"),
  topBannerImageId: integer("top_banner_image_id"),
  bottomBannerImageId: integer("bottom_banner_image_id"),
  customInstructions: text("custom_instructions").default(""),
  
  // AI Section Prompts - base prompts for different description sections
  promptIntro: text("prompt_intro"),
  promptFeatures: text("prompt_features"),
  promptSafety: text("prompt_safety"),
  promptCare: text("prompt_care"),
  promptWarranty: text("prompt_warranty"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertAiPromptSettingsSchema = createInsertSchema(aiPromptSettings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// Types
export type Product = typeof products.$inferSelect;
export type InsertProduct = z.infer<typeof insertProductSchema>;

export type ProductPlatformData = typeof productPlatformData.$inferSelect;
export type InsertProductPlatformData = z.infer<typeof insertProductPlatformDataSchema>;

export type ProductImage = typeof productImages.$inferSelect;
export type InsertProductImage = z.infer<typeof insertProductImageSchema>;

export type ProductAddon = typeof productAddons.$inferSelect;
export type InsertProductAddon = z.infer<typeof insertProductAddonSchema>;

export type AddonImage = typeof addonImages.$inferSelect;
export type InsertAddonImage = z.infer<typeof insertAddonImageSchema>;

export type ProductAddonAssignment = typeof productAddonAssignments.$inferSelect;
export type InsertProductAddonAssignment = z.infer<typeof insertProductAddonAssignmentSchema>;

export type DescriptionTemplate = typeof descriptionTemplates.$inferSelect;
export type InsertDescriptionTemplate = z.infer<typeof insertDescriptionTemplateSchema>;

export type DescriptionTemplateCategory = typeof descriptionTemplateCategories.$inferSelect;
export type InsertDescriptionTemplateCategory = z.infer<typeof insertDescriptionTemplateCategorySchema>;

export type ProductDescriptionTemplate = typeof productDescriptionTemplates.$inferSelect;
export type InsertProductDescriptionTemplate = z.infer<typeof insertProductDescriptionTemplateSchema>;

export type AiGenerationRequest = typeof aiGenerationRequests.$inferSelect;
export type InsertAiGenerationRequest = z.infer<typeof insertAiGenerationRequestSchema>;

export type AiPromptSettings = typeof aiPromptSettings.$inferSelect;
export type InsertAiPromptSettings = z.infer<typeof insertAiPromptSettingsSchema>;

// ==================== Product Creator Schema ====================
// Dedykowany schemat PostgreSQL dla modułu Kreator Produktu
export const productCreatorSchema = pgSchema("product_creator");

// Product Creator - Dictionary Tables
export const productCreatorDictionaries = productCreatorSchema.table("dictionaries", {
  id: serial("id").primaryKey(),
  
  // Typ słownika: 'product_type', 'unit', 'color', 'product_group', 'door', 'leg', 'fabric'
  dictionaryType: varchar("dictionary_type", { length: 50 }).notNull(),
  
  // Kod techniczny (wielkie litery, bez polskich znaków) - np. SUPRA, BOCZKI, BIALY
  code: varchar("code", { length: 100 }).notNull(),
  
  // Nazwa (do wyświetlenia)
  name: text("name").notNull(),
  
  // Nazwa czytelna/opisowa - np. "Dąb Lancelot", "Dąb Wotan"
  readableName: text("readable_name"),
  
  // Nazwa krótka (do budowania nazw produktów) - np. "SUPRA", "D1", "N2"
  shortName: text("short_name"),
  
  // Opis
  description: text("description"),
  
  // Kolejność sortowania
  sortOrder: integer("sort_order").default(0),
  
  // Kategoria (opcjonalna)
  category: text("category"),
  
  // Ścieżka do zdjęcia
  imageUrl: text("image_url"),
  
  // Kolor HEX dla etykiet (np. #6366F1)
  color: varchar("color", { length: 7 }),
  
  // Status
  isActive: boolean("is_active").default(true),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertProductCreatorDictionarySchema = createInsertSchema(productCreatorDictionaries).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type ProductCreatorDictionary = typeof productCreatorDictionaries.$inferSelect;
export type InsertProductCreatorDictionary = z.infer<typeof insertProductCreatorDictionarySchema>;

// Enum dla typów słowników
export const dictionaryTypeSchema = z.enum([
  "product_type",     // Rodzaj produktu: SUPRA, TRES, VB, BINI
  "element_type",     // Rodzaj elementu: BOCZKI, BOK-L, BOK-P, POLKA, DRZWI, PLECY, WG, WD
  "unit",            // Jednostki miary: szt, m, m², m³, kg, l, mb, t, opak
  "color",           // Kolor: BIALY, KASZMIR, WOTAN, CZARNY, etc.
  "product_group",   // Grupa produktów: A1, A2, A3, A4
  "door",            // Drzwi: D1, D2, D3, D4
  "leg",             // Nóżki: N1, N2, N3, N4
  "fabric",          // Tkaniny: ES1, ES2
  "material",        // Materiały: płyta meblowa, sklejka, itp.
  "dimension_length", // Długości: 500, 600, 700, 800, 900, 1000 mm
  "dimension_width",  // Szerokości: 300, 320, 340, 360 mm
  "dimension_height", // Wysokości: 450, 820, 2050 mm
]);

export type DictionaryType = z.infer<typeof dictionaryTypeSchema>;

// Product Matrices - Matryce do generowania produktów
export const productMatrices = productCreatorSchema.table("product_matrices", {
  id: serial("id").primaryKey(),
  
  // Nazwa matryce
  name: text("name").notNull(),
  
  // Prefix i suffix dla nazwy produktu
  namePrefix: text("name_prefix"),
  nameSuffix: text("name_suffix"),
  
  // ===== PODSTAWOWE =====
  // Typ produktu (pojedyncza wartość ze słownika lub własna)
  productType: text("product_type"),
  
  // Drzwi (opcjonalne)
  doors: text("doors"),
  
  // Nóżki (opcjonalne)
  legs: text("legs"),
  
  // Materiał (opcjonalne)
  material: text("material"),
  
  // ===== ROZMIARY =====
  // Osobne pola dla wymiarów (arrays of numbers)
  lengths: decimal("lengths", { precision: 10, scale: 2 }).array().notNull().default(sql`ARRAY[]::numeric[]`),
  widths: decimal("widths", { precision: 10, scale: 2 }).array().notNull().default(sql`ARRAY[]::numeric[]`),
  heights: decimal("heights", { precision: 10, scale: 2 }).array().notNull().default(sql`ARRAY[]::numeric[]`),
  
  // Grupy produktu dla każdego rozmiaru (array - każdy indeks odpowiada rozmiarowi)
  productGroups: text("product_groups").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // Modyfikatory ceny dla każdego rozmiaru (array - każdy indeks odpowiada rozmiarowi)
  // Wartość może być procentem lub kwotą stałą w zależności od priceModifierTypes
  priceModifiers: decimal("price_modifiers", { precision: 10, scale: 2 }).array().notNull().default(sql`ARRAY[]::numeric[]`),
  
  // Typy modyfikatorów ceny ("percent" lub "fixed") dla każdego rozmiaru
  // np. ["percent", "fixed", "percent"] oznacza: pierwszy rozmiar +x%, drugi +x PLN, trzeci +x%
  priceModifierTypes: text("price_modifier_types").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // ===== KOLORY =====
  // Lista kolorów (nazwy) - dla kompatybilności wstecznej
  colors: text("colors").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // Zdjęcia przypisane do kolorów jako JSON: {colorName: ["url1", "url2"]}
  colorImages: jsonb("color_images").notNull().default(sql`'{}'::jsonb`),
  
  // Kolory zaznaczone do generowania (jeśli puste, generuje wszystkie kolory)
  selectedColors: text("selected_colors").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // Daty ostatniej generacji dla każdego koloru jako JSON: {colorName: "2025-10-28T14:30:00.000Z"}
  colorLastGenerated: jsonb("color_last_generated").notNull().default(sql`'{}'::jsonb`),
  
  // Opcje kolorystyczne dla każdego koloru jako JSON: {colorName: ["LISTWA-P-WOTAN", "DRZWI-CZARNY"]}
  colorOptions: jsonb("color_options").notNull().default(sql`'{}'::jsonb`),
  
  // ===== OPIS =====
  // Opis jako dokument Tiptap (JSON)
  descriptionDoc: jsonb("description_doc"),
  
  // Opis jako HTML (generowany z descriptionDoc)
  descriptionHtml: text("description_html"),
  
  // Prompt dla AI do generowania opisów
  aiPrompt: text("ai_prompt"),
  
  // Czy używać AI do generowania opisów
  useAiGeneration: boolean("use_ai_generation").default(false),
  
  // ===== SZABLON OPISU =====
  // Szablon opisu produktu (FK do description_templates)
  templateId: integer("template_id").references(() => descriptionTemplates.id, { onDelete: "set null" }),
  
  // Konfiguracja AI dla różnych sekcji opisu
  // Format JSON: { "intro": { "prompt": "...", "enabled": true }, "features": { "prompt": "...", "enabled": true } }
  aiConfig: jsonb("ai_config").default(sql`'{}'::jsonb`),
  
  // Czy produkt ma gwarancję (wpływa na generowanie sekcji {{ai.warranty}})
  hasWarranty: boolean("has_warranty").default(true),
  
  // ===== WYMIARY I CENA =====
  // Domyślna cena
  defaultPrice: decimal("default_price", { precision: 10, scale: 2 }),
  
  // Twórca matryce
  createdBy: integer("created_by").references(() => users.id, { onDelete: "set null" }),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertProductMatrixSchema = createInsertSchema(productMatrices).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type ProductMatrix = typeof productMatrices.$inferSelect;
export type InsertProductMatrix = z.infer<typeof insertProductMatrixSchema>;

// ===== PRODUCT ACCESSORIES SYSTEM =====
// System dodatków/akcesoriów do produktów (okucia, płyty, nóżki, uchwyty itp.)

// Grupy dodatków (np. "Okucia", "Płyty meblowe", "Nóżki")
export const accessoryGroups = catalogSchema.table("accessory_groups", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(), // np. "Okucia", "Płyty meblowe"
  code: text("code").notNull().unique(), // np. "hardware", "boards", "legs"
  category: text("category"), // Kategoria grupy (np. "Tkaniny", "Okucia", "Płyty meblowe")
  description: text("description"), // Krótki opis grupy
  displayOrder: integer("display_order").default(0).notNull(), // Kolejność wyświetlania
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_accessory_groups_code").on(table.code),
  index("idx_accessory_groups_active").on(table.isActive),
  index("idx_accessory_groups_category").on(table.category),
]);

export const insertAccessoryGroupSchema = createInsertSchema(accessoryGroups).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type AccessoryGroup = typeof accessoryGroups.$inferSelect;
export type InsertAccessoryGroup = z.infer<typeof insertAccessoryGroupSchema>;

// Dodatki (konkretne elementy w grupach)
export const accessories = catalogSchema.table("accessories", {
  id: serial("id").primaryKey(),
  groupId: integer("group_id").references(() => accessoryGroups.id, { onDelete: "cascade" }),
  warehouseMaterialId: integer("warehouse_material_id"), // Link to warehouse.materials
  name: text("name").notNull(), // np. "Zawiasy Silent System"
  code: text("code").notNull().unique(), // np. "silent_hinges"
  alpmaCode: text("alpma_code"), // Własny kod Alpma
  description: text("description"), // Szczegółowy opis dodatku
  price: decimal("price", { precision: 10, scale: 2 }), // Cena from warehouse or manual
  imageUrl: text("image_url"), // URL do zdjęcia dodatku
  displayOrder: integer("display_order").default(0).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_accessories_group").on(table.groupId),
  index("idx_accessories_code").on(table.code),
  index("idx_accessories_active").on(table.isActive),
  index("idx_accessories_warehouse_material").on(table.warehouseMaterialId),
]);

export const insertAccessorySchema = createInsertSchema(accessories).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type Accessory = typeof accessories.$inferSelect;
export type InsertAccessory = z.infer<typeof insertAccessorySchema>;

// Przypisanie GRUP dodatków do matryc produktowych (relacja many-to-many)
// Teraz przypisujemy całe grupy zamiast pojedynczych akcesoriów
export const matrixAccessories = catalogSchema.table("matrix_accessories", {
  id: serial("id").primaryKey(),
  matrixId: integer("matrix_id").references(() => productMatrices.id, { onDelete: "cascade" }).notNull(),
  groupId: integer("group_id").references(() => accessoryGroups.id, { onDelete: "cascade" }).notNull(), // Zmienione: groupId zamiast accessoryId
  displayOrder: integer("display_order").default(0).notNull(), // Kolejność wyświetlania w opisie
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_matrix_accessories_matrix").on(table.matrixId),
  index("idx_matrix_accessories_group").on(table.groupId), // Zmienione: indeks na groupId
  uniqueIndex("idx_matrix_accessories_unique").on(table.matrixId, table.groupId), // Zmienione: unique constraint na matrixId + groupId
]);

export const insertMatrixAccessorySchema = createInsertSchema(matrixAccessories).omit({
  id: true,
  createdAt: true,
});

export type MatrixAccessory = typeof matrixAccessories.$inferSelect;
export type InsertMatrixAccessory = z.infer<typeof insertMatrixAccessorySchema>;

// Sync Health Stats - Monitorowanie stanu synchronizacji
export const syncHealthStats = pgTable("sync_health_stats", {
  id: serial("id").primaryKey(),
  
  // Typ synchronizacji
  syncType: varchar("sync_type", { length: 50 }).notNull(), // allegro, shoper, odoo_queue
  
  // Ostatnia synchronizacja
  lastSyncAt: timestamp("last_sync_at"),
  lastSuccessAt: timestamp("last_success_at"),
  lastFailureAt: timestamp("last_failure_at"),
  
  // Statystyki (24h)
  successCount24h: integer("success_count_24h").default(0),
  failureCount24h: integer("failure_count_24h").default(0),
  totalSynced24h: integer("total_synced_24h").default(0),
  
  // Aktualny status
  healthStatus: varchar("health_status", { length: 20 }).notNull().default("unknown"), // healthy, degraded, critical, unknown
  consecutiveFailures: integer("consecutive_failures").default(0),
  
  // Ostatni błąd
  lastError: text("last_error"),
  lastErrorContext: jsonb("last_error_context"),
  
  // Kolejka (dla Odoo)
  queuePending: integer("queue_pending").default(0),
  queueFailed: integer("queue_failed").default(0),
  
  // Auto-healing
  lastAutoHealAt: timestamp("last_auto_heal_at"),
  autoHealCount: integer("auto_heal_count").default(0),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_sync_health_type").on(table.syncType),
  index("idx_sync_health_status").on(table.healthStatus),
  uniqueIndex("idx_sync_health_unique_type").on(table.syncType),
]);

export const insertSyncHealthStatsSchema = createInsertSchema(syncHealthStats).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type SyncHealthStats = typeof syncHealthStats.$inferSelect;
export type InsertSyncHealthStats = z.infer<typeof insertSyncHealthStatsSchema>;

// ===== FILE STORAGE SETTINGS =====
// External file server configuration (SFTP, S3, etc.)
export const fileStorageSettings = pgTable("file_storage_settings", {
  id: serial("id").primaryKey(),
  
  // Provider type
  provider: varchar("provider", { length: 20 }).notNull().default("local"), // local, sftp, s3
  
  // SFTP/Server settings
  host: text("host"), // np. files.alpsys.pl
  port: integer("port").default(22), // SFTP port
  username: text("username"), // np. oms-sftp
  
  // Base URL for accessing files
  baseUrl: text("base_url"), // np. https://files.alpsys.pl
  
  // Path prefix on server
  pathPrefix: text("path_prefix").default("/OMS"), // np. /OMS
  
  // Is this configuration active
  isActive: boolean("is_active").default(false),
  
  // Connection test results
  lastConnectionTest: timestamp("last_connection_test"),
  connectionStatus: varchar("connection_status", { length: 20 }).default("unknown"), // connected, failed, unknown
  lastError: text("last_error"),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertFileStorageSettingsSchema = createInsertSchema(fileStorageSettings).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type FileStorageSettings = typeof fileStorageSettings.$inferSelect;
export type InsertFileStorageSettings = z.infer<typeof insertFileStorageSettingsSchema>;

// ===== PRODUCT SETS (ZESTAWY) =====

// Set Matrices - Szablony/matrycy zestawów
export const setMatrices = productCreatorSchema.table("set_matrices", {
  id: serial("id").primaryKey(),
  
  // Nazwa matrycy
  name: text("name").notNull(),
  namePrefix: text("name_prefix"), // "Zestaw"
  nameSuffix: text("name_suffix"),
  
  // Grupa produktów dla zestawu (ze słownika product_group)
  productGroup: text("product_group"),
  
  // Wymiary zestawu (ze słowników dimension_length, dimension_width, dimension_height)
  length: text("length"), // Długość - wartość ze słownika (np. "900")
  width: text("width"), // Szerokość - wartość ze słownika (np. "300")
  height: text("height"), // Wysokość - wartość ze słownika (np. "2050")
  
  // Drzwi i nogi (ze słownika door, leg)
  doors: text("doors"), // np. "D1", "D2" ze słownika door
  legs: text("legs"), // np. "N1", "N2" ze słownika leg
  
  // Zdjęcie poglądowe
  imageUrl: text("image_url"),
  
  // Ustawienia zestawu
  defaultDepth: decimal("default_depth", { precision: 10, scale: 2 }), // Głębokość zestawu (mm)
  hookLength: decimal("hook_length", { precision: 10, scale: 2 }), // Długość wieszaka (mm)
  
  // Komponenty zestawu (JSON array)
  // [{componentType: "SZAFKA", length: 60, width: 30, quantity: 1, name: "Szafka"},
  //  {componentType: "PANEL", length: 50, width: 30, quantity: 2, name: "Panel"},
  //  {componentType: "WIESZAK", length: 50, width: null, quantity: 1, name: "Wieszak"}]
  components: jsonb("components").notNull().default(sql`'[]'::jsonb`),
  
  // Cennik
  basePriceModifier: decimal("base_price_modifier", { precision: 10, scale: 2 }).default("0"),
  priceModifierType: text("price_modifier_type").default("percent"), // "percent" lub "fixed"
  
  // ===== KOLORY =====
  // Lista kolorów (nazwy)
  colors: text("colors").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // Zdjęcia przypisane do kolorów jako JSON: {colorName: ["url1", "url2"]}
  colorImages: jsonb("color_images").notNull().default(sql`'{}'::jsonb`),
  
  // Kolory zaznaczone do generowania (jeśli puste, generuje wszystkie kolory)
  selectedColors: text("selected_colors").array().notNull().default(sql`ARRAY[]::text[]`),
  
  // Opcje kolorystyczne dla każdego koloru jako JSON: {colorName: ["LISTWA-P-WOTAN", "DRZWI-CZARNY"]}
  colorOptions: jsonb("color_options").notNull().default(sql`'{}'::jsonb`),
  
  // Generowanie
  useAiGeneration: boolean("use_ai_generation").default(false),
  aiPrompt: text("ai_prompt"),
  descriptionTemplateId: integer("description_template_id").references(() => descriptionTemplates.id, { onDelete: "set null" }),
  
  // Enhanced descriptions - build descriptions from component products
  useEnhancedDescription: boolean("use_enhanced_description").default(false),
  
  // Status
  isActive: boolean("is_active").default(true),
  
  // Generation tracking
  lastGenerationStatus: varchar("last_generation_status", { length: 20 }), // 'success', 'partial', 'failed', 'never', null
  lastGenerationAt: timestamp("last_generation_at"),
  lastMissingComponents: jsonb("last_missing_components"), // Array of {componentType, length?, width?}
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Product Sets - Wygenerowane zestawy
export const productSets = productCreatorSchema.table("product_sets", {
  id: serial("id").primaryKey(),
  setMatrixId: integer("set_matrix_id").references(() => setMatrices.id, { onDelete: "cascade" }),
  
  // Identyfikatory
  sku: text("sku").notNull().unique(),
  title: text("title").notNull(),
  shortDescription: text("short_description"),
  fullDescription: text("full_description"),
  
  // Parametry tego zestawu
  color: text("color"),
  depth: decimal("depth", { precision: 10, scale: 2 }),
  panelCount: integer("panel_count"),
  hookLength: decimal("hook_length", { precision: 10, scale: 2 }),
  
  // Cena
  basePrice: decimal("base_price", { precision: 10, scale: 2 }),
  calculatedPrice: decimal("calculated_price", { precision: 10, scale: 2 }),
  
  // Zdjęcia (URLs)
  images: text("images").array().default(sql`ARRAY[]::text[]`),
  
  // Generowanie
  generatedFromMatrix: boolean("generated_from_matrix").default(false),
  combinationKey: text("combination_key"), // Unikalny klucz: matrixId:color
  
  // Status
  isActive: boolean("is_active").default(true),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  uniqueMatrixCombination: uniqueIndex("unique_set_matrix_combination").on(table.setMatrixId, table.combinationKey),
}));

// Set Product Links - Powiązania produktów z zestawami
export const setProductLinks = productCreatorSchema.table("set_product_links", {
  id: serial("id").primaryKey(),
  setId: integer("set_id").notNull().references(() => productSets.id, { onDelete: "cascade" }),
  productId: integer("product_id").references(() => products.id, { onDelete: "cascade" }), // Nullable - komponenty mogą nie mieć przypisanych produktów
  
  componentType: text("component_type").notNull(), // "cabinet", "panel", "hook", etc.
  quantity: integer("quantity").default(1),
  position: integer("position").default(0), // Kolejność w zestawie
  
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_set_product_links_set").on(table.setId),
  index("idx_set_product_links_product").on(table.productId),
]);

// Insert schemas
export const insertSetMatrixSchema = createInsertSchema(setMatrices).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertProductSetSchema = createInsertSchema(productSets).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertSetProductLinkSchema = createInsertSchema(setProductLinks).omit({
  id: true,
  createdAt: true,
});

// Types
export type SetMatrix = typeof setMatrices.$inferSelect;
export type InsertSetMatrix = z.infer<typeof insertSetMatrixSchema>;

export type ProductSet = typeof productSets.$inferSelect;
export type InsertProductSet = z.infer<typeof insertProductSetSchema>;

export type SetProductLink = typeof setProductLinks.$inferSelect;
export type InsertSetProductLink = z.infer<typeof insertSetProductLinkSchema>;

// ==================== Warehouse Schema ====================
// Dedykowany schemat PostgreSQL dla magazynu surowców
export const warehouseSchema = pgSchema("warehouse");

// Warehouse Categories - Konfigurowalne kategorie magazynu (np. "Okucia", "Płyty meblowe", "Śruby")
export const warehouseCategories = warehouseSchema.table("categories", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(), // Nazwa wyświetlana: "Okucia", "Płyty meblowe"
  code: text("code").notNull().unique(), // Kod URL: "okucia", "plyty_meblowe"
  icon: text("icon").default("Box"), // Nazwa ikony z lucide-react
  iconColor: text("icon_color").default("text-gray-600/70"), // Kolor ikony
  description: text("description"), // Opcjonalny opis kategorii
  displayOrder: integer("display_order").default(0).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_warehouse_categories_code").on(table.code),
  index("idx_warehouse_categories_active").on(table.isActive),
  index("idx_warehouse_categories_order").on(table.displayOrder),
]);

// Material Groups - Grupy materiałów (np. "Zawiasy VB", "Haki wieszaka", "Płyta biała")
export const materialGroups = warehouseSchema.table("material_groups", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(), // np. "Zawiasy VB", "Haki wieszaka"
  code: text("code").notNull().unique(), // np. "hinges_vb", "hooks_hanger"
  category: text("category"), // Kod kategorii: "okucia", "plyty_meblowe", "sruby", "tkaniny", "pianki"
  description: text("description"), // Opis grupy
  
  // Galeria zdjęć dla grupy
  gallery: text("gallery").array().notNull().default(sql`ARRAY[]::text[]`),
  primaryImage: text("primary_image"), // Główne zdjęcie grupy
  
  displayOrder: integer("display_order").default(0).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_material_groups_code").on(table.code),
  index("idx_material_groups_category").on(table.category),
  index("idx_material_groups_active").on(table.isActive),
]);

// Materials - Konkretne materiały/surowce
export const materials = warehouseSchema.table("materials", {
  id: serial("id").primaryKey(),
  groupId: integer("group_id").references(() => materialGroups.id, { onDelete: "cascade" }),
  
  // Podstawowe informacje
  name: text("name").notNull(), // np. "Zawias Silent System 110°"
  internalCode: text("internal_code").notNull().unique(), // Kod wewnętrzny Alpma
  supplierCode: text("supplier_code"), // Kod dostawcy
  
  // Opis i specyfikacja
  description: text("description"),
  specifications: jsonb("specifications"), // Dodatkowe parametry jako JSON
  
  // Jednostka miary
  unitOfMeasure: varchar("unit_of_measure", { length: 20 }).default("szt"), // szt, m, m², kg, l
  
  // Cena
  price: decimal("price", { precision: 10, scale: 2 }), // Cena materiału
  
  // Galeria (ścieżki do zdjęć jako tablica)
  gallery: text("gallery").array().notNull().default(sql`ARRAY[]::text[]`),
  primaryImageUrl: text("primary_image_url"), // Główne zdjęcie
  
  // Status
  displayOrder: integer("display_order").default(0).notNull(),
  isActive: boolean("is_active").default(true).notNull(),
  
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => [
  index("idx_materials_group").on(table.groupId),
  index("idx_materials_internal_code").on(table.internalCode),
  index("idx_materials_supplier_code").on(table.supplierCode),
  index("idx_materials_active").on(table.isActive),
]);

// Material Media - Galeria zdjęć dla materiałów
export const materialMedia = warehouseSchema.table("material_media", {
  id: serial("id").primaryKey(),
  materialId: integer("material_id").references(() => materials.id, { onDelete: "cascade" }).notNull(),
  
  url: text("url").notNull(), // Ścieżka do pliku
  filename: text("filename").notNull(),
  
  // Metadane
  altText: text("alt_text"),
  caption: text("caption"),
  
  // Kolejność i typ
  displayOrder: integer("display_order").default(0).notNull(),
  isPrimary: boolean("is_primary").default(false),
  
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_material_media_material").on(table.materialId),
  index("idx_material_media_order").on(table.displayOrder),
]);

// Material Accessory Links - Powiązania materiałów z akcesoriami
export const materialAccessoryLinks = warehouseSchema.table("material_accessory_links", {
  id: serial("id").primaryKey(),
  materialId: integer("material_id").references(() => materials.id, { onDelete: "cascade" }).notNull(),
  accessoryId: integer("accessory_id").notNull(), // Reference to catalog.accessories
  
  quantity: integer("quantity").default(1), // Ilość materiału na jedno akcesorium
  notes: text("notes"), // Notatki (np. "na 1 szafę 180cm")
  
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => [
  index("idx_material_accessory_material").on(table.materialId),
  index("idx_material_accessory_accessory").on(table.accessoryId),
  uniqueIndex("idx_material_accessory_unique").on(table.materialId, table.accessoryId),
]);

// Insert schemas
export const insertWarehouseCategorySchema = createInsertSchema(warehouseCategories).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertMaterialGroupSchema = createInsertSchema(materialGroups).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertMaterialSchema = createInsertSchema(materials).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertMaterialMediaSchema = createInsertSchema(materialMedia).omit({
  id: true,
  createdAt: true,
});

export const insertMaterialAccessoryLinkSchema = createInsertSchema(materialAccessoryLinks).omit({
  id: true,
  createdAt: true,
});

// Types
export type WarehouseCategory = typeof warehouseCategories.$inferSelect;
export type InsertWarehouseCategory = z.infer<typeof insertWarehouseCategorySchema>;

export type MaterialGroup = typeof materialGroups.$inferSelect;
export type InsertMaterialGroup = z.infer<typeof insertMaterialGroupSchema>;

export type Material = typeof materials.$inferSelect;
export type InsertMaterial = z.infer<typeof insertMaterialSchema>;

export type MaterialMedia = typeof materialMedia.$inferSelect;
export type InsertMaterialMedia = z.infer<typeof insertMaterialMediaSchema>;

export type MaterialAccessoryLink = typeof materialAccessoryLinks.$inferSelect;
export type InsertMaterialAccessoryLink = z.infer<typeof insertMaterialAccessoryLinkSchema>;
