import { pool } from './postgres.js';
import axios from 'axios';

async function getAccessToken(): Promise<string | null> {
  const result = await pool.query(`
    SELECT access_token FROM allegro_connections WHERE is_active = true LIMIT 1
  `);
  return result.rows[0]?.access_token || null;
}

async function fetchOrderFromAllegro(orderId: string, accessToken: string): Promise<any> {
  try {
    const response = await axios.get(
      `https://api.allegro.pl/order/checkout-forms/${orderId}`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        }
      }
    );
    return response.data;
  } catch (error: any) {
    console.error(`❌ Błąd pobierania zamówienia ${orderId}:`, error.message);
    return null;
  }
}

async function updateIncompleteOrders() {
  const client = await pool.connect();
  
  try {
    console.log('🔄 Aktualizacja niekompletnych zamówień...\n');

    const accessToken = await getAccessToken();
    if (!accessToken) {
      console.error('❌ Brak access token');
      return;
    }

    // Pobierz zamówienia bez produktów
    const result = await client.query(`
      SELECT o.id, o.order_number, o.source_order_id
      FROM commerce.orders o
      LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
      WHERE o.source = 'ALLEGRO'
      GROUP BY o.id, o.order_number, o.source_order_id
      HAVING COUNT(oi.id) = 0
      ORDER BY o.id DESC
      LIMIT 10
    `);

    console.log(`📊 Znaleziono ${result.rows.length} zamówień bez produktów\n`);

    for (const order of result.rows) {
      console.log(`\n🔍 Przetwarzanie zamówienia #${order.order_number} (${order.source_order_id})...`);
      
      const orderData = await fetchOrderFromAllegro(order.source_order_id, accessToken);
      
      if (!orderData) {
        console.log(`❌ Nie udało się pobrać danych dla #${order.order_number}`);
        continue;
      }

      if (!orderData.lineItems || orderData.lineItems.length === 0) {
        console.log(`⚠️ Zamówienie #${order.order_number} nie ma produktów w API`);
        continue;
      }

      console.log(`✅ Pobrano ${orderData.lineItems.length} produktów z API`);
      
      await client.query('BEGIN');
      
      try {
        // Zaktualizuj raw_payload
        await client.query(`
          UPDATE commerce.orders
          SET raw_payload = $1, updated_at = NOW()
          WHERE id = $2
        `, [JSON.stringify(orderData), order.id]);
        
        // Dodaj produkty
        for (const item of orderData.lineItems) {
          const externalId = item.offer?.external?.id;
          const imageUrl = item.imageUrl;
          
          await client.query(`
            INSERT INTO commerce.order_items (
              order_id, offer_external_id, name, quantity, unit_price, price, image_url, raw_data
            ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
          `, [
            order.id,
            externalId,
            item.offer?.name,
            item.quantity || 1,
            parseFloat(item.price?.amount || '0'),
            parseFloat(item.price?.amount || '0') * (item.quantity || 1),
            imageUrl,
            JSON.stringify(item)
          ]);
        }
        
        await client.query('COMMIT');
        console.log(`✅ Zaktualizowano zamówienie #${order.order_number} - dodano ${orderData.lineItems.length} produktów`);
        
      } catch (error) {
        await client.query('ROLLBACK');
        console.error(`❌ Błąd aktualizacji zamówienia #${order.order_number}:`, error);
      }
      
      await new Promise(resolve => setTimeout(resolve, 500));
    }

    console.log('\n✅ Zakończono aktualizację zamówień');

  } catch (error) {
    console.error('❌ Błąd:', error);
  } finally {
    client.release();
    await pool.end();
  }
}

updateIncompleteOrders();
