import {
  type AllegroConnection,
  type InsertAllegroConnection,
  type AllegroOrder,
  type InsertAllegroOrder,
  type SyncSettings,
  type InsertSyncSettings,
  type ProductCreatorDictionary,
  type InsertProductCreatorDictionary,
  type DictionaryType,
  type ProductMatrix,
  type InsertProductMatrix,
  type SetMatrix,
  type InsertSetMatrix,
  type ProductSet,
  type InsertProductSet,
  type SetProductLink,
  type InsertSetProductLink,
} from "@shared/schema";
import { randomUUID } from "crypto";
import { pool } from "./postgres";

export interface IStorage {
  getAllegroConnection(): Promise<AllegroConnection | undefined>;
  createOrUpdateConnection(
    connection: InsertAllegroConnection
  ): Promise<AllegroConnection>;
  updateConnectionTokens(
    id: string,
    accessToken: string,
    refreshToken: string,
    expiresAt: Date
  ): Promise<void>;
  
  getOrders(): Promise<AllegroOrder[]>;
  getOrderByAllegroId(allegroOrderId: string): Promise<AllegroOrder | undefined>;
  createOrUpdateOrder(order: InsertAllegroOrder): Promise<AllegroOrder>;
  
  getSyncSettings(): Promise<SyncSettings | undefined>;
  createOrUpdateSyncSettings(
    settings: InsertSyncSettings
  ): Promise<SyncSettings>;
  
  // Product Creator Dictionaries
  getDictionaries(type?: DictionaryType): Promise<ProductCreatorDictionary[]>;
  getDictionaryById(id: number): Promise<ProductCreatorDictionary | undefined>;
  createDictionary(dictionary: InsertProductCreatorDictionary): Promise<ProductCreatorDictionary>;
  updateDictionary(id: number, dictionary: Partial<InsertProductCreatorDictionary>): Promise<ProductCreatorDictionary>;
  deleteDictionary(id: number): Promise<void>;
  
  // Product Matrices
  getProductMatrices(): Promise<ProductMatrix[]>;
  getProductMatrixById(id: number): Promise<ProductMatrix | undefined>;
  createProductMatrix(matrix: InsertProductMatrix): Promise<ProductMatrix>;
  updateProductMatrix(id: number, matrix: Partial<InsertProductMatrix>): Promise<ProductMatrix>;
  deleteProductMatrix(id: number): Promise<void>;
  
  // Set Matrices (Matrycy Zestawów)
  getSetMatrices(): Promise<SetMatrix[]>;
  getSetMatrixById(id: number): Promise<SetMatrix | undefined>;
  createSetMatrix(matrix: InsertSetMatrix): Promise<SetMatrix>;
  updateSetMatrix(id: number, matrix: Partial<InsertSetMatrix>): Promise<SetMatrix>;
  deleteSetMatrix(id: number): Promise<void>;
  
  // Product Sets (Wygenerowane Zestawy)
  getProductSets(): Promise<ProductSet[]>;
  getProductSetById(id: number): Promise<ProductSet | undefined>;
  getProductSetBySku(sku: string): Promise<ProductSet | undefined>;
  createProductSet(set: InsertProductSet): Promise<ProductSet>;
  updateProductSet(id: number, set: Partial<InsertProductSet>): Promise<ProductSet>;
  deleteProductSet(id: number): Promise<void>;
  
  // Set Product Links (Powiązania Produktów z Zestawami)
  getSetProductLinks(setId: number): Promise<SetProductLink[]>;
  getProductSetsForProduct(productId: number): Promise<ProductSet[]>;
  createSetProductLink(link: InsertSetProductLink): Promise<SetProductLink>;
  deleteSetProductLink(id: number): Promise<void>;
  deleteSetProductLinksBySetId(setId: number): Promise<void>;
}

export class MemStorage implements IStorage {
  private connection: AllegroConnection | undefined;
  private orders: Map<string, AllegroOrder>;
  private syncSettings: SyncSettings | undefined;

  constructor() {
    this.orders = new Map();
  }

  async getAllegroConnection(): Promise<AllegroConnection | undefined> {
    return this.connection;
  }

  async createOrUpdateConnection(
    insertConnection: InsertAllegroConnection
  ): Promise<AllegroConnection> {
    const id = this.connection?.id || randomUUID();
    const connection: AllegroConnection = {
      id,
      clientId: insertConnection.clientId,
      clientSecret: insertConnection.clientSecret,
      accessToken: insertConnection.accessToken ?? null,
      refreshToken: insertConnection.refreshToken ?? null,
      tokenExpiresAt: insertConnection.tokenExpiresAt ?? null,
      isActive: insertConnection.isActive ?? null,
      connectionError: null,
      lastErrorAt: null,
      requiresReauth: null,
      createdAt: this.connection?.createdAt || new Date(),
      updatedAt: new Date(),
    };
    this.connection = connection;
    return connection;
  }

  async updateConnectionTokens(
    id: string,
    accessToken: string,
    refreshToken: string,
    expiresAt: Date
  ): Promise<void> {
    if (this.connection && this.connection.id === id) {
      this.connection = {
        ...this.connection,
        accessToken,
        refreshToken,
        tokenExpiresAt: expiresAt,
        isActive: true,
        updatedAt: new Date(),
      };
    }
  }

  async getOrders(): Promise<AllegroOrder[]> {
    return Array.from(this.orders.values()).sort(
      (a, b) =>
        new Date(b.orderDate).getTime() - new Date(a.orderDate).getTime()
    );
  }

  async getOrderByAllegroId(
    allegroOrderId: string
  ): Promise<AllegroOrder | undefined> {
    return Array.from(this.orders.values()).find(
      (order) => order.allegroOrderId === allegroOrderId
    );
  }

  async createOrUpdateOrder(
    insertOrder: InsertAllegroOrder
  ): Promise<AllegroOrder> {
    const existing = await this.getOrderByAllegroId(
      insertOrder.allegroOrderId
    );
    const id = existing?.id || insertOrder.id || randomUUID();
    
    const order: AllegroOrder = {
      id,
      allegroOrderId: insertOrder.allegroOrderId,
      buyerLogin: insertOrder.buyerLogin,
      buyerEmail: insertOrder.buyerEmail ?? null,
      totalAmount: insertOrder.totalAmount,
      currency: insertOrder.currency ?? null,
      paymentStatus: insertOrder.paymentStatus,
      fulfillmentStatus: insertOrder.fulfillmentStatus ?? null,
      itemsCount: insertOrder.itemsCount,
      orderDate: insertOrder.orderDate,
      paymentDate: insertOrder.paymentDate ?? null,
      rawData: insertOrder.rawData ?? null,
      createdAt: existing?.createdAt || new Date(),
      updatedAt: new Date(),
    };
    
    this.orders.set(id, order);
    return order;
  }

  async getSyncSettings(): Promise<SyncSettings | undefined> {
    return this.syncSettings;
  }

  async createOrUpdateSyncSettings(
    insertSettings: InsertSyncSettings
  ): Promise<SyncSettings> {
    const id = this.syncSettings?.id || randomUUID();
    const settings: SyncSettings = {
      id,
      autoRefreshEnabled: insertSettings.autoRefreshEnabled ?? null,
      refreshIntervalMinutes: insertSettings.refreshIntervalMinutes ?? null,
      lastSyncAt: insertSettings.lastSyncAt ?? null,
      fullSyncEnabled: insertSettings.fullSyncEnabled ?? null,
      fullSyncWindowHours: insertSettings.fullSyncWindowHours ?? null,
      shoperAutoRefreshEnabled: insertSettings.shoperAutoRefreshEnabled ?? null,
      shoperRefreshIntervalMinutes: insertSettings.shoperRefreshIntervalMinutes ?? null,
      lastShoperSyncAt: insertSettings.lastShoperSyncAt ?? null,
      shoperFullSyncEnabled: insertSettings.shoperFullSyncEnabled ?? null,
      shoperFullSyncWindowHours: insertSettings.shoperFullSyncWindowHours ?? null,
      nightlySyncEnabled: insertSettings.nightlySyncEnabled ?? null,
      nightlySyncHour: insertSettings.nightlySyncHour ?? null,
      nightlySyncWindowDays: insertSettings.nightlySyncWindowDays ?? null,
      lastNightlySyncAt: insertSettings.lastNightlySyncAt ?? null,
      createdAt: this.syncSettings?.createdAt || new Date(),
      updatedAt: new Date(),
    };
    this.syncSettings = settings;
    return settings;
  }

  // Product Creator Dictionaries
  async getDictionaries(type?: DictionaryType): Promise<ProductCreatorDictionary[]> {
    let query = 'SELECT * FROM product_creator.dictionaries';
    const params: any[] = [];
    
    if (type) {
      query += ' WHERE dictionary_type = $1';
      params.push(type);
    }
    
    query += ' ORDER BY sort_order ASC, id ASC';
    
    const result = await pool.query(query, params);
    return result.rows.map(row => ({
      id: row.id,
      dictionaryType: row.dictionary_type,
      code: row.code,
      name: row.name,
      readableName: row.readable_name,
      shortName: row.short_name,
      description: row.description,
      color: row.color,
      sortOrder: row.sort_order,
      category: row.category,
      imageUrl: row.image_url,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    }));
  }

  async getDictionaryById(id: number): Promise<ProductCreatorDictionary | undefined> {
    const result = await pool.query(
      'SELECT * FROM product_creator.dictionaries WHERE id = $1',
      [id]
    );
    
    if (result.rows.length === 0) {
      return undefined;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      dictionaryType: row.dictionary_type,
      code: row.code,
      name: row.name,
      readableName: row.readable_name,
      shortName: row.short_name,
      description: row.description,
      color: row.color,
      sortOrder: row.sort_order,
      category: row.category,
      imageUrl: row.image_url,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async createDictionary(insertDict: InsertProductCreatorDictionary): Promise<ProductCreatorDictionary> {
    const result = await pool.query(
      `INSERT INTO product_creator.dictionaries 
        (dictionary_type, code, name, readable_name, short_name, description, color, sort_order, category, image_url, is_active)
       VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
       RETURNING *`,
      [
        insertDict.dictionaryType,
        insertDict.code,
        insertDict.name,
        insertDict.readableName ?? null,
        insertDict.shortName ?? insertDict.code, // Default to code if no shortName provided
        insertDict.description ?? null,
        insertDict.color ?? null,
        insertDict.sortOrder ?? 0,
        insertDict.category ?? null,
        insertDict.imageUrl ?? null,
        insertDict.isActive ?? true,
      ]
    );
    
    const row = result.rows[0];
    return {
      id: row.id,
      dictionaryType: row.dictionary_type,
      code: row.code,
      name: row.name,
      readableName: row.readable_name,
      shortName: row.short_name,
      description: row.description,
      color: row.color,
      sortOrder: row.sort_order,
      category: row.category,
      imageUrl: row.image_url,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async updateDictionary(id: number, updates: Partial<InsertProductCreatorDictionary>): Promise<ProductCreatorDictionary> {
    const setParts: string[] = [];
    const values: any[] = [];
    let paramIndex = 1;
    
    if (updates.dictionaryType !== undefined) {
      setParts.push(`dictionary_type = $${paramIndex++}`);
      values.push(updates.dictionaryType);
    }
    if (updates.code !== undefined) {
      setParts.push(`code = $${paramIndex++}`);
      values.push(updates.code);
    }
    if (updates.name !== undefined) {
      setParts.push(`name = $${paramIndex++}`);
      values.push(updates.name);
    }
    if (updates.readableName !== undefined) {
      setParts.push(`readable_name = $${paramIndex++}`);
      values.push(updates.readableName);
    }
    if (updates.shortName !== undefined) {
      setParts.push(`short_name = $${paramIndex++}`);
      values.push(updates.shortName);
    }
    if (updates.description !== undefined) {
      setParts.push(`description = $${paramIndex++}`);
      values.push(updates.description);
    }
    if (updates.color !== undefined) {
      setParts.push(`color = $${paramIndex++}`);
      values.push(updates.color);
    }
    if (updates.sortOrder !== undefined) {
      setParts.push(`sort_order = $${paramIndex++}`);
      values.push(updates.sortOrder);
    }
    if (updates.category !== undefined) {
      setParts.push(`category = $${paramIndex++}`);
      values.push(updates.category);
    }
    if (updates.imageUrl !== undefined) {
      setParts.push(`image_url = $${paramIndex++}`);
      values.push(updates.imageUrl);
    }
    if (updates.isActive !== undefined) {
      setParts.push(`is_active = $${paramIndex++}`);
      values.push(updates.isActive);
    }
    
    setParts.push(`updated_at = NOW()`);
    values.push(id);
    
    const result = await pool.query(
      `UPDATE product_creator.dictionaries 
       SET ${setParts.join(', ')}
       WHERE id = $${paramIndex}
       RETURNING *`,
      values
    );
    
    if (result.rows.length === 0) {
      throw new Error(`Dictionary with id ${id} not found`);
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      dictionaryType: row.dictionary_type,
      code: row.code,
      name: row.name,
      readableName: row.readable_name,
      shortName: row.short_name,
      description: row.description,
      color: row.color,
      sortOrder: row.sort_order,
      category: row.category,
      imageUrl: row.image_url,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async deleteDictionary(id: number): Promise<void> {
    await pool.query(
      'DELETE FROM product_creator.dictionaries WHERE id = $1',
      [id]
    );
  }

  // Product Matrices
  async getProductMatrices(options?: {
    sort?: string;
    order?: 'asc' | 'desc';
    search?: string;
    productType?: string;
    limit?: number;
    offset?: number;
  }): Promise<ProductMatrix[]> {
    const { 
      sort = 'updated_at', 
      order = 'desc', 
      search = '', 
      productType = '', 
      limit = 25, 
      offset = 0 
    } = options || {};

    // Build WHERE clause
    const whereClauses: string[] = [];
    const params: any[] = [];
    let paramIndex = 1;

    if (search) {
      whereClauses.push(`name ILIKE $${paramIndex}`);
      params.push(`%${search}%`);
      paramIndex++;
    }

    if (productType) {
      whereClauses.push(`product_type = $${paramIndex}`);
      params.push(productType);
      paramIndex++;
    }

    const whereClause = whereClauses.length > 0 ? `WHERE ${whereClauses.join(' AND ')}` : '';

    // Map sort field to database column
    const sortColumnMap: { [key: string]: string } = {
      'name': 'name',
      'productType': 'product_type',
      'updatedAt': 'updated_at',
      'createdAt': 'created_at'
    };
    const sortColumn = sortColumnMap[sort] || 'updated_at';

    // Build query
    const query = `
      SELECT * FROM product_creator.product_matrices
      ${whereClause}
      ORDER BY ${sortColumn} ${order.toUpperCase()}
      LIMIT $${paramIndex} OFFSET $${paramIndex + 1}
    `;
    params.push(limit, offset);

    const result = await pool.query(query, params);
    
    return result.rows.map(row => ({
      id: row.id,
      name: row.name,
      namePrefix: row.name_prefix,
      nameSuffix: row.name_suffix,
      productType: row.product_type,
      productGroups: row.product_groups || [],
      priceModifiers: row.price_modifiers || [],
      priceModifierTypes: row.price_modifier_types || [],
      doors: row.doors,
      legs: row.legs,
      material: row.material,
      lengths: row.lengths || [],
      widths: row.widths || [],
      heights: row.heights || [],
      colors: row.colors || [],
      colorImages: row.color_images || {},
      selectedColors: row.selected_colors || [],
      colorLastGenerated: row.color_last_generated || {},
      colorOptions: row.color_options || {},
      hasWarranty: row.has_warranty ?? true,
      descriptionDoc: row.description_doc,
      descriptionHtml: row.description_html,
      aiPrompt: row.ai_prompt,
      useAiGeneration: row.use_ai_generation,
      defaultPrice: row.default_price,
      templateId: row.template_id,
      aiConfig: row.ai_config,
      createdBy: row.created_by,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    }));
  }

  async getProductMatricesCount(options?: {
    search?: string;
    productType?: string;
  }): Promise<number> {
    const { search = '', productType = '' } = options || {};

    // Build WHERE clause
    const whereClauses: string[] = [];
    const params: any[] = [];
    let paramIndex = 1;

    if (search) {
      whereClauses.push(`name ILIKE $${paramIndex}`);
      params.push(`%${search}%`);
      paramIndex++;
    }

    if (productType) {
      whereClauses.push(`product_type = $${paramIndex}`);
      params.push(productType);
      paramIndex++;
    }

    const whereClause = whereClauses.length > 0 ? `WHERE ${whereClauses.join(' AND ')}` : '';

    const query = `
      SELECT COUNT(*) as count
      FROM product_creator.product_matrices
      ${whereClause}
    `;

    const result = await pool.query(query, params);
    return parseInt(result.rows[0].count, 10);
  }

  async getProductMatrixById(id: number): Promise<ProductMatrix | undefined> {
    const result = await pool.query(
      'SELECT * FROM product_creator.product_matrices WHERE id = $1',
      [id]
    );
    
    if (result.rows.length === 0) {
      return undefined;
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      namePrefix: row.name_prefix,
      nameSuffix: row.name_suffix,
      productType: row.product_type,
      productGroups: row.product_groups || [],
      priceModifiers: row.price_modifiers || [],
      priceModifierTypes: row.price_modifier_types || [],
      doors: row.doors,
      legs: row.legs,
      material: row.material,
      lengths: row.lengths || [],
      widths: row.widths || [],
      heights: row.heights || [],
      colors: row.colors || [],
      colorImages: row.color_images || {},
      selectedColors: row.selected_colors || [],
      colorLastGenerated: row.color_last_generated || {},
      colorOptions: row.color_options || {},
      hasWarranty: row.has_warranty ?? true,
      descriptionDoc: row.description_doc,
      descriptionHtml: row.description_html,
      aiPrompt: row.ai_prompt,
      useAiGeneration: row.use_ai_generation,
      defaultPrice: row.default_price,
      templateId: row.template_id,
      aiConfig: row.ai_config || {},
      createdBy: row.created_by,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async createProductMatrix(matrix: InsertProductMatrix): Promise<ProductMatrix> {
    // Helper function to sanitize numeric arrays - convert empty strings and "null" strings to null and filter them out
    const sanitizeNumericArray = (arr: any[] | undefined): any[] => {
      if (!arr || !Array.isArray(arr)) return [];
      return arr.map(v => (v === "" || v === "null") ? null : v).filter(v => v !== null);
    };
    
    const result = await pool.query(
      `INSERT INTO product_creator.product_matrices 
        (name, name_prefix, name_suffix, product_type, doors, legs, material,
         lengths, widths, heights, product_groups, price_modifiers, price_modifier_types, colors, color_images,
         selected_colors, color_last_generated, color_options,
         description_doc, description_html, ai_prompt, use_ai_generation,
         default_price, template_id, ai_config, created_by)
       VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $20, $21, $22, $23, $24, $25, $26)
       RETURNING *`,
      [
        matrix.name,
        matrix.namePrefix ?? null,
        matrix.nameSuffix ?? null,
        matrix.productType ?? null,
        matrix.doors ?? null,
        matrix.legs ?? null,
        matrix.material ?? null,
        sanitizeNumericArray(matrix.lengths),
        sanitizeNumericArray(matrix.widths),
        sanitizeNumericArray(matrix.heights),
        matrix.productGroups ?? [],
        sanitizeNumericArray(matrix.priceModifiers),
        matrix.priceModifierTypes ?? [],
        matrix.colors ?? [],
        (matrix as any).colorImages ? JSON.stringify((matrix as any).colorImages) : '{}',
        (matrix as any).selectedColors ?? [],
        (matrix as any).colorLastGenerated ? JSON.stringify((matrix as any).colorLastGenerated) : '{}',
        (matrix as any).colorOptions ? JSON.stringify((matrix as any).colorOptions) : '{}',
        matrix.descriptionDoc ?? null,
        matrix.descriptionHtml ?? null,
        matrix.aiPrompt ?? null,
        matrix.useAiGeneration ?? false,
        matrix.defaultPrice === "" ? null : (matrix.defaultPrice ?? null),
        (matrix as any).templateId ?? null,
        (matrix as any).aiConfig ? JSON.stringify((matrix as any).aiConfig) : null,
        matrix.createdBy ?? null,
      ]
    );
    
    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      namePrefix: row.name_prefix,
      nameSuffix: row.name_suffix,
      productType: row.product_type,
      productGroups: row.product_groups || [],
      priceModifiers: row.price_modifiers || [],
      priceModifierTypes: row.price_modifier_types || [],
      doors: row.doors,
      legs: row.legs,
      material: row.material,
      lengths: row.lengths || [],
      widths: row.widths || [],
      heights: row.heights || [],
      colors: row.colors || [],
      colorImages: row.color_images || {},
      selectedColors: row.selected_colors || [],
      colorLastGenerated: row.color_last_generated || {},
      colorOptions: row.color_options || {},
      hasWarranty: row.has_warranty ?? true,
      descriptionDoc: row.description_doc,
      descriptionHtml: row.description_html,
      aiPrompt: row.ai_prompt,
      useAiGeneration: row.use_ai_generation,
      defaultPrice: row.default_price,
      templateId: row.template_id,
      aiConfig: row.ai_config,
      createdBy: row.created_by,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async updateProductMatrix(id: number, updates: Partial<InsertProductMatrix>): Promise<ProductMatrix> {
    const setParts: string[] = [];
    const values: any[] = [];
    let paramIndex = 1;
    
    // Helper function to sanitize numeric arrays - convert empty strings and "null" strings to null and filter them out
    const sanitizeNumericArray = (arr: any[]): any[] => {
      if (!Array.isArray(arr)) return arr;
      return arr.map(v => (v === "" || v === "null") ? null : v).filter(v => v !== null);
    };
    
    if (updates.name !== undefined) {
      setParts.push(`name = $${paramIndex++}`);
      values.push(updates.name);
    }
    if (updates.namePrefix !== undefined) {
      setParts.push(`name_prefix = $${paramIndex++}`);
      values.push(updates.namePrefix);
    }
    if (updates.nameSuffix !== undefined) {
      setParts.push(`name_suffix = $${paramIndex++}`);
      values.push(updates.nameSuffix);
    }
    if (updates.productType !== undefined) {
      setParts.push(`product_type = $${paramIndex++}`);
      values.push(updates.productType);
    }
    if (updates.productGroups !== undefined) {
      setParts.push(`product_groups = $${paramIndex++}`);
      values.push(updates.productGroups);
    }
    if (updates.doors !== undefined) {
      setParts.push(`doors = $${paramIndex++}`);
      values.push(updates.doors);
    }
    if (updates.legs !== undefined) {
      setParts.push(`legs = $${paramIndex++}`);
      values.push(updates.legs);
    }
    if (updates.material !== undefined) {
      console.log(`🔧 Material update: value="${updates.material}", type=${typeof updates.material}`);
      setParts.push(`material = $${paramIndex++}`);
      values.push(updates.material);
    }
    if (updates.lengths !== undefined) {
      setParts.push(`lengths = $${paramIndex++}`);
      values.push(sanitizeNumericArray(updates.lengths));
    }
    if (updates.widths !== undefined) {
      setParts.push(`widths = $${paramIndex++}`);
      values.push(sanitizeNumericArray(updates.widths));
    }
    if (updates.heights !== undefined) {
      setParts.push(`heights = $${paramIndex++}`);
      values.push(sanitizeNumericArray(updates.heights));
    }
    if (updates.colors !== undefined) {
      setParts.push(`colors = $${paramIndex++}`);
      values.push(updates.colors);
    }
    if ((updates as any).colorImages !== undefined) {
      setParts.push(`color_images = $${paramIndex++}`);
      values.push(JSON.stringify((updates as any).colorImages));
    }
    if (updates.priceModifiers !== undefined) {
      setParts.push(`price_modifiers = $${paramIndex++}`);
      values.push(sanitizeNumericArray(updates.priceModifiers));
    }
    if (updates.priceModifierTypes !== undefined) {
      setParts.push(`price_modifier_types = $${paramIndex++}`);
      values.push(updates.priceModifierTypes);
    }
    if (updates.descriptionDoc !== undefined) {
      setParts.push(`description_doc = $${paramIndex++}`);
      values.push(updates.descriptionDoc);
    }
    if (updates.descriptionHtml !== undefined) {
      setParts.push(`description_html = $${paramIndex++}`);
      values.push(updates.descriptionHtml);
    }
    if (updates.aiPrompt !== undefined) {
      setParts.push(`ai_prompt = $${paramIndex++}`);
      values.push(updates.aiPrompt);
    }
    if (updates.useAiGeneration !== undefined) {
      setParts.push(`use_ai_generation = $${paramIndex++}`);
      values.push(updates.useAiGeneration);
    }
    if (updates.defaultPrice !== undefined) {
      setParts.push(`default_price = $${paramIndex++}`);
      // Convert empty string to null for numeric field
      values.push(updates.defaultPrice === "" ? null : updates.defaultPrice);
    }
    if ((updates as any).templateId !== undefined) {
      setParts.push(`template_id = $${paramIndex++}`);
      values.push((updates as any).templateId);
    }
    if ((updates as any).aiConfig !== undefined) {
      setParts.push(`ai_config = $${paramIndex++}`);
      values.push((updates as any).aiConfig ? JSON.stringify((updates as any).aiConfig) : null);
    }
    if ((updates as any).selectedColors !== undefined) {
      setParts.push(`selected_colors = $${paramIndex++}`);
      values.push((updates as any).selectedColors);
    }
    if ((updates as any).colorLastGenerated !== undefined) {
      setParts.push(`color_last_generated = $${paramIndex++}`);
      values.push((updates as any).colorLastGenerated ? JSON.stringify((updates as any).colorLastGenerated) : null);
    }
    if ((updates as any).colorOptions !== undefined) {
      setParts.push(`color_options = $${paramIndex++}`);
      values.push((updates as any).colorOptions ? JSON.stringify((updates as any).colorOptions) : null);
    }
    
    setParts.push(`updated_at = NOW()`);
    values.push(id);
    
    console.log(`🔍 UPDATE query parts: ${setParts.join(', ')}`);
    console.log(`🔍 UPDATE values:`, values);
    
    const result = await pool.query(
      `UPDATE product_creator.product_matrices 
       SET ${setParts.join(', ')}
       WHERE id = $${paramIndex}
       RETURNING *`,
      values
    );
    
    if (result.rows.length === 0) {
      throw new Error(`Product matrix with id ${id} not found`);
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      namePrefix: row.name_prefix,
      nameSuffix: row.name_suffix,
      productType: row.product_type,
      productGroups: row.product_groups || [],
      priceModifiers: row.price_modifiers || [],
      priceModifierTypes: row.price_modifier_types || [],
      doors: row.doors,
      legs: row.legs,
      material: row.material,
      lengths: row.lengths || [],
      widths: row.widths || [],
      heights: row.heights || [],
      colors: row.colors || [],
      colorImages: row.color_images || {},
      selectedColors: row.selected_colors || [],
      colorLastGenerated: row.color_last_generated || {},
      colorOptions: row.color_options || {},
      hasWarranty: row.has_warranty ?? true,
      descriptionDoc: row.description_doc,
      descriptionHtml: row.description_html,
      aiPrompt: row.ai_prompt,
      useAiGeneration: row.use_ai_generation,
      defaultPrice: row.default_price,
      templateId: row.template_id,
      aiConfig: row.ai_config,
      createdBy: row.created_by,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async deleteProductMatrix(id: number): Promise<void> {
    await pool.query(
      'DELETE FROM product_creator.product_matrices WHERE id = $1',
      [id]
    );
  }

  async getSetMatrices(): Promise<SetMatrix[]> {
    const result = await pool.query(
      'SELECT * FROM product_creator.set_matrices ORDER BY id DESC'
    );
    return result.rows.map(row => ({
      id: row.id,
      name: row.name,
      productGroup: row.product_group,
      length: row.length,
      width: row.width,
      height: row.height,
      doors: row.doors,
      legs: row.legs,
      imageUrl: row.image_url,
      namePrefix: row.name_prefix,
      nameSuffix: row.name_suffix,
      defaultDepth: row.default_depth,
      hookLength: row.hook_length,
      components: row.components,
      basePriceModifier: row.base_price_modifier,
      priceModifierType: row.price_modifier_type,
      colors: row.colors,
      colorImages: row.color_images,
      colorOptions: row.color_options,
      selectedColors: row.selected_colors,
      useAiGeneration: row.use_ai_generation,
      aiPrompt: row.ai_prompt,
      descriptionTemplateId: row.description_template_id,
      useEnhancedDescription: row.use_enhanced_description,
      isActive: row.is_active,
      lastGenerationStatus: row.last_generation_status,
      lastGenerationAt: row.last_generation_at,
      lastMissingComponents: row.last_missing_components,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    }));
  }

  async getSetMatrixById(id: number): Promise<SetMatrix | undefined> {
    const result = await pool.query(
      'SELECT * FROM product_creator.set_matrices WHERE id = $1',
      [id]
    );
    if (result.rows.length === 0) return undefined;
    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      productGroup: row.product_group,
      length: row.length,
      width: row.width,
      height: row.height,
      doors: row.doors,
      legs: row.legs,
      imageUrl: row.image_url,
      namePrefix: row.name_prefix,
      nameSuffix: row.name_suffix,
      defaultDepth: row.default_depth,
      hookLength: row.hook_length,
      components: row.components,
      basePriceModifier: row.base_price_modifier,
      priceModifierType: row.price_modifier_type,
      colors: row.colors,
      colorImages: row.color_images,
      colorOptions: row.color_options,
      selectedColors: row.selected_colors,
      useAiGeneration: row.use_ai_generation,
      aiPrompt: row.ai_prompt,
      descriptionTemplateId: row.description_template_id,
      useEnhancedDescription: row.use_enhanced_description,
      isActive: row.is_active,
      lastGenerationStatus: row.last_generation_status,
      lastGenerationAt: row.last_generation_at,
      lastMissingComponents: row.last_missing_components,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async createSetMatrix(matrix: InsertSetMatrix): Promise<SetMatrix> {
    const result = await pool.query(
      `INSERT INTO product_creator.set_matrices 
        (name, product_group, length, width, height, doors, legs, image_url, name_prefix, name_suffix, 
         default_depth, hook_length, 
         components, base_price_modifier, price_modifier_type, colors, color_images, color_options, selected_colors,
         use_ai_generation, ai_prompt, description_template_id, use_enhanced_description, is_active)
       VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $20, $21, $22, $23, $24)
       RETURNING *`,
      [
        matrix.name,
        matrix.productGroup ?? null,
        (matrix as any).length ?? null,
        (matrix as any).width ?? null,
        (matrix as any).height ?? null,
        (matrix as any).doors ?? null,
        (matrix as any).legs ?? null,
        matrix.imageUrl ?? null,
        matrix.namePrefix ?? null,
        matrix.nameSuffix ?? null,
        matrix.defaultDepth ?? null,
        matrix.hookLength ?? null,
        typeof matrix.components === 'string' ? matrix.components : JSON.stringify(matrix.components ?? []),
        matrix.basePriceModifier ?? '0',
        matrix.priceModifierType ?? 'percent',
        typeof matrix.colors === 'string' ? matrix.colors : JSON.stringify(matrix.colors ?? []),
        typeof matrix.colorImages === 'string' ? matrix.colorImages : JSON.stringify(matrix.colorImages ?? {}),
        typeof matrix.colorOptions === 'string' ? matrix.colorOptions : JSON.stringify(matrix.colorOptions ?? {}),
        typeof matrix.selectedColors === 'string' ? matrix.selectedColors : JSON.stringify(matrix.selectedColors ?? []),
        matrix.useAiGeneration ?? false,
        matrix.aiPrompt ?? null,
        matrix.descriptionTemplateId ?? null,
        (matrix as any).useEnhancedDescription ?? false,
        matrix.isActive ?? true,
      ]
    );
    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      productGroup: row.product_group,
      length: row.length,
      width: row.width,
      height: row.height,
      doors: row.doors,
      legs: row.legs,
      imageUrl: row.image_url,
      namePrefix: row.name_prefix,
      nameSuffix: row.name_suffix,
      defaultDepth: row.default_depth,
      hookLength: row.hook_length,
      components: row.components,
      basePriceModifier: row.base_price_modifier,
      priceModifierType: row.price_modifier_type,
      colors: row.colors,
      colorImages: row.color_images,
      colorOptions: row.color_options,
      selectedColors: row.selected_colors,
      useAiGeneration: row.use_ai_generation,
      aiPrompt: row.ai_prompt,
      descriptionTemplateId: row.description_template_id,
      useEnhancedDescription: row.use_enhanced_description,
      isActive: row.is_active,
      lastGenerationStatus: row.last_generation_status,
      lastGenerationAt: row.last_generation_at,
      lastMissingComponents: row.last_missing_components,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async updateSetMatrix(id: number, updates: Partial<InsertSetMatrix>): Promise<SetMatrix> {
    const setParts: string[] = [];
    const values: any[] = [];
    let paramIndex = 1;

    if (updates.name !== undefined) {
      setParts.push(`name = $${paramIndex++}`);
      values.push(updates.name);
    }
    if (updates.productGroup !== undefined) {
      setParts.push(`product_group = $${paramIndex++}`);
      values.push(updates.productGroup);
    }
    if ((updates as any).length !== undefined) {
      setParts.push(`length = $${paramIndex++}`);
      values.push((updates as any).length);
    }
    if ((updates as any).width !== undefined) {
      setParts.push(`width = $${paramIndex++}`);
      values.push((updates as any).width);
    }
    if ((updates as any).height !== undefined) {
      setParts.push(`height = $${paramIndex++}`);
      values.push((updates as any).height);
    }
    if ((updates as any).doors !== undefined) {
      setParts.push(`doors = $${paramIndex++}`);
      values.push((updates as any).doors);
    }
    if ((updates as any).legs !== undefined) {
      setParts.push(`legs = $${paramIndex++}`);
      values.push((updates as any).legs);
    }
    if (updates.imageUrl !== undefined) {
      setParts.push(`image_url = $${paramIndex++}`);
      values.push(updates.imageUrl);
    }
    if (updates.namePrefix !== undefined) {
      setParts.push(`name_prefix = $${paramIndex++}`);
      values.push(updates.namePrefix);
    }
    if (updates.nameSuffix !== undefined) {
      setParts.push(`name_suffix = $${paramIndex++}`);
      values.push(updates.nameSuffix);
    }
    if (updates.defaultDepth !== undefined) {
      setParts.push(`default_depth = $${paramIndex++}`);
      values.push(updates.defaultDepth);
    }
    if (updates.hookLength !== undefined) {
      setParts.push(`hook_length = $${paramIndex++}`);
      values.push(updates.hookLength);
    }
    if (updates.components !== undefined) {
      setParts.push(`components = $${paramIndex++}`);
      values.push(typeof updates.components === 'string' ? updates.components : JSON.stringify(updates.components));
    }
    if (updates.basePriceModifier !== undefined) {
      setParts.push(`base_price_modifier = $${paramIndex++}`);
      values.push(updates.basePriceModifier);
    }
    if (updates.priceModifierType !== undefined) {
      setParts.push(`price_modifier_type = $${paramIndex++}`);
      values.push(updates.priceModifierType);
    }
    if (updates.colors !== undefined) {
      setParts.push(`colors = $${paramIndex++}`);
      values.push(updates.colors);
    }
    if (updates.colorImages !== undefined) {
      setParts.push(`color_images = $${paramIndex++}`);
      values.push(typeof updates.colorImages === 'string' ? updates.colorImages : JSON.stringify(updates.colorImages));
    }
    if (updates.colorOptions !== undefined) {
      setParts.push(`color_options = $${paramIndex++}`);
      values.push(typeof updates.colorOptions === 'string' ? updates.colorOptions : JSON.stringify(updates.colorOptions));
    }
    if (updates.selectedColors !== undefined) {
      setParts.push(`selected_colors = $${paramIndex++}`);
      values.push(updates.selectedColors);
    }
    if (updates.useAiGeneration !== undefined) {
      setParts.push(`use_ai_generation = $${paramIndex++}`);
      values.push(updates.useAiGeneration);
    }
    if (updates.aiPrompt !== undefined) {
      setParts.push(`ai_prompt = $${paramIndex++}`);
      values.push(updates.aiPrompt);
    }
    if (updates.descriptionTemplateId !== undefined) {
      setParts.push(`description_template_id = $${paramIndex++}`);
      values.push(updates.descriptionTemplateId);
    }
    if ((updates as any).useEnhancedDescription !== undefined) {
      setParts.push(`use_enhanced_description = $${paramIndex++}`);
      values.push((updates as any).useEnhancedDescription);
    }
    if (updates.isActive !== undefined) {
      setParts.push(`is_active = $${paramIndex++}`);
      values.push(updates.isActive);
    }

    setParts.push(`updated_at = NOW()`);
    values.push(id);

    const result = await pool.query(
      `UPDATE product_creator.set_matrices 
       SET ${setParts.join(', ')}
       WHERE id = $${paramIndex}
       RETURNING *`,
      values
    );

    if (result.rows.length === 0) {
      throw new Error(`Set matrix with id ${id} not found`);
    }

    const row = result.rows[0];
    return {
      id: row.id,
      name: row.name,
      productGroup: row.product_group,
      length: row.length,
      width: row.width,
      height: row.height,
      doors: row.doors,
      legs: row.legs,
      imageUrl: row.image_url,
      namePrefix: row.name_prefix,
      nameSuffix: row.name_suffix,
      defaultDepth: row.default_depth,
      hookLength: row.hook_length,
      components: row.components,
      basePriceModifier: row.base_price_modifier,
      priceModifierType: row.price_modifier_type,
      colors: row.colors,
      colorImages: row.color_images,
      colorOptions: row.color_options,
      selectedColors: row.selected_colors,
      useAiGeneration: row.use_ai_generation,
      aiPrompt: row.ai_prompt,
      descriptionTemplateId: row.description_template_id,
      useEnhancedDescription: row.use_enhanced_description,
      isActive: row.is_active,
      lastGenerationStatus: row.last_generation_status,
      lastGenerationAt: row.last_generation_at,
      lastMissingComponents: row.last_missing_components,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async deleteSetMatrix(id: number): Promise<void> {
    await pool.query(
      'DELETE FROM product_creator.set_matrices WHERE id = $1',
      [id]
    );
  }

  async getProductSets(): Promise<ProductSet[]> {
    const result = await pool.query(
      'SELECT * FROM product_creator.product_sets ORDER BY id DESC'
    );
    return result.rows.map(row => ({
      id: row.id,
      setMatrixId: row.set_matrix_id,
      sku: row.sku,
      title: row.title,
      shortDescription: row.short_description,
      fullDescription: row.full_description,
      color: row.color,
      depth: row.depth,
      panelCount: row.panel_count,
      hookLength: row.hook_length,
      basePrice: row.base_price,
      calculatedPrice: row.calculated_price,
      images: row.images,
      generatedFromMatrix: row.generated_from_matrix,
      combinationKey: row.combination_key,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    }));
  }

  async getProductSetById(id: number): Promise<ProductSet | undefined> {
    const result = await pool.query(
      'SELECT * FROM product_creator.product_sets WHERE id = $1',
      [id]
    );
    if (result.rows.length === 0) return undefined;
    const row = result.rows[0];
    return {
      id: row.id,
      setMatrixId: row.set_matrix_id,
      sku: row.sku,
      title: row.title,
      shortDescription: row.short_description,
      fullDescription: row.full_description,
      color: row.color,
      depth: row.depth,
      panelCount: row.panel_count,
      hookLength: row.hook_length,
      basePrice: row.base_price,
      calculatedPrice: row.calculated_price,
      images: row.images,
      generatedFromMatrix: row.generated_from_matrix,
      combinationKey: row.combination_key,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async getProductSetBySku(sku: string): Promise<ProductSet | undefined> {
    const result = await pool.query(
      'SELECT * FROM product_creator.product_sets WHERE sku = $1',
      [sku]
    );
    if (result.rows.length === 0) return undefined;
    const row = result.rows[0];
    return {
      id: row.id,
      setMatrixId: row.set_matrix_id,
      sku: row.sku,
      title: row.title,
      shortDescription: row.short_description,
      fullDescription: row.full_description,
      color: row.color,
      depth: row.depth,
      panelCount: row.panel_count,
      hookLength: row.hook_length,
      basePrice: row.base_price,
      calculatedPrice: row.calculated_price,
      images: row.images,
      generatedFromMatrix: row.generated_from_matrix,
      combinationKey: row.combination_key,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async createProductSet(set: InsertProductSet): Promise<ProductSet> {
    const result = await pool.query(
      `INSERT INTO product_creator.product_sets 
        (set_matrix_id, sku, title, short_description, full_description, color,
         depth, panel_count, hook_length, base_price, calculated_price, images,
         generated_from_matrix, combination_key, is_active)
       VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15)
       RETURNING *`,
      [
        set.setMatrixId ?? null,
        set.sku,
        set.title,
        set.shortDescription ?? null,
        set.fullDescription ?? null,
        set.color ?? null,
        set.depth ?? null,
        set.panelCount ?? null,
        set.hookLength ?? null,
        set.basePrice ?? null,
        set.calculatedPrice ?? null,
        set.images ?? [],
        set.generatedFromMatrix ?? false,
        set.combinationKey ?? null,
        set.isActive ?? true,
      ]
    );
    const row = result.rows[0];
    return {
      id: row.id,
      setMatrixId: row.set_matrix_id,
      sku: row.sku,
      title: row.title,
      shortDescription: row.short_description,
      fullDescription: row.full_description,
      color: row.color,
      depth: row.depth,
      panelCount: row.panel_count,
      hookLength: row.hook_length,
      basePrice: row.base_price,
      calculatedPrice: row.calculated_price,
      images: row.images,
      generatedFromMatrix: row.generated_from_matrix,
      combinationKey: row.combination_key,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async updateProductSet(id: number, updates: Partial<InsertProductSet>): Promise<ProductSet> {
    const setParts: string[] = [];
    const values: any[] = [];
    let paramIndex = 1;

    if (updates.sku !== undefined) {
      setParts.push(`sku = $${paramIndex++}`);
      values.push(updates.sku);
    }
    if (updates.title !== undefined) {
      setParts.push(`title = $${paramIndex++}`);
      values.push(updates.title);
    }
    if (updates.shortDescription !== undefined) {
      setParts.push(`short_description = $${paramIndex++}`);
      values.push(updates.shortDescription);
    }
    if (updates.fullDescription !== undefined) {
      setParts.push(`full_description = $${paramIndex++}`);
      values.push(updates.fullDescription);
    }
    if (updates.color !== undefined) {
      setParts.push(`color = $${paramIndex++}`);
      values.push(updates.color);
    }
    if (updates.depth !== undefined) {
      setParts.push(`depth = $${paramIndex++}`);
      values.push(updates.depth);
    }
    if (updates.panelCount !== undefined) {
      setParts.push(`panel_count = $${paramIndex++}`);
      values.push(updates.panelCount);
    }
    if (updates.hookLength !== undefined) {
      setParts.push(`hook_length = $${paramIndex++}`);
      values.push(updates.hookLength);
    }
    if (updates.basePrice !== undefined) {
      setParts.push(`base_price = $${paramIndex++}`);
      values.push(updates.basePrice);
    }
    if (updates.calculatedPrice !== undefined) {
      setParts.push(`calculated_price = $${paramIndex++}`);
      values.push(updates.calculatedPrice);
    }
    if (updates.images !== undefined) {
      setParts.push(`images = $${paramIndex++}`);
      values.push(updates.images);
    }
    if (updates.generatedFromMatrix !== undefined) {
      setParts.push(`generated_from_matrix = $${paramIndex++}`);
      values.push(updates.generatedFromMatrix);
    }
    if (updates.combinationKey !== undefined) {
      setParts.push(`combination_key = $${paramIndex++}`);
      values.push(updates.combinationKey);
    }
    if (updates.isActive !== undefined) {
      setParts.push(`is_active = $${paramIndex++}`);
      values.push(updates.isActive);
    }

    setParts.push(`updated_at = NOW()`);
    values.push(id);

    const result = await pool.query(
      `UPDATE product_creator.product_sets 
       SET ${setParts.join(', ')}
       WHERE id = $${paramIndex}
       RETURNING *`,
      values
    );

    if (result.rows.length === 0) {
      throw new Error(`Product set with id ${id} not found`);
    }

    const row = result.rows[0];
    return {
      id: row.id,
      setMatrixId: row.set_matrix_id,
      sku: row.sku,
      title: row.title,
      shortDescription: row.short_description,
      fullDescription: row.full_description,
      color: row.color,
      depth: row.depth,
      panelCount: row.panel_count,
      hookLength: row.hook_length,
      basePrice: row.base_price,
      calculatedPrice: row.calculated_price,
      images: row.images,
      generatedFromMatrix: row.generated_from_matrix,
      combinationKey: row.combination_key,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  }

  async deleteProductSet(id: number): Promise<void> {
    await pool.query(
      'DELETE FROM product_creator.product_sets WHERE id = $1',
      [id]
    );
  }

  async getSetProductLinks(setId: number): Promise<SetProductLink[]> {
    const result = await pool.query(
      'SELECT * FROM product_creator.set_product_links WHERE set_id = $1',
      [setId]
    );
    return result.rows.map(row => ({
      id: row.id,
      setId: row.set_id,
      productId: row.product_id,
      componentType: row.component_type,
      quantity: row.quantity,
      position: row.position,
      createdAt: row.created_at,
    }));
  }

  async getProductSetsForProduct(productId: number): Promise<ProductSet[]> {
    const result = await pool.query(
      `SELECT ps.* FROM product_creator.product_sets ps
       INNER JOIN product_creator.set_product_links spl ON ps.id = spl.set_id
       WHERE spl.product_id = $1
       ORDER BY ps.id DESC`,
      [productId]
    );
    return result.rows.map(row => ({
      id: row.id,
      setMatrixId: row.set_matrix_id,
      sku: row.sku,
      title: row.title,
      shortDescription: row.short_description,
      fullDescription: row.full_description,
      color: row.color,
      depth: row.depth,
      panelCount: row.panel_count,
      hookLength: row.hook_length,
      basePrice: row.base_price,
      calculatedPrice: row.calculated_price,
      images: row.images,
      generatedFromMatrix: row.generated_from_matrix,
      combinationKey: row.combination_key,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    }));
  }

  async createSetProductLink(link: InsertSetProductLink): Promise<SetProductLink> {
    const result = await pool.query(
      `INSERT INTO product_creator.set_product_links 
        (set_id, product_id, component_type, quantity, position)
       VALUES ($1, $2, $3, $4, $5)
       RETURNING *`,
      [
        link.setId,
        link.productId,
        link.componentType,
        link.quantity ?? 1,
        link.position ?? 0,
      ]
    );
    const row = result.rows[0];
    return {
      id: row.id,
      setId: row.set_id,
      productId: row.product_id,
      componentType: row.component_type,
      quantity: row.quantity,
      position: row.position,
      createdAt: row.created_at,
    };
  }

  async deleteSetProductLink(id: number): Promise<void> {
    await pool.query(
      'DELETE FROM product_creator.set_product_links WHERE id = $1',
      [id]
    );
  }

  async deleteSetProductLinksBySetId(setId: number): Promise<void> {
    await pool.query(
      'DELETE FROM product_creator.set_product_links WHERE set_id = $1',
      [setId]
    );
  }
}

export const storage = new MemStorage();
