import { pool } from './postgres';
import { createOdooClient } from './odoo-client';

interface OrderToSync {
  id: number;
  orderNumber: number;
  source: string;
  sourceOrderId: string;
  orderDate: string | null;  // TO_CHAR zwraca string, nie Date!
  status: string | null;
  buyerLogin: string | null;
  buyerEmail: string | null;
  buyerCompany: string | null;
  buyerFirstName: string | null;
  buyerLastName: string | null;
  buyerPhone: string | null;
  buyerAddress: string | null;
  buyerZip: string | null;
  buyerCity: string | null;
  buyerCountryCode: string | null;
  paymentId: string | null;
  paymentStatus: string | null;
  paymentProvider: string | null;
  paymentAmount: string | null;
  paymentCurrency: string | null;
  deliveryMethod: string | null;
  deliveryAmount: string | null;
  deliveryCurrency: string | null;
  totalToPayAmount: string | null;
  totalToPayCurrency: string | null;
  buyerNotes: string | null;
  invoiceRequired: boolean | null;
  trackingNumbers: string[] | null;
  marketplaceUrl?: string | null;
  items?: Array<{
    name: string;
    quantity: string;
    price: string;
    sku?: string;
  }>;
}

export async function addOrderToSyncQueue(
  orderNumber: number,
  source: string,
  operation: 'create' | 'update' = 'create'
): Promise<void> {
  const client = await pool.connect();

  try {
    // Sprawdź czy zamówienie NIE jest właśnie przetwarzane lub już zakończone niedawno (< 2 min)
    const checkResult = await client.query(`
      SELECT status, processed_at FROM odoo_sync_queue
      WHERE order_number = $1
      LIMIT 1
    `, [orderNumber]);

    if (checkResult.rows.length > 0) {
      const { status, processed_at } = checkResult.rows[0];
      
      // Jeśli zamówienie zostało zakończone w ciągu ostatnich 2 minut, nie dodawaj ponownie
      if (status === 'completed' && processed_at) {
        const completedAt = new Date(processed_at);
        const now = new Date();
        const minutesAgo = (now.getTime() - completedAt.getTime()) / 1000 / 60;
        if (minutesAgo < 2) {
          // console.log(`⏭️  Pominięto #${orderNumber} - niedawno zakończone (${minutesAgo.toFixed(1)} min temu)`);
          return;
        }
      }
    }

    await client.query(`
      INSERT INTO odoo_sync_queue (
        order_number, source, operation, status, attempts
      ) VALUES ($1, $2, $3, 'pending', 0)
      ON CONFLICT (order_number) DO UPDATE SET
        operation = EXCLUDED.operation,
        status = 'pending',
        attempts = 0,
        scheduled_at = NOW(),
        updated_at = NOW(),
        odoo_order_id = CASE 
          WHEN EXCLUDED.operation = 'update' THEN odoo_sync_queue.odoo_order_id
          ELSE NULL
        END
      WHERE odoo_sync_queue.status NOT IN ('completed', 'processing')
        OR odoo_sync_queue.processed_at < NOW() - INTERVAL '2 minutes'
    `, [orderNumber, source, operation]);

    console.log(`✅ Dodano #${orderNumber} do kolejki Odoo (${operation})`);
  } catch (error) {
    // Szczegółowe logowanie błędów
    console.error(`❌ BŁĄD: Nie można dodać #${orderNumber} do kolejki Odoo:`);
    console.error(`   Operacja: ${operation}, Source: ${source}, OrderNumber: ${orderNumber}`);
    
    if (error instanceof Error) {
      console.error(`   Komunikat: ${error.message}`);
      if ('code' in error) {
        console.error(`   Kod błędu PostgreSQL: ${(error as any).code}`);
      }
      if ('detail' in error) {
        console.error(`   Szczegóły: ${(error as any).detail}`);
      }
    } else {
      console.error(`   Nieznany błąd:`, error);
    }
    
    // Re-throw błąd aby wywołujący mógł go obsłużyć
    throw error;
  } finally {
    client.release();
  }
}

export async function addOrdersToSyncQueue(orderNumbers: number[]): Promise<void> {
  const client = await pool.connect();

  try {
    for (const orderNumber of orderNumbers) {
      const orderResult = await client.query(`
        SELECT order_number, source
        FROM commerce.orders
        WHERE order_number = $1
      `, [orderNumber]);

      if (orderResult.rows.length > 0) {
        const order = orderResult.rows[0];
        await addOrderToSyncQueue(order.order_number, order.source);
      }
    }
  } finally {
    client.release();
  }
}

export async function processSyncQueue(): Promise<{
  processed: number;
  succeeded: number;
  failed: number;
}> {
  console.log('🔧 processSyncQueue() started');
  const odooClient = await createOdooClient();
  console.log('🔧 createOdooClient() returned:', odooClient ? 'OdooClient instance' : 'null');
  
  if (!odooClient) {
    console.warn('⚠️  Odoo client not configured, skipping sync');
    return { processed: 0, succeeded: 0, failed: 0 };
  }

  const client = await pool.connect();
  let processed = 0;
  let succeeded = 0;
  let failed = 0;
  let lockAcquired = false;

  try {
    // Blokada PostgreSQL Advisory Lock (zapobiega równoległemu przetwarzaniu)
    const lockResult = await client.query(`SELECT pg_try_advisory_lock(123456789)`);
    lockAcquired = lockResult.rows[0]?.pg_try_advisory_lock;
    
    if (!lockAcquired) {
      console.log('🔒 Inna instancja processSyncQueue() już działa - pomijam');
      return { processed: 0, succeeded: 0, failed: 0 };
    }
    
    console.log('🔓 Zdobyto blokadę processSyncQueue()');
    
    try {
    const configResult = await client.query(`
      SELECT auto_confirm_orders, oms_domain FROM odoo_config WHERE id = 1
    `);
    const autoConfirmOrders = configResult.rows[0]?.auto_confirm_orders || false;
    const omsDomain = configResult.rows[0]?.oms_domain || 'alp-oms.replit.app';

    // Użyj SELECT FOR UPDATE SKIP LOCKED żeby zapobiec duplikatom
    const queueResult = await client.query(`
      SELECT * FROM odoo_sync_queue
      WHERE status = 'pending' AND attempts < max_attempts
      ORDER BY scheduled_at ASC
      LIMIT 50
      FOR UPDATE SKIP LOCKED
    `);

    for (const queueItem of queueResult.rows) {
      processed++;
      const startTime = Date.now();

      try {
        const orderResult = await client.query(`
          SELECT 
            o.id,
            o.order_number as "orderNumber",
            o.source,
            o.source_order_id as "sourceOrderId",
            CASE 
              WHEN o.source = 'ALLEGRO' THEN 
                TO_CHAR(o.order_date, 'YYYY-MM-DD HH24:MI:SS')::text
              WHEN o.source = 'SHOPER' THEN 
                TO_CHAR(o.order_date AT TIME ZONE 'Europe/Warsaw' AT TIME ZONE 'UTC', 'YYYY-MM-DD HH24:MI:SS')::text
              ELSE 
                TO_CHAR(o.order_date, 'YYYY-MM-DD HH24:MI:SS')::text
            END as "orderDate",
            o.status,
            o.payment_status as "paymentStatus",
            o.buyer_login as "buyerLogin",
            o.buyer_email as "buyerEmail",
            o.buyer_company as "buyerCompany",
            o.buyer_first_name as "buyerFirstName",
            o.buyer_last_name as "buyerLastName",
            o.buyer_phone as "buyerPhone",
            o.buyer_address as "buyerAddress",
            o.buyer_zip as "buyerZip",
            o.buyer_city as "buyerCity",
            o.buyer_country_code as "buyerCountryCode",
            o.delivery_method as "deliveryMethod",
            o.delivery_amount as "deliveryPrice",
            o.invoice_required as "invoiceRequired",
            o.total_to_pay_amount as "totalToPayAmount",
            COALESCE(
              json_agg(
                json_build_object(
                  'name', oi.name,
                  'quantity', oi.quantity::text,
                  'price', oi.unit_price::text,
                  'sku', COALESCE(
                    oi.raw_data->>'code',
                    oi.raw_data->'offer'->'external'->>'id',
                    oi.offer_external_id
                  ),
                  'image_url', oi.image_url
                )
                ORDER BY oi.id
              ) FILTER (WHERE oi.id IS NOT NULL),
              '[]'
            ) as items
          FROM commerce.orders o
          LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
          WHERE o.order_number = $1
          GROUP BY o.id, o.order_number, o.source, o.source_order_id, 
                   o.order_date, o.status, o.payment_status,
                   o.buyer_login, o.buyer_email, o.buyer_company,
                   o.buyer_first_name, o.buyer_last_name, o.buyer_phone, o.buyer_address,
                   o.buyer_zip, o.buyer_city, o.buyer_country_code, o.delivery_method,
                   o.delivery_amount, o.invoice_required, o.total_to_pay_amount
        `, [queueItem.order_number]);

        if (orderResult.rows.length === 0) {
          throw new Error('Order not found');
        }

        const order = orderResult.rows[0] as OrderToSync;
        
        // DEBUG: Sprawdź typ orderDate
        console.log(`🔍 DEBUG orderDate for #${queueItem.order_number}:`, {
          value: order.orderDate,
          type: typeof order.orderDate,
          isDate: false, // TO_CHAR zwraca string, nie Date
        });

        // Buduj URL marketplace (dla CREATE i UPDATE)
        let marketplaceUrl: string | undefined;
        if (order.source === 'ALLEGRO') {
          marketplaceUrl = `https://allegro.pl/moje-allegro/sprzedaz/zamowienia/${order.sourceOrderId}`;
        } else if (order.source === 'SHOPER') {
          // Pobierz konfigurację Shoper
          const shoperConfig = await client.query(`SELECT orders_url FROM shoper_config LIMIT 1`);
          if (shoperConfig.rows[0]?.orders_url) {
            marketplaceUrl = `${shoperConfig.rows[0].orders_url}${order.sourceOrderId}`;
          }
        }
        console.log(`🏪 [ODOO MARKETPLACE URL] #${order.orderNumber}: ${marketplaceUrl || 'brak'}`);
        
        // Dodaj marketplaceUrl do obiektu order
        order.marketplaceUrl = marketplaceUrl;

        let odooOrderId: number;
        let updateData: any = null; // Do logowania UPDATE

        // PROSTA LOGIKA: 
        // - Jeśli zamówienie MA odoo_order_id → UPDATE
        // - Jeśli zamówienie NIE MA odoo_order_id → CREATE
        const hasOdooId = !!queueItem.odoo_order_id;
        const operation = hasOdooId ? 'update' : 'create';

        if (operation === 'create') {
          odooOrderId = await odooClient.createSaleOrder(order);
          
          // ZAPISZ odoo_order_id ZARAZ po utworzeniu, PRZED potwierdzeniem
          await client.query(`
            UPDATE odoo_sync_queue
            SET odoo_order_id = $1, updated_at = NOW()
            WHERE id = $2
          `, [odooOrderId, queueItem.id]);

          await client.query(`
            UPDATE commerce.orders
            SET odoo_order_id = $1, updated_at = NOW()
            WHERE order_number = $2
          `, [odooOrderId, queueItem.order_number]);
          
          if (autoConfirmOrders) {
            await odooClient.confirmSaleOrder(odooOrderId);
            
            // 🔧 FIX: Po potwierdzeniu, Odoo nadpisuje date_order na bieżącą datę
            // Musimy przywrócić oryginalną datę zamówienia
            if (order.orderDate) {
              await odooClient.updateSaleOrder(odooOrderId, {
                date_order: order.orderDate
              } as any);
              console.log(`✅ Przywrócono oryginalną datę zamówienia: ${order.orderDate}`);
            }
          }
        } else {
          if (!queueItem.odoo_order_id) {
            throw new Error('Odoo order ID not found for update operation');
          }

          // Pobierz historię zmian dla tego zamówienia
          const changesResult = await client.query(`
            SELECT 
              field_changed, old_value, new_value, detected_at
            FROM public.order_changes
            WHERE order_id = $1
            ORDER BY detected_at DESC
            LIMIT 20
          `, [queueItem.order_number]);

          // Przygotuj notatkę z historią zmian
          let note = `Updated from Alpma OMS at ${new Date().toISOString()}`;
          
          if (changesResult.rows.length > 0) {
            note += '\n\n=== Historia zmian ===\n';
            const fieldLabels: Record<string, string> = {
              status: 'Status zamówienia',
              payment_status: 'Status płatności',
              payment_amount: 'Kwota płatności',
              refund_amount: 'Kwota zwrotu',
              has_returns: 'Zwroty produktów',
              tracking_numbers: 'Numery przesyłek',
              buyer_address: 'Adres kupującego',
              buyer_city: 'Miasto',
              buyer_zip: 'Kod pocztowy',
            };

            for (const change of changesResult.rows) {
              const fieldName = fieldLabels[change.field_changed] || change.field_changed;
              const date = new Date(change.detected_at).toLocaleString('pl-PL');
              note += `\n[${date}] ${fieldName}: "${change.old_value}" → "${change.new_value}"`;
            }
          }

          // Oblicz URL do zamówienia w OMS (z konfiguracji)
          const omsUrl = `https://${omsDomain}/order/${order.orderNumber}`;
          console.log(`🔗 [ODOO URL UPDATE] Generuję URL dla #${order.orderNumber}: ${omsUrl} (domena: ${omsDomain})`);

          // Mapuj status zamówienia
          const orderStatus = order.status || 'NEW';
          
          // Mapuj status płatności
          const paymentStatus = order.paymentStatus || 'PENDING';

          // Aktualizuj datę zamówienia, notatkę i pola OMS
          // Data już jest sformatowana z PostgreSQL (YYYY-MM-DD HH24:MI:SS)
          updateData = { 
            note: note,
            client_order_ref: `#${order.orderNumber}`,
            x_studio_url_w_oms: omsUrl,
            x_studio_status_w_oms: orderStatus,
            x_studio_patnosc_w_oms: paymentStatus,
            x_studio_numer_w_oms: order.orderNumber,
            x_studio_url_marketplace: order.marketplaceUrl,
          };
          
          // Dodaj datę zamówienia jeśli jest dostępna
          if (order.orderDate) {
            updateData.date_order = order.orderDate;  // orderDate to już string z TO_CHAR
            console.log(`📅 [ODOO UPDATE] #${order.orderNumber}: date_order=${updateData.date_order}`);
          }
          
          await odooClient.updateSaleOrder(queueItem.odoo_order_id, updateData);
          odooOrderId = queueItem.odoo_order_id;
        }

        const duration = Date.now() - startTime;

        await client.query(`
          UPDATE odoo_sync_queue
          SET status = 'completed', processed_at = NOW(), odoo_order_id = $1, updated_at = NOW()
          WHERE id = $2
        `, [odooOrderId, queueItem.id]);

        await client.query(`
          UPDATE commerce.orders
          SET odoo_order_id = $1, updated_at = NOW()
          WHERE id = $2
        `, [odooOrderId, queueItem.order_number]);

        // Przygotuj payload do logowania (CREATE wysyła cały obiekt, UPDATE używa updateData)
        const logPayload = operation === 'create' ? order : updateData;

        await client.query(`
          INSERT INTO odoo_sync_logs (
            queue_id, order_number, operation, status, message,
            request_payload, odoo_order_id, duration
          ) VALUES ($1, $2, $3, 'success', $4, $5, $6, $7)
        `, [
          queueItem.id,
          queueItem.order_number,
          operation,
          `Successfully synced to Odoo order #${odooOrderId}`,
          JSON.stringify(logPayload),
          odooOrderId,
          duration,
        ]);

        succeeded++;
        console.log(`✅ Synced #${queueItem.order_number} to Odoo (order #${odooOrderId}) in ${duration}ms`);
      } catch (error: unknown) {
        failed++;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        const duration = Date.now() - startTime;

        // Jeśli zamówienie zostało usunięte z Odoo, automatycznie zmień na CREATE
        const isDeletedInOdoo = errorMessage.includes('Record does not exist') || 
                                errorMessage.includes('has been deleted');
        
        if (isDeletedInOdoo) {
          console.log(`🔄 Zamówienie #${queueItem.order_number} usunięte z Odoo - zmieniam na CREATE`);
          await client.query(`
            UPDATE odoo_sync_queue
            SET 
              operation = 'create',
              odoo_order_id = NULL,
              attempts = 0,
              status = 'pending',
              last_error = NULL,
              updated_at = NOW()
            WHERE id = $1
          `, [queueItem.id]);
          
          // Wyczyść też odoo_order_id w commerce.orders
          await client.query(`
            UPDATE commerce.orders
            SET odoo_order_id = NULL
            WHERE id = $1
          `, [queueItem.order_number]);
          
          continue; // Pomiń logowanie błędu - to nie jest błąd, tylko automatyczna naprawa
        }

        await client.query(`
          UPDATE odoo_sync_queue
          SET 
            status = CASE 
              WHEN attempts + 1 >= max_attempts THEN 'failed'
              ELSE 'pending'
            END,
            attempts = attempts + 1,
            last_error = $1,
            updated_at = NOW()
          WHERE id = $2
        `, [errorMessage, queueItem.id]);

        await client.query(`
          INSERT INTO odoo_sync_logs (
            queue_id, order_number, operation, status, message, error_details, duration
          ) VALUES ($1, $2, $3, 'error', $4, $5, $6)
        `, [
          queueItem.id,
          queueItem.order_number,
          queueItem.operation,
          errorMessage,
          JSON.stringify({ error: errorMessage, attempt: queueItem.attempts + 1 }),
          duration,
        ]);

        console.error(`❌ Failed to sync #${queueItem.order_number} to Odoo (attempt ${queueItem.attempts + 1}):`, errorMessage);
      }
    }
    
    } catch (innerError) {
      console.error('❌ Error processing Odoo sync queue:', innerError);
    } finally {
      // Odblokuj advisory lock tylko jeśli został zdobyty
      if (lockAcquired) {
        await client.query(`SELECT pg_advisory_unlock(123456789)`);
        console.log('🔓 Zwolniono blokadę processSyncQueue()');
      }
    }
  } catch (error) {
    console.error('❌ Error in processSyncQueue:', error);
  } finally {
    client.release();
  }

  if (processed > 0) {
    console.log(`📊 Odoo sync: ${succeeded} succeeded, ${failed} failed out of ${processed} processed`);
  }

  return { processed, succeeded, failed };
}

export async function syncRecentOrdersToOdoo(timeWindowMinutes: number = 10): Promise<void> {
  const client = await pool.connect();

  try {
    const result = await client.query(`
      SELECT order_number, source
      FROM commerce.orders
      WHERE (updated_at >= NOW() - INTERVAL '${timeWindowMinutes} minutes'
         OR created_at >= NOW() - INTERVAL '${timeWindowMinutes} minutes')
        AND order_number IS NOT NULL
      ORDER BY 
        CASE 
          WHEN updated_at > created_at THEN updated_at 
          ELSE created_at 
        END DESC
    `);

    console.log(`🔄 Found ${result.rows.length} orders updated in last ${timeWindowMinutes} minutes for Odoo sync`);

    for (const order of result.rows) {
      // PROSTA LOGIKA: Sprawdź czy zamówienie ma odoo_order_id w commerce.orders
      const odooCheck = await client.query(`
        SELECT odoo_order_id FROM commerce.orders
        WHERE order_number = $1
        LIMIT 1
      `, [order.order_number]);

      // Jeśli zamówienie ma odoo_order_id → UPDATE, jeśli nie → CREATE
      const operation = odooCheck.rows[0]?.odoo_order_id ? 'update' : 'create';

      await addOrderToSyncQueue(order.order_number, order.source, operation);
    }

    await processSyncQueue();
  } catch (error) {
    console.error('❌ Error syncing recent orders to Odoo:', error);
  } finally {
    client.release();
  }
}

export async function getOdooSyncStatus(): Promise<{
  pending: number;
  processing: number;
  completed: number;
  failed: number;
  lastSyncAt: string | null;
}> {
  const client = await pool.connect();

  try {
    const statsResult = await client.query(`
      SELECT 
        COUNT(*) FILTER (WHERE status = 'pending') as pending,
        COUNT(*) FILTER (WHERE status = 'processing') as processing,
        COUNT(*) FILTER (WHERE status = 'completed') as completed,
        COUNT(*) FILTER (WHERE status = 'failed') as failed,
        MAX(processed_at) as last_sync_at
      FROM odoo_sync_queue
    `);

    const stats = statsResult.rows[0];

    return {
      pending: parseInt(stats.pending || '0'),
      processing: parseInt(stats.processing || '0'),
      completed: parseInt(stats.completed || '0'),
      failed: parseInt(stats.failed || '0'),
      lastSyncAt: stats.last_sync_at,
    };
  } finally {
    client.release();
  }
}

export async function getOdooSyncLogs(limit: number = 100): Promise<any[]> {
  const client = await pool.connect();

  try {
    const result = await client.query(`
      SELECT * FROM odoo_sync_logs
      ORDER BY created_at DESC
      LIMIT $1
    `, [limit]);

    return result.rows;
  } finally {
    client.release();
  }
}

export async function retryFailedSync(queueId: number): Promise<void> {
  const client = await pool.connect();

  try {
    await client.query(`
      UPDATE odoo_sync_queue
      SET status = 'pending', attempts = 0, last_error = NULL, updated_at = NOW()
      WHERE id = $1
    `, [queueId]);

    console.log(`🔄 Reset queue item ${queueId} for retry`);
  } finally {
    client.release();
  }
}
