import xmlrpc from 'xmlrpc';
import { logOdooError } from './error-logger';

interface CommerceOrder {
  id: number;
  orderNumber: number;
  source: string;
  sourceOrderId: string;
  orderDate: string | null;  // TO_CHAR zwraca string, nie Date!
  status: string | null;
  paymentStatus: string | null;
  buyerLogin: string | null;
  buyerEmail: string | null;
  buyerCompany: string | null;
  buyerFirstName: string | null;
  buyerLastName: string | null;
  buyerPhone: string | null;
  buyerAddress: string | null;
  buyerZip: string | null;
  buyerCity: string | null;
  buyerCountryCode: string | null;
  paymentId: string | null;
  paymentProvider: string | null;
  paymentAmount: string | null;
  paymentCurrency: string | null;
  deliveryMethod: string | null;
  deliveryAmount: string | null;
  deliveryCurrency: string | null;
  totalToPayAmount: string | null;
  totalToPayCurrency: string | null;
  buyerNotes: string | null;
  invoiceRequired: boolean | null;
  trackingNumbers: string[] | null;
  marketplaceUrl?: string | null;
  items?: Array<{
    name: string;
    quantity: string;
    price: string;
    sku?: string;
    image_url?: string;
  }>;
}

interface OdooConfig {
  url: string;
  database: string;
  username: string;
  apiKey: string;
}

interface OdooOrderLine {
  product_id: number | false;
  name: string;
  product_uom_qty: number;
  price_unit: number;
}

interface OdooOrderData {
  partner_id: number | false;
  origin: string;
  client_order_ref: string;
  date_order: string;
  amount_total: number;
  order_line: [number, number, OdooOrderLine][];
  note?: string;
  // Własne pola z Odoo Studio (prefix x_studio_)
  x_studio_url_w_oms?: string;        // URL w OMS (link do zamówienia)
  x_studio_status_w_oms?: string;     // Status w OMS (status zamówienia)
  x_studio_patnosc_w_oms?: string;    // Płatność w OMS (status płatności)
  x_studio_numer_w_oms?: number;      // Numer zamówienia w OMS (czysty numer)
  x_studio_url_marketplace?: string;  // URL do zamówienia na marketplace (Allegro/Shoper)
}

export class OdooClient {
  private config: OdooConfig;
  private uid: number | null = null;
  private commonClient: xmlrpc.Client;
  private objectClient: xmlrpc.Client;

  constructor(config: OdooConfig) {
    // Automatyczna migracja na nową domenę Cloudflare Tunnel alpma.app
    if (config.url.includes('tunnel.pyjam.as')) {
      // Migracja ze starej domeny Cloudflare
      config.url = config.url.replace('tunnel.pyjam.as', 'alpma.app');
      console.log(`✅ Automatycznie zaktualizowano URL Odoo: stara domena → ${config.url}`);
    } else if (config.url.includes('100.99.76.111')) {
      // Podmiana lokalnego IP na Cloudflare Tunnel (alpma.app)
      config.url = 'https://alpma.app';
      console.log(`✅ Automatycznie użyto Cloudflare Tunnel: ${config.url}`);
    }
    
    this.config = config;
    
    const url = new URL(this.config.url);
    const basePath = url.pathname.endsWith('/') ? url.pathname.slice(0, -1) : url.pathname;
    const commonPath = `${basePath}/xmlrpc/2/common`;
    const objectPath = `${basePath}/xmlrpc/2/object`;
    const isHttps = url.protocol === 'https:';
    const defaultPort = isHttps ? 443 : 8069;
    
    const clientOptions = {
      host: url.hostname,
      port: parseInt(url.port) || defaultPort,
      path: commonPath,
    };

    const objectOptions = {
      host: url.hostname,
      port: parseInt(url.port) || defaultPort,
      path: objectPath,
    };
    
    // Use secure client for HTTPS URLs
    if (isHttps) {
      this.commonClient = xmlrpc.createSecureClient(clientOptions);
      this.objectClient = xmlrpc.createSecureClient(objectOptions);
    } else {
      this.commonClient = xmlrpc.createClient(clientOptions);
      this.objectClient = xmlrpc.createClient(objectOptions);
    }
  }

  async authenticate(): Promise<number> {
    return new Promise((resolve, reject) => {
      this.commonClient.methodCall(
        'authenticate',
        [this.config.database, this.config.username, this.config.apiKey, {}],
        async (error, value) => {
          if (error) {
            const errorMsg = error && typeof error === 'object' && 'message' in error ? String(error.message) : String(error);
            const authError = new Error(`Odoo authentication failed: ${errorMsg}`);
            
            // Log to error logs system
            await logOdooError('Odoo authentication failed', error, {
              database: this.config.database,
              username: this.config.username,
              url: this.config.url
            }).catch(logErr => console.error('Failed to log Odoo auth error:', logErr));
            
            reject(authError);
          } else if (!value) {
            const credError = new Error('Odoo authentication failed: Invalid credentials');
            
            await logOdooError('Odoo authentication failed: Invalid credentials', credError, {
              database: this.config.database,
              username: this.config.username
            }).catch(logErr => console.error('Failed to log Odoo auth error:', logErr));
            
            reject(credError);
          } else {
            this.uid = value as number;
            resolve(this.uid);
          }
        }
      );
    });
  }

  async execute(model: string, method: string, params: any[]): Promise<any> {
    if (!this.uid) {
      await this.authenticate();
    }

    return new Promise((resolve, reject) => {
      this.objectClient.methodCall(
        'execute_kw',
        [
          this.config.database,
          this.uid,
          this.config.apiKey,
          model,
          method,
          params,
        ],
        async (error, value) => {
          if (error) {
            const errorMsg = error && typeof error === 'object' && 'message' in error ? String(error.message) : String(error);
            const execError = new Error(`Odoo execute failed: ${errorMsg}`);
            
            // Log to error logs system
            await logOdooError(`Odoo execute failed: ${model}.${method}`, error, {
              model,
              method,
              paramsCount: params.length
            }).catch(logErr => console.error('Failed to log Odoo execute error:', logErr));
            
            reject(execError);
          } else {
            resolve(value);
          }
        }
      );
    });
  }

  async findOrCreatePartner(
    email: string, 
    name?: string,
    phone?: string,
    street?: string,
    zip?: string,
    city?: string,
    countryCode?: string
  ): Promise<number> {
    try {
      const partners = await this.execute('res.partner', 'search_read', [
        [['email', '=', email]],
        ['id', 'name', 'email'],
      ]);

      if (partners && partners.length > 0) {
        console.log(`👤 Found existing partner: ${partners[0].name} (ID: ${partners[0].id})`);
        return partners[0].id;
      }

      const partnerData: any = {
        name: name || email,
        email: email,
        customer_rank: 1,
      };

      if (phone) partnerData.phone = phone;
      if (street) partnerData.street = street;
      if (zip) partnerData.zip = zip;
      if (city) partnerData.city = city;
      if (countryCode) {
        const countries = await this.execute('res.country', 'search_read', [
          [['code', '=', countryCode.toUpperCase()]],
          ['id'],
        ]);
        if (countries && countries.length > 0) {
          partnerData.country_id = countries[0].id;
        }
      }

      const partnerId = await this.execute('res.partner', 'create', [partnerData]);

      console.log(`👤 Created new partner: ${name || email} (ID: ${partnerId})`);
      return partnerId;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      throw new Error(`Failed to find or create partner: ${errorMessage}`);
    }
  }

  async findOrCreateProduct(productName: string, price: number, sku?: string, imageUrl?: string): Promise<number> {
    try {
      // KROK 1: Jeśli mamy SKU, szukaj NAJPIERW po SKU (najbardziej precyzyjne)
      if (sku) {
        const productsBySku = await this.execute('product.product', 'search_read', [
          ['|', '|',
            ['default_code', '=', sku],
            ['barcode', '=', sku],
            ['x_studio_sku', '=', sku]
          ],
          ['id', 'name', 'default_code', 'barcode'],
        ]);

        if (productsBySku && productsBySku.length > 0) {
          console.log(`📦 Found existing product by SKU "${sku}": ${productsBySku[0].name} (ID: ${productsBySku[0].id})`);
          return productsBySku[0].id;
        }
      }

      // KROK 2: Jeśli nie znaleziono po SKU, szukaj po nazwie
      const productsByName = await this.execute('product.product', 'search_read', [
        [['name', '=', productName]],
        ['id', 'name', 'default_code'],
      ]);

      if (productsByName && productsByName.length > 0) {
        console.log(`📦 Found existing product by name: ${productsByName[0].name} (ID: ${productsByName[0].id})`);
        return productsByName[0].id;
      }

      const productData: any = {
        name: productName,
        type: 'consu',
        list_price: price,
        sale_ok: true,
        purchase_ok: false,
      };

      // Dodaj SKU jeśli dostępny (Internal Reference + Kod kreskowy + pole niestandardowe)
      if (sku) {
        productData.default_code = sku;   // Internal Reference w Odoo
        productData.barcode = sku;        // Kod kreskowy
        productData.x_studio_sku = sku;   // Pole niestandardowe SKU
      }

      // Dodaj zdjęcie jeśli dostępne
      if (imageUrl) {
        try {
          const imageBase64 = await this.downloadImageAsBase64(imageUrl);
          if (imageBase64) {
            productData.image_1920 = imageBase64;
            console.log(`🖼️  Dodano zdjęcie produktu z: ${imageUrl}`);
          }
        } catch (imageError) {
          console.warn(`⚠️  Nie udało się pobrać zdjęcia: ${imageUrl}`, imageError);
        }
      }

      const productId = await this.execute('product.product', 'create', [productData]);

      const skuInfo = sku ? ` | SKU: ${sku}` : '';
      const imageInfo = imageUrl ? ' | Zdjęcie: ✓' : '';
      console.log(`📦 Created new product: ${productName} (ID: ${productId}) - Price: ${price} PLN${skuInfo}${imageInfo}`);
      return productId;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      throw new Error(`Failed to find or create product: ${errorMessage}`);
    }
  }

  private async downloadImageAsBase64(imageUrl: string): Promise<string | null> {
    try {
      // Walidacja: sprawdź czy to pełny URL (http/https)
      if (!imageUrl.startsWith('http://') && !imageUrl.startsWith('https://')) {
        console.warn(`⚠️  Pominięto nieprawidłowy URL obrazka: ${imageUrl} (brak protokołu http/https)`);
        return null;
      }

      const response = await fetch(imageUrl);
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }
      
      const arrayBuffer = await response.arrayBuffer();
      const buffer = Buffer.from(arrayBuffer);
      return buffer.toString('base64');
    } catch (error) {
      console.error(`Failed to download image from ${imageUrl}:`, error);
      return null;
    }
  }

  async createSaleOrder(order: CommerceOrder): Promise<number> {
    try {
      // 🛡️ OCHRONA PRZED DUPLIKATAMI: Sprawdź czy zamówienie już istnieje w Odoo
      // Szukamy po TRZECH kryteriach (renumeracja może zmienić order_number!):
      // 1. client_order_ref - aktualny numer (po renumeracji)
      // 2. origin - source + source_order_id (NIGDY się nie zmienia)
      // 3. x_studio_numer_w_oms - numer zamówienia w OMS (unikalny identyfikator)
      
      const originRef = `${order.source}-${order.sourceOrderId}`;
      
      const existingOrders = await this.execute('sale.order', 'search_read', [
        ['|', '|',
          ['client_order_ref', '=', `#${order.orderNumber}`],
          ['origin', '=', originRef],
          ['x_studio_numer_w_oms', '=', order.orderNumber]
        ],
        ['id', 'name', 'client_order_ref', 'origin', 'x_studio_numer_w_oms'],
      ]);

      if (existingOrders && existingOrders.length > 0) {
        const existingId = existingOrders[0].id;
        let matchedBy = 'unknown';
        if (existingOrders[0].origin === originRef) matchedBy = 'origin';
        else if (existingOrders[0].client_order_ref === `#${order.orderNumber}`) matchedBy = 'client_order_ref';
        else if (existingOrders[0].x_studio_numer_w_oms === order.orderNumber) matchedBy = 'x_studio_numer_w_oms';
        
        console.log(`⚠️  Order #${order.orderNumber} (${originRef}) already exists in Odoo (ID: ${existingId}, matched by: ${matchedBy}) - returning existing ID instead of creating duplicate`);
        return existingId;
      }

      const customerName = [
        order.buyerFirstName,
        order.buyerLastName
      ].filter(Boolean).join(' ') || order.buyerLogin || order.buyerEmail || 'Unknown Customer';

      const partnerId = await this.findOrCreatePartner(
        order.buyerEmail || 'unknown@example.com',
        customerName,
        order.buyerPhone || undefined,
        order.buyerAddress || undefined,
        order.buyerZip || undefined,
        order.buyerCity || undefined,
        order.buyerCountryCode || undefined
      );

      const orderLines: [number, number, OdooOrderLine][] = [];

      if (order.items && order.items.length > 0) {
        for (const item of order.items) {
          const productPrice = parseFloat(item.price || '0');
          const productId = await this.findOrCreateProduct(
            item.name || 'Unknown Product',
            productPrice,
            item.sku, // SKU produktu
            item.image_url // URL zdjęcia produktu
          );

          orderLines.push([
            0,
            0,
            {
              product_id: productId,
              name: item.name || 'Unknown Product',
              product_uom_qty: parseInt(item.quantity || '1'),
              price_unit: productPrice,
            },
          ]);
        }
      }

      // Data już jest w formacie YYYY-MM-DD HH24:MI:SS z PostgreSQL (TO_CHAR)
      // Odoo akceptuje dokładnie ten format - nie konwertuj przez JavaScript Date!
      
      // 🔍 DEBUG: Sprawdź wartość order.orderDate
      console.log(`🔍 [DEBUG orderDate] #${order.orderNumber}:`, {
        value: order.orderDate,
        type: typeof order.orderDate,
        isNull: order.orderDate === null,
        isUndefined: order.orderDate === undefined,
        isFalsy: !order.orderDate
      });
      
      // NAPRAWIONA LOGIKA:  
      // 1. Sprawdź czy orderDate istnieje i nie jest pusty string
      // 2. Jeśli brak - RZUĆ BŁĄD zamiast używać fallback!
      const isEmptyString = typeof order.orderDate === 'string' && order.orderDate.trim() === '';
      if (!order.orderDate || isEmptyString) {
        const errorMsg = `❌ BRAK order.orderDate dla zamówienia #${order.orderNumber}! Nie można utworzyć w Odoo bez daty zamówienia.`;
        console.error(errorMsg);
        throw new Error(errorMsg);
      }
      
      // order.orderDate to zawsze string z TO_CHAR w SQL
      const odooDateFormat = order.orderDate;
      
      console.log(`📅 [ODOO DATE] #${order.orderNumber}: ${odooDateFormat} ✅ (from order_date)`);
      // Oblicz URL do zamówienia w OMS (z konfiguracji)
      const omsDomain = this.config.omsDomain || 'alp-oms.replit.app';
      const omsUrl = `https://${omsDomain}/order/${order.orderNumber}`;
      console.log(`🔗 [ODOO URL] Generuję URL dla #${order.orderNumber}: ${omsUrl} (domena: ${omsDomain})`);

      // Mapuj status zamówienia
      const orderStatus = order.status || 'NEW';
      
      // Mapuj status płatności
      const paymentStatus = order.paymentStatus || 'PENDING';

      const orderData: OdooOrderData = {
        partner_id: partnerId,
        origin: `${order.source}-${order.sourceOrderId}`, // Unikalny identyfikator (nie zmienia się po renumeracji!)
        client_order_ref: `#${order.orderNumber}`,
        date_order: odooDateFormat,
        amount_total: parseFloat(order.totalToPayAmount || '0'),
        order_line: orderLines,
        note: `Imported from Alpma OMS - ${order.source} Order #${order.orderNumber}`,
        // Nowe pola dla Odoo
        x_studio_url_w_oms: omsUrl,
        x_studio_status_w_oms: orderStatus,
        x_studio_patnosc_w_oms: paymentStatus,
        x_studio_numer_w_oms: order.orderNumber,
        x_studio_url_marketplace: order.marketplaceUrl ?? undefined,
      };

      console.log(`🔍 [ODOO CREATE] Wysyłam do Odoo:`, JSON.stringify({
        order_number: order.orderNumber,
        date_order: orderData.date_order,
        partner_id: orderData.partner_id,
        amount_total: orderData.amount_total
      }, null, 2));

      const orderId = await this.execute('sale.order', 'create', [orderData]);

      // Sprawdź jaką datę Odoo faktycznie zapisało
      const createdOrder = await this.execute('sale.order', 'read', [
        [orderId],
        ['id', 'name', 'date_order', 'create_date']
      ]);
      
      console.log(`✅ Created Odoo sale order ${orderId} for #${order.orderNumber}`);
      if (createdOrder && createdOrder[0]) {
        console.log(`🔍 [ODOO RESPONSE] Odoo zapisało:`, JSON.stringify({
          odoo_id: createdOrder[0].id,
          date_order: createdOrder[0].date_order,
          create_date: createdOrder[0].create_date
        }, null, 2));
      }
      return orderId;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      throw new Error(`Failed to create sale order: ${errorMessage}`);
    }
  }

  async confirmSaleOrder(odooOrderId: number): Promise<boolean> {
    try {
      const result = await this.execute('sale.order', 'action_confirm', [[odooOrderId]]);
      console.log(`✅ Confirmed Odoo sale order ${odooOrderId} (changed from quotation to sale order)`);
      return result;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      
      // Jeśli zamówienie już jest potwierdzone, nie traktuj tego jako błąd
      if (errorMessage.includes('nie są w stanie wymagającym potwierdzenia') || 
          errorMessage.includes('not in a state requiring confirmation')) {
        console.log(`ℹ️  Odoo sale order ${odooOrderId} is already confirmed`);
        return true;
      }
      
      throw new Error(`Failed to confirm sale order: ${errorMessage}`);
    }
  }

  async updateSaleOrder(odooOrderId: number, updates: Partial<OdooOrderData>): Promise<boolean> {
    try {
      console.log(`🔍 [ODOO UPDATE] Wysyłam do Odoo order #${odooOrderId}:`, JSON.stringify({
        x_studio_status_w_oms: updates.x_studio_status_w_oms,
        x_studio_patnosc_w_oms: updates.x_studio_patnosc_w_oms,
        x_studio_numer_w_oms: updates.x_studio_numer_w_oms,
        client_order_ref: updates.client_order_ref,
        date_order: updates.date_order,
        note_length: updates.note?.length || 0
      }, null, 2));
      
      const result = await this.execute('sale.order', 'write', [
        [odooOrderId],
        updates,
      ]);

      // WERYFIKACJA: Odczytaj dane PO aktualizacji i sprawdź co Odoo faktycznie zapisało
      const verifyData = await this.execute('sale.order', 'read', [
        [odooOrderId],
        ['x_studio_status_w_oms', 'x_studio_patnosc_w_oms', 'x_studio_numer_w_oms', 'client_order_ref', 'date_order', 'state']
      ]);
      
      if (verifyData && verifyData[0]) {
        console.log(`✅ [ODOO VERIFY] Order #${odooOrderId} PO aktualizacji:`, JSON.stringify({
          x_studio_status_w_oms: verifyData[0].x_studio_status_w_oms,
          x_studio_patnosc_w_oms: verifyData[0].x_studio_patnosc_w_oms,
          x_studio_numer_w_oms: verifyData[0].x_studio_numer_w_oms,
          client_order_ref: verifyData[0].client_order_ref,
          date_order: verifyData[0].date_order,
          state: verifyData[0].state
        }, null, 2));
        
        // Sprawdź czy date_order się zgadza
        if (updates.date_order && verifyData[0].date_order !== updates.date_order) {
          console.warn(`⚠️ [ODOO WARN] date_order nie została zaktualizowana! Wysłano: ${updates.date_order}, Odoo ma: ${verifyData[0].date_order}, Stan zamówienia: ${verifyData[0].state}`);
        }
      }

      console.log(`✅ Updated Odoo sale order ${odooOrderId}`);
      return result;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      throw new Error(`Failed to update sale order: ${errorMessage}`);
    }
  }

  async getSaleOrder(odooOrderId: number): Promise<any> {
    try {
      const orders = await this.execute('sale.order', 'read', [
        [odooOrderId],
        ['id', 'name', 'partner_id', 'amount_total', 'state'],
      ]);

      return orders && orders.length > 0 ? orders[0] : null;
    } catch (error: unknown) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      throw new Error(`Failed to get sale order: ${errorMessage}`);
    }
  }

  async testConnection(): Promise<{ success: boolean; version?: any; error?: string; details?: any }> {
    const details: any = {
      url: this.config.url,
      database: this.config.database,
      username: this.config.username,
      steps: []
    };

    try {
      // Step 1: Test version endpoint
      details.steps.push({ step: 'version', status: 'trying' });
      const version = await new Promise((resolve, reject) => {
        const timeout = setTimeout(() => {
          reject(new Error('Connection timeout (10s) - Odoo nie odpowiada'));
        }, 10000);

        this.commonClient.methodCall('version', [], (error, value) => {
          clearTimeout(timeout);
          if (error) reject(error);
          else resolve(value);
        });
      });
      details.steps.push({ step: 'version', status: 'success', value: version });

      // Step 2: Test authentication
      details.steps.push({ step: 'authenticate', status: 'trying' });
      await this.authenticate();
      details.steps.push({ step: 'authenticate', status: 'success', uid: this.uid });

      return { success: true, version, details };
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      details.steps.push({ 
        step: details.steps[details.steps.length - 1]?.step || 'unknown', 
        status: 'failed', 
        error: errorMessage 
      });

      return {
        success: false,
        error: errorMessage,
        details
      };
    }
  }
}

export async function getOdooUserUrl(): Promise<string | null> {
  // Zwraca URL dla linków klikanych przez użytkowników (Tailscale)
  const { pool } = await import('./postgres.js');
  const client = await pool.connect();

  try {
    const result = await client.query(`
      SELECT user_url, url
      FROM odoo_config
      WHERE is_active = true
      LIMIT 1
    `);

    if (result.rows.length === 0) {
      return null;
    }

    // Użyj user_url jeśli jest ustawiony, w przeciwnym razie użyj url
    return result.rows[0].user_url || result.rows[0].url;
  } finally {
    client.release();
  }
}

export async function createOdooClient(): Promise<OdooClient | null> {
  const apiKey = process.env.ODOO_API_KEY;

  if (!apiKey) {
    console.warn('⚠️  ODOO_API_KEY not configured');
    return null;
  }

  // Import pool to get config from database
  const { pool } = await import('./postgres.js');
  const client = await pool.connect();

  try {
    const result = await client.query(`
      SELECT url, database, username
      FROM odoo_config
      WHERE is_active = true
      LIMIT 1
    `);

    if (result.rows.length === 0) {
      console.warn('⚠️  No active Odoo configuration found');
      return null;
    }

    const config = result.rows[0];

    // Automatyczna migracja na nową domenę Cloudflare Tunnel
    let odooUrl = config.url;
    console.log(`🔍 DEBUG createOdooClient: Original URL = "${odooUrl}"`);
    console.log(`🔍 DEBUG createOdooClient: includes tunnel.pyjam.as = ${odooUrl.includes('tunnel.pyjam.as')}`);
    
    if (odooUrl.includes('tunnel.pyjam.as')) {
      odooUrl = odooUrl.replace('tunnel.pyjam.as', 'alpma.app');
      console.log(`✅ Automatycznie zaktualizowano URL Odoo: ${odooUrl}`);
    } else if (odooUrl.includes('100.99.76.111')) {
      odooUrl = 'https://alpma.app/odoo';
      console.log(`✅ Automatycznie użyto Cloudflare Tunnel: ${odooUrl}`);
    }
    
    console.log(`🔍 DEBUG createOdooClient: Final URL = "${odooUrl}"`);

    return new OdooClient({
      url: odooUrl,
      database: config.database,
      username: config.username,
      apiKey,
    });
  } finally {
    client.release();
  }
}
