import { pool } from './postgres.js';
import axios from 'axios';
import { saveOrderToCommerce } from './postgres.js';

async function getAccessToken(): Promise<string | null> {
  const result = await pool.query(`
    SELECT access_token FROM allegro_connections WHERE is_active = true LIMIT 1
  `);
  return result.rows[0]?.access_token || null;
}

async function fetchOrderFromAllegro(orderId: string, accessToken: string): Promise<any> {
  try {
    const response = await axios.get(
      `https://api.allegro.pl/order/checkout-forms/${orderId}`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: 'application/vnd.allegro.public.v1+json',
        }
      }
    );
    return response.data;
  } catch (error: any) {
    console.error(`❌ Błąd pobierania zamówienia ${orderId}:`, error.message);
    return null;
  }
}

async function fixIncompleteOrders() {
  try {
    console.log('🔄 Naprawianie niekompletnych zamówień...\n');

    const accessToken = await getAccessToken();
    if (!accessToken) {
      console.error('❌ Brak access token');
      return;
    }

    // Pobierz zamówienia bez produktów
    const result = await pool.query(`
      SELECT o.id, o.order_number, o.source_order_id, o.source
      FROM commerce.orders o
      LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
      WHERE o.source = 'ALLEGRO'
      GROUP BY o.id, o.order_number, o.source_order_id, o.source
      HAVING COUNT(oi.id) = 0
      ORDER BY o.id DESC
      LIMIT 10
    `);

    console.log(`📊 Znaleziono ${result.rows.length} zamówień bez produktów\n`);

    for (const order of result.rows) {
      console.log(`\n🔍 Przetwarzanie zamówienia #${order.order_number} (${order.source_order_id})...`);
      
      const orderData = await fetchOrderFromAllegro(order.source_order_id, accessToken);
      
      if (!orderData) {
        console.log(`❌ Nie udało się pobrać danych dla #${order.order_number}`);
        continue;
      }

      if (!orderData.lineItems || orderData.lineItems.length === 0) {
        console.log(`⚠️ Zamówienie #${order.order_number} nie ma produktów w API`);
        continue;
      }

      console.log(`✅ Pobrano ${orderData.lineItems.length} produktów`);
      
      // Usuń stare niepełne zamówienie
      await pool.query('DELETE FROM commerce.orders WHERE id = $1', [order.id]);
      console.log(`🗑️ Usunięto stare niepełne zamówienie`);
      
      // Zapisz ponownie z pełnymi danymi
      await saveOrderToCommerce('ALLEGRO', order.source_order_id, orderData, [], [], accessToken);
      console.log(`✅ Ponownie zapisano zamówienie #${order.order_number} z ${orderData.lineItems.length} produktami`);
      
      await new Promise(resolve => setTimeout(resolve, 500));
    }

    console.log('\n✅ Zakończono naprawianie zamówień');

  } catch (error) {
    console.error('❌ Błąd:', error);
  } finally {
    await pool.end();
  }
}

fixIncompleteOrders();
