import { pool } from "./postgres";
import type { InsertErrorLog } from "@shared/schema";

// Error types for categorization
export type ErrorType = 
  | "allegro"
  | "shoper"
  | "odoo"
  | "database"
  | "api"
  | "webhook"
  | "sync"
  | "auth"
  | "general";

// Severity levels
export type ErrorSeverity = "error" | "warning" | "info";

interface LogErrorOptions {
  type: ErrorType;
  message: string;
  error?: Error | any;
  context?: Record<string, any>;
  severity?: ErrorSeverity;
}

/**
 * Centralized error logging function
 * Logs errors to database and console
 */
export async function logError(options: LogErrorOptions): Promise<void> {
  const {
    type,
    message,
    error,
    context = {},
    severity = "error",
  } = options;

  // Truncate message if too long (prevent JWT tokens from breaking UI)
  let truncatedMessage = message;
  if (truncatedMessage.includes('Invalid refresh token:')) {
    truncatedMessage = 'Invalid refresh token - token wygasł lub został unieważniony';
  } else if (truncatedMessage.length > 500) {
    truncatedMessage = truncatedMessage.substring(0, 500) + '...';
  }

  // Extract stack trace if error object provided
  let stackTrace: string | undefined;
  if (error) {
    if (error.stack) {
      stackTrace = error.stack;
    } else if (error.response?.data) {
      // For axios errors, include response data
      stackTrace = JSON.stringify(error.response.data, null, 2);
    }
    
    // Truncate stack trace to reasonable length
    if (stackTrace && stackTrace.length > 5000) {
      stackTrace = stackTrace.substring(0, 5000) + '\n... (truncated)';
    }
  }

  try {
    // Log to console
    const consolePrefix = severity === "error" ? "❌" : severity === "warning" ? "⚠️" : "ℹ️";
    console.error(`${consolePrefix} [${type.toUpperCase()}] ${truncatedMessage}`);
    
    if (error && error.stack) {
      console.error("Stack trace:", error.stack);
    }
    
    if (Object.keys(context).length > 0) {
      console.error("Context:", context);
    }

    // Save to database
    await pool.query(`
      INSERT INTO error_logs (type, message, stack_trace, context, severity)
      VALUES ($1, $2, $3, $4, $5)
    `, [
      type,
      truncatedMessage,
      stackTrace || null,
      context ? JSON.stringify(context) : null,
      severity,
    ]);
  } catch (dbError) {
    // If database logging fails, at least log to console
    console.error("❌ Failed to save error log to database:", dbError);
    console.error("Original error:", truncatedMessage);
  }
}

/**
 * Log Allegro-related errors
 */
export async function logAllegroError(message: string, error?: any, context?: Record<string, any>) {
  return logError({ type: "allegro", message, error, context });
}

/**
 * Log Shoper-related errors
 */
export async function logShoperError(message: string, error?: any, context?: Record<string, any>) {
  return logError({ type: "shoper", message, error, context });
}

/**
 * Log Odoo-related errors
 */
export async function logOdooError(message: string, error?: any, context?: Record<string, any>) {
  return logError({ type: "odoo", message, error, context });
}

/**
 * Log Database-related errors
 */
export async function logDatabaseError(message: string, error?: any, context?: Record<string, any>) {
  return logError({ type: "database", message, error, context });
}

/**
 * Log API-related errors
 */
export async function logApiError(message: string, error?: any, context?: Record<string, any>) {
  return logError({ type: "api", message, error, context });
}

/**
 * Log Webhook-related errors
 */
export async function logWebhookError(message: string, error?: any, context?: Record<string, any>) {
  return logError({ type: "webhook", message, error, context });
}

/**
 * Log Sync-related errors
 */
export async function logSyncError(message: string, error?: any, context?: Record<string, any>) {
  return logError({ type: "sync", message, error, context });
}

/**
 * Log Authentication-related errors
 */
export async function logAuthError(message: string, error?: any, context?: Record<string, any>) {
  return logError({ type: "auth", message, error, context });
}

/**
 * Clean up old error logs (older than specified days)
 */
export async function cleanupOldErrorLogs(daysOld: number = 30): Promise<number> {
  try {
    const result = await pool.query(`
      DELETE FROM error_logs
      WHERE timestamp < NOW() - INTERVAL '${daysOld} days'
      RETURNING id
    `);
    
    const deletedCount = result.rowCount || 0;
    if (deletedCount > 0) {
      console.log(`🧹 Cleaned up ${deletedCount} old error logs (older than ${daysOld} days)`);
    }
    
    return deletedCount;
  } catch (error) {
    console.error("❌ Failed to cleanup old error logs:", error);
    return 0;
  }
}
