import { pool } from '../server/postgres.js';
import { promises as fs } from 'fs';
import path from 'path';

async function backupDatabase() {
  const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
  const backupDir = path.join(process.cwd(), 'backups');
  const backupFile = path.join(backupDir, `backup-${timestamp}.sql`);
  
  try {
    // Utwórz folder backups jeśli nie istnieje
    await fs.mkdir(backupDir, { recursive: true });
    
    console.log('🔄 Rozpoczynam backup bazy danych...');
    
    // Lista wszystkich schematów do backupu
    const schemas = [
      'public',     // Tabele systemowe (users, sessions, configs, itp.)
      'commerce',   // Główna tabela zamówień
      'catalog',    // Produkty
      'allegro',    // Dane surowe z Allegro
      'shoper'      // Dane surowe z Shoper
    ];
    
    let sqlDump = `-- PostgreSQL Database Backup
-- Generated: ${new Date().toISOString()}
-- Database: Alpma OMS

`;
    
    for (const schema of schemas) {
      console.log(`📦 Backup schematu: ${schema}`);
      
      // Sprawdź czy schemat istnieje
      const schemaExists = await pool.query(
        `SELECT schema_name FROM information_schema.schemata WHERE schema_name = $1`,
        [schema]
      );
      
      if (schemaExists.rows.length === 0) {
        console.log(`⚠️  Schemat ${schema} nie istnieje - pomijam`);
        continue;
      }
      
      sqlDump += `\n-- ==========================================\n`;
      sqlDump += `-- Schema: ${schema}\n`;
      sqlDump += `-- ==========================================\n\n`;
      
      // Pobierz wszystkie tabele w schemacie
      const tables = await pool.query(
        `SELECT table_name 
         FROM information_schema.tables 
         WHERE table_schema = $1 
         AND table_type = 'BASE TABLE'
         ORDER BY table_name`,
        [schema]
      );
      
      for (const { table_name } of tables.rows) {
        console.log(`  📄 Backup tabeli: ${schema}.${table_name}`);
        
        // Pobierz strukturę tabeli
        const createTable = await pool.query(
          `SELECT 
            'CREATE TABLE IF NOT EXISTS ' || $1 || '.' || $2 || ' (' || 
            string_agg(
              column_name || ' ' || 
              data_type || 
              CASE 
                WHEN character_maximum_length IS NOT NULL 
                THEN '(' || character_maximum_length || ')'
                WHEN data_type = 'numeric' AND numeric_precision IS NOT NULL
                THEN '(' || numeric_precision || ',' || numeric_scale || ')'
                ELSE ''
              END ||
              CASE WHEN is_nullable = 'NO' THEN ' NOT NULL' ELSE '' END,
              ', '
              ORDER BY ordinal_position
            ) || ');' as create_statement
           FROM information_schema.columns
           WHERE table_schema = $1 AND table_name = $2
           GROUP BY table_schema, table_name`,
          [schema, table_name]
        );
        
        if (createTable.rows[0]) {
          sqlDump += `-- Table: ${schema}.${table_name}\n`;
          sqlDump += `DROP TABLE IF EXISTS ${schema}.${table_name} CASCADE;\n`;
          sqlDump += `${createTable.rows[0].create_statement}\n\n`;
        }
        
        // Pobierz dane z tabeli
        const data = await pool.query(`SELECT * FROM ${schema}.${table_name}`);
        
        if (data.rows.length > 0) {
          sqlDump += `-- Data for: ${schema}.${table_name}\n`;
          
          const columns = Object.keys(data.rows[0]);
          const columnList = columns.join(', ');
          
          for (const row of data.rows) {
            const values = columns.map(col => {
              const val = row[col];
              if (val === null) return 'NULL';
              if (typeof val === 'string') {
                return `'${val.replace(/'/g, "''")}'`;
              }
              if (typeof val === 'object') {
                return `'${JSON.stringify(val).replace(/'/g, "''")}'`;
              }
              return val;
            }).join(', ');
            
            sqlDump += `INSERT INTO ${schema}.${table_name} (${columnList}) VALUES (${values});\n`;
          }
          
          sqlDump += `\n`;
        }
      }
    }
    
    // Zapisz do pliku
    await fs.writeFile(backupFile, sqlDump, 'utf-8');
    
    const stats = await fs.stat(backupFile);
    const fileSizeMB = (stats.size / (1024 * 1024)).toFixed(2);
    
    console.log('\n✅ Backup zakończony pomyślnie!');
    console.log(`📁 Plik: ${backupFile}`);
    console.log(`📊 Rozmiar: ${fileSizeMB} MB`);
    
    // Pokaż listę wszystkich backupów
    const files = await fs.readdir(backupDir);
    const backupFiles = files.filter(f => f.startsWith('backup-') && f.endsWith('.sql'));
    
    console.log(`\n📚 Wszystkie backupy (${backupFiles.length}):`);
    for (const file of backupFiles.sort().reverse()) {
      const filePath = path.join(backupDir, file);
      const fileStats = await fs.stat(filePath);
      const sizeMB = (fileStats.size / (1024 * 1024)).toFixed(2);
      console.log(`  - ${file} (${sizeMB} MB)`);
    }
    
  } catch (error) {
    console.error('❌ Błąd podczas tworzenia backupu:', error);
    throw error;
  } finally {
    await pool.end();
  }
}

backupDatabase().catch(console.error);
