import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { Plus, Pencil, Trash2, Image as ImageIcon } from "lucide-react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

type MaterialGroup = {
  id: number;
  name: string;
  code: string;
  category: string | null;
  description: string | null;
  displayOrder: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
};

type Material = {
  id: number;
  groupId: number | null;
  name: string;
  internalCode: string;
  supplierCode: string | null;
  description: string | null;
  specifications: any;
  unitOfMeasure: string;
  gallery: string[];
  primaryImageUrl: string | null;
  displayOrder: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
  groupName?: string;
  groupCode?: string;
};

export default function OkuciaPage() {
  const { toast } = useToast();
  const [selectedGroupId, setSelectedGroupId] = useState<number | null>(null);
  const [isGroupDialogOpen, setIsGroupDialogOpen] = useState(false);
  const [isMaterialDialogOpen, setIsMaterialDialogOpen] = useState(false);
  const [editingGroup, setEditingGroup] = useState<MaterialGroup | null>(null);
  const [editingMaterial, setEditingMaterial] = useState<Material | null>(null);
  
  // Form state for groups
  const [groupForm, setGroupForm] = useState({
    name: "",
    code: "",
    description: "",
    displayOrder: 0,
  });
  
  // Form state for materials
  const [materialForm, setMaterialForm] = useState({
    groupId: null as number | null,
    name: "",
    internalCode: "",
    supplierCode: "",
    description: "",
    unitOfMeasure: "szt",
    displayOrder: 0,
  });

  // Fetch groups (filtered by category 'okucia')
  const { data: groups = [], isLoading: groupsLoading } = useQuery<MaterialGroup[]>({
    queryKey: ["/api/warehouse/material-groups"],
  });

  // Filter groups by 'okucia' category
  const okuciaGroups = groups.filter(g => g.category === 'okucia');

  // Fetch materials
  const { data: materials = [], isLoading: materialsLoading } = useQuery<Material[]>({
    queryKey: ["/api/warehouse/materials", { category: 'okucia' }],
  });

  // Filter materials by selected group
  const filteredMaterials = selectedGroupId
    ? materials.filter((m) => m.groupId === selectedGroupId)
    : materials;

  // Create/Update Group Mutation
  const groupMutation = useMutation({
    mutationFn: async (data: Partial<MaterialGroup>) => {
      if (editingGroup) {
        return apiRequest("PATCH", `/api/warehouse/material-groups/${editingGroup.id}`, data);
      } else {
        return apiRequest("POST", "/api/warehouse/material-groups", { ...data, category: 'okucia' });
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      setIsGroupDialogOpen(false);
      setEditingGroup(null);
      resetGroupForm();
      toast({
        title: "Sukces",
        description: editingGroup ? "Grupa zaktualizowana" : "Grupa utworzona",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zapisać grupy",
        variant: "destructive",
      });
    },
  });

  // Delete Group Mutation
  const deleteGroupMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/warehouse/material-groups/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      toast({ title: "Sukces", description: "Grupa usunięta" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć grupy",
        variant: "destructive",
      });
    },
  });

  // Create/Update Material Mutation
  const materialMutation = useMutation({
    mutationFn: async (data: Partial<Material>) => {
      if (editingMaterial) {
        return apiRequest("PATCH", `/api/warehouse/materials/${editingMaterial.id}`, data);
      } else {
        return apiRequest("POST", "/api/warehouse/materials", data);
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/materials", { category: 'okucia' }] });
      setIsMaterialDialogOpen(false);
      setEditingMaterial(null);
      resetMaterialForm();
      toast({
        title: "Sukces",
        description: editingMaterial ? "Materiał zaktualizowany" : "Materiał utworzony",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zapisać materiału",
        variant: "destructive",
      });
    },
  });

  // Delete Material Mutation
  const deleteMaterialMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/warehouse/materials/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/materials", { category: 'okucia' }] });
      toast({ title: "Sukces", description: "Materiał usunięty" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć materiału",
        variant: "destructive",
      });
    },
  });

  const resetGroupForm = () => {
    setGroupForm({
      name: "",
      code: "",
      description: "",
      displayOrder: 0,
    });
  };

  const resetMaterialForm = () => {
    setMaterialForm({
      groupId: null,
      name: "",
      internalCode: "",
      supplierCode: "",
      description: "",
      unitOfMeasure: "szt",
      displayOrder: 0,
    });
  };

  const openGroupDialog = (group?: MaterialGroup) => {
    if (group) {
      setEditingGroup(group);
      setGroupForm({
        name: group.name,
        code: group.code,
        description: group.description || "",
        displayOrder: group.displayOrder,
      });
    } else {
      setEditingGroup(null);
      resetGroupForm();
    }
    setIsGroupDialogOpen(true);
  };

  const openMaterialDialog = (material?: Material) => {
    if (material) {
      setEditingMaterial(material);
      setMaterialForm({
        groupId: material.groupId,
        name: material.name,
        internalCode: material.internalCode,
        supplierCode: material.supplierCode || "",
        description: material.description || "",
        unitOfMeasure: material.unitOfMeasure,
        displayOrder: material.displayOrder,
      });
    } else {
      setEditingMaterial(null);
      resetMaterialForm();
    }
    setIsMaterialDialogOpen(true);
  };

  const handleGroupSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    groupMutation.mutate(groupForm);
  };

  const handleMaterialSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    materialMutation.mutate(materialForm);
  };

  if (groupsLoading || materialsLoading) {
    return <div className="p-6">Ładowanie...</div>;
  }

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-3xl font-bold">Magazyn - Okucia</h1>
        <div className="flex gap-2">
          <Button
            data-testid="button-add-group"
            onClick={() => openGroupDialog()}
          >
            <Plus className="w-4 h-4 mr-2" />
            Dodaj grupę
          </Button>
          <Button
            data-testid="button-add-material"
            onClick={() => openMaterialDialog()}
            disabled={okuciaGroups.length === 0}
          >
            <Plus className="w-4 h-4 mr-2" />
            Dodaj materiał
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-12 gap-6">
        {/* Grupy - Lewa kolumna */}
        <div className="lg:col-span-3">
          <Card>
            <CardHeader>
              <CardTitle>Grupy okuć</CardTitle>
            </CardHeader>
            <CardContent className="space-y-2">
              <Button
                data-testid="button-all-groups"
                variant={selectedGroupId === null ? "default" : "outline"}
                className="w-full justify-start"
                onClick={() => setSelectedGroupId(null)}
              >
                Wszystkie ({materials.length})
              </Button>
              {okuciaGroups.map((group) => {
                const groupMaterialsCount = materials.filter(m => m.groupId === group.id).length;
                return (
                  <div
                    key={group.id}
                    className="flex items-center gap-2"
                  >
                    <Button
                      data-testid={`button-group-${group.id}`}
                      variant={selectedGroupId === group.id ? "default" : "outline"}
                      className="flex-1 justify-start"
                      onClick={() => setSelectedGroupId(group.id)}
                    >
                      {group.name} ({groupMaterialsCount})
                    </Button>
                    <Button
                      data-testid={`button-edit-group-${group.id}`}
                      size="icon"
                      variant="ghost"
                      onClick={() => openGroupDialog(group)}
                    >
                      <Pencil className="w-4 h-4" />
                    </Button>
                    <Button
                      data-testid={`button-delete-group-${group.id}`}
                      size="icon"
                      variant="ghost"
                      onClick={() => {
                        if (confirm("Czy na pewno chcesz usunąć tę grupę?")) {
                          deleteGroupMutation.mutate(group.id);
                        }
                      }}
                    >
                      <Trash2 className="w-4 h-4" />
                    </Button>
                  </div>
                );
              })}
            </CardContent>
          </Card>
        </div>

        {/* Materiały - Prawa kolumna */}
        <div className="lg:col-span-9">
          <Card>
            <CardHeader>
              <CardTitle>
                {selectedGroupId
                  ? `Materiały - ${okuciaGroups.find(g => g.id === selectedGroupId)?.name}`
                  : "Wszystkie materiały"}
              </CardTitle>
            </CardHeader>
            <CardContent>
              {filteredMaterials.length === 0 ? (
                <div className="text-center text-muted-foreground py-8">
                  Brak materiałów w tej grupie
                </div>
              ) : (
                <div className="space-y-4">
                  {filteredMaterials.map((material) => (
                    <Card key={material.id}>
                      <CardContent className="p-4">
                        <div className="flex items-start justify-between gap-4">
                          <div className="flex gap-4 flex-1">
                            {material.primaryImageUrl && (
                              <div className="w-24 h-24 bg-muted rounded flex-shrink-0 flex items-center justify-center overflow-hidden">
                                <img
                                  src={material.primaryImageUrl}
                                  alt={material.name}
                                  className="w-full h-full object-cover"
                                />
                              </div>
                            )}
                            {!material.primaryImageUrl && (
                              <div className="w-24 h-24 bg-muted rounded flex-shrink-0 flex items-center justify-center">
                                <ImageIcon className="w-8 h-8 text-muted-foreground" />
                              </div>
                            )}
                            <div className="flex-1">
                              <div className="flex items-start justify-between">
                                <div>
                                  <h3 className="font-semibold text-lg" data-testid={`text-material-name-${material.id}`}>
                                    {material.name}
                                  </h3>
                                  <p className="text-sm text-muted-foreground">
                                    Kod wewnętrzny: {material.internalCode}
                                  </p>
                                  {material.supplierCode && (
                                    <p className="text-sm text-muted-foreground">
                                      Kod dostawcy: {material.supplierCode}
                                    </p>
                                  )}
                                  {material.groupName && (
                                    <p className="text-sm text-muted-foreground">
                                      Grupa: {material.groupName}
                                    </p>
                                  )}
                                  <p className="text-sm text-muted-foreground">
                                    Jednostka: {material.unitOfMeasure}
                                  </p>
                                  {material.description && (
                                    <p className="text-sm mt-2">{material.description}</p>
                                  )}
                                </div>
                              </div>
                            </div>
                          </div>
                          <div className="flex gap-2">
                            <Button
                              data-testid={`button-edit-material-${material.id}`}
                              size="icon"
                              variant="ghost"
                              onClick={() => openMaterialDialog(material)}
                            >
                              <Pencil className="w-4 h-4" />
                            </Button>
                            <Button
                              data-testid={`button-delete-material-${material.id}`}
                              size="icon"
                              variant="ghost"
                              onClick={() => {
                                if (confirm("Czy na pewno chcesz usunąć ten materiał?")) {
                                  deleteMaterialMutation.mutate(material.id);
                                }
                              }}
                            >
                              <Trash2 className="w-4 h-4" />
                            </Button>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>

      {/* Group Dialog */}
      <Dialog open={isGroupDialogOpen} onOpenChange={setIsGroupDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>
              {editingGroup ? "Edytuj grupę" : "Dodaj grupę"}
            </DialogTitle>
            <DialogDescription>
              Uzupełnij dane grupy okuć
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleGroupSubmit}>
            <div className="space-y-4 py-4">
              <div>
                <Label htmlFor="group-name">Nazwa *</Label>
                <Input
                  id="group-name"
                  data-testid="input-group-name"
                  value={groupForm.name}
                  onChange={(e) => setGroupForm({ ...groupForm, name: e.target.value })}
                  required
                />
              </div>
              <div>
                <Label htmlFor="group-code">Kod *</Label>
                <Input
                  id="group-code"
                  data-testid="input-group-code"
                  value={groupForm.code}
                  onChange={(e) => setGroupForm({ ...groupForm, code: e.target.value })}
                  required
                />
              </div>
              <div>
                <Label htmlFor="group-description">Opis</Label>
                <Textarea
                  id="group-description"
                  data-testid="input-group-description"
                  value={groupForm.description}
                  onChange={(e) => setGroupForm({ ...groupForm, description: e.target.value })}
                />
              </div>
              <div>
                <Label htmlFor="group-display-order">Kolejność wyświetlania</Label>
                <Input
                  id="group-display-order"
                  data-testid="input-group-display-order"
                  type="number"
                  value={groupForm.displayOrder}
                  onChange={(e) => setGroupForm({ ...groupForm, displayOrder: parseInt(e.target.value) || 0 })}
                />
              </div>
            </div>
            <DialogFooter>
              <Button
                type="button"
                variant="outline"
                onClick={() => setIsGroupDialogOpen(false)}
                data-testid="button-cancel-group"
              >
                Anuluj
              </Button>
              <Button type="submit" data-testid="button-save-group">
                Zapisz
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      {/* Material Dialog */}
      <Dialog open={isMaterialDialogOpen} onOpenChange={setIsMaterialDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>
              {editingMaterial ? "Edytuj materiał" : "Dodaj materiał"}
            </DialogTitle>
            <DialogDescription>
              Uzupełnij dane materiału
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleMaterialSubmit}>
            <div className="space-y-4 py-4">
              <div>
                <Label htmlFor="material-group">Grupa *</Label>
                <Select
                  value={materialForm.groupId?.toString() || ""}
                  onValueChange={(value) => setMaterialForm({ ...materialForm, groupId: parseInt(value) })}
                  required
                >
                  <SelectTrigger data-testid="select-material-group">
                    <SelectValue placeholder="Wybierz grupę" />
                  </SelectTrigger>
                  <SelectContent>
                    {okuciaGroups.map((group) => (
                      <SelectItem key={group.id} value={group.id.toString()}>
                        {group.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="material-name">Nazwa *</Label>
                <Input
                  id="material-name"
                  data-testid="input-material-name"
                  value={materialForm.name}
                  onChange={(e) => setMaterialForm({ ...materialForm, name: e.target.value })}
                  required
                />
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="material-internal-code">Kod wewnętrzny *</Label>
                  <Input
                    id="material-internal-code"
                    data-testid="input-material-internal-code"
                    value={materialForm.internalCode}
                    onChange={(e) => setMaterialForm({ ...materialForm, internalCode: e.target.value })}
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="material-supplier-code">Kod dostawcy</Label>
                  <Input
                    id="material-supplier-code"
                    data-testid="input-material-supplier-code"
                    value={materialForm.supplierCode}
                    onChange={(e) => setMaterialForm({ ...materialForm, supplierCode: e.target.value })}
                  />
                </div>
              </div>
              <div>
                <Label htmlFor="material-description">Opis</Label>
                <Textarea
                  id="material-description"
                  data-testid="input-material-description"
                  value={materialForm.description}
                  onChange={(e) => setMaterialForm({ ...materialForm, description: e.target.value })}
                />
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="material-unit">Jednostka miary</Label>
                  <Input
                    id="material-unit"
                    data-testid="input-material-unit"
                    value={materialForm.unitOfMeasure}
                    onChange={(e) => setMaterialForm({ ...materialForm, unitOfMeasure: e.target.value })}
                  />
                </div>
                <div>
                  <Label htmlFor="material-display-order">Kolejność wyświetlania</Label>
                  <Input
                    id="material-display-order"
                    data-testid="input-material-display-order"
                    type="number"
                    value={materialForm.displayOrder}
                    onChange={(e) => setMaterialForm({ ...materialForm, displayOrder: parseInt(e.target.value) || 0 })}
                  />
                </div>
              </div>
            </div>
            <DialogFooter>
              <Button
                type="button"
                variant="outline"
                onClick={() => setIsMaterialDialogOpen(false)}
                data-testid="button-cancel-material"
              >
                Anuluj
              </Button>
              <Button type="submit" data-testid="button-save-material">
                Zapisz
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
