import { useState, useEffect, useCallback } from "react";
import { useRoute } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useToast } from "@/hooks/use-toast";
import { Search, X, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, Loader2, Download, Trash2, LayoutGrid, List, AlignJustify, Plus, FolderPlus, Upload, Edit, Image as ImageIcon, Star, Copy, MoreVertical } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger, DialogFooter } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { CSVImportDialog } from "@/components/csv-import-dialog";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger, DropdownMenuSeparator } from "@/components/ui/dropdown-menu";
import { useDropzone } from "react-dropzone";
import { GenerationLogsDialog } from "@/components/generation-logs-dialog";
import { CompactListItem } from "@/components/compact-list-item";

interface Material {
  id: number;
  groupId: number | null;
  name: string;
  internalCode: string;
  supplierCode: string | null;
  description: string | null;
  specifications: any;
  unitOfMeasure: string;
  price?: string | null;
  gallery: string[];
  primaryImageUrl: string | null;
  displayOrder: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
  groupName?: string;
  groupCode?: string;
  groupCategory?: string;
}

interface SearchResponse {
  materials: Material[];
  pagination: {
    page: number;
    limit: number;
    total: number;
    totalPages: number;
    offset: number; // Globalny offset dla numeracji
  };
}

const CATEGORY_LABELS: Record<string, string> = {
  okucia: "Okucia",
  plyty_meblowe: "Płyty meblowe",
  sruby: "Śruby",
  tkaniny: "Tkaniny",
  pianki: "Pianki",
};

const STORAGE_KEY_PREFIX = 'warehouse-filters-';

type ViewMode = 'list' | 'compact' | 'grid';

// Helper function to generate slug from name
function generateSlug(name: string): string {
  // Map Polish characters to ASCII equivalents
  const polishMap: Record<string, string> = {
    'ą': 'a', 'ć': 'c', 'ę': 'e', 'ł': 'l', 'ń': 'n',
    'ó': 'o', 'ś': 's', 'ź': 'z', 'ż': 'z',
    'Ą': 'A', 'Ć': 'C', 'Ę': 'E', 'Ł': 'L', 'Ń': 'N',
    'Ó': 'O', 'Ś': 'S', 'Ź': 'Z', 'Ż': 'Z'
  };
  
  return name
    .split('')
    .map(char => polishMap[char] || char)
    .join('')
    .toLowerCase()
    .replace(/[^a-z0-9]+/g, '_')
    .replace(/^_+|_+$/g, '')
    .replace(/_+/g, '_');
}

export default function WarehouseCategoryPage() {
  const { toast } = useToast();
  const [, params] = useRoute("/warehouse/:category/:groupCode?");
  const category = params?.category || 'okucia';
  const groupCode = params?.groupCode || null;

  // Load filters from localStorage
  const loadFilters = () => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY_PREFIX + category);
      if (saved) {
        return JSON.parse(saved);
      }
    } catch (error) {
      console.error('Error loading filters from localStorage:', error);
    }
    return {};
  };

  const savedFilters = loadFilters();

  const [searchQuery, setSearchQuery] = useState(savedFilters.searchQuery || "");
  const [page, setPage] = useState(savedFilters.page || 1);
  const [limit, setLimit] = useState(savedFilters.limit || 25);
  const [sortBy, setSortBy] = useState(savedFilters.sortBy || 'name');
  const [sortOrder, setSortOrder] = useState<'ASC' | 'DESC'>(savedFilters.sortOrder || 'ASC');
  const [viewMode, setViewMode] = useState<ViewMode>(savedFilters.viewMode || 'list');
  const [selectedMaterials, setSelectedMaterials] = useState<number[]>([]);
  
  // Dialog states
  const [isAddGroupDialogOpen, setIsAddGroupDialogOpen] = useState(false);
  const [isEditGroupDialogOpen, setIsEditGroupDialogOpen] = useState(false);
  const [isAddMaterialDialogOpen, setIsAddMaterialDialogOpen] = useState(false);
  const [isEditMaterialDialogOpen, setIsEditMaterialDialogOpen] = useState(false);
  
  // Form states for new group
  const [newGroupName, setNewGroupName] = useState("");
  const [newGroupCode, setNewGroupCode] = useState("");
  const [newGroupDescription, setNewGroupDescription] = useState("");
  
  // Form states for edit group
  const [editingGroup, setEditingGroup] = useState<any | null>(null);
  const [editGroupName, setEditGroupName] = useState("");
  const [editGroupCode, setEditGroupCode] = useState("");
  const [editGroupDescription, setEditGroupDescription] = useState("");
  const [editGroupDisplayOrder, setEditGroupDisplayOrder] = useState("0");
  
  // Form states for new material
  const [newMaterialName, setNewMaterialName] = useState("");
  const [newMaterialInternalCode, setNewMaterialInternalCode] = useState("");
  const [newMaterialSupplierCode, setNewMaterialSupplierCode] = useState("");
  const [newMaterialDescription, setNewMaterialDescription] = useState("");
  const [newMaterialUnitOfMeasure, setNewMaterialUnitOfMeasure] = useState("szt");
  const [newMaterialPrice, setNewMaterialPrice] = useState("");
  const [newMaterialGroupId, setNewMaterialGroupId] = useState<number | null>(null);
  const [isInternalCodeManuallyEdited, setIsInternalCodeManuallyEdited] = useState(false);
  
  // Form states for edit material
  const [editingMaterial, setEditingMaterial] = useState<Material | null>(null);
  const [editMaterialName, setEditMaterialName] = useState("");
  const [editMaterialInternalCode, setEditMaterialInternalCode] = useState("");
  const [editMaterialSupplierCode, setEditMaterialSupplierCode] = useState("");
  const [editMaterialDescription, setEditMaterialDescription] = useState("");
  const [editMaterialUnitOfMeasure, setEditMaterialUnitOfMeasure] = useState("szt");
  const [editMaterialPrice, setEditMaterialPrice] = useState("");
  const [editMaterialGroupId, setEditMaterialGroupId] = useState<number | null>(null);
  const [editMaterialGallery, setEditMaterialGallery] = useState<string[]>([]);
  const [isEditInternalCodeManuallyEdited, setIsEditInternalCodeManuallyEdited] = useState(true);
  const [isDragging, setIsDragging] = useState(false);
  const [uploadHoverMaterials, setUploadHoverMaterials] = useState<Record<number, boolean>>({});
  const [uploadingMaterials, setUploadingMaterials] = useState<Record<number, boolean>>({});
  
  // Duplicate material states
  const [duplicateMaterial, setDuplicateMaterial] = useState<Material | null>(null);
  const [duplicateTargetGroupId, setDuplicateTargetGroupId] = useState<number | null>(null);
  const [showDuplicateDialog, setShowDuplicateDialog] = useState(false);
  
  // Bulk edit states
  const [showBulkEditPriceDialog, setShowBulkEditPriceDialog] = useState(false);
  const [showBulkEditUnitDialog, setShowBulkEditUnitDialog] = useState(false);
  const [bulkEditPrice, setBulkEditPrice] = useState("");
  const [bulkEditUnit, setBulkEditUnit] = useState("szt");
  
  // Bulk import states
  const [bulkImportFiles, setBulkImportFiles] = useState<File[]>([]);
  const [bulkImportSessionId, setBulkImportSessionId] = useState<string>('');
  const [showBulkImportLogs, setShowBulkImportLogs] = useState(false);
  
  // Auto-generate internal code from name
  useEffect(() => {
    if (!isInternalCodeManuallyEdited && newMaterialName.trim()) {
      const generatedCode = generateSlug(newMaterialName);
      setNewMaterialInternalCode(generatedCode);
    }
  }, [newMaterialName, isInternalCodeManuallyEdited]);

  // Save filters to localStorage
  useEffect(() => {
    const filters = {
      searchQuery,
      page,
      limit,
      sortBy,
      sortOrder,
      viewMode,
    };
    localStorage.setItem(STORAGE_KEY_PREFIX + category, JSON.stringify(filters));
  }, [searchQuery, page, limit, sortBy, sortOrder, viewMode, category]);

  // Reset page when search or groupCode changes
  useEffect(() => {
    setPage(1);
    setSelectedMaterials([]);
  }, [searchQuery, groupCode]);

  // Build query parameters
  const queryParams: Record<string, string> = {
    category,
    page: page.toString(),
    limit: limit.toString(),
    sortBy,
    sortOrder,
  };

  if (groupCode) {
    queryParams.groupCode = groupCode;
  }

  if (searchQuery) {
    queryParams.search = searchQuery;
  }

  const queryString = new URLSearchParams(queryParams).toString();

  const { data, isLoading } = useQuery<SearchResponse>({
    queryKey: [`/api/warehouse/materials/search?${queryString}`],
  });

  const materials = data?.materials || [];
  const pagination = data?.pagination;

  // Load material groups for the current category
  const { data: allGroups = [] } = useQuery<any[]>({
    queryKey: ["/api/warehouse/material-groups"],
  });

  const categoryGroups = allGroups
    .filter(g => g.category === category && g.isActive)
    .sort((a, b) => {
      if (a.displayOrder !== b.displayOrder) {
        return a.displayOrder - b.displayOrder;
      }
      return a.name.localeCompare(b.name);
    });

  // Load units of measure from dictionaries
  const { data: units = [] } = useQuery<any[]>({
    queryKey: ["/api/dictionaries?type=unit"],
  });

  // Parse search query into AND groups (separated by semicolons)
  // Each AND group may contain OR terms (separated by commas)
  const getSearchGroups = () => {
    if (!searchQuery) return [];
    const groups: string[] = [];
    searchQuery.split(';').forEach((group: string) => {
      const trimmed = group.trim();
      if (trimmed) groups.push(trimmed);
    });
    return groups;
  };

  const searchGroups = getSearchGroups();

  // Remove a specific AND group
  const removeSearchGroup = (groupToRemove: string) => {
    const updatedGroups = searchGroups.filter(g => g !== groupToRemove);
    setSearchQuery(updatedGroups.join('; '));
  };

  const clearAllFilters = () => {
    setSearchQuery("");
  };

  // Bulk delete mutation
  const bulkDeleteMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", "/api/warehouse/materials/bulk-delete", {
        materialIds: selectedMaterials,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Materiały usunięte",
        description: `Pomyślnie usunięto ${selectedMaterials.length} materiałów.`,
      });
      setSelectedMaterials([]);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć materiałów.",
        variant: "destructive",
      });
    },
  });

  // Add material group mutation
  const addGroupMutation = useMutation({
    mutationFn: async (data: { name: string; code: string; category: string; description?: string }) => {
      const response = await apiRequest("POST", "/api/warehouse/material-groups", data);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      toast({
        title: "Grupa utworzona",
        description: "Grupa materiałów została pomyślnie utworzona.",
      });
      setIsAddGroupDialogOpen(false);
      setNewGroupName("");
      setNewGroupCode("");
      setNewGroupDescription("");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia grupy",
        description: error.message || "Nie udało się utworzyć grupy materiałów.",
        variant: "destructive",
      });
    },
  });

  // Edit material group mutation
  const editGroupMutation = useMutation({
    mutationFn: async (data: { id: number; name: string; code: string; description?: string; displayOrder: number }) => {
      const response = await apiRequest("PATCH", `/api/warehouse/material-groups/${data.id}`, data);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      toast({
        title: "Grupa zaktualizowana",
        description: "Grupa materiałów została pomyślnie zaktualizowana.",
      });
      setIsEditGroupDialogOpen(false);
      setEditingGroup(null);
      setEditGroupName("");
      setEditGroupCode("");
      setEditGroupDescription("");
      setEditGroupDisplayOrder("0");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji grupy",
        description: error.message || "Nie udało się zaktualizować grupy materiałów.",
        variant: "destructive",
      });
    },
  });

  // Add material mutation
  const addMaterialMutation = useMutation({
    mutationFn: async (data: { 
      name: string; 
      internalCode: string;
      supplierCode?: string;
      description?: string;
      unitOfMeasure: string;
      price?: number;
      groupId?: number;
    }) => {
      const response = await apiRequest("POST", "/api/warehouse/materials", data);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/warehouse/materials/search?${queryString}`] });
      toast({
        title: "Materiał utworzony",
        description: "Materiał został pomyślnie utworzony.",
      });
      setIsAddMaterialDialogOpen(false);
      setNewMaterialName("");
      setNewMaterialInternalCode("");
      setNewMaterialSupplierCode("");
      setNewMaterialDescription("");
      setNewMaterialUnitOfMeasure("szt");
      setNewMaterialPrice("");
      setNewMaterialGroupId(null);
      setIsInternalCodeManuallyEdited(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia materiału",
        description: error.message || "Nie udało się utworzyć materiału.",
        variant: "destructive",
      });
    },
  });

  // Edit material mutation
  const editMaterialMutation = useMutation({
    mutationFn: async (data: { 
      id: number;
      groupId?: number | null;
      name?: string; 
      internalCode?: string;
      supplierCode?: string;
      description?: string;
      unitOfMeasure?: string;
      price?: number | null;
      gallery?: string[];
    }) => {
      const response = await apiRequest("PATCH", `/api/warehouse/materials/${data.id}`, data);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Materiał zaktualizowany",
        description: "Materiał został pomyślnie zaktualizowany.",
      });
      setIsEditMaterialDialogOpen(false);
      setEditingMaterial(null);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji materiału",
        description: error.message || "Nie udało się zaktualizować materiału.",
        variant: "destructive",
      });
    },
  });

  // Duplicate material mutation
  const duplicateMaterialMutation = useMutation({
    mutationFn: async (data: { id: number; targetGroupId: number }) => {
      const response = await apiRequest("POST", `/api/warehouse/materials/${data.id}/duplicate`, {
        targetGroupId: data.targetGroupId,
      });
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Materiał zduplikowany",
        description: "Materiał został pomyślnie zduplikowany.",
      });
      setShowDuplicateDialog(false);
      setDuplicateMaterial(null);
      setDuplicateTargetGroupId(null);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd duplikacji materiału",
        description: error.message || "Nie udało się zduplikować materiału.",
        variant: "destructive",
      });
    },
  });

  // Upload material image mutation
  const uploadMaterialImageMutation = useMutation({
    mutationFn: async ({ materialId, file }: { materialId: number; file: File }) => {
      const formData = new FormData();
      formData.append('image', file);
      
      const response = await fetch(`/api/warehouse/materials/${materialId}/images`, {
        method: 'POST',
        body: formData,
      });
      
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Upload failed');
      }
      
      return await response.json();
    },
    onMutate: ({ materialId }) => {
      setUploadingMaterials(prev => ({ ...prev, [materialId]: true }));
    },
    onSuccess: (data, { materialId }) => {
      if (editingMaterial && data.material) {
        setEditMaterialGallery(data.material.gallery || []);
      }
      // Invalidate cache to refresh material list
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Zdjęcie dodane",
        description: "Zdjęcie zostało pomyślnie dodane.",
      });
      setUploadingMaterials(prev => ({ ...prev, [materialId]: false }));
    },
    onError: (error: any, { materialId }) => {
      toast({
        title: "Błąd uploadu",
        description: error.message || "Nie udało się dodać zdjęcia.",
        variant: "destructive",
      });
      setUploadingMaterials(prev => ({ ...prev, [materialId]: false }));
    },
  });

  // Delete material image mutation
  const deleteMaterialImageMutation = useMutation({
    mutationFn: async ({ materialId, imageUrl }: { materialId: number; imageUrl: string }) => {
      const response = await apiRequest("DELETE", `/api/warehouse/materials/${materialId}/images`, { imageUrl });
      return await response.json();
    },
    onSuccess: (data) => {
      if (editingMaterial && data.material) {
        setEditMaterialGallery(data.material.gallery || []);
      }
      // Invalidate cache to refresh material list
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Zdjęcie usunięte",
        description: "Zdjęcie zostało pomyślnie usunięte.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć zdjęcia.",
        variant: "destructive",
      });
    },
  });

  // Set primary material image mutation
  const setPrimaryImageMutation = useMutation({
    mutationFn: async ({ materialId, imageUrl }: { materialId: number; imageUrl: string }) => {
      const response = await apiRequest("PATCH", `/api/warehouse/materials/${materialId}/primary-image`, { imageUrl });
      return await response.json();
    },
    onSuccess: (data) => {
      // Update local dialog state with new primary image and gallery
      if (editingMaterial && data.material) {
        setEditingMaterial({
          ...editingMaterial,
          primaryImageUrl: data.material.primaryImageUrl,
          gallery: data.material.gallery,
        });
        // Also update gallery state for consistency
        setEditMaterialGallery(data.material.gallery || []);
      }
      // Invalidate cache to refresh material list
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Główne zdjęcie ustawione",
        description: "Główne zdjęcie zostało pomyślnie ustawione.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd ustawiania zdjęcia",
        description: error.message || "Nie udało się ustawić głównego zdjęcia.",
        variant: "destructive",
      });
    },
  });

  // Export mutation
  const exportMutation = useMutation({
    mutationFn: async () => {
      const response = await fetch("/api/warehouse/materials/export", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          materialIds: selectedMaterials,
          format: "csv",
        }),
      });

      if (!response.ok) {
        throw new Error("Export failed");
      }

      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = `materialy-${category}-${new Date().toISOString().split('T')[0]}.csv`;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);
    },
    onSuccess: () => {
      toast({
        title: "Eksport zakończony",
        description: `Wyeksportowano ${selectedMaterials.length} materiałów.`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd eksportu",
        description: error.message || "Nie udało się wyeksportować materiałów.",
        variant: "destructive",
      });
    },
  });

  // Bulk update price mutation
  const bulkUpdatePriceMutation = useMutation({
    mutationFn: async (price: string) => {
      const response = await apiRequest("POST", "/api/warehouse/materials/bulk-update", {
        materialIds: selectedMaterials,
        updates: { price }
      });
      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Ceny zaktualizowane",
        description: `Zaktualizowano cenę dla ${data.updatedCount} materiałów.`,
      });
      setSelectedMaterials([]);
      setShowBulkEditPriceDialog(false);
      setBulkEditPrice("");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować cen.",
        variant: "destructive",
      });
    },
  });

  // Bulk update unit mutation
  const bulkUpdateUnitMutation = useMutation({
    mutationFn: async (unit: string) => {
      const response = await apiRequest("POST", "/api/warehouse/materials/bulk-update", {
        materialIds: selectedMaterials,
        updates: { unit_of_measure: unit }
      });
      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      toast({
        title: "Jednostki zaktualizowane",
        description: `Zaktualizowano jednostkę miary dla ${data.updatedCount} materiałów.`,
      });
      setSelectedMaterials([]);
      setShowBulkEditUnitDialog(false);
      setBulkEditUnit("szt");
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować jednostek.",
        variant: "destructive",
      });
    },
  });

  // Bulk import mutation
  const bulkImportMutation = useMutation({
    mutationFn: async (sessionId: string) => {
      const formData = new FormData();
      bulkImportFiles.forEach((file) => {
        formData.append('images', file);
      });
      
      // Use currently selected group from URL
      if (groupCode) {
        const selectedGroup = categoryGroups.find(g => g.code === groupCode);
        if (selectedGroup) {
          formData.append('groupId', selectedGroup.id.toString());
        }
      }
      
      formData.append('category', category);
      formData.append('sessionId', sessionId);

      const response = await fetch('/api/warehouse/materials/bulk-import', {
        method: 'POST',
        body: formData,
        credentials: 'include',
      });

      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(errorText || response.statusText);
      }

      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({
        predicate: (query) => {
          const key = query.queryKey[0];
          return typeof key === 'string' && key.includes('/api/warehouse/materials/search');
        }
      });
      setBulkImportFiles([]);
      toast({
        title: "Sukces",
        description: `Zaimportowano ${data.created} materiałów${data.errors > 0 ? `, ${data.errors} błędów` : ''}`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaimportować materiałów",
        variant: "destructive",
      });
    },
  });

  // Drag & drop for bulk import
  const onDropBulk = useCallback((acceptedFiles: File[]) => {
    console.log('📦 Dropped files:', acceptedFiles.length, acceptedFiles.map(f => f.name));
    setBulkImportFiles(prev => [...prev, ...acceptedFiles]);
  }, []);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop: onDropBulk,
    accept: {
      'image/*': ['.png', '.jpg', '.jpeg', '.gif', '.webp']
    },
    multiple: true
  });

  const handleBulkImport = () => {
    console.log('🚀 Bulk import clicked, files:', bulkImportFiles.length);
    console.log('📍 Current category:', category, 'groupCode:', groupCode);
    if (bulkImportFiles.length === 0) {
      toast({
        title: "Błąd",
        description: "Wybierz przynajmniej jedno zdjęcie",
        variant: "destructive",
      });
      return;
    }
    
    // Generate session ID and show progress dialog
    const sessionId = `bulk-import-${Date.now()}-${Math.random().toString(36).substring(7)}`;
    setBulkImportSessionId(sessionId);
    setShowBulkImportLogs(true);
    bulkImportMutation.mutate(sessionId);
  };

  const removeBulkFile = (index: number) => {
    setBulkImportFiles(prev => prev.filter((_, i) => i !== index));
  };

  const handleBulkDelete = () => {
    if (selectedMaterials.length === 0) return;
    if (confirm(`Czy na pewno chcesz usunąć ${selectedMaterials.length} materiałów?`)) {
      bulkDeleteMutation.mutate();
    }
  };

  const handleExport = () => {
    if (selectedMaterials.length === 0) {
      toast({
        title: "Brak zaznaczonych materiałów",
        description: "Zaznacz materiały do eksportu.",
        variant: "destructive",
      });
      return;
    }
    exportMutation.mutate();
  };

  const toggleSelectAll = () => {
    if (selectedMaterials.length === materials.length) {
      setSelectedMaterials([]);
    } else {
      setSelectedMaterials(materials.map(m => m.id));
    }
  };

  const toggleSelectMaterial = (id: number) => {
    setSelectedMaterials(prev =>
      prev.includes(id) ? prev.filter(i => i !== id) : [...prev, id]
    );
  };

  const handleAddGroup = () => {
    if (!newGroupName.trim() || !newGroupCode.trim()) {
      toast({
        title: "Błąd walidacji",
        description: "Nazwa i kod grupy są wymagane.",
        variant: "destructive",
      });
      return;
    }
    
    addGroupMutation.mutate({
      name: newGroupName,
      code: newGroupCode,
      category: category,
      description: newGroupDescription || undefined,
    });
  };

  const handleEditGroup = (group: any) => {
    setEditingGroup(group);
    setEditGroupName(group.name);
    setEditGroupCode(group.code);
    setEditGroupDescription(group.description || "");
    setEditGroupDisplayOrder(group.displayOrder?.toString() || "0");
    setIsEditGroupDialogOpen(true);
  };

  const handleUpdateGroup = () => {
    if (!editGroupName.trim() || !editGroupCode.trim()) {
      toast({
        title: "Błąd walidacji",
        description: "Nazwa i kod grupy są wymagane.",
        variant: "destructive",
      });
      return;
    }
    
    if (!editingGroup) return;
    
    editGroupMutation.mutate({
      id: editingGroup.id,
      name: editGroupName,
      code: editGroupCode,
      description: editGroupDescription || undefined,
      displayOrder: parseInt(editGroupDisplayOrder) || 0,
    });
  };

  const handleAddMaterial = () => {
    if (!newMaterialName.trim() || !newMaterialInternalCode.trim()) {
      toast({
        title: "Błąd walidacji",
        description: "Nazwa i kod wewnętrzny są wymagane.",
        variant: "destructive",
      });
      return;
    }
    
    addMaterialMutation.mutate({
      name: newMaterialName,
      internalCode: newMaterialInternalCode,
      supplierCode: newMaterialSupplierCode || undefined,
      description: newMaterialDescription || undefined,
      unitOfMeasure: newMaterialUnitOfMeasure,
      price: newMaterialPrice ? parseFloat(newMaterialPrice) : undefined,
      groupId: newMaterialGroupId || undefined,
    });
  };

  const handleEditMaterial = (material: Material) => {
    setEditingMaterial(material);
    setEditMaterialName(material.name);
    setEditMaterialInternalCode(material.internalCode);
    setEditMaterialSupplierCode(material.supplierCode || "");
    setEditMaterialDescription(material.description || "");
    setEditMaterialUnitOfMeasure(material.unitOfMeasure);
    setEditMaterialPrice(material.price || "");
    setEditMaterialGroupId(material.groupId);
    setEditMaterialGallery(material.gallery || []);
    setIsEditInternalCodeManuallyEdited(true);
    setIsEditMaterialDialogOpen(true);
  };

  const handleUpdateMaterial = () => {
    if (!editMaterialName.trim() || !editMaterialInternalCode.trim()) {
      toast({
        title: "Błąd walidacji",
        description: "Nazwa i kod wewnętrzny są wymagane.",
        variant: "destructive",
      });
      return;
    }
    
    if (!editingMaterial) return;
    
    editMaterialMutation.mutate({
      id: editingMaterial.id,
      name: editMaterialName,
      internalCode: editMaterialInternalCode,
      supplierCode: editMaterialSupplierCode || undefined,
      description: editMaterialDescription || undefined,
      unitOfMeasure: editMaterialUnitOfMeasure,
      price: editMaterialPrice ? parseFloat(editMaterialPrice) : null,
      groupId: editMaterialGroupId || null,
      gallery: editMaterialGallery,
    });
  };

  const handleDuplicateSameGroup = (material: Material) => {
    if (!material.groupId) {
      toast({
        title: "Błąd duplikacji",
        description: "Materiał musi należeć do grupy.",
        variant: "destructive",
      });
      return;
    }
    
    duplicateMaterialMutation.mutate({
      id: material.id,
      targetGroupId: material.groupId,
    });
  };

  const handleDuplicateToGroup = (material: Material) => {
    setDuplicateMaterial(material);
    setDuplicateTargetGroupId(material.groupId);
    setShowDuplicateDialog(true);
  };

  const handleConfirmDuplicate = () => {
    if (!duplicateMaterial || !duplicateTargetGroupId) {
      toast({
        title: "Błąd walidacji",
        description: "Wybierz grupę docelową.",
        variant: "destructive",
      });
      return;
    }
    
    duplicateMaterialMutation.mutate({
      id: duplicateMaterial.id,
      targetGroupId: duplicateTargetGroupId,
    });
  };

  const handleImageUpload = (files: FileList | null, material?: Material) => {
    const targetMaterial = material || editingMaterial;
    if (!files || files.length === 0 || !targetMaterial) return;
    
    Array.from(files).forEach(file => {
      uploadMaterialImageMutation.mutate({
        materialId: targetMaterial.id,
        file,
      });
    });
  };

  const handleImageDelete = (imageUrl: string) => {
    if (!editingMaterial) return;
    
    deleteMaterialImageMutation.mutate({
      materialId: editingMaterial.id,
      imageUrl,
    });
  };

  const handleSetPrimaryImage = (imageUrl: string) => {
    if (!editingMaterial) return;
    
    setPrimaryImageMutation.mutate({
      materialId: editingMaterial.id,
      imageUrl,
    });
  };

  // Set default group when dialog opens and reset manual edit flag
  useEffect(() => {
    if (isAddMaterialDialogOpen) {
      setIsInternalCodeManuallyEdited(false);
      
      if (groupCode && categoryGroups.length > 0) {
        const currentGroup = categoryGroups.find(g => g.code === groupCode);
        if (currentGroup) {
          setNewMaterialGroupId(currentGroup.id);
        }
      }
    }
  }, [isAddMaterialDialogOpen, groupCode, categoryGroups]);

  return (
    <WarehouseLayout 
      category={category} 
      onEditGroup={handleEditGroup}
      onAddGroup={() => setIsAddGroupDialogOpen(true)}
    >
      <div className="p-6 space-y-6">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <h1 className="text-2xl sm:text-3xl font-bold">
            {CATEGORY_LABELS[category] || category}
            {groupCode && ` - ${materials[0]?.groupName || groupCode}`}
          </h1>
          <div className="flex flex-wrap gap-2">
            {/* Action buttons */}
            {!groupCode && (
              <Button
                variant="outline"
                size="sm"
                onClick={() => setIsAddGroupDialogOpen(true)}
                data-testid="button-add-group"
                className="border-blue-500 text-blue-600 hover:bg-blue-50 dark:border-blue-400 dark:text-blue-400 dark:hover:bg-blue-950"
              >
                <FolderPlus className="w-4 h-4 mr-2" />
                <span className="hidden sm:inline">Dodaj grupę</span>
                <span className="sm:hidden">Grupa</span>
              </Button>
            )}
            <Button
              variant="default"
              size="sm"
              onClick={() => setIsAddMaterialDialogOpen(true)}
              data-testid="button-add-material"
              className="bg-green-600 hover:bg-green-700 dark:bg-green-700 dark:hover:bg-green-800"
            >
              <Plus className="w-4 h-4 mr-2" />
              <span className="hidden sm:inline">{groupCode ? `Dodaj materiał do grupy` : 'Dodaj materiał'}</span>
              <span className="sm:hidden">Materiał</span>
            </Button>
            <CSVImportDialog
              endpoint="/api/warehouse/materials/import-csv"
              templateCsv="name,internal_code,supplier_code,group_code,description,unit_of_measure,display_order\nZawias 35mm,ZAW-35-STD,SUP-123,zawiasy_std,Zawias standardowy 35mm,szt,0"
              templateFilename={`template_materialy_${category}.csv`}
              invalidateQueryKey={[`/api/warehouse/materials/search?${queryString}`]}
              title="Import materiałów z CSV"
              description="Zaimportuj materiały z pliku CSV. Wymagane kolumny: name, internal_code"
            >
              <Button
                variant="outline"
                size="sm"
                data-testid="button-import-csv"
              >
                <Upload className="w-4 h-4 mr-2" />
                <span className="hidden sm:inline">Import CSV</span>
                <span className="sm:hidden">Import</span>
              </Button>
            </CSVImportDialog>
            
            {/* View mode buttons */}
            <div className="flex gap-1 ml-2 border-l pl-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setViewMode('list')}
                data-testid="button-view-list"
                className={viewMode === 'list' ? 'bg-accent' : ''}
              >
                <List className="w-4 h-4" />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setViewMode('compact')}
                data-testid="button-view-compact"
                className={viewMode === 'compact' ? 'bg-accent' : ''}
              >
                <AlignJustify className="w-4 h-4" />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setViewMode('grid')}
                data-testid="button-view-grid"
                className={viewMode === 'grid' ? 'bg-accent' : ''}
              >
                <LayoutGrid className="w-4 h-4" />
              </Button>
            </div>
          </div>
        </div>

        {/* Filters and Bulk Import - side by side */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          {/* Filters */}
          <Card>
            <CardHeader>
              <CardTitle>Filtry</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              {/* Search Input */}
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Szukaj... (użyj ; dla AND, , dla OR)"
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-9"
                  data-testid="input-search"
                />
              </div>

              {/* Active Filters */}
              {searchGroups.length > 0 && (
                <div className="flex flex-wrap gap-2">
                  {searchGroups.map((group, index) => (
                    <Badge
                      key={index}
                      variant="secondary"
                      className="gap-1"
                      data-testid={`badge-search-${index}`}
                    >
                      {group}
                      <X
                        className="h-3 w-3 cursor-pointer hover:text-destructive"
                        onClick={() => removeSearchGroup(group)}
                      />
                    </Badge>
                  ))}
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={clearAllFilters}
                    data-testid="button-clear-filters"
                  >
                    Wyczyść wszystkie
                  </Button>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Bulk Import */}
          <Card>
            <CardHeader>
              <CardTitle>Importuj zdjęcia</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div
                {...getRootProps()}
                className={`border-2 border-dashed p-3 text-center cursor-pointer hover-elevate rounded-md ${
                  isDragActive ? 'border-primary bg-primary/10' : 'border-border'
                }`}
                data-testid="dropzone-bulk-import"
              >
                <input {...getInputProps()} />
                <Upload className="h-6 w-6 mx-auto mb-1 text-muted-foreground" />
                {isDragActive ? (
                  <p className="text-xs">Upuść zdjęcia tutaj...</p>
                ) : (
                  <p className="text-xs text-muted-foreground">
                    Przeciągnij zdjęcia lub kliknij
                  </p>
                )}
              </div>

              {bulkImportFiles.length > 0 && (
                <div className="space-y-2">
                  <div className="flex items-center justify-between">
                    <span className="text-xs font-medium">Plików: {bulkImportFiles.length}</span>
                    <Button
                      size="sm"
                      variant="ghost"
                      onClick={() => setBulkImportFiles([])}
                      data-testid="button-clear-bulk-files"
                      className="h-6 text-xs"
                    >
                      <X className="h-3 w-3 mr-1" />
                      Wyczyść
                    </Button>
                  </div>
                  <div className="flex gap-2 flex-wrap">
                    <Button
                      size="sm"
                      onClick={handleBulkImport}
                      disabled={bulkImportMutation.isPending}
                      data-testid="button-bulk-import"
                    >
                      {bulkImportMutation.isPending ? 'Importowanie...' : `Importuj ${bulkImportFiles.length}`}
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => document.getElementById('bulk-file-input')?.click()}
                      disabled={bulkImportMutation.isPending}
                      data-testid="button-add-more-files"
                    >
                      <Plus className="h-3 w-3 mr-1" />
                      Dodaj więcej
                    </Button>
                    <input
                      id="bulk-file-input"
                      type="file"
                      multiple
                      accept="image/*"
                      className="hidden"
                      onChange={(e) => {
                        const files = Array.from(e.target.files || []);
                        setBulkImportFiles(prev => [...prev, ...files]);
                        e.target.value = '';
                      }}
                    />
                  </div>
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        {/* Bulk Actions */}
        {selectedMaterials.length > 0 && (
          <Card>
            <CardContent className="py-4">
              <div className="flex items-center justify-between flex-wrap gap-3">
                <span className="text-sm font-medium">
                  Zaznaczono: {selectedMaterials.length} materiałów
                </span>
                <div className="flex gap-2 flex-wrap">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setShowBulkEditPriceDialog(true)}
                    data-testid="button-bulk-edit-price"
                  >
                    <Edit className="w-4 h-4 mr-2" />
                    Zmień cenę
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setShowBulkEditUnitDialog(true)}
                    data-testid="button-bulk-edit-unit"
                  >
                    <Edit className="w-4 h-4 mr-2" />
                    Zmień jednostkę
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={handleExport}
                    disabled={exportMutation.isPending}
                    data-testid="button-export-selected"
                  >
                    {exportMutation.isPending ? (
                      <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                    ) : (
                      <Download className="w-4 h-4 mr-2" />
                    )}
                    Eksportuj
                  </Button>
                  <Button
                    variant="destructive"
                    size="sm"
                    onClick={handleBulkDelete}
                    disabled={bulkDeleteMutation.isPending}
                    data-testid="button-delete-selected"
                  >
                    {bulkDeleteMutation.isPending ? (
                      <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                    ) : (
                      <Trash2 className="w-4 h-4 mr-2" />
                    )}
                    Usuń
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Pagination Top */}
        {pagination && (
          <div className="flex items-center justify-between">
            <div className="text-sm text-muted-foreground">
              Pokazano {((page - 1) * limit) + 1} - {Math.min(page * limit, pagination.total)} z {pagination.total}
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage(1)}
                disabled={page === 1}
                data-testid="button-page-first"
              >
                <ChevronsLeft className="w-4 h-4" />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage((p: number) => Math.max(1, p - 1))}
                disabled={page === 1}
                data-testid="button-page-prev"
              >
                <ChevronLeft className="w-4 h-4" />
              </Button>
              <span className="text-sm">
                Strona {page} z {pagination.totalPages}
              </span>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage((p: number) => Math.min(pagination.totalPages, p + 1))}
                disabled={page === pagination.totalPages}
                data-testid="button-page-next"
              >
                <ChevronRight className="w-4 h-4" />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage(pagination.totalPages)}
                disabled={page === pagination.totalPages}
                data-testid="button-page-last"
              >
                <ChevronsRight className="w-4 h-4" />
              </Button>
              <Select value={limit.toString()} onValueChange={(v) => { setLimit(parseInt(v)); setPage(1); }}>
                <SelectTrigger className="w-24" data-testid="select-page-size">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="10">10</SelectItem>
                  <SelectItem value="25">25</SelectItem>
                  <SelectItem value="50">50</SelectItem>
                  <SelectItem value="100">100</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        )}

        {/* Materials List */}
        <Card>
          <CardContent className="p-4">
            {isLoading ? (
              <div className="flex items-center justify-center py-8">
                <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
              </div>
            ) : materials.length === 0 ? (
              <div className="text-center text-muted-foreground py-8">
                Brak materiałów
              </div>
            ) : (
              <div className="space-y-2">
                {/* Select All */}
                <div className="flex items-center gap-2 pb-2 border-b">
                  <Checkbox
                    checked={materials.length > 0 && selectedMaterials.length === materials.length}
                    onCheckedChange={toggleSelectAll}
                    data-testid="checkbox-select-all"
                  />
                  <span className="text-sm font-medium">Zaznacz wszystkie</span>
                </div>

                {/* List View */}
                {viewMode === 'list' && materials.map((material) => {
                  const imageUrl = material.primaryImageUrl || (material.gallery && material.gallery.length > 0 ? material.gallery[0] : null);
                  const isHovering = uploadHoverMaterials[material.id] || false;
                  const isUploading = uploadingMaterials[material.id] || false;
                  
                  return (
                    <div
                      key={material.id}
                      className="flex items-start gap-3 p-2 border-b hover-elevate cursor-pointer"
                      data-testid={`material-${material.id}`}
                      onClick={() => handleEditMaterial(material)}
                    >
                      <div onClick={(e) => e.stopPropagation()}>
                        <Checkbox
                          checked={selectedMaterials.includes(material.id)}
                          onCheckedChange={() => toggleSelectMaterial(material.id)}
                          data-testid={`checkbox-material-${material.id}`}
                        />
                      </div>
                      
                      {/* Image/Upload Area */}
                      <div 
                        onClick={(e) => e.stopPropagation()}
                        onDragOver={(e) => {
                          e.preventDefault();
                          e.stopPropagation();
                          setUploadHoverMaterials(prev => ({ ...prev, [material.id]: true }));
                        }}
                        onDragLeave={(e) => {
                          e.preventDefault();
                          e.stopPropagation();
                          setUploadHoverMaterials(prev => ({ ...prev, [material.id]: false }));
                        }}
                        onDrop={(e) => {
                          e.preventDefault();
                          e.stopPropagation();
                          setUploadHoverMaterials(prev => ({ ...prev, [material.id]: false }));
                          const files = e.dataTransfer.files;
                          if (files.length > 0) {
                            setEditingMaterial(material);
                            setEditMaterialGallery(material.gallery || []);
                            handleImageUpload(files, material);
                          }
                        }}
                        className={`w-12 h-12 flex-shrink-0 border ${
                          isHovering ? 'border-primary bg-primary/10' : 'border-dashed border-muted-foreground/30'
                        } flex items-center justify-center overflow-hidden transition-colors relative`}
                      >
                        {isUploading ? (
                          <div className="absolute inset-0 bg-background/80 flex items-center justify-center">
                            <Loader2 className="w-5 h-5 animate-spin text-primary" />
                          </div>
                        ) : null}
                        {imageUrl ? (
                          <img 
                            src={imageUrl} 
                            alt={material.name}
                            className="w-full h-full object-cover"
                          />
                        ) : (
                          <Upload className="w-5 h-5 text-muted-foreground" />
                        )}
                      </div>
                      
                      <div className="flex-1">
                        <h3 className="text-sm font-semibold">{material.name}</h3>
                        <p className="text-xs text-muted-foreground">
                          Kod: {material.internalCode}
                          {material.supplierCode && ` | Dostawca: ${material.supplierCode}`}
                          {material.groupName && ` | Grupa: ${material.groupName}`}
                        </p>
                        {material.description && (
                          <p className="text-sm mt-1">{material.description}</p>
                        )}
                      </div>
                      <div className="flex flex-col items-end gap-1">
                        <div className="text-sm text-muted-foreground">
                          {material.unitOfMeasure}
                        </div>
                        {material.price && (
                          <div className="text-sm font-medium">
                            {parseFloat(material.price).toFixed(2)} zł
                          </div>
                        )}
                      </div>
                      <div onClick={(e) => e.stopPropagation()}>
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button
                              variant="ghost"
                              size="icon"
                              data-testid={`button-menu-material-${material.id}`}
                              className="flex-shrink-0"
                            >
                              <MoreVertical className="w-4 h-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuItem onClick={() => handleEditMaterial(material)}>
                              <Edit className="w-4 h-4 mr-2" />
                              Edytuj
                            </DropdownMenuItem>
                            <DropdownMenuSeparator />
                            <DropdownMenuItem onClick={() => handleDuplicateSameGroup(material)}>
                              <Copy className="w-4 h-4 mr-2" />
                              Duplikuj w tej samej grupie
                            </DropdownMenuItem>
                            <DropdownMenuItem onClick={() => handleDuplicateToGroup(material)}>
                              <Copy className="w-4 h-4 mr-2" />
                              Duplikuj do innej grupy
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </div>
                    </div>
                  );
                })}

                {/* Compact View */}
                {viewMode === 'compact' && materials.map((material, idx) => {
                  const imageUrl = material.primaryImageUrl || (material.gallery && material.gallery.length > 0 ? material.gallery[0] : null);
                  const globalIndex = (pagination?.offset || 0) + idx;
                  
                  return (
                    <CompactListItem
                      key={material.id}
                      index={globalIndex}
                      imageUrl={imageUrl}
                      primaryText={material.name}
                      secondaryText={material.internalCode}
                      price={material.price}
                      selectable
                      selected={selectedMaterials.includes(material.id)}
                      onSelect={() => toggleSelectMaterial(material.id)}
                      onClick={() => handleEditMaterial(material)}
                      testId={`material-${material.id}`}
                      menuItems={
                        <>
                          <DropdownMenuItem onClick={() => handleEditMaterial(material)}>
                            <Edit className="w-4 h-4 mr-2" />
                            Edytuj
                          </DropdownMenuItem>
                          <DropdownMenuSeparator />
                          <DropdownMenuItem onClick={() => handleDuplicateSameGroup(material)}>
                            <Copy className="w-4 h-4 mr-2" />
                            Duplikuj w tej samej grupie
                          </DropdownMenuItem>
                          <DropdownMenuItem onClick={() => handleDuplicateToGroup(material)}>
                            <Copy className="w-4 h-4 mr-2" />
                            Duplikuj do innej grupy
                          </DropdownMenuItem>
                        </>
                      }
                    />
                  );
                })}

                {/* Grid View */}
                {viewMode === 'grid' && (
                  <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-2">
                    {materials.map((material) => {
                      const imageUrl = material.primaryImageUrl || (material.gallery && material.gallery.length > 0 ? material.gallery[0] : null);
                      return (
                        <div
                          key={material.id}
                          className="border overflow-hidden hover-elevate cursor-pointer"
                          data-testid={`material-${material.id}`}
                          onClick={() => handleEditMaterial(material)}
                        >
                          <div className="aspect-square bg-muted flex items-center justify-center">
                            {imageUrl ? (
                              <img 
                                src={imageUrl} 
                                alt={material.name}
                                className="w-full h-full object-cover"
                              />
                            ) : (
                              <Upload className="w-8 h-8 text-muted-foreground" />
                            )}
                          </div>
                          <div className="p-2">
                            <div className="flex items-start gap-1 mb-1">
                              <div onClick={(e) => e.stopPropagation()}>
                                <Checkbox
                                  checked={selectedMaterials.includes(material.id)}
                                  onCheckedChange={() => toggleSelectMaterial(material.id)}
                                  data-testid={`checkbox-material-${material.id}`}
                                />
                              </div>
                              <div className="flex-1 min-w-0">
                                <h3 className="font-semibold text-xs truncate">{material.name}</h3>
                                <p className="text-xs text-muted-foreground truncate">{material.internalCode}</p>
                              </div>
                              <div onClick={(e) => e.stopPropagation()}>
                                <DropdownMenu>
                                  <DropdownMenuTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-6 w-6"
                                      data-testid={`button-menu-material-${material.id}`}
                                    >
                                      <MoreVertical className="w-3 h-3" />
                                    </Button>
                                  </DropdownMenuTrigger>
                                  <DropdownMenuContent align="end">
                                    <DropdownMenuItem onClick={() => handleEditMaterial(material)}>
                                      <Edit className="w-4 h-4 mr-2" />
                                      Edytuj
                                    </DropdownMenuItem>
                                    <DropdownMenuSeparator />
                                    <DropdownMenuItem onClick={() => handleDuplicateSameGroup(material)}>
                                      <Copy className="w-4 h-4 mr-2" />
                                      Duplikuj w tej samej grupie
                                    </DropdownMenuItem>
                                    <DropdownMenuItem onClick={() => handleDuplicateToGroup(material)}>
                                      <Copy className="w-4 h-4 mr-2" />
                                      Duplikuj do innej grupy
                                    </DropdownMenuItem>
                                  </DropdownMenuContent>
                                </DropdownMenu>
                              </div>
                            </div>
                            {material.price && (
                              <div className="text-xs font-medium">
                                {parseFloat(material.price).toFixed(2)} zł
                              </div>
                            )}
                          </div>
                        </div>
                      );
                    })}
                  </div>
                )}
              </div>
            )}
          </CardContent>
        </Card>

        {/* Pagination Bottom */}
        {pagination && (
          <div className="flex items-center justify-between">
            <div className="text-sm text-muted-foreground">
              Pokazano {((page - 1) * limit) + 1} - {Math.min(page * limit, pagination.total)} z {pagination.total}
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage(1)}
                disabled={page === 1}
                data-testid="button-page-first-bottom"
              >
                <ChevronsLeft className="w-4 h-4" />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage((p: number) => Math.max(1, p - 1))}
                disabled={page === 1}
                data-testid="button-page-prev-bottom"
              >
                <ChevronLeft className="w-4 h-4" />
              </Button>
              <span className="text-sm">
                Strona {page} z {pagination.totalPages}
              </span>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage((p: number) => Math.min(pagination.totalPages, p + 1))}
                disabled={page === pagination.totalPages}
                data-testid="button-page-next-bottom"
              >
                <ChevronRight className="w-4 h-4" />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setPage(pagination.totalPages)}
                disabled={page === pagination.totalPages}
                data-testid="button-page-last-bottom"
              >
                <ChevronsRight className="w-4 h-4" />
              </Button>
              <Select value={limit.toString()} onValueChange={(v) => { setLimit(parseInt(v)); setPage(1); }}>
                <SelectTrigger className="w-24" data-testid="select-page-size-bottom">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="10">10</SelectItem>
                  <SelectItem value="25">25</SelectItem>
                  <SelectItem value="50">50</SelectItem>
                  <SelectItem value="100">100</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        )}
      </div>

      {/* Add Group Dialog */}
      <Dialog open={isAddGroupDialogOpen} onOpenChange={setIsAddGroupDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Dodaj grupę materiałów</DialogTitle>
            <DialogDescription>
              Utwórz nową grupę materiałów dla kategorii {CATEGORY_LABELS[category]}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="group-name">Nazwa grupy *</Label>
              <Input
                id="group-name"
                value={newGroupName}
                onChange={(e) => {
                  const name = e.target.value;
                  setNewGroupName(name);
                  // Auto-generate code from name
                  setNewGroupCode(generateSlug(name));
                }}
                placeholder="np. Zawiasy standardowe"
                data-testid="input-group-name"
              />
            </div>
            <div>
              <Label htmlFor="group-code">Kod grupy *</Label>
              <Input
                id="group-code"
                value={newGroupCode}
                onChange={(e) => setNewGroupCode(e.target.value)}
                placeholder="np. zawiasy_std"
                data-testid="input-group-code"
              />
            </div>
            <div>
              <Label htmlFor="group-description">Opis (opcjonalnie)</Label>
              <Textarea
                id="group-description"
                value={newGroupDescription}
                onChange={(e) => setNewGroupDescription(e.target.value)}
                placeholder="Opis grupy materiałów"
                data-testid="input-group-description"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setIsAddGroupDialogOpen(false)}
              data-testid="button-cancel-group"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleAddGroup}
              disabled={addGroupMutation.isPending}
              data-testid="button-submit-group"
            >
              {addGroupMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Utwórz grupę
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Group Dialog */}
      <Dialog open={isEditGroupDialogOpen} onOpenChange={setIsEditGroupDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Edytuj grupę materiałów</DialogTitle>
            <DialogDescription>
              Edytuj grupę materiałów: {editingGroup?.name}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="edit-group-name">Nazwa grupy *</Label>
              <Input
                id="edit-group-name"
                value={editGroupName}
                onChange={(e) => setEditGroupName(e.target.value)}
                placeholder="np. Zawiasy standardowe"
                data-testid="input-edit-group-name"
              />
            </div>
            <div>
              <Label htmlFor="edit-group-code">Kod grupy *</Label>
              <Input
                id="edit-group-code"
                value={editGroupCode}
                onChange={(e) => setEditGroupCode(e.target.value)}
                placeholder="np. zawiasy_std"
                data-testid="input-edit-group-code"
              />
            </div>
            <div>
              <Label htmlFor="edit-group-display-order">Kolejność wyświetlania</Label>
              <Input
                id="edit-group-display-order"
                type="number"
                value={editGroupDisplayOrder}
                onChange={(e) => setEditGroupDisplayOrder(e.target.value)}
                placeholder="0"
                data-testid="input-edit-group-display-order"
              />
              <p className="text-xs text-muted-foreground mt-1">
                Niższy numer = wyżej na liście
              </p>
            </div>
            <div>
              <Label htmlFor="edit-group-description">Opis (opcjonalnie)</Label>
              <Textarea
                id="edit-group-description"
                value={editGroupDescription}
                onChange={(e) => setEditGroupDescription(e.target.value)}
                placeholder="Opis grupy materiałów"
                data-testid="input-edit-group-description"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setIsEditGroupDialogOpen(false)}
              data-testid="button-cancel-edit-group"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleUpdateGroup}
              disabled={editGroupMutation.isPending}
              data-testid="button-update-group"
            >
              {editGroupMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Zapisz zmiany
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Add Material Dialog */}
      <Dialog open={isAddMaterialDialogOpen} onOpenChange={setIsAddMaterialDialogOpen}>
        <DialogContent className="max-w-[95vw] sm:max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Dodaj materiał</DialogTitle>
            <DialogDescription>
              {groupCode 
                ? `Dodajesz materiał do grupy: ${categoryGroups.find(g => g.code === groupCode)?.name || groupCode}`
                : 'Utwórz nowy materiał w kategorii ' + (CATEGORY_LABELS[category] || category)
              }
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="material-name">Nazwa materiału *</Label>
              <Input
                id="material-name"
                value={newMaterialName}
                onChange={(e) => setNewMaterialName(e.target.value)}
                placeholder="np. Zawias 35mm"
                data-testid="input-material-name"
              />
            </div>
            <div>
              <Label htmlFor="material-internal-code">Kod wewnętrzny *</Label>
              <Input
                id="material-internal-code"
                value={newMaterialInternalCode}
                onChange={(e) => {
                  setNewMaterialInternalCode(e.target.value);
                  setIsInternalCodeManuallyEdited(true);
                }}
                placeholder="np. ZAW-35-STD"
                data-testid="input-material-internal-code"
              />
              <p className="text-xs text-muted-foreground mt-1">
                Kod generowany automatycznie z nazwy. Możesz go edytować.
              </p>
            </div>
            <div>
              <Label htmlFor="material-group">
                Grupa materiałów {groupCode ? '' : '(opcjonalnie)'}
              </Label>
              <Select 
                value={newMaterialGroupId?.toString() || "none"} 
                onValueChange={(value) => setNewMaterialGroupId(value === "none" ? null : parseInt(value))}
                disabled={!!groupCode}
              >
                <SelectTrigger id="material-group" data-testid="select-material-group">
                  <SelectValue placeholder="Bez grupy" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="none">Bez grupy</SelectItem>
                  {categoryGroups.map((group) => (
                    <SelectItem key={group.id} value={group.id.toString()}>
                      {group.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              {groupCode && (
                <p className="text-xs text-muted-foreground mt-1">
                  Grupa jest automatycznie wybrana, ponieważ dodajesz materiał z poziomu grupy
                </p>
              )}
            </div>
            <div>
              <Label htmlFor="material-supplier-code">Kod dostawcy (opcjonalnie)</Label>
              <Input
                id="material-supplier-code"
                value={newMaterialSupplierCode}
                onChange={(e) => setNewMaterialSupplierCode(e.target.value)}
                placeholder="np. SUP-12345"
                data-testid="input-material-supplier-code"
              />
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label htmlFor="material-unit">Jednostka miary</Label>
                <Select value={newMaterialUnitOfMeasure} onValueChange={setNewMaterialUnitOfMeasure}>
                  <SelectTrigger id="material-unit" data-testid="select-material-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {units.map((unit: any) => (
                      <SelectItem key={unit.id} value={unit.code}>
                        {unit.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="material-price">Cena (opcjonalnie)</Label>
                <Input
                  id="material-price"
                  type="number"
                  step="0.01"
                  min="0"
                  value={newMaterialPrice}
                  onChange={(e) => setNewMaterialPrice(e.target.value)}
                  placeholder="0.00"
                  data-testid="input-material-price"
                />
              </div>
            </div>
            <div>
              <Label htmlFor="material-description">Opis (opcjonalnie)</Label>
              <Textarea
                id="material-description"
                value={newMaterialDescription}
                onChange={(e) => setNewMaterialDescription(e.target.value)}
                placeholder="Opis materiału"
                data-testid="input-material-description"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setIsAddMaterialDialogOpen(false)}
              data-testid="button-cancel-material"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleAddMaterial}
              disabled={addMaterialMutation.isPending}
              data-testid="button-submit-material"
            >
              {addMaterialMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Utwórz materiał
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Material Dialog */}
      <Dialog open={isEditMaterialDialogOpen} onOpenChange={setIsEditMaterialDialogOpen}>
        <DialogContent className="max-w-[95vw] sm:max-w-2xl max-h-[90vh] overflow-y-auto p-4">
          <DialogHeader className="pb-2">
            <DialogTitle className="text-base">Edytuj materiał</DialogTitle>
            <DialogDescription className="text-xs">
              Edytuj materiał: {editingMaterial?.name}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-3">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
              <div>
                <Label htmlFor="edit-material-name" className="text-xs">Nazwa materiału *</Label>
                <Input
                  id="edit-material-name"
                  value={editMaterialName}
                  onChange={(e) => setEditMaterialName(e.target.value)}
                  placeholder="np. Zawias 35mm"
                  className="h-8 text-sm"
                  data-testid="input-edit-material-name"
                />
              </div>
              <div>
                <Label htmlFor="edit-material-internal-code" className="text-xs">Kod wewnętrzny *</Label>
                <Input
                  id="edit-material-internal-code"
                  value={editMaterialInternalCode}
                  onChange={(e) => setEditMaterialInternalCode(e.target.value)}
                  placeholder="np. ZAW-35-STD"
                  className="h-8 text-sm"
                  data-testid="input-edit-material-internal-code"
                />
              </div>
              <div>
                <Label htmlFor="edit-material-supplier-code" className="text-xs">Kod dostawcy</Label>
                <Input
                  id="edit-material-supplier-code"
                  value={editMaterialSupplierCode}
                  onChange={(e) => setEditMaterialSupplierCode(e.target.value)}
                  placeholder="np. SUP-12345"
                  className="h-8 text-sm"
                  data-testid="input-edit-material-supplier-code"
                />
              </div>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
              <div>
                <Label htmlFor="edit-material-group" className="text-xs">Grupa materiałów</Label>
                <Select 
                  value={editMaterialGroupId?.toString() || "none"} 
                  onValueChange={(value) => setEditMaterialGroupId(value === "none" ? null : parseInt(value))}
                >
                  <SelectTrigger id="edit-material-group" className="h-8 text-sm" data-testid="select-edit-material-group">
                    <SelectValue placeholder="Bez grupy" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Bez grupy</SelectItem>
                    {categoryGroups.map((group) => (
                      <SelectItem key={group.id} value={group.id.toString()}>
                        {group.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="edit-material-unit" className="text-xs">Jednostka miary</Label>
                <Select value={editMaterialUnitOfMeasure} onValueChange={setEditMaterialUnitOfMeasure}>
                  <SelectTrigger id="edit-material-unit" className="h-8 text-sm" data-testid="select-edit-material-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {units.map((unit: any) => (
                      <SelectItem key={unit.id} value={unit.code}>
                        {unit.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="edit-material-price" className="text-xs">Cena</Label>
                <Input
                  id="edit-material-price"
                  type="number"
                  step="0.01"
                  min="0"
                  value={editMaterialPrice}
                  onChange={(e) => setEditMaterialPrice(e.target.value)}
                  placeholder="0.00"
                  className="h-8 text-sm"
                  data-testid="input-edit-material-price"
                />
              </div>
            </div>
            <div>
              <Label htmlFor="edit-material-description" className="text-xs">Opis (opcjonalnie)</Label>
              <Textarea
                id="edit-material-description"
                value={editMaterialDescription}
                onChange={(e) => setEditMaterialDescription(e.target.value)}
                placeholder="Opis materiału"
                className="text-sm min-h-[60px]"
                data-testid="input-edit-material-description"
              />
            </div>

            {/* Image Gallery Section */}
            <div className="border-t pt-3">
              <Label className="text-xs font-semibold mb-2 block">Galeria zdjęć</Label>
              
              {/* Drag & Drop Upload Area */}
              <div 
                className={`border-2 border-dashed rounded-lg p-3 text-center hover-elevate cursor-pointer mb-3 transition-colors ${
                  isDragging ? 'border-primary bg-primary/10' : 'border-border'
                }`}
                onDragEnter={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                  setIsDragging(true);
                }}
                onDragLeave={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                  setIsDragging(false);
                }}
                onDragOver={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                }}
                onDrop={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                  setIsDragging(false);
                  handleImageUpload(e.dataTransfer.files);
                }}
                onClick={() => {
                  const input = document.createElement('input');
                  input.type = 'file';
                  input.accept = 'image/*';
                  input.multiple = true;
                  input.onchange = (e) => {
                    const target = e.target as HTMLInputElement;
                    handleImageUpload(target.files);
                  };
                  input.click();
                }}
                data-testid="dropzone-material-images"
              >
                <Upload className={`w-5 h-5 mx-auto mb-1 ${isDragging ? 'text-primary' : 'text-muted-foreground'}`} />
                <p className={`text-xs ${isDragging ? 'text-primary font-medium' : 'text-muted-foreground'}`}>
                  {isDragging ? 'Upuść zdjęcia tutaj' : 'Przeciągnij i upuść zdjęcia lub kliknij'}
                </p>
              </div>

              {/* Gallery Grid */}
              {editMaterialGallery.length > 0 && (
                <div className="grid grid-cols-4 gap-2">
                  {editMaterialGallery.map((imageUrl, index) => {
                    const isPrimary = editingMaterial?.primaryImageUrl === imageUrl;
                    return (
                      <div 
                        key={index} 
                        className={`relative group aspect-square rounded-md overflow-hidden border ${
                          isPrimary ? 'border-primary border-2' : 'border-border'
                        }`}
                      >
                        {isPrimary && (
                          <div className="absolute top-1 right-1 z-10">
                            <Badge variant="default" className="gap-0.5 h-5 px-1.5 text-[10px]">
                              <Star className="w-2.5 h-2.5 fill-current" />
                              Główne
                            </Badge>
                          </div>
                        )}
                        <img 
                          src={imageUrl} 
                          alt={`Material image ${index + 1}`} 
                          className="w-full h-full object-cover"
                        />
                        <div className="absolute inset-0 bg-black/60 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center gap-1">
                          {!isPrimary && (
                            <Button
                              variant="secondary"
                              size="icon"
                              className="h-7 w-7"
                              onClick={() => handleSetPrimaryImage(imageUrl)}
                              disabled={setPrimaryImageMutation.isPending}
                              data-testid={`button-set-primary-${index}`}
                              title="Ustaw jako główne"
                            >
                              <Star className="w-3 h-3" />
                            </Button>
                          )}
                          <Button
                            variant="destructive"
                            size="icon"
                            className="h-7 w-7"
                            onClick={() => handleImageDelete(imageUrl)}
                            disabled={deleteMaterialImageMutation.isPending}
                            data-testid={`button-delete-image-${index}`}
                            title="Usuń zdjęcie"
                          >
                            <Trash2 className="w-3 h-3" />
                          </Button>
                        </div>
                      </div>
                    );
                  })}
                </div>
              )}

              {uploadMaterialImageMutation.isPending && (
                <div className="flex items-center justify-center py-3">
                  <Loader2 className="w-4 h-4 animate-spin text-muted-foreground" />
                  <span className="ml-2 text-xs text-muted-foreground">Uploading...</span>
                </div>
              )}
            </div>
          </div>
          <DialogFooter className="gap-2 pt-3">
            <Button
              variant="outline"
              size="sm"
              onClick={() => setIsEditMaterialDialogOpen(false)}
              data-testid="button-cancel-edit-material"
            >
              Anuluj
            </Button>
            <Button
              size="sm"
              onClick={handleUpdateMaterial}
              disabled={editMaterialMutation.isPending}
              data-testid="button-update-material"
            >
              {editMaterialMutation.isPending && <Loader2 className="w-3 h-3 mr-1 animate-spin" />}
              Zapisz zmiany
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Duplicate Material Dialog */}
      <Dialog open={showDuplicateDialog} onOpenChange={setShowDuplicateDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Duplikuj materiał</DialogTitle>
            <DialogDescription>
              Wybierz grupę docelową dla kopii materiału: {duplicateMaterial?.name}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="duplicate-target-group">Grupa docelowa *</Label>
              <Select 
                value={duplicateTargetGroupId?.toString() || ""} 
                onValueChange={(value) => setDuplicateTargetGroupId(parseInt(value))}
              >
                <SelectTrigger id="duplicate-target-group" data-testid="select-duplicate-target-group">
                  <SelectValue placeholder="Wybierz grupę" />
                </SelectTrigger>
                <SelectContent>
                  {categoryGroups.map((group) => (
                    <SelectItem key={group.id} value={group.id.toString()}>
                      {group.name}
                      {group.id === duplicateMaterial?.groupId && " (aktualna)"}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setShowDuplicateDialog(false);
                setDuplicateMaterial(null);
                setDuplicateTargetGroupId(null);
              }}
              data-testid="button-cancel-duplicate"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleConfirmDuplicate}
              disabled={duplicateMaterialMutation.isPending}
              data-testid="button-confirm-duplicate"
            >
              {duplicateMaterialMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Duplikuj
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Edit Price Dialog */}
      <Dialog open={showBulkEditPriceDialog} onOpenChange={setShowBulkEditPriceDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Zmień cenę materiałów</DialogTitle>
            <DialogDescription>
              Ustaw nową cenę dla {selectedMaterials.length} zaznaczonych materiałów
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="bulk-price">Nowa cena (zł) *</Label>
              <Input
                id="bulk-price"
                type="number"
                step="0.01"
                min="0"
                value={bulkEditPrice}
                onChange={(e) => setBulkEditPrice(e.target.value)}
                placeholder="np. 12.50"
                data-testid="input-bulk-price"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setShowBulkEditPriceDialog(false);
                setBulkEditPrice("");
              }}
              data-testid="button-cancel-bulk-price"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => bulkUpdatePriceMutation.mutate(bulkEditPrice)}
              disabled={bulkUpdatePriceMutation.isPending || !bulkEditPrice}
              data-testid="button-confirm-bulk-price"
            >
              {bulkUpdatePriceMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Zmień cenę
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Edit Unit Dialog */}
      <Dialog open={showBulkEditUnitDialog} onOpenChange={setShowBulkEditUnitDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Zmień jednostkę miary</DialogTitle>
            <DialogDescription>
              Ustaw nową jednostkę miary dla {selectedMaterials.length} zaznaczonych materiałów
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="bulk-unit">Nowa jednostka miary *</Label>
              <Select value={bulkEditUnit} onValueChange={setBulkEditUnit}>
                <SelectTrigger id="bulk-unit" data-testid="select-bulk-unit">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {units.map((unit: any) => (
                    <SelectItem key={unit.id} value={unit.code}>
                      {unit.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setShowBulkEditUnitDialog(false);
                setBulkEditUnit("szt");
              }}
              data-testid="button-cancel-bulk-unit"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => bulkUpdateUnitMutation.mutate(bulkEditUnit)}
              disabled={bulkUpdateUnitMutation.isPending}
              data-testid="button-confirm-bulk-unit"
            >
              {bulkUpdateUnitMutation.isPending && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              Zmień jednostkę
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk import progress dialog */}
      <GenerationLogsDialog
        open={showBulkImportLogs}
        onOpenChange={setShowBulkImportLogs}
        sessionId={bulkImportSessionId}
        title="Import materiałów"
        description="Podgląd na żywo procesu importu materiałów"
      />
    </WarehouseLayout>
  );
}
