import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { Plus, Trash2, Edit2, GripVertical, Save } from "lucide-react";
import * as Icons from "lucide-react";
import type { WarehouseCategory } from "@shared/schema";

// Lista dostępnych ikon z lucide-react
const AVAILABLE_ICONS = [
  "Box", "Package", "Wrench", "Hammer", "Cog", "Boxes", "FileBox",
  "Cylinder", "Container", "Truck", "Warehouse", "Archive",
  "Layers", "ShoppingBag", "ShoppingCart", "PackageCheck"
];

// Lista dostępnych kolorów ikon
const ICON_COLORS = [
  { value: "text-gray-600/70", label: "Szary" },
  { value: "text-amber-600/70", label: "Bursztynowy" },
  { value: "text-blue-600/70", label: "Niebieski" },
  { value: "text-purple-600/70", label: "Fioletowy" },
  { value: "text-green-600/70", label: "Zielony" },
  { value: "text-red-600/70", label: "Czerwony" },
  { value: "text-orange-600/70", label: "Pomarańczowy" },
  { value: "text-pink-600/70", label: "Różowy" },
  { value: "text-slate-600/70", label: "Grafitowy" },
];

export default function WarehouseCategoriesManager() {
  const { toast } = useToast();
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [editingCategory, setEditingCategory] = useState<WarehouseCategory | null>(null);
  const [draggedItem, setDraggedItem] = useState<number | null>(null);

  const [formData, setFormData] = useState({
    name: "",
    code: "",
    icon: "Box",
    iconColor: "text-gray-600/70",
    description: "",
  });

  const { data: categories = [], isLoading } = useQuery<WarehouseCategory[]>({
    queryKey: ["/api/warehouse/categories"],
  });

  const addMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", "/api/warehouse/categories", {
        ...formData,
        displayOrder: categories.length + 1,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/categories"] });
      toast({
        title: "Kategoria dodana",
        description: `Kategoria "${formData.name}" została pomyślnie dodana.`,
      });
      setIsAddDialogOpen(false);
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd dodawania",
        description: error.message || "Nie udało się dodać kategorii.",
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async () => {
      if (!editingCategory) return;
      return apiRequest("PATCH", `/api/warehouse/categories/${editingCategory.id}`, formData);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/categories"] });
      toast({
        title: "Kategoria zaktualizowana",
        description: "Zmiany zostały zapisane.",
      });
      setIsEditDialogOpen(false);
      setEditingCategory(null);
      resetForm();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd aktualizacji",
        description: error.message || "Nie udało się zaktualizować kategorii.",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/warehouse/categories/${id}`, {});
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/categories"] });
      toast({
        title: "Kategoria usunięta",
        description: "Kategoria została pomyślnie usunięta.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć kategorii.",
        variant: "destructive",
      });
    },
  });

  const reorderMutation = useMutation({
    mutationFn: async (newOrder: number[]) => {
      return apiRequest("PATCH", "/api/warehouse/categories/reorder", {
        categoryIds: newOrder,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/categories"] });
      toast({
        title: "Kolejność zmieniona",
        description: "Nowa kolejność kategorii została zapisana.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd zmiany kolejności",
        description: error.message || "Nie udało się zmienić kolejności.",
        variant: "destructive",
      });
    },
  });

  const resetForm = () => {
    setFormData({
      name: "",
      code: "",
      icon: "Box",
      iconColor: "text-gray-600/70",
      description: "",
    });
  };

  const handleAdd = () => {
    if (!formData.name || !formData.code) {
      toast({
        title: "Błąd walidacji",
        description: "Nazwa i kod są wymagane.",
        variant: "destructive",
      });
      return;
    }
    addMutation.mutate();
  };

  const handleEdit = (category: WarehouseCategory) => {
    setEditingCategory(category);
    setFormData({
      name: category.name || "",
      code: category.code || "",
      icon: category.icon || "Box",
      iconColor: category.iconColor || "text-gray-600/70",
      description: category.description || "",
    });
    setIsEditDialogOpen(true);
  };

  const handleUpdate = () => {
    if (!formData.name || !formData.code) {
      toast({
        title: "Błąd walidacji",
        description: "Nazwa i kod są wymagane.",
        variant: "destructive",
      });
      return;
    }
    updateMutation.mutate();
  };

  const handleDelete = (id: number, name: string) => {
    if (confirm(`Czy na pewno chcesz usunąć kategorię "${name}"?`)) {
      deleteMutation.mutate(id);
    }
  };

  const handleDragStart = (id: number) => {
    setDraggedItem(id);
  };

  const handleDragOver = (e: React.DragEvent, targetId: number) => {
    e.preventDefault();
    if (draggedItem === null || draggedItem === targetId) return;

    const draggedIndex = categories.findIndex(c => c.id === draggedItem);
    const targetIndex = categories.findIndex(c => c.id === targetId);

    if (draggedIndex === -1 || targetIndex === -1) return;

    // Reorder array
    const newCategories = [...categories];
    const [removed] = newCategories.splice(draggedIndex, 1);
    newCategories.splice(targetIndex, 0, removed);

    // Extract new order of IDs
    const newOrder = newCategories.map(c => c.id);
    reorderMutation.mutate(newOrder);
  };

  const handleDragEnd = () => {
    setDraggedItem(null);
  };

  const IconComponent = (iconName: string) => {
    const Icon = (Icons as any)[iconName] || Icons.Box;
    return <Icon className="h-5 w-5" />;
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Zarządzanie kategoriami magazynu</h1>
          <p className="text-muted-foreground mt-1">
            Konfiguruj kategorie surowców wyświetlane w menu magazynu
          </p>
        </div>
        <Button onClick={() => setIsAddDialogOpen(true)} data-testid="button-add-category">
          <Plus className="w-4 h-4 mr-2" />
          Dodaj kategorię
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Kategorie</CardTitle>
          <CardDescription>
            Przeciągnij i upuść aby zmienić kolejność kategorii w menu
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="text-center text-muted-foreground py-8">Ładowanie...</div>
          ) : categories.length === 0 ? (
            <div className="text-center text-muted-foreground py-8">
              Brak kategorii. Dodaj pierwszą kategorię używając przycisku powyżej.
            </div>
          ) : (
            <div className="space-y-2">
              {categories.map((category) => (
                <div
                  key={category.id}
                  draggable
                  onDragStart={() => handleDragStart(category.id)}
                  onDragOver={(e) => handleDragOver(e, category.id)}
                  onDragEnd={handleDragEnd}
                  className="flex items-center gap-4 p-4 border rounded-md hover-elevate cursor-move"
                  data-testid={`category-${category.id}`}
                >
                  <GripVertical className="h-5 w-5 text-muted-foreground" />
                  <div className={category.iconColor || "text-gray-600/70"}>
                    {IconComponent(category.icon || "Box")}
                  </div>
                  <div className="flex-1">
                    <div className="font-semibold">{category.name || ""}</div>
                    <div className="text-sm text-muted-foreground">
                      Kod: {category.code || ""}
                      {category.description && ` | ${category.description}`}
                    </div>
                  </div>
                  <div className="flex gap-2">
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => handleEdit(category)}
                      data-testid={`button-edit-${category.id}`}
                    >
                      <Edit2 className="w-4 h-4" />
                    </Button>
                    <Button
                      variant="destructive"
                      size="sm"
                      onClick={() => handleDelete(category.id, category.name)}
                      data-testid={`button-delete-${category.id}`}
                    >
                      <Trash2 className="w-4 h-4" />
                    </Button>
                  </div>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>

      {/* Add Dialog */}
      <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Dodaj nową kategorię</DialogTitle>
            <DialogDescription>
              Wypełnij poniższe dane aby utworzyć nową kategorię magazynu
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="name">Nazwa *</Label>
              <Input
                id="name"
                placeholder="np. Okucia"
                value={formData.name}
                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                data-testid="input-name"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="code">Kod URL *</Label>
              <Input
                id="code"
                placeholder="np. okucia"
                value={formData.code}
                onChange={(e) => setFormData({ ...formData, code: e.target.value.toLowerCase().replace(/\s+/g, '_') })}
                data-testid="input-code"
              />
              <p className="text-xs text-muted-foreground">
                Kod używany w URL (np. /warehouse/okucia)
              </p>
            </div>
            <div className="space-y-2">
              <Label htmlFor="icon">Ikona</Label>
              <Select value={formData.icon} onValueChange={(v) => setFormData({ ...formData, icon: v })}>
                <SelectTrigger data-testid="select-icon">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {AVAILABLE_ICONS.map((iconName) => {
                    const Icon = (Icons as any)[iconName];
                    return (
                      <SelectItem key={iconName} value={iconName}>
                        <div className="flex items-center gap-2">
                          {Icon && <Icon className="h-4 w-4" />}
                          <span>{iconName}</span>
                        </div>
                      </SelectItem>
                    );
                  })}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="iconColor">Kolor ikony</Label>
              <Select value={formData.iconColor} onValueChange={(v) => setFormData({ ...formData, iconColor: v })}>
                <SelectTrigger data-testid="select-icon-color">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {ICON_COLORS.map((color) => (
                    <SelectItem key={color.value} value={color.value}>
                      <div className="flex items-center gap-2">
                        <div className={`h-4 w-4 rounded-full ${color.value.replace('/70', '')}`} />
                        <span>{color.label}</span>
                      </div>
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="description">Opis (opcjonalny)</Label>
              <Textarea
                id="description"
                placeholder="Opis kategorii..."
                value={formData.description}
                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                data-testid="input-description"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setIsAddDialogOpen(false);
                resetForm();
              }}
              data-testid="button-cancel"
            >
              Anuluj
            </Button>
            <Button onClick={handleAdd} disabled={addMutation.isPending} data-testid="button-save">
              <Save className="w-4 h-4 mr-2" />
              {addMutation.isPending ? "Dodawanie..." : "Dodaj kategorię"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Dialog */}
      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Edytuj kategorię</DialogTitle>
            <DialogDescription>
              Wprowadź zmiany w kategorii magazynu
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="edit-name">Nazwa *</Label>
              <Input
                id="edit-name"
                placeholder="np. Okucia"
                value={formData.name}
                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                data-testid="input-edit-name"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="edit-code">Kod URL *</Label>
              <Input
                id="edit-code"
                placeholder="np. okucia"
                value={formData.code}
                onChange={(e) => setFormData({ ...formData, code: e.target.value.toLowerCase().replace(/\s+/g, '_') })}
                data-testid="input-edit-code"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="edit-icon">Ikona</Label>
              <Select value={formData.icon} onValueChange={(v) => setFormData({ ...formData, icon: v })}>
                <SelectTrigger data-testid="select-edit-icon">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {AVAILABLE_ICONS.map((iconName) => {
                    const Icon = (Icons as any)[iconName];
                    return (
                      <SelectItem key={iconName} value={iconName}>
                        <div className="flex items-center gap-2">
                          {Icon && <Icon className="h-4 w-4" />}
                          <span>{iconName}</span>
                        </div>
                      </SelectItem>
                    );
                  })}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="edit-iconColor">Kolor ikony</Label>
              <Select value={formData.iconColor} onValueChange={(v) => setFormData({ ...formData, iconColor: v })}>
                <SelectTrigger data-testid="select-edit-icon-color">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {ICON_COLORS.map((color) => (
                    <SelectItem key={color.value} value={color.value}>
                      <div className="flex items-center gap-2">
                        <div className={`h-4 w-4 rounded-full ${color.value.replace('/70', '')}`} />
                        <span>{color.label}</span>
                      </div>
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="edit-description">Opis (opcjonalny)</Label>
              <Textarea
                id="edit-description"
                placeholder="Opis kategorii..."
                value={formData.description}
                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                data-testid="input-edit-description"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setIsEditDialogOpen(false);
                setEditingCategory(null);
                resetForm();
              }}
              data-testid="button-cancel-edit"
            >
              Anuluj
            </Button>
            <Button onClick={handleUpdate} disabled={updateMutation.isPending} data-testid="button-update">
              <Save className="w-4 h-4 mr-2" />
              {updateMutation.isPending ? "Zapisywanie..." : "Zapisz zmiany"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
