import { useState, useEffect, useCallback, useMemo } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation, useRoute } from "wouter";
import { useDropzone } from "react-dropzone";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { useToast } from "@/hooks/use-toast";
import { useAuth } from "@/hooks/use-auth";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { ArrowLeft, Trash2, Plus, Upload, X, Eye, Palette, Check, ChevronsUpDown, Save, Play, Loader2, ExternalLink, Package2 } from "lucide-react";
import { Link } from "wouter";
import { Checkbox } from "@/components/ui/checkbox";
import { DictionaryComboboxWithAdd } from "@/components/dictionary-combobox-with-add";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import { cn } from "@/lib/utils";

interface SetMatrix {
  id: number;
  name: string;
  namePrefix?: string | null;
  nameSuffix?: string | null;
  productGroup: string | null;
  length?: string | null;
  width?: string | null;
  height?: string | null;
  doors?: string | null;
  legs?: string | null;
  imageUrl?: string | null;
  basePriceModifier?: string | null;
  priceModifierType?: string | null;
  components: {
    name: string;
    componentType: string;
    length: number | null;
    width: number | null;
    quantity: number;
  }[];
  colors?: string[];
  colorImages?: Record<string, string[]>;
  colorOptions?: Record<string, string[]>;
  selectedColors?: string[];
  useEnhancedDescription?: boolean;
}

interface Dictionary {
  id: number;
  dictionaryType?: string;
  code: string;
  name: string;
  shortName?: string | null;
  readableName: string | null;
  color?: string | null;
  isActive: boolean;
}

// Helper function to determine text color based on background luminance
function getTextColorForBackground(hexColor: string | null | undefined): string {
  if (!hexColor || !hexColor.startsWith('#')) return 'black';
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  // Calculate luminance using standard formula
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance > 128 ? 'black' : 'white';
}

export default function SetMatrixSettingsPage() {
  const { toast } = useToast();
  const { user } = useAuth();
  const [, setLocation] = useLocation();
  const [, params] = useRoute("/set-matrices/settings/:id");
  const [showDeleteDialog, setShowDeleteDialog] = useState(false);
  const [expandedColorImages, setExpandedColorImages] = useState<Set<number>>(new Set());
  const [colors, setColors] = useState<string[]>([]);
  const [selectedColors, setSelectedColors] = useState<string[]>([]);
  const [colorOpen, setColorOpen] = useState(false);
  const [colorSearchInput, setColorSearchInput] = useState("");
  const [colorOpenRows, setColorOpenRows] = useState<Set<number>>(new Set());
  const [colorSearchInputs, setColorSearchInputs] = useState<Map<number, string>>(new Map());
  const [colorOptionsMatrix, setColorOptionsMatrix] = useState<Record<string, string[]>>({});
  const [draggedImage, setDraggedImage] = useState<{ color: string; index: number } | null>(null);
  
  // Generation states
  const [isGenerating, setIsGenerating] = useState(false);
  const [showPreviewDialog, setShowPreviewDialog] = useState(false);
  const [previewData, setPreviewData] = useState<any>(null);
  
  // Form state
  const [name, setName] = useState("");
  const [namePrefix, setNamePrefix] = useState("");
  const [nameSuffix, setNameSuffix] = useState("");
  const [productGroup, setProductGroup] = useState("");
  const [length, setLength] = useState("");
  const [width, setWidth] = useState("");
  const [height, setHeight] = useState("");
  const [doors, setDoors] = useState("");
  const [legs, setLegs] = useState("");
  const [imageUrl, setImageUrl] = useState("");
  const [basePriceModifier, setBasePriceModifier] = useState("0");
  const [priceModifierType, setPriceModifierType] = useState("percent");
  const [components, setComponents] = useState<{
    name: string;
    componentType: string;
    length: string;
    width: string;
    quantity: number;
  }[]>([]);
  const [useEnhancedDescription, setUseEnhancedDescription] = useState(false);
  
  const matrixId = params?.id === "new" ? null : params?.id ? parseInt(params.id) : null;
  const isNew = params?.id === "new";

  const { data: matrix, isLoading } = useQuery<SetMatrix>({
    queryKey: [`/api/set-matrices/${matrixId}`],
    enabled: !isNew && !!matrixId && !!user,
  });

  // Fetch generated sets for this matrix
  interface ProductSet {
    id: number;
    setMatrixId: number | null;
    sku: string;
    title: string;
    color: string | null;
    panelCount: number | null;
    basePrice: string | null;
    calculatedPrice: string | null;
    isActive: boolean;
    createdAt: string;
  }
  
  const { data: generatedSets = [], isLoading: isLoadingSets } = useQuery<ProductSet[]>({
    queryKey: [`/api/set-matrices/${matrixId}/sets`],
    enabled: !isNew && !!matrixId && !!user,
  });

  const { data: colorDictionary = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "color" }],
    enabled: !!user,
  });

  // Fetch product groups dictionary
  const { data: productGroups = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "product_group" }],
    enabled: !!user,
  });

  // Fetch product types dictionary
  const { data: productTypes = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "product_type" }],
    enabled: !!user,
  });

  // Fetch dimension dictionaries
  const { data: dimensionLengths = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "dimension_length" }],
    enabled: !!user,
  });

  const { data: dimensionWidths = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "dimension_width" }],
    enabled: !!user,
  });

  const { data: dimensionHeights = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "dimension_height" }],
    enabled: !!user,
  });

  const { data: doorOptions = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "door" }],
    enabled: !!user,
  });

  const { data: legOptions = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "leg" }],
    enabled: !!user,
  });

  // Fetch unit and color options for UNIT-COLOR combinations
  const { data: unitOptions = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "unit" }],
    enabled: !!user,
  });

  const { data: colorOptionsDict = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "color" }],
    enabled: !!user,
  });

  // Generate UNIT-COLOR combinations
  const availableColorOptions = useMemo(() => {
    if (!unitOptions || !colorOptionsDict) return [];
    const combinations: Array<{ value: string; label: string }> = [];
    unitOptions.forEach(unit => {
      colorOptionsDict.forEach(color => {
        combinations.push({
          value: `${unit.code}-${color.code}`,
          label: `${unit.code}-${color.code}`
        });
      });
    });
    return combinations;
  }, [unitOptions, colorOptionsDict]);

  // Generate preview name from current form values
  const previewName = useMemo(() => {
    const nameParts: string[] = [];
    
    // Helper function to get shortName from dictionary
    const getShortName = (code: string, dictionaries: Dictionary[]): string => {
      const item = dictionaries.find(d => d.code === code);
      return item?.shortName || code;
    };
    
    // 1. Prefix
    if (namePrefix.trim()) nameParts.push(namePrefix.trim());
    
    // 2. Grupa produktów (use shortName)
    if (productGroup) {
      const shortName = getShortName(productGroup, productGroups);
      nameParts.push(shortName);
    }
    
    // 3. Długość × Szerokość (konwersja mm -> cm)
    if (length && width) {
      const lengthCm = Math.round(parseFloat(length) / 10);
      const widthCm = Math.round(parseFloat(width) / 10);
      nameParts.push(`${lengthCm}x${widthCm}cm`);
    }
    
    // 4. Rodzaje produktów z komponentów (use shortName)
    const typeNamesList: string[] = [];
    let totalPanelCount = 0;
    
    components.forEach((comp) => {
      const compType = comp.componentType || '';
      const compLength = comp.length ? Math.round(parseFloat(comp.length) / 10) : null;
      
      if (compType === 'PANEL') {
        totalPanelCount += (comp.quantity || 0);
      } else if (compType && compLength) {
        const shortName = getShortName(compType, productTypes);
        typeNamesList.push(`${shortName}${compLength}`);
      } else if (compType) {
        const shortName = getShortName(compType, productTypes);
        typeNamesList.push(shortName);
      }
    });
    
    typeNamesList.forEach(type => nameParts.push(type));
    
    // 5. Liczba paneli (z komponentów)
    if (totalPanelCount > 0) {
      nameParts.push(`${totalPanelCount} Panel${totalPanelCount > 1 ? 'e' : ''}`);
    }
    
    // 6. Drzwi (use shortName)
    if (doors) {
      const shortName = getShortName(doors, doorOptions);
      nameParts.push(shortName);
    }
    
    // 7. Nogi (use shortName)
    if (legs) {
      const shortName = getShortName(legs, legOptions);
      nameParts.push(shortName);
    }
    
    // 8. Kolor - pokazujemy przykładowy pierwszy kolor
    if (colors.length > 0) {
      nameParts.push(colors[0]);
    }
    
    // 9. Suffix
    if (nameSuffix.trim()) nameParts.push(nameSuffix.trim());
    
    return nameParts.filter(Boolean).join(' ').trim();
  }, [namePrefix, productGroup, productGroups, length, width, components, productTypes, doors, doorOptions, legs, legOptions, colors, nameSuffix]);

  // Update form state when matrix data changes
  useEffect(() => {
    if (matrix) {
      setName(matrix.name);
      setNamePrefix(matrix.namePrefix || "");
      setNameSuffix(matrix.nameSuffix || "");
      setProductGroup(matrix.productGroup || "");
      setLength(matrix.length || "");
      setWidth(matrix.width || "");
      setHeight(matrix.height || "");
      setDoors(matrix.doors || "");
      setLegs(matrix.legs || "");
      setImageUrl(matrix.imageUrl || "");
      setBasePriceModifier(matrix.basePriceModifier || "0");
      setPriceModifierType(matrix.priceModifierType || "percent");
      setColors(matrix.colors || []);
      setSelectedColors(matrix.selectedColors || []);
      setColorOptionsMatrix(matrix.colorOptions || {});
      setUseEnhancedDescription(matrix.useEnhancedDescription || false);
      
      // Convert components for editing
      const componentsData = Array.isArray(matrix.components) 
        ? matrix.components.map(comp => ({
            ...comp,
            length: comp.length ? String(comp.length) : "",
            width: comp.width ? String(comp.width) : "",
          }))
        : [];
      setComponents(componentsData);
    }
  }, [matrix]);

  const createMutation = useMutation({
    mutationFn: async (data: { name: string; productGroup: string | null; imageUrl: string | null; components: any[]; colors: string[]; colorOptions?: Record<string, string[]> }) => {
      const response = await fetch("/api/set-matrices", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(data),
      });
      if (!response.ok) {
        throw new Error(await response.text());
      }
      return response.json();
    },
    onSuccess: (newMatrix) => {
      queryClient.invalidateQueries({ queryKey: ["/api/set-matrices"] });
      toast({
        title: "Sukces",
        description: "Matryca została utworzona",
      });
      setLocation(`/set-matrices/settings/${newMatrix.id}`);
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async (data: { name: string; productGroup: string | null; imageUrl: string | null; components: any[]; colors?: string[]; colorOptions?: Record<string, string[]> }) => {
      const response = await fetch(`/api/set-matrices/${matrixId}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(data),
      });
      if (!response.ok) {
        throw new Error(await response.text());
      }
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/set-matrices"] });
      queryClient.invalidateQueries({ queryKey: [`/api/set-matrices/${matrixId}`] });
      toast({
        title: "Sukces",
        description: "Matryca została zaktualizowana",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const updateMatrixMutation = useMutation({
    mutationFn: async (updates: Partial<SetMatrix>) => {
      const response = await apiRequest("PUT", `/api/set-matrices/${matrixId}`, updates);
      return await response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/set-matrices/${matrixId}`] });
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await fetch(`/api/set-matrices/${id}`, {
        method: "DELETE",
      });
      if (!response.ok) {
        throw new Error(await response.text());
      }
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/set-matrices"] });
      toast({
        title: "Sukces",
        description: "Matryca została usunięta",
      });
      setLocation("/set-matrices");
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const previewMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("POST", `/api/set-matrices/${id}/preview-generation`, {});
    },
    onSuccess: (data: any) => {
      setPreviewData(data);
      setShowPreviewDialog(true);
      toast({
        title: "Podgląd wygenerowany",
        description: data?.preview?.setName ? `Podgląd dla: ${data.preview.setName}` : "Podgląd wygenerowany pomyślnie",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd generowania podglądu",
        description: error.message || "Nie udało się wygenerować podglądu.",
        variant: "destructive",
      });
    },
  });

  const generateSetsMutation = useMutation({
    mutationFn: async (id: number) => {
      const sessionId = `generate-sets-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
      return apiRequest("POST", `/api/set-matrices/${id}/generate`, { sessionId });
    },
    onSuccess: (data: any) => {
      setIsGenerating(false);
      setShowPreviewDialog(false);
      queryClient.invalidateQueries({ queryKey: [`/api/set-matrices/${matrixId}/sets`] });
      toast({
        title: "Zestawy wygenerowane",
        description: `Pomyślnie wygenerowano ${data.setsGenerated} zestawów produktów.`,
      });
    },
    onError: (error: any) => {
      setIsGenerating(false);
      toast({
        title: "Błąd generowania",
        description: error.message || "Nie udało się wygenerować zestawów.",
        variant: "destructive",
      });
    },
  });

  const handleStartGeneration = () => {
    if (isNew || !matrixId) {
      toast({
        title: "Zapisz najpierw",
        description: "Musisz najpierw zapisać matrycę przed rozpoczęciem generowania.",
        variant: "destructive",
      });
      return;
    }

    setLocation(`/set-matrices/${matrixId}/generate`);
  };

  const handleGenerate = () => {
    if (isNew || !matrixId) {
      toast({
        title: "Zapisz najpierw",
        description: "Musisz najpierw zapisać matrycę przed generowaniem zestawów.",
        variant: "destructive",
      });
      return;
    }

    setIsGenerating(true);
    generateSetsMutation.mutate(matrixId, {
      onSettled: () => setIsGenerating(false),
    });
  };

  const handleSave = () => {
    if (!name.trim()) {
      toast({
        title: "Błąd",
        description: "Nazwa matrycy jest wymagana",
        variant: "destructive",
      });
      return;
    }

    // Convert string values to appropriate types for database
    const processedComponents = components.map(comp => ({
      name: comp.name || "",
      componentType: comp.componentType,
      length: comp.length ? parseInt(comp.length) : null,
      width: comp.width ? parseInt(comp.width) : null,
      quantity: comp.quantity,
    }));

    const data = {
      name: name.trim(),
      namePrefix: namePrefix || null,
      nameSuffix: nameSuffix || null,
      productGroup: productGroup || null,
      length: length || null,
      width: width || null,
      height: height || null,
      doors: doors || null,
      legs: legs || null,
      imageUrl: imageUrl.trim() || null,
      basePriceModifier: basePriceModifier || "0",
      priceModifierType: priceModifierType || "percent",
      components: processedComponents,
      colors: colors,
      selectedColors: selectedColors,
      colorOptions: colorOptionsMatrix,
      useEnhancedDescription: useEnhancedDescription,
    };

    if (isNew) {
      createMutation.mutate(data);
    } else {
      updateMutation.mutate(data);
    }
  };

  const handleDelete = () => {
    if (matrixId) {
      deleteMutation.mutate(matrixId);
    }
  };

  const handleAddColor = (colorCode: string) => {
    if (!colorCode || colors.includes(colorCode)) return;
    
    const newColors = [...colors, colorCode];
    setColors(newColors);
    
    if (!isNew) {
      updateMatrixMutation.mutate({ colors: newColors });
    }
    setColorOpen(false);
    setColorSearchInput("");
  };

  const handleRemoveColor = (colorToRemove: string) => {
    const newColors = colors.filter(c => c !== colorToRemove);
    setColors(newColors);
    
    // Also remove from selectedColors
    const newSelectedColors = selectedColors.filter(c => c !== colorToRemove);
    setSelectedColors(newSelectedColors);
    
    if (!isNew) {
      // Also remove color images and color options for this color
      const newColorImages = { ...(matrix?.colorImages || {}) };
      delete newColorImages[colorToRemove];
      
      const newColorOptions = { ...colorOptionsMatrix };
      delete newColorOptions[colorToRemove];
      setColorOptionsMatrix(newColorOptions);
      
      updateMatrixMutation.mutate({ 
        colors: newColors,
        selectedColors: newSelectedColors,
        colorImages: newColorImages,
        colorOptions: newColorOptions
      });
    } else {
      // For new matrices, just update local state
      const newColorOptions = { ...colorOptionsMatrix };
      delete newColorOptions[colorToRemove];
      setColorOptionsMatrix(newColorOptions);
    }
  };
  
  const toggleColorForGeneration = async (colorName: string) => {
    const newSelectedColors = selectedColors.includes(colorName)
      ? selectedColors.filter(c => c !== colorName)
      : [...selectedColors, colorName];
    
    setSelectedColors(newSelectedColors);
    
    // Auto-save to backend
    if (!isNew && matrixId) {
      try {
        await apiRequest("PUT", `/api/set-matrices/${matrixId}`, {
          selectedColors: newSelectedColors
        });
        queryClient.invalidateQueries({ queryKey: [`/api/set-matrices/${matrixId}`] });
      } catch (error) {
        console.error("Failed to auto-save selected colors:", error);
        toast({
          title: "Błąd",
          description: "Nie udało się zapisać wybranych kolorów",
          variant: "destructive",
        });
      }
    }
  };

  const handleColorImageUpload = async (colorName: string, files: FileList | null) => {
    if (!files || files.length === 0 || !matrixId) return;

    for (const file of Array.from(files)) {
      const formData = new FormData();
      formData.append('image', file);
      formData.append('colorName', colorName);

      try {
        const response = await fetch(`/api/set-matrices/${matrixId}/color-images`, {
          method: 'POST',
          body: formData,
          credentials: 'include',
        });

        if (!response.ok) {
          const text = await response.text();
          throw new Error(`${response.status}: ${text}`);
        }

        await response.json();
        
        queryClient.invalidateQueries({ queryKey: [`/api/set-matrices/${matrixId}`] });

        toast({
          title: "Zdjęcie dodane",
          description: `Dodano zdjęcie dla koloru ${colorName}`,
        });
      } catch (error: any) {
        toast({
          title: "Błąd uploadu",
          description: error.message || "Nie udało się dodać zdjęcia",
          variant: "destructive",
        });
      }
    }
  };

  const handleColorImageDelete = async (colorName: string, imageUrl: string) => {
    if (!matrixId) return;

    const filename = imageUrl.split('/').pop();
    if (!filename) return;

    try {
      const response = await fetch(`/api/set-matrices/${matrixId}/color-images/${encodeURIComponent(colorName)}/${encodeURIComponent(filename)}`, {
        method: 'DELETE',
        credentials: 'include',
      });

      if (!response.ok) {
        const text = await response.text();
        throw new Error(`${response.status}: ${text}`);
      }

      queryClient.invalidateQueries({ queryKey: [`/api/set-matrices/${matrixId}`] });

      toast({
        title: "Zdjęcie usunięte",
        description: `Usunięto zdjęcie dla koloru ${colorName}`,
      });
    } catch (error: any) {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć zdjęcia",
        variant: "destructive",
      });
    }
  };

  const handleImageReorder = async (colorName: string, fromIndex: number, toIndex: number) => {
    if (!matrixId || !matrix?.colorImages || fromIndex === toIndex) return;
    
    const currentImages = [...(matrix.colorImages[colorName] || [])];
    if (fromIndex < 0 || fromIndex >= currentImages.length || toIndex < 0 || toIndex >= currentImages.length) return;
    
    // Reorder array
    const [movedImage] = currentImages.splice(fromIndex, 1);
    currentImages.splice(toIndex, 0, movedImage);
    
    // Update matrix with new order
    const updatedColorImages = {
      ...matrix.colorImages,
      [colorName]: currentImages
    };
    
    try {
      await updateMatrixMutation.mutateAsync({ colorImages: updatedColorImages });
      
      toast({
        title: "Kolejność zmieniona",
        description: `Zmieniono kolejność zdjęć dla koloru ${colorName}`,
      });
    } catch (error: any) {
      toast({
        title: "Błąd zmiany kolejności",
        description: error.message || "Nie udało się zmienić kolejności",
        variant: "destructive",
      });
    }
  };

  const handleAddComponent = () => {
    setComponents([
      ...components,
      {
        name: "",
        componentType: "",
        length: "",
        width: "",
        quantity: 1,
      },
    ]);
  };

  const handleRemoveComponent = (index: number) => {
    setComponents(components.filter((_, i) => i !== index));
  };

  const handleUpdateComponent = (index: number, field: string, value: any) => {
    const updated = [...components];
    updated[index] = { ...updated[index], [field]: value };
    setComponents(updated);
  };

  const onDrop = useCallback((acceptedFiles: File[]) => {
    const file = acceptedFiles[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = () => {
        setImageUrl(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  }, []);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop,
    accept: {
      'image/*': ['.png', '.jpg', '.jpeg', '.gif', '.webp']
    },
    maxFiles: 1,
    multiple: false
  });

  const handleRemoveImage = () => {
    setImageUrl("");
  };

  if (isLoading && !isNew) {
    return (
      <div className="p-3 sm:p-6">
        <p className="text-muted-foreground">Ładowanie...</p>
      </div>
    );
  }

  const colorImages = matrix?.colorImages || {};

  const filteredColors = colorDictionary.filter(item => {
    const searchLower = colorSearchInput.toLowerCase();
    return item.code.toLowerCase().includes(searchLower) ||
           (item.readableName || "").toLowerCase().includes(searchLower);
  });

  return (
    <div className="p-3 sm:p-6 space-y-4 sm:space-y-6">
      <div className="flex flex-col sm:flex-row gap-3 sm:gap-0 sm:items-center sm:justify-between">
        <div className="flex items-center gap-3">
          <Button
            variant="outline"
            size="icon"
            onClick={() => setLocation("/set-matrices")}
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div>
            <h1 className="text-2xl sm:text-3xl font-bold">
              {isNew ? "Nowa Matryca Zestawu" : name || "Edytuj matrycę"}
            </h1>
            <p className="text-muted-foreground mt-1 text-sm sm:text-base">
              {isNew ? "Utwórz nowy szablon zestawu produktów" : "Edytuj szablon zestawu produktów"}
            </p>
          </div>
        </div>
        <div className="flex gap-2">
          {!isNew && (
            <Button
              onClick={handleStartGeneration}
              disabled={isGenerating}
              data-testid="button-generate-sets"
              className="flex-1 sm:flex-none"
              variant="default"
            >
              <Play className="h-4 w-4 mr-2" />
              Generuj zestawy
            </Button>
          )}
          <Button
            onClick={handleSave}
            disabled={createMutation.isPending || updateMutation.isPending}
            data-testid="button-save-matrix"
            className="flex-1 sm:flex-none"
          >
            <Save className="h-4 w-4 mr-2" />
            {createMutation.isPending || updateMutation.isPending
              ? "Zapisywanie..."
              : isNew
              ? "Utwórz matrycę"
              : "Zapisz zmiany"}
          </Button>
          {!isNew && (
            <Button
              variant="destructive"
              onClick={() => setShowDeleteDialog(true)}
              data-testid="button-delete-matrix"
              className="flex-1 sm:flex-none"
            >
              <Trash2 className="h-4 w-4 mr-2" />
              Usuń
            </Button>
          )}
        </div>
      </div>

      {/* Podstawowe informacje i zdjęcie */}
      <Card>
        <CardContent className="p-4 sm:p-6">
          {/* 3 kolumny: 2 z polami formularza + 1 ze zdjęciem */}
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-4">
            {/* Kolumna 1 - Nazwa i Prefix */}
            <div className="space-y-4">
              <div>
                <Label htmlFor="name">Nazwa matrycy</Label>
                <Input
                  id="name"
                  value={name}
                  onChange={(e) => setName(e.target.value)}
                  placeholder="np. Zestaw kuchenny podstawowy"
                  data-testid="input-matrix-name"
                />
              </div>

              <div>
                <Label htmlFor="namePrefix">Prefix nazwy</Label>
                <Input
                  id="namePrefix"
                  value={namePrefix}
                  onChange={(e) => setNamePrefix(e.target.value)}
                  placeholder="np. Garderoba"
                  data-testid="input-name-prefix"
                />
              </div>
            </div>

            {/* Kolumna 2 - Grupa i Suffix */}
            <div className="space-y-4">
              <div>
                <Label htmlFor="productGroup">Grupa produktów</Label>
                <DictionaryComboboxWithAdd
                  items={productGroups}
                  value={productGroup}
                  onChange={setProductGroup}
                  dictionaryType="product_group"
                  placeholder="Wybierz grupę produktów"
                  searchPlaceholder="Szukaj grupy..."
                  emptyText="Nie znaleziono grup"
                  testId="select-product-group"
                  valueField="code"
                  displayField="name"
                />
              </div>

              <div>
                <Label htmlFor="nameSuffix">Suffix nazwy</Label>
                <Input
                  id="nameSuffix"
                  value={nameSuffix}
                  onChange={(e) => setNameSuffix(e.target.value)}
                  placeholder="np. Premium"
                  data-testid="input-name-suffix"
                />
              </div>
            </div>

            {/* Kolumna 3 - Zdjęcie poglądowe */}
            <div>
              <Label>Zdjęcie poglądowe</Label>
              {imageUrl ? (
                <div className="space-y-2">
                  <div className="relative inline-block">
                    <img 
                      src={imageUrl} 
                      alt="Podgląd" 
                      className="h-20 w-auto object-contain border"
                    />
                    <Button
                      type="button"
                      variant="destructive"
                      size="icon"
                      className="absolute -top-2 -right-2 h-6 w-6"
                      onClick={handleRemoveImage}
                      data-testid="button-remove-image"
                    >
                      <X className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
              ) : (
                <div
                  {...getRootProps()}
                  className={`border-2 border-dashed p-4 text-center cursor-pointer transition-colors ${
                    isDragActive ? 'border-primary bg-primary/5' : 'border-muted-foreground/25 hover:border-primary/50'
                  }`}
                  data-testid="dropzone-image"
                >
                  <input {...getInputProps()} />
                  <Upload className="h-6 w-6 mx-auto mb-2 text-muted-foreground" />
                  {isDragActive ? (
                    <p className="text-sm text-muted-foreground">Upuść zdjęcie tutaj...</p>
                  ) : (
                    <>
                      <p className="text-sm text-muted-foreground mb-1">
                        Przeciągnij i upuść zdjęcie lub kliknij aby wybrać
                      </p>
                      <p className="text-xs text-muted-foreground">
                        PNG, JPG, GIF, WEBP (maks. 5MB)
                      </p>
                    </>
                  )}
                </div>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* SEKCJA WYMIARÓW, DRZWI, NOGI I CENY */}
      <Card>
        <CardHeader>
          <CardTitle>Parametry zestawu i generowanie nazwy</CardTitle>
          <CardDescription>Wymiary, komponenty i cena - wpływają na nazwę produktu</CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Wymiary */}
          <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
            <div>
              <Label htmlFor="length">Długość (mm)</Label>
              <DictionaryComboboxWithAdd
                items={dimensionLengths}
                value={length}
                onChange={setLength}
                dictionaryType="dimension_length"
                placeholder="Wybierz długość"
                searchPlaceholder="Szukaj długości..."
                emptyText="Nie znaleziono długości"
                testId="select-length"
                valueField="code"
                displayField="name"
              />
            </div>
            <div>
              <Label htmlFor="width">Szerokość (mm)</Label>
              <DictionaryComboboxWithAdd
                items={dimensionWidths}
                value={width}
                onChange={setWidth}
                dictionaryType="dimension_width"
                placeholder="Wybierz szerokość"
                searchPlaceholder="Szukaj szerokości..."
                emptyText="Nie znaleziono szerokości"
                testId="select-width"
                valueField="code"
                displayField="name"
              />
            </div>
            <div>
              <Label htmlFor="height">Wysokość (mm)</Label>
              <DictionaryComboboxWithAdd
                items={dimensionHeights}
                value={height}
                onChange={setHeight}
                dictionaryType="dimension_height"
                placeholder="Wybierz wysokość"
                searchPlaceholder="Szukaj wysokości..."
                emptyText="Nie znaleziono wysokości"
                testId="select-height"
                valueField="code"
                displayField="name"
              />
            </div>
          </div>

          {/* Drzwi i Nogi */}
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
            <div>
              <Label htmlFor="doors">Drzwi</Label>
              <DictionaryComboboxWithAdd
                items={doorOptions}
                value={doors}
                onChange={setDoors}
                dictionaryType="door"
                placeholder="Wybierz drzwi"
                searchPlaceholder="Szukaj drzwi..."
                emptyText="Nie znaleziono drzwi"
                testId="select-doors"
                valueField="code"
                displayField="name"
              />
            </div>
            <div>
              <Label htmlFor="legs">Nogi</Label>
              <DictionaryComboboxWithAdd
                items={legOptions}
                value={legs}
                onChange={setLegs}
                dictionaryType="leg"
                placeholder="Wybierz nogi"
                searchPlaceholder="Szukaj nóg..."
                emptyText="Nie znaleziono nóg"
                testId="select-legs"
                valueField="code"
                displayField="name"
              />
            </div>
          </div>

          {/* Modyfikatory ceny */}
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
            <div>
              <Label htmlFor="basePriceModifier">Modyfikator ceny</Label>
              <Input
                id="basePriceModifier"
                type="number"
                step="0.01"
                value={basePriceModifier}
                onChange={(e) => setBasePriceModifier(e.target.value)}
                placeholder="np. 10"
                data-testid="input-base-price-modifier"
              />
            </div>
            <div>
              <Label htmlFor="priceModifierType">Typ modyfikatora</Label>
              <select
                id="priceModifierType"
                value={priceModifierType}
                onChange={(e) => setPriceModifierType(e.target.value)}
                className="flex h-9 w-full rounded-md border border-input bg-transparent px-3 py-1 text-sm shadow-sm transition-colors file:border-0 file:bg-transparent file:text-sm file:font-medium placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring disabled:cursor-not-allowed disabled:opacity-50"
                data-testid="select-price-modifier-type"
              >
                <option value="percent">Procent (%)</option>
                <option value="fixed">Kwota (PLN)</option>
              </select>
            </div>
          </div>

          {/* Opcje generowania */}
          <div className="flex items-start space-x-2 pt-4">
            <Checkbox
              id="useEnhancedDescription"
              checked={useEnhancedDescription}
              onCheckedChange={(checked) => setUseEnhancedDescription(checked === true)}
              data-testid="checkbox-use-enhanced-description"
            />
            <div className="space-y-1">
              <Label
                htmlFor="useEnhancedDescription"
                className="text-sm font-medium leading-none cursor-pointer"
              >
                Użyj rozszerzonych opisów AI
              </Label>
              <p className="text-xs text-muted-foreground">
                Automatycznie generuj szczegółowe opisy zestawów na podstawie opisów AI komponentów
              </p>
            </div>
          </div>

          {/* Live preview nazwy */}
          <div className="border-t pt-6">
            <Label className="text-base font-semibold mb-3 block">Podgląd wygenerowanej nazwy</Label>
            <div className="bg-muted p-4 rounded-md space-y-2">
              <div className="font-mono text-sm break-all" data-testid="text-name-preview">
                {previewName || <span className="text-muted-foreground italic">Wprowadź dane aby zobaczyć podgląd...</span>}
              </div>
              <div className="text-xs text-muted-foreground">
                Liczba znaków: <span className="font-semibold">{previewName.length}</span>
                {previewName.length > 100 && <span className="text-destructive ml-2">(Uwaga: nazwa może być zbyt długa)</span>}
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* SEKCJA KOLORÓW */}
      <Card>
        <CardHeader className="p-3 sm:p-4">
          <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-3">
            <Popover open={colorOpen} onOpenChange={setColorOpen}>
              <PopoverTrigger asChild>
                <Button
                  variant="default"
                  size="default"
                  role="combobox"
                  aria-expanded={colorOpen}
                  className="w-full sm:w-auto"
                  data-testid="button-add-color"
                >
                  <Plus className="h-4 w-4 mr-2" />
                  Dodaj kolor
                  <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                </Button>
              </PopoverTrigger>
              <PopoverContent className="w-[250px] p-0">
                <Command>
                  <CommandInput
                    placeholder="Szukaj koloru..."
                    value={colorSearchInput}
                    onValueChange={setColorSearchInput}
                  />
                  <CommandList>
                    <CommandEmpty>Brak wyników</CommandEmpty>
                    <CommandGroup>
                      {filteredColors.map((color) => (
                        <CommandItem
                          key={color.id}
                          value={color.code}
                          onSelect={() => handleAddColor(color.code)}
                          disabled={colors.includes(color.code)}
                        >
                          <Check
                            className={cn(
                              "mr-2 h-4 w-4",
                              colors.includes(color.code) ? "opacity-100" : "opacity-0"
                            )}
                          />
                          <div className="flex flex-col">
                            <span className="font-medium">{color.code}</span>
                            <span className="text-xs text-muted-foreground">
                              {color.readableName}
                            </span>
                          </div>
                        </CommandItem>
                      ))}
                    </CommandGroup>
                  </CommandList>
                </Command>
              </PopoverContent>
            </Popover>
            <div className="flex items-center gap-2">
              <Palette className="h-4 w-4" />
              <div>
                <CardTitle className="text-base sm:text-lg">Kolory ({colors.length})</CardTitle>
                <CardDescription className="text-xs">
                  Zarządzaj kolorami i ich zdjęciami dla generatora zestawów
                </CardDescription>
              </div>
            </div>
          </div>
        </CardHeader>
        <CardContent className="p-3 sm:p-4">
          {colors.length > 0 ? (
            <div className="border rounded-lg overflow-x-auto">
              <Table className="min-w-[700px]">
                <TableHeader>
                  <TableRow>
                    <TableHead className="w-10 py-1 px-2 h-8">#</TableHead>
                    <TableHead className="min-w-[150px] py-1 px-1.5 h-8 text-xs">Kolor</TableHead>
                    <TableHead className="w-10 py-1 px-1 h-8 text-xs text-center">Gen.</TableHead>
                    <TableHead className="min-w-[160px] py-1 px-2 text-xs h-8">Opcje</TableHead>
                    <TableHead className="min-w-[130px] py-1 px-2 text-xs h-8">Zdjęcia</TableHead>
                    <TableHead className="w-10 py-1 px-2 h-8"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {colors.map((color, index) => {
                    const colorDict = colorDictionary.find(c => c.code === color);
                    const isSelectedForGeneration = selectedColors.includes(color);
                    return (
                    <TableRow 
                      key={index}
                      className={isSelectedForGeneration ? 'bg-green-50 dark:bg-green-900/20' : ''}
                    >
                      <TableCell className="font-mono text-muted-foreground py-1 px-2">
                        {index + 1}
                      </TableCell>
                      <TableCell className="py-1 px-1.5">
                        <Popover 
                          open={colorOpenRows.has(index)} 
                          onOpenChange={(open) => {
                            if (open) {
                              setColorOpenRows(prev => {
                                const newSet = new Set(prev);
                                newSet.add(index);
                                return newSet;
                              });
                            } else {
                              setColorSearchInputs(prev => {
                                const newMap = new Map(prev);
                                newMap.delete(index);
                                return newMap;
                              });
                              setColorOpenRows(prev => {
                                const newSet = new Set(prev);
                                newSet.delete(index);
                                return newSet;
                              });
                            }
                          }}
                        >
                          <PopoverTrigger asChild>
                            <Button
                              variant="outline"
                              role="combobox"
                              aria-expanded={colorOpenRows.has(index)}
                              className="w-full justify-between"
                              style={colorDict?.color ? {
                                backgroundColor: colorDict.color,
                                color: getTextColorForBackground(colorDict.color)
                              } : undefined}
                              data-testid={`select-color-${index}`}
                            >
                              {color}
                              <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                            </Button>
                          </PopoverTrigger>
                          <PopoverContent className="w-[300px] p-0">
                            <Command shouldFilter={false}>
                              <CommandInput 
                                placeholder="Szukaj koloru..." 
                                value={colorSearchInputs.get(index) || ""}
                                onValueChange={(value) => {
                                  const newMap = new Map(colorSearchInputs);
                                  newMap.set(index, value);
                                  setColorSearchInputs(newMap);
                                }}
                              />
                              <CommandList>
                                <CommandEmpty>Nie znaleziono koloru</CommandEmpty>
                                <CommandGroup>
                                  {colorDictionary
                                    .filter((opt) => {
                                      const searchTerm = (colorSearchInputs.get(index) || "").toLowerCase();
                                      if (!searchTerm) return true;
                                      const name = (opt.name || "").toLowerCase();
                                      const readableName = (opt.readableName || "").toLowerCase();
                                      const code = (opt.code || "").toLowerCase();
                                      return name.includes(searchTerm) || 
                                             readableName.includes(searchTerm) || 
                                             code.includes(searchTerm);
                                    })
                                    .map((opt) => (
                                      <CommandItem
                                        key={opt.id}
                                        value={opt.code}
                                        onSelect={() => {
                                          const newColors = [...colors];
                                          newColors[index] = opt.code;
                                          setColors(newColors);
                                          const newSet = new Set(colorOpenRows);
                                          newSet.delete(index);
                                          setColorOpenRows(newSet);
                                          const newMap = new Map(colorSearchInputs);
                                          newMap.delete(index);
                                          setColorSearchInputs(newMap);
                                        }}
                                      >
                                        <Check
                                          className={`mr-2 h-4 w-4 ${color === opt.code ? "opacity-100" : "opacity-0"}`}
                                        />
                                        {opt.code}
                                      </CommandItem>
                                    ))}
                                </CommandGroup>
                              </CommandList>
                            </Command>
                          </PopoverContent>
                        </Popover>
                      </TableCell>
                      <TableCell className="py-1 px-1 text-center">
                        <Checkbox
                          className="h-3.5 w-3.5"
                          checked={selectedColors.includes(color)}
                          onCheckedChange={() => toggleColorForGeneration(color)}
                          data-testid={`checkbox-generate-${index}`}
                        />
                      </TableCell>
                      <TableCell className="py-1 px-2">
                        <div className="flex flex-wrap gap-1">
                          {(colorOptionsMatrix[color] || []).map((option, optIdx) => (
                            <Badge 
                              key={optIdx} 
                              variant="secondary" 
                              className="text-xs"
                              data-testid={`badge-color-option-${index}-${optIdx}`}
                            >
                              {option}
                              <button
                                className="ml-1 hover:text-destructive"
                                onClick={() => {
                                  const newOptions = [...(colorOptionsMatrix[color] || [])];
                                  newOptions.splice(optIdx, 1);
                                  setColorOptionsMatrix({
                                    ...colorOptionsMatrix,
                                    [color]: newOptions
                                  });
                                }}
                                data-testid={`button-remove-option-${index}-${optIdx}`}
                              >
                                <X className="w-3 h-3" />
                              </button>
                            </Badge>
                          ))}
                          <Popover>
                            <PopoverTrigger asChild>
                              <Button
                                variant="outline"
                                size="icon"
                                className="h-6 w-6"
                                data-testid={`button-add-color-option-${index}`}
                              >
                                <Plus className="w-3 h-3" />
                              </Button>
                            </PopoverTrigger>
                            <PopoverContent className="w-80 p-0" align="start">
                              <Command>
                                <CommandInput placeholder="Szukaj opcji..." />
                                <CommandEmpty>Brak wyników</CommandEmpty>
                                <CommandList>
                                  <CommandGroup heading="Dostępne opcje UNIT-COLOR">
                                    {availableColorOptions
                                      .filter(opt => !(colorOptionsMatrix[color] || []).includes(opt.value))
                                      .map((opt) => (
                                        <CommandItem
                                          key={opt.value}
                                          value={opt.value}
                                          onSelect={(selectedValue) => {
                                            const newOptions = [...(colorOptionsMatrix[color] || []), selectedValue];
                                            setColorOptionsMatrix({
                                              ...colorOptionsMatrix,
                                              [color]: newOptions
                                            });
                                          }}
                                        >
                                          {opt.label}
                                        </CommandItem>
                                      ))}
                                    {availableColorOptions.length === 0 && (
                                      <CommandItem disabled>
                                        <span className="text-muted-foreground text-xs italic">
                                          Ładowanie opcji...
                                        </span>
                                      </CommandItem>
                                    )}
                                  </CommandGroup>
                                </CommandList>
                              </Command>
                            </PopoverContent>
                          </Popover>
                        </div>
                      </TableCell>
                      <TableCell className="p-0">
                        <div className="flex items-center gap-1.5 flex-wrap pt-1 px-2">
                          {(() => {
                            const images = colorImages[color] || [];
                            const isExpanded = expandedColorImages.has(index);
                            const visibleImages = isExpanded ? images : images.slice(0, 2);
                            const remainingCount = images.length - 2;
                            
                            return (
                              <>
                                {visibleImages.map((imageUrl, imgIndex) => {
                                  const actualIndex = isExpanded ? imgIndex : imgIndex;
                                  const isDragging = draggedImage?.color === color && draggedImage?.index === actualIndex;
                                  
                                  return (
                                    <div 
                                      key={imgIndex} 
                                      className={`relative group cursor-move flex-shrink-0 w-12 h-12 rounded border overflow-hidden transition-opacity ${isDragging ? 'opacity-50' : ''}`}
                                      draggable={true}
                                      onDragStart={(e) => {
                                        setDraggedImage({ color, index: actualIndex });
                                        e.dataTransfer.effectAllowed = 'move';
                                      }}
                                      onDragOver={(e) => {
                                        e.preventDefault();
                                        e.dataTransfer.dropEffect = 'move';
                                      }}
                                      onDrop={(e) => {
                                        e.preventDefault();
                                        if (draggedImage && draggedImage.color === color && draggedImage.index !== actualIndex) {
                                          handleImageReorder(color, draggedImage.index, actualIndex);
                                        }
                                        setDraggedImage(null);
                                      }}
                                      onDragEnd={() => setDraggedImage(null)}
                                    >
                                      <img
                                        src={imageUrl}
                                        alt={`${color} ${actualIndex + 1}`}
                                        className="w-full h-full object-cover pointer-events-none"
                                      />
                                      
                                      <div className="absolute inset-0 bg-black/60 rounded opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center pointer-events-none">
                                        <span className="text-white text-2xl font-bold">
                                          {actualIndex + 1}
                                        </span>
                                      </div>
                                      
                                      {actualIndex === 0 && (
                                        <span className="absolute top-1 left-1 bg-primary text-primary-foreground text-xs px-1.5 py-0.5 rounded-sm font-medium z-10 pointer-events-none">
                                          1
                                        </span>
                                      )}
                                      <Button
                                        variant="destructive"
                                        size="icon"
                                        className="absolute bottom-6 left-1 w-5 h-5 opacity-0 group-hover:opacity-100 transition-opacity z-10"
                                        onClick={() => handleColorImageDelete(color, imageUrl)}
                                        data-testid={`button-delete-image-${index}-${actualIndex}`}
                                      >
                                        <X className="w-3 h-3" />
                                      </Button>
                                    </div>
                                  );
                                })}
                                
                                {!isExpanded && remainingCount > 0 && (
                                  <Button
                                    variant="outline"
                                    size="sm"
                                    className="h-12 px-2 text-xs"
                                    onClick={() => {
                                      setExpandedColorImages(prev => {
                                        const newSet = new Set(prev);
                                        newSet.add(index);
                                        return newSet;
                                      });
                                    }}
                                    data-testid={`button-show-more-images-${index}`}
                                  >
                                    +{remainingCount}
                                  </Button>
                                )}
                                
                                {isExpanded && images.length > 2 && (
                                  <Button
                                    variant="outline"
                                    size="sm"
                                    className="h-12 px-2 text-xs"
                                    onClick={() => {
                                      setExpandedColorImages(prev => {
                                        const newSet = new Set(prev);
                                        newSet.delete(index);
                                        return newSet;
                                      });
                                    }}
                                    data-testid={`button-show-less-images-${index}`}
                                  >
                                    <Eye className="w-3 h-3" />
                                  </Button>
                                )}
                                
                                {!isNew && (
                                  <label
                                    className="w-12 h-12 border-2 border-dashed rounded flex items-center justify-center cursor-pointer hover:border-primary hover:bg-accent/50 transition-colors flex-shrink-0"
                                    data-testid={`label-upload-${index}`}
                                    onDragOver={(e) => {
                                      e.preventDefault();
                                      e.currentTarget.classList.add('border-primary', 'bg-accent');
                                    }}
                                    onDragLeave={(e) => {
                                      e.preventDefault();
                                      e.currentTarget.classList.remove('border-primary', 'bg-accent');
                                    }}
                                    onDrop={(e) => {
                                      e.preventDefault();
                                      e.currentTarget.classList.remove('border-primary', 'bg-accent');
                                      handleColorImageUpload(color, e.dataTransfer.files);
                                    }}
                                  >
                                    <Upload className="w-4 h-4 text-muted-foreground" />
                                    <input
                                      type="file"
                                      accept="image/*"
                                      multiple
                                      className="hidden"
                                      onChange={(e) => handleColorImageUpload(color, e.target.files)}
                                      data-testid={`input-upload-${index}`}
                                    />
                                  </label>
                                )}
                              </>
                            );
                          })()}
                        </div>
                      </TableCell>
                      <TableCell className="py-1 px-2">
                        <Button
                          variant="ghost"
                          size="icon"
                          className="h-6 w-6"
                          onClick={() => handleRemoveColor(color)}
                          data-testid={`button-remove-color-${index}`}
                        >
                          <X className="h-3 w-3" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  );
                  })}
                </TableBody>
              </Table>
            </div>
          ) : (
            <div className="p-6 text-center text-sm text-muted-foreground">
              <p className="mb-3">Brak kolorów. Użyj przycisku "Dodaj kolor" powyżej aby dodać kolory ze słownika.</p>
            </div>
          )}
          {colors.length > 0 && !isNew && (
            <p className="text-xs text-muted-foreground mt-3">
              Przeciągnij i upuść zdjęcia lub kliknij ikonę + aby dodać. Kliknij X aby usunąć kolor.
            </p>
          )}
        </CardContent>
      </Card>

      {/* SEKCJA KOMPONENTÓW */}
      <Card>
        <CardContent className="p-4 sm:p-6 space-y-6">
          <div className="space-y-3">
            <Label>Komponenty zestawu</Label>

            {!Array.isArray(components) || components.length === 0 ? (
              <div className="text-center py-6 border-2 border-dashed">
                <p className="text-sm text-muted-foreground">
                  Brak komponentów. Kliknij "Dodaj komponent" poniżej aby dodać.
                </p>
              </div>
            ) : (
              <div className="space-y-2">
                {components.map((component, index) => (
                  <div key={index} className="flex items-end gap-2 p-2 even:bg-muted/30" data-testid={`component-${index}`}>
                    <div className="flex-1 grid grid-cols-4 gap-2">
                      <div>
                        <Label className="text-xs">Rodzaj</Label>
                        <DictionaryComboboxWithAdd
                          items={productTypes}
                          value={component.componentType}
                          onChange={(value) =>
                            handleUpdateComponent(index, "componentType", value)
                          }
                          dictionaryType="product_type"
                          placeholder="Rodzaj"
                          searchPlaceholder="Szukaj typu..."
                          emptyText="Nie znaleziono"
                          testId={`select-component-type-${index}`}
                          valueField="code"
                          displayField="name"
                          backgroundColor={
                            component.componentType 
                              ? productTypes.find(t => t.code === component.componentType)?.color || undefined
                              : undefined
                          }
                        />
                      </div>
                      <div>
                        <Label className="text-xs">Długość</Label>
                        <DictionaryComboboxWithAdd
                          items={dimensionLengths}
                          value={component.length}
                          onChange={(value) =>
                            handleUpdateComponent(index, "length", value)
                          }
                          dictionaryType="dimension_length"
                          placeholder="Długość"
                          searchPlaceholder="Szukaj..."
                          emptyText="Nie znaleziono"
                          testId={`select-component-length-${index}`}
                          valueField="code"
                          displayField="name"
                        />
                      </div>
                      <div>
                        <Label className="text-xs">Szerokość</Label>
                        <DictionaryComboboxWithAdd
                          items={dimensionWidths}
                          value={component.width}
                          onChange={(value) =>
                            handleUpdateComponent(index, "width", value)
                          }
                          dictionaryType="dimension_width"
                          placeholder="Szerokość"
                          searchPlaceholder="Szukaj..."
                          emptyText="Nie znaleziono"
                          testId={`select-component-width-${index}`}
                          valueField="code"
                          displayField="name"
                        />
                      </div>
                      <div>
                        <Label className="text-xs">Ilość</Label>
                        <Input
                          type="number"
                          min="1"
                          value={component.quantity}
                          onChange={(e) =>
                            handleUpdateComponent(
                              index,
                              "quantity",
                              parseInt(e.target.value) || 1
                            )
                          }
                          data-testid={`input-component-quantity-${index}`}
                          className="h-8"
                        />
                      </div>
                    </div>
                    <Button
                      type="button"
                      variant="ghost"
                      size="icon"
                      onClick={() => handleRemoveComponent(index)}
                      data-testid={`button-remove-component-${index}`}
                      className="h-8 w-8 shrink-0"
                    >
                      <X className="h-4 w-4" />
                    </Button>
                  </div>
                ))}
              </div>
            )}
            
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={handleAddComponent}
              data-testid="button-add-component"
              className="w-full sm:w-auto"
            >
              <Plus className="h-4 w-4 mr-1" />
              Dodaj komponent
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Wygenerowane zestawy w katalogu */}
      {!isNew && (
        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle className="flex items-center gap-2">
                  <Package2 className="h-5 w-5" />
                  Wygenerowane zestawy w katalogu {!isLoadingSets && `(${generatedSets.length})`}
                </CardTitle>
                <CardDescription>
                  Lista zestawów produktów wygenerowanych z tej matrycy
                </CardDescription>
              </div>
              <Link href="/catalog-sets">
                <Button variant="outline" size="sm">
                  <ExternalLink className="h-4 w-4 mr-2" />
                  Wszystkie zestawy
                </Button>
              </Link>
            </div>
          </CardHeader>
          <CardContent>
            {isLoadingSets ? (
              <div className="flex items-center justify-center py-8">
                <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
              </div>
            ) : generatedSets.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                <p>Brak wygenerowanych zestawów dla tej matrycy</p>
                <p className="text-sm mt-2">Użyj przycisku "Generuj zestawy" aby utworzyć zestawy</p>
              </div>
            ) : (
              <div className="rounded-md border">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>SKU</TableHead>
                      <TableHead>Tytuł</TableHead>
                      <TableHead>Kolor</TableHead>
                      <TableHead className="text-center">Panele</TableHead>
                      <TableHead className="text-right">Cena</TableHead>
                      <TableHead className="text-center">Status</TableHead>
                      <TableHead className="text-right">Akcje</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {generatedSets.map((set) => (
                      <TableRow key={set.id}>
                        <TableCell className="font-mono text-xs">{set.sku}</TableCell>
                        <TableCell>
                          <Link href={`/catalog-sets/${set.id}`}>
                            <span className="hover-elevate cursor-pointer px-1 font-medium">
                              {set.title}
                            </span>
                          </Link>
                        </TableCell>
                        <TableCell>
                          {set.color ? (() => {
                            const colorDict = colorDictionary.find(c => c.code === set.color);
                            const bgColor = colorDict?.color || '#94A3B8';
                            const textColor = getTextColorForBackground(bgColor);
                            return (
                              <Badge 
                                variant="outline" 
                                className="text-xs h-5 px-1.5 border-transparent" 
                                style={{ backgroundColor: bgColor, color: textColor }}
                              >
                                {set.color}
                              </Badge>
                            );
                          })() : (
                            <span className="text-muted-foreground text-xs">-</span>
                          )}
                        </TableCell>
                        <TableCell className="text-center">
                          {set.panelCount || '-'}
                        </TableCell>
                        <TableCell className="text-right">
                          {set.basePrice || set.calculatedPrice ? (
                            <span className="text-sm">
                              {Number(set.basePrice || set.calculatedPrice).toFixed(2)} PLN
                            </span>
                          ) : (
                            <span className="text-muted-foreground text-xs">-</span>
                          )}
                        </TableCell>
                        <TableCell className="text-center">
                          <Badge variant={set.isActive ? "default" : "secondary"} className="text-xs">
                            {set.isActive ? "Aktywny" : "Nieaktywny"}
                          </Badge>
                        </TableCell>
                        <TableCell className="text-right">
                          <Link href={`/catalog-sets/${set.id}`}>
                            <Button variant="ghost" size="sm" data-testid={`button-view-set-${set.id}`}>
                              <Eye className="h-4 w-4 mr-1" />
                              Zobacz
                            </Button>
                          </Link>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            )}
          </CardContent>
        </Card>
      )}

      {/* Przyciski na dole strony */}
      <div className="flex flex-col sm:flex-row gap-3 sm:gap-0 sm:justify-end pt-4 border-t">
        <div className="flex gap-2">
          {!isNew && (
            <Button
              onClick={handleStartGeneration}
              disabled={isGenerating}
              data-testid="button-generate-sets-bottom"
              className="flex-1 sm:flex-none"
              variant="default"
            >
              <Play className="h-4 w-4 mr-2" />
              Generuj zestawy
            </Button>
          )}
          <Button
            onClick={handleSave}
            disabled={createMutation.isPending || updateMutation.isPending}
            data-testid="button-save-matrix-bottom"
            className="flex-1 sm:flex-none"
          >
            <Save className="h-4 w-4 mr-2" />
            {createMutation.isPending || updateMutation.isPending
              ? "Zapisywanie..."
              : isNew
              ? "Utwórz matrycę"
              : "Zapisz zmiany"}
          </Button>
          {!isNew && (
            <Button
              variant="destructive"
              onClick={() => setShowDeleteDialog(true)}
              data-testid="button-delete-matrix-bottom"
              className="flex-1 sm:flex-none"
            >
              <Trash2 className="h-4 w-4 mr-2" />
              Usuń
            </Button>
          )}
        </div>
      </div>

      <AlertDialog open={showDeleteDialog} onOpenChange={setShowDeleteDialog}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Czy na pewno usunąć?</AlertDialogTitle>
            <AlertDialogDescription>
              Ta operacja jest nieodwracalna. Matryca "{name}" zostanie trwale usunięta.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDelete}
              data-testid="button-confirm-delete"
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
            >
              Usuń
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      {/* Preview Dialog */}
      <AlertDialog open={showPreviewDialog} onOpenChange={setShowPreviewDialog}>
        <AlertDialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
          <AlertDialogHeader>
            <AlertDialogTitle>Podgląd generowania zestawu</AlertDialogTitle>
            <AlertDialogDescription>
              Sprawdź szczegóły przed wygenerowaniem zestawów produktów
            </AlertDialogDescription>
          </AlertDialogHeader>
          
          {!previewData || !previewData.preview ? (
            <div className="py-8 text-center text-muted-foreground">
              Ładowanie podglądu...
            </div>
          ) : (
            <div className="space-y-4 py-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label className="text-muted-foreground">Nazwa zestawu:</Label>
                  <div className="font-semibold">{previewData.preview.setName}</div>
                </div>
                <div>
                  <Label className="text-muted-foreground">SKU:</Label>
                  <div className="font-mono text-sm">{previewData.preview.sku}</div>
                </div>
                <div>
                  <Label className="text-muted-foreground">Kolor:</Label>
                  <div>{previewData.preview.color || "Brak"}</div>
                </div>
                <div>
                  <Label className="text-muted-foreground">Szacowana cena:</Label>
                  <div className="font-semibold">{previewData.preview.estimatedPrice} PLN</div>
                </div>
              </div>

              <div className="border-t pt-4">
                <Label className="text-muted-foreground mb-2 block">
                  Komponenty zestawu ({previewData.preview.totalComponents}):
                </Label>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="w-12">#</TableHead>
                      <TableHead>Nazwa</TableHead>
                      <TableHead>Typ</TableHead>
                      <TableHead>Wymiary</TableHead>
                      <TableHead className="w-20">Ilość</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {previewData.preview.components.map((comp: any, idx: number) => (
                      <TableRow key={idx}>
                        <TableCell className="font-medium">{comp.position}</TableCell>
                        <TableCell>{comp.name}</TableCell>
                        <TableCell>
                          <Badge variant="outline" className="text-xs">
                            {comp.componentType}
                          </Badge>
                        </TableCell>
                        <TableCell>
                          {comp.length && comp.width 
                            ? `${comp.length} × ${comp.width} mm`
                            : comp.length 
                            ? `${comp.length} mm`
                            : "–"}
                        </TableCell>
                        <TableCell className="text-center">{comp.quantity}</TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>

              <div className="bg-accent/20 border border-accent/40 rounded-lg p-4 text-sm">
                <div className="font-medium mb-1">ℹ️ Informacja</div>
                <div className="text-muted-foreground">
                  Zostanie wygenerowany zestaw dla każdego zaznaczonego koloru.
                  Jeśli nie zaznaczono żadnego koloru, zostanie wygenerowany jeden podstawowy zestaw.
                </div>
              </div>
            </div>
          )}

          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-preview">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleGenerate}
              data-testid="button-confirm-generate"
              disabled={isGenerating}
            >
              {isGenerating ? (
                <>
                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  Generowanie...
                </>
              ) : (
                <>
                  <Play className="h-4 w-4 mr-2" />
                  Generuj zestawy
                </>
              )}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
