import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Plus, Edit, Trash2, Package, Copy, Image as ImageIcon, Search, X, ChevronDown, ChevronUp } from "lucide-react";
import { useState, useEffect, useMemo } from "react";
import { useLocation } from "wouter";
import { useToast } from "@/hooks/use-toast";
import { useAuth } from "@/hooks/use-auth";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";

interface SetMatrix {
  id: number;
  name: string;
  namePrefix?: string | null;
  nameSuffix?: string | null;
  productGroup: string | null;
  length?: string | null;
  width?: string | null;
  height?: string | null;
  doors?: string | null;
  legs?: string | null;
  imageUrl?: string | null;
  components: {
    name: string;
    componentType: string;
    length: number | null;
    width: number | null;
    quantity: number;
  }[];
  colors?: string[];
}

interface Dictionary {
  id: number;
  code: string;
  name: string;
  shortName?: string | null;
}

const STORAGE_KEY = 'set-matrices-filters';

export default function SetMatricesPage() {
  const { toast } = useToast();
  const { user } = useAuth();
  const [, setLocation] = useLocation();
  const [matrixToDelete, setMatrixToDelete] = useState<number | null>(null);

  // Load filters from localStorage
  const loadFilters = () => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY);
      if (saved) {
        return JSON.parse(saved);
      }
    } catch (error) {
      console.error('Error loading filters from localStorage:', error);
    }
    return {};
  };

  const savedFilters = loadFilters();

  // Filter states
  const [searchQuery, setSearchQuery] = useState(savedFilters.searchQuery || "");
  const [filterGroup, setFilterGroup] = useState<string>(savedFilters.filterGroup || "");
  const [filtersExpanded, setFiltersExpanded] = useState(false);

  const { data: matrices = [], isLoading } = useQuery<SetMatrix[]>({
    queryKey: ["/api/set-matrices"],
    enabled: !!user,
  });

  // Fetch all dictionaries for shortName lookup
  const { data: allDictionaries = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries"],
    enabled: !!user,
  });

  // Fetch product groups for filter dropdown
  const { data: productGroups = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries", { type: "product_group" }],
    enabled: !!user,
  });

  // Save filters to localStorage
  useEffect(() => {
    try {
      const filters = {
        searchQuery,
        filterGroup,
      };
      localStorage.setItem(STORAGE_KEY, JSON.stringify(filters));
    } catch (error) {
      console.error('Error saving filters to localStorage:', error);
    }
  }, [searchQuery, filterGroup]);

  const duplicateMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await fetch(`/api/set-matrices/${id}/duplicate`, {
        method: "POST",
      });
      if (!response.ok) {
        throw new Error(await response.text());
      }
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/set-matrices"] });
      toast({
        title: "Sukces",
        description: "Matryca została zduplikowana",
      });
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await fetch(`/api/set-matrices/${id}`, {
        method: "DELETE",
      });
      if (!response.ok) {
        throw new Error(await response.text());
      }
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/set-matrices"] });
      toast({
        title: "Sukces",
        description: "Matryca została usunięta",
      });
      setMatrixToDelete(null);
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message,
        variant: "destructive",
      });
      setMatrixToDelete(null);
    },
  });

  const handleDelete = () => {
    if (matrixToDelete) {
      deleteMutation.mutate(matrixToDelete);
    }
  };

  // Parse search query into individual search phrases
  const getSearchPhrases = () => {
    if (!searchQuery) return [];
    const phrases: string[] = [];
    searchQuery.split(';').forEach((phrase: string) => {
      const trimmed = phrase.trim();
      if (trimmed) phrases.push(trimmed);
    });
    return phrases;
  };

  const searchPhrases = getSearchPhrases();

  // Remove individual search phrase
  const removeSearchPhrase = (phrase: string) => {
    const phrases = getSearchPhrases();
    const filtered = phrases.filter(p => p !== phrase);
    setSearchQuery(filtered.join('; '));
  };

  // Clear all filters
  const clearAllFilters = () => {
    setSearchQuery("");
    setFilterGroup("");
  };

  // Count active filters
  const activeFiltersCount = useMemo(() => {
    let count = 0;
    count += searchPhrases.length;
    if (filterGroup) count++;
    return count;
  }, [searchPhrases, filterGroup]);

  // Helper function to get shortName from dictionary
  function getShortName(code: string): string {
    const item = allDictionaries.find(d => d.code === code);
    return item?.shortName || code;
  }

  // Generate preview name for a matrix (function declaration for hoisting)
  function generatePreviewName(matrix: SetMatrix): string {
    const nameParts: string[] = [];
    
    // 1. Prefix
    if (matrix.namePrefix?.trim()) nameParts.push(matrix.namePrefix.trim());
    
    // 2. Grupa produktów (use shortName)
    if (matrix.productGroup) {
      nameParts.push(getShortName(matrix.productGroup));
    }
    
    // 3. Długość × Szerokość (konwersja mm -> cm)
    if (matrix.length && matrix.width) {
      const lengthCm = Math.round(parseFloat(matrix.length) / 10);
      const widthCm = Math.round(parseFloat(matrix.width) / 10);
      nameParts.push(`${lengthCm}x${widthCm}cm`);
    }
    
    // 4. Rodzaje produktów z komponentów (use shortName)
    const typeNames: string[] = [];
    let totalPanelCount = 0;
    
    matrix.components?.forEach((comp) => {
      const compType = comp.componentType || '';
      const compLength = comp.length ? Math.round(parseFloat(String(comp.length)) / 10) : null;
      
      if (compType === 'PANEL') {
        totalPanelCount += (comp.quantity || 0);
      } else if (compType && compLength) {
        const shortName = getShortName(compType);
        typeNames.push(`${shortName}${compLength}`);
      } else if (compType) {
        const shortName = getShortName(compType);
        typeNames.push(shortName);
      }
    });
    
    typeNames.forEach(type => nameParts.push(type));
    
    // 5. Liczba paneli (z komponentów)
    if (totalPanelCount > 0) {
      nameParts.push(`${totalPanelCount} Panel${totalPanelCount > 1 ? 'e' : ''}`);
    }
    
    // 6. Drzwi (use shortName)
    if (matrix.doors) {
      nameParts.push(getShortName(matrix.doors));
    }
    
    // 7. Nogi (use shortName)
    if (matrix.legs) {
      nameParts.push(getShortName(matrix.legs));
    }
    
    // 8. Kolor - pokazujemy przykładowy pierwszy kolor
    if (matrix.colors && matrix.colors.length > 0) {
      nameParts.push(matrix.colors[0]);
    }
    
    // 9. Suffix
    if (matrix.nameSuffix?.trim()) nameParts.push(matrix.nameSuffix.trim());
    
    return nameParts.filter(Boolean).join(' ').trim();
  }

  // Apply filters to matrices
  const filteredMatrices = useMemo(() => {
    return matrices.filter(matrix => {
      // Text search with AND logic (semicolon)
      if (searchQuery) {
        const terms = searchQuery.split(';').map((t: string) => t.trim()).filter(Boolean);
        const matches = terms.every((term: string) => {
          const lowerTerm = term.toLowerCase();
          return (
            matrix.name.toLowerCase().includes(lowerTerm) ||
            matrix.productGroup?.toLowerCase().includes(lowerTerm) ||
            matrix.namePrefix?.toLowerCase().includes(lowerTerm) ||
            matrix.nameSuffix?.toLowerCase().includes(lowerTerm) ||
            generatePreviewName(matrix).toLowerCase().includes(lowerTerm)
          );
        });
        if (!matches) return false;
      }

      // Product group filter
      if (filterGroup && matrix.productGroup !== filterGroup) {
        return false;
      }

      return true;
    });
  }, [matrices, searchQuery, filterGroup, allDictionaries]);

  return (
    <div className="p-3 sm:p-6 space-y-4 sm:space-y-6">
      <Card>
        <CardHeader className="p-3 sm:p-4">
          <div className="flex flex-col sm:flex-row gap-3 sm:gap-0 sm:justify-between sm:items-center">
            <div className="flex items-center gap-2">
              <Package className="h-5 w-5 text-primary" />
              <div>
                <CardTitle className="text-lg sm:text-xl">Matryce Zestawów</CardTitle>
                <p className="text-xs sm:text-sm text-muted-foreground mt-0.5">
                  Szablony do generowania zestawów produktów
                </p>
              </div>
            </div>
            <Button
              onClick={() => setLocation("/set-matrices/settings/new")}
              data-testid="button-create-matrix"
              className="w-full sm:w-auto"
            >
              <Plus className="h-4 w-4 mr-2" />
              Nowa Matryca
            </Button>
          </div>
        </CardHeader>
      </Card>

      {/* Search Bar */}
      <Card className="bg-primary/5 border-primary/20">
        <CardContent className="py-3 md:py-4">
          <div className="flex items-center gap-2">
            <Search className="h-4 w-4 text-primary shrink-0" />
            {searchQuery && (
              <Button
                variant="ghost"
                size="icon"
                className="h-6 w-6 hover:bg-primary/10 shrink-0"
                onClick={() => setSearchQuery("")}
                data-testid="button-clear-search"
              >
                <X className="h-3.5 w-3.5 text-muted-foreground" />
              </Button>
            )}
            <Input
              placeholder="Szukaj (użyj ; aby połączyć filtry np. F2;SUPRA)"
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="h-9 md:h-10 text-xs md:text-sm bg-background border-primary/30 focus-visible:ring-primary px-3"
              data-testid="input-search-matrix"
            />
          </div>
        </CardContent>
      </Card>

      {/* Collapsible Filters Panel */}
      <Card>
        <CardHeader className="cursor-pointer py-1 px-3" onClick={() => setFiltersExpanded(!filtersExpanded)}>
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <h3 className="text-sm font-medium">Filtry</h3>
              {activeFiltersCount > 0 && (
                <Badge variant="default" className="h-5 text-xs">
                  {activeFiltersCount}
                </Badge>
              )}
            </div>
            {filtersExpanded ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
          </div>
        </CardHeader>
        {filtersExpanded && (
          <CardContent className="px-3 pb-3">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
              <div>
                <label className="text-xs font-medium mb-1.5 block">Grupa produktów</label>
                <Select value={filterGroup || "all"} onValueChange={(val) => setFilterGroup(val === "all" ? "" : val)}>
                  <SelectTrigger className="h-9">
                    <SelectValue placeholder="Wszystkie grupy" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie grupy</SelectItem>
                    {productGroups.map(group => (
                      <SelectItem key={group.id} value={group.code}>
                        {group.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="flex items-end">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={clearAllFilters}
                  disabled={activeFiltersCount === 0}
                  className="h-9"
                >
                  Wyczyść filtry
                </Button>
              </div>
            </div>
          </CardContent>
        )}
      </Card>

      {/* Active Filters Display */}
      {(activeFiltersCount > 0 || filteredMatrices.length > 0) && (
        <div className="flex flex-wrap items-center gap-2">
          <span className="text-sm text-muted-foreground">
            Znaleziono: <strong>{filteredMatrices.length}</strong> {filteredMatrices.length === 1 ? 'matryca' : filteredMatrices.length < 5 ? 'matryc' : 'matryc'}
          </span>
          {searchPhrases.map((phrase, index) => (
            <Badge 
              key={`search-${index}`}
              variant="default" 
              className="gap-1.5 pr-1 h-6"
              data-testid={`badge-search-phrase-${index}`}
            >
              <Search className="h-3 w-3" />
              {phrase}
              <Button
                variant="ghost"
                size="sm"
                className="h-4 w-4 p-0 hover:bg-primary-foreground/20"
                onClick={() => removeSearchPhrase(phrase)}
                data-testid={`button-remove-search-${index}`}
              >
                <X className="h-3 w-3" />
              </Button>
            </Badge>
          ))}
          {filterGroup && (
            <Badge variant="default" className="gap-1.5 pr-1 h-6">
              Grupa: {productGroups.find(g => g.code === filterGroup)?.name || filterGroup}
              <Button
                variant="ghost"
                size="sm"
                className="h-4 w-4 p-0 hover:bg-primary-foreground/20"
                onClick={() => setFilterGroup("")}
              >
                <X className="h-3 w-3" />
              </Button>
            </Badge>
          )}
        </div>
      )}

      {isLoading ? (
        <Card>
          <CardContent className="p-6">
            <p className="text-center text-muted-foreground">Ładowanie...</p>
          </CardContent>
        </Card>
      ) : filteredMatrices.length === 0 ? (
        <Card>
          <CardContent className="p-6">
            <div className="text-center py-8">
              <Package className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <p className="text-muted-foreground mb-4">
                {matrices.length === 0 ? 'Brak matryc zestawów' : 'Nie znaleziono matryc pasujących do filtrów'}
              </p>
              {matrices.length === 0 ? (
                <Button
                  onClick={() => setLocation("/set-matrices/settings/new")}
                  data-testid="button-create-first-matrix"
                >
                  <Plus className="h-4 w-4 mr-2" />
                  Utwórz pierwszą matrycę
                </Button>
              ) : (
                <Button
                  variant="outline"
                  onClick={clearAllFilters}
                  data-testid="button-clear-filters"
                >
                  Wyczyść filtry
                </Button>
              )}
            </div>
          </CardContent>
        </Card>
      ) : (
        <Card>
          <CardContent className="p-0">
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="w-[50px]">ID</TableHead>
                    <TableHead>Nazwa</TableHead>
                    <TableHead className="hidden lg:table-cell">Nazwa wygenerowana</TableHead>
                    <TableHead className="hidden sm:table-cell">Grupa produktów</TableHead>
                    <TableHead className="text-center">Komponenty</TableHead>
                    <TableHead className="text-right w-[140px]">Akcje</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredMatrices.map((matrix) => (
                    <TableRow
                      key={matrix.id}
                      data-testid={`matrix-row-${matrix.id}`}
                      className="hover-elevate cursor-pointer"
                      onClick={() => setLocation(`/set-matrices/settings/${matrix.id}`)}
                    >
                      <TableCell className="font-medium text-xs text-muted-foreground">
                        {matrix.id}
                      </TableCell>
                      <TableCell className="font-medium">
                        <div className="flex items-center gap-2">
                          {matrix.imageUrl ? (
                            <img 
                              src={matrix.imageUrl} 
                              alt={matrix.name}
                              className="h-[30px] w-[30px] object-cover border"
                              data-testid={`image-matrix-${matrix.id}`}
                            />
                          ) : (
                            <div className="h-[30px] w-[30px] border bg-muted flex items-center justify-center">
                              <ImageIcon className="h-3 w-3 text-muted-foreground" />
                            </div>
                          )}
                          <span>{matrix.name}</span>
                        </div>
                      </TableCell>
                      <TableCell className="hidden lg:table-cell">
                        <div className="font-mono text-xs text-muted-foreground max-w-md truncate" title={generatePreviewName(matrix)}>
                          {generatePreviewName(matrix) || <span className="italic">-</span>}
                        </div>
                      </TableCell>
                      <TableCell className="hidden sm:table-cell">
                        {matrix.productGroup ? (
                          <Badge variant="secondary" className="text-xs">
                            {matrix.productGroup}
                          </Badge>
                        ) : (
                          <span className="text-xs text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell className="text-center">
                        <Badge variant="outline" className="text-xs">
                          {matrix.components?.length || 0}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-right">
                        <div className="flex justify-end gap-1">
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={(e) => {
                              e.stopPropagation();
                              setLocation(`/set-matrices/settings/${matrix.id}`);
                            }}
                            data-testid={`button-edit-matrix-${matrix.id}`}
                            className="h-8 px-2"
                          >
                            <Edit className="h-3.5 w-3.5 mr-1" />
                            Edytuj
                          </Button>
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={(e) => {
                              e.stopPropagation();
                              duplicateMutation.mutate(matrix.id);
                            }}
                            data-testid={`button-duplicate-matrix-${matrix.id}`}
                            className="h-8 px-2"
                            disabled={duplicateMutation.isPending}
                          >
                            <Copy className="h-3.5 w-3.5" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={(e) => {
                              e.stopPropagation();
                              setMatrixToDelete(matrix.id);
                            }}
                            data-testid={`button-delete-matrix-${matrix.id}`}
                            className="h-8 px-2 text-destructive hover:text-destructive"
                          >
                            <Trash2 className="h-3.5 w-3.5" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          </CardContent>
        </Card>
      )}

      <AlertDialog open={!!matrixToDelete} onOpenChange={() => setMatrixToDelete(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Czy na pewno usunąć?</AlertDialogTitle>
            <AlertDialogDescription>
              Ta operacja jest nieodwracalna. Matryca zostanie trwale usunięta.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDelete}
              data-testid="button-confirm-delete"
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
            >
              Usuń
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
