import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Server, CheckCircle2, AlertCircle, RefreshCw, Clock, Package, ShoppingCart } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { formatDistanceToNow } from "date-fns";
import { pl } from "date-fns/locale";

export default function OdooConfig() {
  const { toast } = useToast();
  const [syncInterval, setSyncInterval] = useState("3");
  const [isActive, setIsActive] = useState(true);
  const [autoConfirmOrders, setAutoConfirmOrders] = useState(false);
  const [odooUrl, setOdooUrl] = useState("");
  const [odooUserUrl, setOdooUserUrl] = useState("");
  const [omsDomain, setOmsDomain] = useState("alp-oms.replit.app");
  const [shoperOrdersUrl, setShoperOrdersUrl] = useState("https://alpmeb.pl/admin/orders/");
  const [shoperProductsUrl, setShoperProductsUrl] = useState("https://sklep378098.shoparena.pl/admin/products/edit/id/");

  const { data: config, isLoading: configLoading } = useQuery<{
    configured: boolean;
    url?: string;
    user_url?: string;
    oms_domain?: string;
    database?: string;
    username?: string;
    sync_interval_minutes?: number;
    is_active?: boolean;
    auto_confirm_orders?: boolean;
    last_sync_at?: string;
    last_sync_status?: string;
  }>({
    queryKey: ["/api/odoo/config"],
  });

  const { data: shoperConfig, isLoading: shoperConfigLoading } = useQuery<{
    configured: boolean;
    orders_url?: string;
    products_url?: string;
  }>({
    queryKey: ["/api/shoper/config"],
  });

  const { data: syncStatus, isLoading: statusLoading } = useQuery<{
    pending: number;
    processing: number;
    completed: number;
    failed: number;
    lastSyncAt: string | null;
  }>({
    queryKey: ["/api/odoo/sync/status"],
    refetchInterval: 10000, // Refresh every 10 seconds
  });

  const { data: syncLogs, isLoading: logsLoading } = useQuery<Array<{
    id: number;
    order_number: number;
    operation: string;
    status: string;
    message: string;
    odoo_order_id: number | null;
    created_at: string;
  }>>({
    queryKey: ["/api/odoo/sync/logs"],
  });

  useEffect(() => {
    if (config) {
      setSyncInterval(String(config.sync_interval_minutes || 3));
      setIsActive(config.is_active !== false);
      setAutoConfirmOrders(config.auto_confirm_orders || false);
      setOdooUrl(config.url || "");
      setOdooUserUrl(config.user_url || "");
      setOmsDomain(config.oms_domain || "alp-oms.replit.app");
    }
  }, [config]);

  useEffect(() => {
    if (shoperConfig) {
      setShoperOrdersUrl(shoperConfig.orders_url || "https://alpmeb.pl/admin/orders/");
      setShoperProductsUrl(shoperConfig.products_url || "https://sklep378098.shoparena.pl/admin/products/edit/id/");
    }
  }, [shoperConfig]);

  const testConnectionMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/odoo/test-connection", {});
    },
    onSuccess: (data: any) => {
      if (data.success) {
        toast({
          title: "Połączenie udane ✅",
          description: `Połączono z Odoo ${data.version?.server_version || ''}`,
        });
      } else {
        // Show detailed diagnostic info
        const details = data.details;
        const steps = details?.steps || [];
        const failedStep = steps.find((s: any) => s.status === 'failed');
        
        let description = data.error || "Nie udało się połączyć z Odoo";
        
        if (failedStep) {
          if (failedStep.step === 'version') {
            description = `❌ Krok 1/2 - Połączenie z Odoo:\n${failedStep.error}\n\n📍 URL: ${details?.url}\n\n💡 Sprawdź czy:\n• Odoo działa na Windows\n• Cloudflare Tunnel jest uruchomiony`;
          } else if (failedStep.step === 'authenticate') {
            description = `❌ Krok 2/2 - Autentykacja:\n${failedStep.error}\n\n💡 Sprawdź ODOO_API_KEY w secrets`;
          }
        }
        
        toast({
          title: "Błąd połączenia z Odoo",
          description: description,
          variant: "destructive",
          duration: 10000, // Show for 10 seconds
        });
        
        // Also log to console for debugging
        console.error('Odoo connection test failed:', data);
      }
    },
  });

  const saveConfigMutation = useMutation({
    mutationFn: async (data: {
      syncIntervalMinutes: number;
      isActive: boolean;
      autoConfirmOrders: boolean;
      url?: string;
      userUrl?: string;
      omsDomain?: string;
    }) => {
      return await apiRequest("POST", "/api/odoo/config", data);
    },
    onSuccess: () => {
      toast({
        title: "Konfiguracja zapisana",
        description: "Ustawienia synchronizacji Odoo zostały zaktualizowane.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/odoo/config"] });
    },
  });

  const manualSyncMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/odoo/sync/manual", {});
    },
    onSuccess: (data: any) => {
      const processed = data?.processed ?? 0;
      const succeeded = data?.succeeded ?? 0;
      const failed = data?.failed ?? 0;

      if (processed === 0) {
        toast({
          title: "Brak zamówień do synchronizacji",
          description: "Wszystkie zamówienia są już zsynchronizowane z Odoo.",
        });
      } else {
        toast({
          title: "Synchronizacja zakończona",
          description: `Przetworzono: ${processed}, Sukces: ${succeeded}, Błędy: ${failed}`,
        });
      }
      queryClient.invalidateQueries({ queryKey: ["/api/odoo/sync/status"] });
      queryClient.invalidateQueries({ queryKey: ["/api/odoo/sync/logs"] });
    },
  });

  const saveShoperConfigMutation = useMutation({
    mutationFn: async (data: {
      ordersUrl: string;
      productsUrl: string;
    }) => {
      return await apiRequest("POST", "/api/shoper/config", data);
    },
    onSuccess: () => {
      toast({
        title: "Konfiguracja zapisana",
        description: "Ustawienia linków Shoper zostały zaktualizowane.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/shoper/config"] });
    },
  });

  const handleSaveConfig = () => {
    saveConfigMutation.mutate({
      syncIntervalMinutes: parseInt(syncInterval),
      isActive,
      autoConfirmOrders,
      url: odooUrl,
      userUrl: odooUserUrl,
      omsDomain,
    });
  };

  const handleSaveShoperConfig = () => {
    saveShoperConfigMutation.mutate({
      ordersUrl: shoperOrdersUrl,
      productsUrl: shoperProductsUrl,
    });
  };

  if (configLoading) {
    return (
      <div className="container mx-auto p-6">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-muted rounded w-1/3"></div>
          <div className="h-64 bg-muted rounded"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold flex items-center gap-2" data-testid="text-page-title">
            <Server className="w-8 h-8" />
            Integracja z Odoo
          </h1>
          <p className="text-muted-foreground mt-1">
            Konfiguracja automatycznej synchronizacji zamówień do Odoo
          </p>
        </div>
        <Button
          onClick={() => testConnectionMutation.mutate()}
          disabled={testConnectionMutation.isPending || !config?.configured}
          data-testid="button-test-connection"
        >
          {testConnectionMutation.isPending ? (
            <>
              <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
              Testowanie...
            </>
          ) : (
            <>
              <CheckCircle2 className="mr-2 h-4 w-4" />
              Test Połączenia
            </>
          )}
        </Button>
      </div>

      {/* Configuration Card */}
      <Card>
        <CardHeader>
          <CardTitle>Konfiguracja Połączenia</CardTitle>
          <CardDescription>
            Dane połączenia są przechowywane w zmiennych środowiskowych
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          {!config?.configured ? (
            <div className="rounded-lg border border-yellow-200 bg-yellow-50 dark:bg-yellow-950 dark:border-yellow-800 p-4">
              <div className="flex">
                <AlertCircle className="h-5 w-5 text-yellow-600 dark:text-yellow-500" />
                <div className="ml-3">
                  <h3 className="text-sm font-medium text-yellow-800 dark:text-yellow-200">
                    Odoo nie jest skonfigurowane
                  </h3>
                  <div className="mt-2 text-sm text-yellow-700 dark:text-yellow-300">
                    <p>Aby rozpocząć synchronizację z Odoo, skonfiguruj następujące zmienne środowiskowe:</p>
                    <ul className="list-disc list-inside mt-2 space-y-1">
                      <li>ODOO_URL</li>
                      <li>ODOO_DATABASE</li>
                      <li>ODOO_USERNAME</li>
                      <li>ODOO_API_KEY</li>
                    </ul>
                  </div>
                </div>
              </div>
            </div>
          ) : (
            <div className="grid gap-4">
              <div className="space-y-2">
                <Label htmlFor="odoo-url">URL Odoo (synchronizacja API)</Label>
                <Input
                  id="odoo-url"
                  type="url"
                  placeholder="https://alpma.app/odoo"
                  value={odooUrl}
                  onChange={(e) => setOdooUrl(e.target.value)}
                  data-testid="input-odoo-url"
                />
                <p className="text-xs text-muted-foreground">
                  URL do synchronizacji przez Cloudflare Tunnel (np. https://alpma.app/odoo)
                </p>
              </div>
              <div className="space-y-2">
                <Label htmlFor="odoo-user-url">URL Odoo (linki dla użytkowników)</Label>
                <Input
                  id="odoo-user-url"
                  type="url"
                  placeholder="http://100.99.76.111:8070/odoo"
                  value={odooUserUrl}
                  onChange={(e) => setOdooUserUrl(e.target.value)}
                  data-testid="input-odoo-user-url"
                />
                <p className="text-xs text-muted-foreground">
                  URL dla linków klikanych przez użytkowników przez Tailscale (np. http://100.99.76.111:8070/odoo)
                </p>
              </div>
              <div className="space-y-2">
                <Label htmlFor="oms-domain">Domena OMS</Label>
                <Input
                  id="oms-domain"
                  type="text"
                  placeholder="alp-oms.replit.app"
                  value={omsDomain}
                  onChange={(e) => setOmsDomain(e.target.value)}
                  data-testid="input-oms-domain"
                />
                <p className="text-xs text-muted-foreground">
                  Domena systemu OMS używana do generowania linków w Odoo (np. alp-oms.replit.app). Zmień jeśli adres serwisu się zmieni.
                </p>
              </div>
              <div>
                <Label>Baza Danych</Label>
                <div className="mt-1 text-sm text-muted-foreground" data-testid="text-odoo-database">
                  {config.database || 'Nie skonfigurowano'}
                </div>
              </div>
              <div>
                <Label>Użytkownik</Label>
                <div className="mt-1 text-sm text-muted-foreground" data-testid="text-odoo-username">
                  {config.username || 'Nie skonfigurowano'}
                </div>
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Shoper Configuration Card */}
      <Card>
        <CardHeader>
          <div className="flex items-center gap-2">
            <ShoppingCart className="w-5 h-5" />
            <CardTitle>Konfiguracja Shoper</CardTitle>
          </div>
          <CardDescription>
            Skonfiguruj linki do platformy Shoper - bazowe URL będzie automatycznie połączone z numerem zamówienia/produktu
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="shoper-orders-url">URL zamówień Shoper</Label>
            <Input
              id="shoper-orders-url"
              type="url"
              placeholder="https://alpmeb.pl/admin/orders/"
              value={shoperOrdersUrl}
              onChange={(e) => setShoperOrdersUrl(e.target.value)}
              data-testid="input-shoper-orders-url"
            />
            <p className="text-xs text-muted-foreground">
              Bazowy URL do zamówień (np. https://alpmeb.pl/admin/orders/). Numer zamówienia zostanie dodany automatycznie na końcu.
            </p>
          </div>
          <div className="space-y-2">
            <Label htmlFor="shoper-products-url">URL produktów Shoper</Label>
            <Input
              id="shoper-products-url"
              type="url"
              placeholder="https://sklep378098.shoparena.pl/admin/products/edit/id/"
              value={shoperProductsUrl}
              onChange={(e) => setShoperProductsUrl(e.target.value)}
              data-testid="input-shoper-products-url"
            />
            <p className="text-xs text-muted-foreground">
              Bazowy URL do produktów (np. https://sklep378098.shoparena.pl/admin/products/edit/id/). ID produktu zostanie dodane automatycznie na końcu.
            </p>
          </div>
          <Button
            onClick={handleSaveShoperConfig}
            disabled={saveShoperConfigMutation.isPending}
            data-testid="button-save-shoper-config"
          >
            {saveShoperConfigMutation.isPending ? "Zapisywanie..." : "Zapisz linki Shoper"}
          </Button>
        </CardContent>
      </Card>

      {/* Sync Settings Card */}
      {config?.configured && (
        <Card>
          <CardHeader>
            <CardTitle>Ustawienia Synchronizacji</CardTitle>
            <CardDescription>
              Konfiguracja automatycznej synchronizacji zamówień
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-6">
            <div className="flex items-center justify-between">
              <div className="space-y-0.5">
                <Label htmlFor="auto-sync">Automatyczna synchronizacja</Label>
                <div className="text-sm text-muted-foreground">
                  Włącz cykliczną synchronizację zamówień do Odoo
                </div>
              </div>
              <Switch
                id="auto-sync"
                checked={isActive}
                onCheckedChange={setIsActive}
                data-testid="switch-auto-sync"
              />
            </div>

            <div className="flex items-center justify-between">
              <div className="space-y-0.5">
                <Label htmlFor="auto-confirm">Automatyczne potwierdzanie zamówień</Label>
                <div className="text-sm text-muted-foreground">
                  Zamówienia w Odoo będą automatycznie potwierdzane (zmiana z oferty na zamówienie)
                </div>
              </div>
              <Switch
                id="auto-confirm"
                checked={autoConfirmOrders}
                onCheckedChange={setAutoConfirmOrders}
                data-testid="switch-auto-confirm"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="sync-interval">
                Interwał synchronizacji (minuty)
              </Label>
              <Select
                value={syncInterval}
                onValueChange={setSyncInterval}
                disabled={!isActive}
              >
                <SelectTrigger data-testid="select-sync-interval">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="1">1 minuta</SelectItem>
                  <SelectItem value="2">2 minuty</SelectItem>
                  <SelectItem value="3">3 minuty</SelectItem>
                  <SelectItem value="5">5 minut</SelectItem>
                  <SelectItem value="10">10 minut</SelectItem>
                  <SelectItem value="15">15 minut</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <Button
              onClick={handleSaveConfig}
              disabled={saveConfigMutation.isPending}
              data-testid="button-save-config"
            >
              {saveConfigMutation.isPending ? "Zapisywanie..." : "Zapisz ustawienia"}
            </Button>
          </CardContent>
        </Card>
      )}

      {/* Sync Status Card */}
      {config?.configured && (
        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle>Status Synchronizacji</CardTitle>
                <CardDescription>
                  Bieżący stan synchronizacji zamówień
                </CardDescription>
              </div>
              <Button
                onClick={() => manualSyncMutation.mutate()}
                disabled={manualSyncMutation.isPending}
                size="sm"
                data-testid="button-manual-sync"
              >
                {manualSyncMutation.isPending ? (
                  <>
                    <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                    Synchronizacja...
                  </>
                ) : (
                  <>
                    <RefreshCw className="mr-2 h-4 w-4" />
                    Synchronizuj teraz
                  </>
                )}
              </Button>
            </div>
          </CardHeader>
          <CardContent>
            {statusLoading ? (
              <div className="animate-pulse space-y-2">
                <div className="h-4 bg-muted rounded w-full"></div>
                <div className="h-4 bg-muted rounded w-3/4"></div>
              </div>
            ) : syncStatus ? (
              <div className="grid gap-4 md:grid-cols-4">
                <div className="flex flex-col gap-1">
                  <div className="flex items-center gap-2">
                    <Clock className="w-4 h-4 text-yellow-600" />
                    <span className="text-sm text-muted-foreground">Oczekujące</span>
                  </div>
                  <div className="text-2xl font-bold" data-testid="text-pending-count">
                    {syncStatus.pending}
                  </div>
                </div>
                <div className="flex flex-col gap-1">
                  <div className="flex items-center gap-2">
                    <CheckCircle2 className="w-4 h-4 text-green-600" />
                    <span className="text-sm text-muted-foreground">Ukończone</span>
                  </div>
                  <div className="text-2xl font-bold" data-testid="text-completed-count">
                    {syncStatus.completed}
                  </div>
                </div>
                <div className="flex flex-col gap-1">
                  <div className="flex items-center gap-2">
                    <AlertCircle className="w-4 h-4 text-red-600" />
                    <span className="text-sm text-muted-foreground">Błędy</span>
                  </div>
                  <div className="text-2xl font-bold" data-testid="text-failed-count">
                    {syncStatus.failed}
                  </div>
                </div>
                <div className="flex flex-col gap-1">
                  <div className="flex items-center gap-2">
                    <Package className="w-4 h-4 text-blue-600" />
                    <span className="text-sm text-muted-foreground">Ostatnia sync</span>
                  </div>
                  <div className="text-sm font-medium" data-testid="text-last-sync">
                    {syncStatus.lastSyncAt
                      ? formatDistanceToNow(new Date(syncStatus.lastSyncAt), {
                          addSuffix: true,
                          locale: pl,
                        })
                      : "Nigdy"}
                  </div>
                </div>
              </div>
            ) : null}
          </CardContent>
        </Card>
      )}

      {/* Sync Logs Card */}
      {config?.configured && syncLogs && syncLogs.length > 0 && (
        <Card>
          <CardHeader>
            <CardTitle>Ostatnie Logi Synchronizacji</CardTitle>
            <CardDescription>
              Historia synchronizacji zamówień (ostatnie 100)
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              {syncLogs.slice(0, 20).map((log) => (
                <div
                  key={log.id}
                  className="flex items-center justify-between p-3 rounded-lg border"
                  data-testid={`log-${log.id}`}
                >
                  <div className="flex items-center gap-3">
                    {log.status === 'success' ? (
                      <CheckCircle2 className="w-4 h-4 text-green-600" />
                    ) : (
                      <AlertCircle className="w-4 h-4 text-red-600" />
                    )}
                    <div>
                      <div className="font-medium">{log.order_number}</div>
                      <div className="text-sm text-muted-foreground">
                        {log.message}
                      </div>
                    </div>
                  </div>
                  <div className="flex items-center gap-3">
                    {log.odoo_order_id && (
                      <Badge variant="secondary">
                        Odoo #{log.odoo_order_id}
                      </Badge>
                    )}
                    <Badge variant={log.operation === 'create' ? 'default' : 'outline'}>
                      {log.operation}
                    </Badge>
                    <span className="text-sm text-muted-foreground">
                      {formatDistanceToNow(new Date(log.created_at), {
                        addSuffix: true,
                        locale: pl,
                      })}
                    </span>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
