import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Skeleton } from "@/components/ui/skeleton";
import { PieChart, Pie, Cell, ResponsiveContainer, Legend, Tooltip } from "recharts";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { DateRangeFilter, type DateRangePreset, getDateRangeFromPreset } from "@/components/date-range-filter";

const COLORS = {
  commission: '#ef4444',
  ads: '#a855f7',
  delivery: '#3b82f6',
  refund: '#f97316',
  listing: '#10b981',
  other: '#6b7280',
};

export default function FeesGrouped() {
  const [dateRange, setDateRange] = useState<DateRangePreset>("last30days");
  
  const range = getDateRangeFromPreset(dateRange);
  const { data: grouped, isLoading } = useQuery<any[]>({
    queryKey: [`/api/allegro/fees/grouped?from=${range.from}&to=${range.to}`],
  });

  const formatAmount = (amount: string | number) => {
    const num = typeof amount === 'string' ? parseFloat(amount) : amount;
    return new Intl.NumberFormat('pl-PL', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(num);
  };

  const getCategoryLabel = (cat: string) => {
    const labels: Record<string, string> = {
      commission: 'Prowizje',
      ads: 'Reklamy',
      delivery: 'Dostawa',
      refund: 'Zwroty',
      listing: 'Wystawienie',
      other: 'Inne',
    };
    return labels[cat] || cat;
  };

  const categoryTotals = grouped?.reduce((acc, item) => {
    const cat = item.category;
    if (!acc[cat]) {
      acc[cat] = { total: 0, count: 0, category: cat };
    }
    acc[cat].total += parseFloat(item.total || '0');
    acc[cat].count += parseInt(item.count || '0', 10);
    return acc;
  }, {} as Record<string, any>);

  const pieData = Object.values(categoryTotals || {}).map((item: any) => ({
    name: getCategoryLabel(item.category),
    value: item.total,
    count: item.count,
  }));

  return (
    <div className="p-6 space-y-6">
      <div>
        <h1 className="text-3xl font-bold" data-testid="heading-fees-grouped">Podział grupowany</h1>
        <p className="text-muted-foreground">Opłaty pogrupowane według kategorii i typów</p>
      </div>

      {/* Date range filter */}
      <Card className="bg-muted/30">
        <CardContent className="pt-6">
          <DateRangeFilter value={dateRange} onChange={setDateRange} />
        </CardContent>
      </Card>

      {isLoading ? (
        <div className="grid gap-6 md:grid-cols-2">
          <Card>
            <CardHeader>
              <Skeleton className="h-6 w-48" />
            </CardHeader>
            <CardContent>
              <Skeleton className="h-64 w-full" />
            </CardContent>
          </Card>
          <Card>
            <CardHeader>
              <Skeleton className="h-6 w-48" />
            </CardHeader>
            <CardContent>
              <Skeleton className="h-64 w-full" />
            </CardContent>
          </Card>
        </div>
      ) : (
        <>
          <div className="grid gap-6 md:grid-cols-2">
            <Card>
              <CardHeader>
                <CardTitle>Podział według kategorii</CardTitle>
                <CardDescription>Procentowy udział poszczególnych kategorii</CardDescription>
              </CardHeader>
              <CardContent className="flex justify-center">
                <ResponsiveContainer width="100%" height={300}>
                  <PieChart>
                    <Pie
                      data={pieData}
                      cx="50%"
                      cy="50%"
                      labelLine={false}
                      label={(entry) => `${entry.name}: ${formatAmount(entry.value)} zł`}
                      outerRadius={100}
                      fill="#8884d8"
                      dataKey="value"
                    >
                      {pieData.map((entry, index) => (
                        <Cell 
                          key={`cell-${index}`} 
                          fill={Object.values(COLORS)[index % Object.values(COLORS).length]} 
                        />
                      ))}
                    </Pie>
                    <Tooltip formatter={(value: any) => `${formatAmount(value)} zł`} />
                    <Legend />
                  </PieChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Podsumowanie kategorii</CardTitle>
                <CardDescription>Suma i liczba opłat w każdej kategorii</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {Object.values(categoryTotals || {}).map((item: any) => (
                    <div key={item.category} className="flex items-center justify-between p-3 bg-muted/30 rounded-md">
                      <div className="flex items-center gap-3">
                        <div 
                          className="w-3 h-3 rounded-full" 
                          style={{ backgroundColor: COLORS[item.category as keyof typeof COLORS] }}
                        />
                        <span className="font-medium">{getCategoryLabel(item.category)}</span>
                      </div>
                      <div className="text-right">
                        <div className="font-bold">{formatAmount(item.total)} zł</div>
                        <div className="text-xs text-muted-foreground">{item.count} opłat</div>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>Szczegółowy podział według typów</CardTitle>
              <CardDescription>Wszystkie typy opłat z statystykami</CardDescription>
            </CardHeader>
            <CardContent>
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Kategoria</TableHead>
                    <TableHead>Typ</TableHead>
                    <TableHead>Nazwa</TableHead>
                    <TableHead className="text-right">Liczba</TableHead>
                    <TableHead className="text-right">Suma</TableHead>
                    <TableHead className="text-right">Średnia</TableHead>
                    <TableHead className="text-right">Min</TableHead>
                    <TableHead className="text-right">Max</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {grouped?.map((fee: any, idx: number) => (
                    <TableRow key={idx}>
                      <TableCell>
                        <span className="inline-flex items-center gap-2">
                          <div 
                            className="w-2 h-2 rounded-full" 
                            style={{ backgroundColor: COLORS[fee.category as keyof typeof COLORS] }}
                          />
                          {getCategoryLabel(fee.category)}
                        </span>
                      </TableCell>
                      <TableCell className="font-mono text-xs">{fee.type_id}</TableCell>
                      <TableCell className="max-w-xs truncate">{fee.type_name}</TableCell>
                      <TableCell className="text-right">{fee.count}</TableCell>
                      <TableCell className="text-right font-medium">{formatAmount(fee.total)} zł</TableCell>
                      <TableCell className="text-right text-muted-foreground">{formatAmount(fee.average)} zł</TableCell>
                      <TableCell className="text-right text-muted-foreground">{formatAmount(fee.min)} zł</TableCell>
                      <TableCell className="text-right text-muted-foreground">{formatAmount(fee.max)} zł</TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </>
      )}
    </div>
  );
}
