import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Skeleton } from "@/components/ui/skeleton";
import { ChevronDown, ChevronRight } from "lucide-react";
import { DateRangeFilter, type DateRangePreset, getDateRangeFromPreset } from "@/components/date-range-filter";

export default function FeesDaily() {
  const [expandedDays, setExpandedDays] = useState<Set<string>>(new Set());
  const [dateRange, setDateRange] = useState<DateRangePreset>("last30days");

  const range = getDateRangeFromPreset(dateRange);
  const { data: summaries, isLoading } = useQuery<any[]>({
    queryKey: [`/api/allegro/fees/summaries?from=${range.from}&to=${range.to}`],
  });

  const formatAmount = (amount: string | number) => {
    const num = typeof amount === 'string' ? parseFloat(amount) : amount;
    return new Intl.NumberFormat('pl-PL', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(num);
  };

  const formatDate = (dateStr: string) => {
    return new Date(dateStr).toLocaleDateString('pl-PL', {
      weekday: 'long',
      year: 'numeric',
      month: 'long',
      day: 'numeric',
    });
  };

  const toggleDay = (date: string) => {
    setExpandedDays(prev => {
      const next = new Set(prev);
      if (next.has(date)) {
        next.delete(date);
      } else {
        next.add(date);
      }
      return next;
    });
  };

  return (
    <div className="p-6 space-y-6">
      <div>
        <h1 className="text-3xl font-bold" data-testid="heading-fees-daily">Szczegółowe dzienne</h1>
        <p className="text-muted-foreground">Podsumowania opłat dla każdego dnia</p>
      </div>

      {/* Date range filter */}
      <Card className="bg-muted/30">
        <CardContent className="pt-6">
          <DateRangeFilter value={dateRange} onChange={setDateRange} />
        </CardContent>
      </Card>

      {isLoading ? (
        <Card>
          <CardContent className="pt-6">
            <div className="space-y-2">
              {[...Array(10)].map((_, i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          </CardContent>
        </Card>
      ) : (
        <div className="space-y-3">
          {summaries?.map((summary: any) => {
            const isExpanded = expandedDays.has(summary.summary_date);
            const hasData = parseFloat(summary.grand_total || '0') > 0;

            return (
              <Card key={summary.id} className={!hasData ? 'opacity-60' : ''}>
                <CardHeader 
                  className="cursor-pointer hover-elevate" 
                  onClick={() => toggleDay(summary.summary_date)}
                >
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      {hasData ? (
                        isExpanded ? <ChevronDown className="h-5 w-5" /> : <ChevronRight className="h-5 w-5" />
                      ) : (
                        <div className="h-5 w-5" />
                      )}
                      <div>
                        <CardTitle className="text-base">{formatDate(summary.summary_date)}</CardTitle>
                        <CardDescription className="mt-1">
                          Łączna kwota: <span className="font-medium">{formatAmount(summary.grand_total)} zł</span>
                        </CardDescription>
                      </div>
                    </div>
                    <div className="text-sm text-muted-foreground">
                      {parseInt(summary.commission_count || '0') + 
                       parseInt(summary.ads_count || '0') + 
                       parseInt(summary.delivery_count || '0') + 
                       parseInt(summary.refunds_count || '0') + 
                       parseInt(summary.other_count || '0')} opłat
                    </div>
                  </div>
                </CardHeader>

                {isExpanded && hasData && (
                  <CardContent>
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>Kategoria</TableHead>
                          <TableHead className="text-right">Liczba</TableHead>
                          <TableHead className="text-right">Suma</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {parseFloat(summary.commission_total || '0') > 0 && (
                          <TableRow>
                            <TableCell className="font-medium">
                              <span className="text-red-600 dark:text-red-400">Prowizje</span>
                            </TableCell>
                            <TableCell className="text-right">{summary.commission_count}</TableCell>
                            <TableCell className="text-right font-medium text-red-600 dark:text-red-400">
                              {formatAmount(summary.commission_total)} zł
                            </TableCell>
                          </TableRow>
                        )}
                        {parseFloat(summary.ads_total || '0') > 0 && (
                          <TableRow>
                            <TableCell className="font-medium">
                              <span className="text-purple-600 dark:text-purple-400">Reklamy</span>
                            </TableCell>
                            <TableCell className="text-right">{summary.ads_count}</TableCell>
                            <TableCell className="text-right font-medium text-purple-600 dark:text-purple-400">
                              {formatAmount(summary.ads_total)} zł
                            </TableCell>
                          </TableRow>
                        )}
                        {parseFloat(summary.delivery_total || '0') > 0 && (
                          <TableRow>
                            <TableCell className="font-medium">
                              <span className="text-blue-600 dark:text-blue-400">Dostawa</span>
                            </TableCell>
                            <TableCell className="text-right">{summary.delivery_count}</TableCell>
                            <TableCell className="text-right font-medium text-blue-600 dark:text-blue-400">
                              {formatAmount(summary.delivery_total)} zł
                            </TableCell>
                          </TableRow>
                        )}
                        {parseFloat(summary.refunds_total || '0') > 0 && (
                          <TableRow>
                            <TableCell className="font-medium">
                              <span className="text-orange-600 dark:text-orange-400">Zwroty</span>
                            </TableCell>
                            <TableCell className="text-right">{summary.refunds_count}</TableCell>
                            <TableCell className="text-right font-medium text-orange-600 dark:text-orange-400">
                              {formatAmount(summary.refunds_total)} zł
                            </TableCell>
                          </TableRow>
                        )}
                        {parseFloat(summary.listing_total || '0') > 0 && (
                          <TableRow>
                            <TableCell className="font-medium">
                              <span className="text-green-600 dark:text-green-400">Wystawienie</span>
                            </TableCell>
                            <TableCell className="text-right">{summary.listing_count}</TableCell>
                            <TableCell className="text-right font-medium text-green-600 dark:text-green-400">
                              {formatAmount(summary.listing_total)} zł
                            </TableCell>
                          </TableRow>
                        )}
                        {parseFloat(summary.other_total || '0') > 0 && (
                          <TableRow>
                            <TableCell className="font-medium text-gray-600 dark:text-gray-400">Inne</TableCell>
                            <TableCell className="text-right">{summary.other_count}</TableCell>
                            <TableCell className="text-right font-medium text-gray-600 dark:text-gray-400">
                              {formatAmount(summary.other_total)} zł
                            </TableCell>
                          </TableRow>
                        )}
                        <TableRow className="font-bold bg-muted/30">
                          <TableCell>RAZEM</TableCell>
                          <TableCell className="text-right">
                            {parseInt(summary.commission_count || '0') + 
                             parseInt(summary.ads_count || '0') + 
                             parseInt(summary.delivery_count || '0') + 
                             parseInt(summary.refunds_count || '0') + 
                             parseInt(summary.other_count || '0')}
                          </TableCell>
                          <TableCell className="text-right">
                            {formatAmount(summary.grand_total)} zł
                          </TableCell>
                        </TableRow>
                      </TableBody>
                    </Table>
                  </CardContent>
                )}
              </Card>
            );
          })}
        </div>
      )}
    </div>
  );
}
