import { useState } from "react";
import { DictionariesSubmenu } from "@/components/dictionaries-submenu";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Button } from "@/components/ui/button";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import { useToast } from "@/hooks/use-toast";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Loader2, Menu } from "lucide-react";

const newDictionaryTypeSchema = z.object({
  type: z.string().min(1, "Typ jest wymagany").regex(/^[a-z_]+$/, "Tylko małe litery i podkreślenia"),
  label: z.string().min(1, "Etykieta jest wymagana"),
  description: z.string().optional(),
});

type NewDictionaryTypeData = z.infer<typeof newDictionaryTypeSchema>;

interface DictionariesLayoutProps {
  children: React.ReactNode;
}

export default function DictionariesLayout({ children }: DictionariesLayoutProps) {
  const { toast } = useToast();
  const [isAddTypeDialogOpen, setIsAddTypeDialogOpen] = useState(false);
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);

  const form = useForm<NewDictionaryTypeData>({
    resolver: zodResolver(newDictionaryTypeSchema),
    defaultValues: {
      type: "",
      label: "",
      description: "",
    },
  });

  const handleAddNewType = () => {
    toast({
      title: "Dodawanie nowych typów słowników",
      description: "Ta funkcjonalność będzie dostępna w przyszłej wersji. Obecnie możesz zarządzać istniejącymi typami słowników.",
    });
    // setIsAddTypeDialogOpen(true);
  };

  const handleSubmitNewType = async (data: NewDictionaryTypeData) => {
    try {
      // This would require backend support to add new dictionary types
      toast({ title: "Nowy typ słownika dodany", description: `Typ: ${data.type}` });
      setIsAddTypeDialogOpen(false);
      form.reset();
    } catch (error) {
      toast({ title: "Błąd dodawania typu", variant: "destructive" });
    }
  };

  return (
    <div className="flex h-full w-full overflow-hidden">
      {/* Left submenu - hidden on mobile, visible on md+ */}
      <div className="hidden md:flex w-64 border-r bg-card flex-shrink-0 overflow-y-auto">
        <DictionariesSubmenu onAddNew={handleAddNewType} />
      </div>

      {/* Main content area */}
      <div className="flex-1 overflow-auto bg-background">
        {/* Mobile menu button */}
        <div className="md:hidden sticky top-0 z-10 bg-background border-b p-2">
          <Sheet open={mobileMenuOpen} onOpenChange={setMobileMenuOpen}>
            <SheetTrigger asChild>
              <Button variant="outline" size="sm" data-testid="button-mobile-dictionaries-menu">
                <Menu className="w-4 h-4 mr-2" />
                Typy słowników
              </Button>
            </SheetTrigger>
            <SheetContent side="left" className="w-64 p-0">
              <DictionariesSubmenu onAddNew={handleAddNewType} />
            </SheetContent>
          </Sheet>
        </div>

        {children}
      </div>

      {/* Add new dictionary type dialog */}
      <Dialog open={isAddTypeDialogOpen} onOpenChange={setIsAddTypeDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Dodaj nowy typ słownika</DialogTitle>
            <DialogDescription>
              Stwórz nowy typ słownika dla swojej organizacji
            </DialogDescription>
          </DialogHeader>

          <Form {...form}>
            <form onSubmit={form.handleSubmit(handleSubmitNewType)} className="space-y-4">
              <FormField
                control={form.control}
                name="type"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Typ *</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. material" data-testid="input-new-type" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="label"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Etykieta *</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. Materiał" data-testid="input-new-label" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} placeholder="np. Materiały wykończeniowe" data-testid="input-new-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex justify-end gap-2">
                <Button 
                  type="button" 
                  variant="outline" 
                  onClick={() => setIsAddTypeDialogOpen(false)}
                  data-testid="button-cancel-new-type"
                >
                  Anuluj
                </Button>
                <Button type="submit" data-testid="button-submit-new-type">
                  Dodaj typ
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
